/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.regionserver;

import java.io.Closeable;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Random;
import java.util.Set;
import java.util.TreeSet;
import java.util.stream.Collectors;
import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.ChoreService;
import org.apache.hadoop.hbase.HBaseTestingUtility;
import org.apache.hadoop.hbase.HConstants;
import org.apache.hadoop.hbase.HRegionLocation;
import org.apache.hadoop.hbase.MetaTableAccessor;
import org.apache.hadoop.hbase.NotServingRegionException;
import org.apache.hadoop.hbase.ScheduledChore;
import org.apache.hadoop.hbase.Stoppable;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.client.Admin;
import org.apache.hadoop.hbase.client.Connection;
import org.apache.hadoop.hbase.client.ConnectionFactory;
import org.apache.hadoop.hbase.client.Get;
import org.apache.hadoop.hbase.client.Put;
import org.apache.hadoop.hbase.client.RegionInfo;
import org.apache.hadoop.hbase.client.RegionLocator;
import org.apache.hadoop.hbase.client.Result;
import org.apache.hadoop.hbase.client.Table;
import org.apache.hadoop.hbase.regionserver.HRegionServer;
import org.apache.hadoop.hbase.regionserver.Store;
import org.apache.hadoop.hbase.shaded.com.google.common.collect.Iterators;
import org.apache.hadoop.hbase.testclassification.LargeTests;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.Pair;
import org.apache.hadoop.hbase.util.PairOfSameType;
import org.apache.hadoop.hbase.util.StoppableImplementation;
import org.apache.hadoop.hbase.util.Threads;
import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.BeforeClass;
import org.junit.Rule;
import org.junit.Test;
import org.junit.experimental.categories.Category;
import org.junit.rules.TestName;

@Category(value={LargeTests.class})
public class TestEndToEndSplitTransaction {
    private static final Log LOG = LogFactory.getLog(TestEndToEndSplitTransaction.class);
    private static final HBaseTestingUtility TEST_UTIL = new HBaseTestingUtility();
    private static final Configuration CONF = TEST_UTIL.getConfiguration();
    @Rule
    public TestName name = new TestName();

    @BeforeClass
    public static void beforeAllTests() throws Exception {
        TEST_UTIL.getConfiguration().setInt("hbase.client.retries.number", 5);
        TEST_UTIL.startMiniCluster();
    }

    @AfterClass
    public static void afterAllTests() throws Exception {
        TEST_UTIL.shutdownMiniCluster();
    }

    @Test
    public void testFromClientSideWhileSplitting() throws Throwable {
        LOG.info((Object)"Starting testFromClientSideWhileSplitting");
        TableName tableName = TableName.valueOf((String)this.name.getMethodName());
        byte[] FAMILY = Bytes.toBytes((String)"family");
        Table table = TEST_UTIL.createTable(tableName, FAMILY);
        StoppableImplementation stopper = new StoppableImplementation();
        RegionSplitter regionSplitter = new RegionSplitter(table);
        RegionChecker regionChecker = new RegionChecker(CONF, stopper, tableName);
        ChoreService choreService = new ChoreService("TEST_SERVER");
        choreService.scheduleChore((ScheduledChore)regionChecker);
        regionSplitter.start();
        regionSplitter.join();
        stopper.stop(null);
        if (regionChecker.ex != null) {
            throw new AssertionError("regionChecker", regionChecker.ex);
        }
        if (regionSplitter.ex != null) {
            throw new AssertionError("regionSplitter", regionSplitter.ex);
        }
        regionChecker.verify();
    }

    public static void log(String msg) {
        LOG.info((Object)msg);
    }

    public static void flushAndBlockUntilDone(Admin admin, HRegionServer rs, byte[] regionName) throws IOException, InterruptedException {
        TestEndToEndSplitTransaction.log("flushing region: " + Bytes.toStringBinary((byte[])regionName));
        admin.flushRegion(regionName);
        TestEndToEndSplitTransaction.log("blocking until flush is complete: " + Bytes.toStringBinary((byte[])regionName));
        Threads.sleepWithoutInterrupt((long)500L);
        while (rs.getOnlineRegion(regionName).getMemStoreSize() > 0L) {
            Threads.sleep((long)50L);
        }
    }

    public static void compactAndBlockUntilDone(Admin admin, HRegionServer rs, byte[] regionName) throws IOException, InterruptedException {
        TestEndToEndSplitTransaction.log("Compacting region: " + Bytes.toStringBinary((byte[])regionName));
        admin.majorCompactRegion(regionName);
        TestEndToEndSplitTransaction.log("blocking until compaction is complete: " + Bytes.toStringBinary((byte[])regionName));
        Threads.sleepWithoutInterrupt((long)500L);
        block0: while (true) {
            for (Store store : rs.getOnlineRegion(regionName).getStores()) {
                if (store.getStorefilesCount() <= 1) continue;
                Threads.sleep((long)50L);
                continue block0;
            }
            break;
        }
    }

    public static void blockUntilRegionSplit(Configuration conf, long timeout, byte[] regionName, boolean waitForDaughters) throws IOException, InterruptedException {
        long start = System.currentTimeMillis();
        TestEndToEndSplitTransaction.log("blocking until region is split:" + Bytes.toStringBinary((byte[])regionName));
        RegionInfo daughterA = null;
        RegionInfo daughterB = null;
        try (Connection conn = ConnectionFactory.createConnection((Configuration)conf);
             Table metaTable = conn.getTable(TableName.META_TABLE_NAME);){
            Result result = null;
            RegionInfo region = null;
            while (System.currentTimeMillis() - start < timeout && (result = metaTable.get(new Get(regionName))) != null) {
                region = MetaTableAccessor.getRegionInfo((Result)result);
                if (region.isSplitParent()) {
                    TestEndToEndSplitTransaction.log("found parent region: " + region.toString());
                    PairOfSameType pair = MetaTableAccessor.getDaughterRegions((Result)result);
                    daughterA = (RegionInfo)pair.getFirst();
                    daughterB = (RegionInfo)pair.getSecond();
                    break;
                }
                Threads.sleep((long)100L);
            }
            if (daughterA == null || daughterB == null) {
                throw new IOException("Failed to get daughters, daughterA=" + daughterA + ", daughterB=" + daughterB + ", timeout=" + timeout + ", result=" + result + ", regionName=" + regionName + ", region=" + region);
            }
            if (waitForDaughters) {
                long rem = timeout - (System.currentTimeMillis() - start);
                TestEndToEndSplitTransaction.blockUntilRegionIsInMeta(conn, rem, daughterA);
                rem = timeout - (System.currentTimeMillis() - start);
                TestEndToEndSplitTransaction.blockUntilRegionIsInMeta(conn, rem, daughterB);
                rem = timeout - (System.currentTimeMillis() - start);
                TestEndToEndSplitTransaction.blockUntilRegionIsOpened(conf, rem, daughterA);
                rem = timeout - (System.currentTimeMillis() - start);
                TestEndToEndSplitTransaction.blockUntilRegionIsOpened(conf, rem, daughterB);
            }
        }
    }

    public static void blockUntilRegionIsInMeta(Connection conn, long timeout, RegionInfo hri) throws IOException, InterruptedException {
        TestEndToEndSplitTransaction.log("blocking until region is in META: " + hri.getRegionNameAsString());
        long start = System.currentTimeMillis();
        while (System.currentTimeMillis() - start < timeout) {
            HRegionLocation loc = MetaTableAccessor.getRegionLocation((Connection)conn, (RegionInfo)hri);
            if (loc != null && !loc.getRegionInfo().isOffline()) {
                TestEndToEndSplitTransaction.log("found region in META: " + hri.getRegionNameAsString());
                break;
            }
            Threads.sleep((long)10L);
        }
    }

    public static void blockUntilRegionIsOpened(Configuration conf, long timeout, RegionInfo hri) throws IOException, InterruptedException {
        TestEndToEndSplitTransaction.log("blocking until region is opened for reading:" + hri.getRegionNameAsString());
        long start = System.currentTimeMillis();
        try (Connection conn = ConnectionFactory.createConnection((Configuration)conf);
             Table table = conn.getTable(hri.getTable());){
            byte[] row = hri.getStartKey();
            if (row == null || row.length <= 0) {
                row = new byte[]{48};
            }
            Get get = new Get(row);
            while (System.currentTimeMillis() - start < timeout) {
                try {
                    table.get(get);
                    break;
                }
                catch (IOException iOException) {
                    Threads.sleep((long)10L);
                }
            }
        }
    }

    static class RegionChecker
    extends ScheduledChore {
        Connection connection;
        Configuration conf;
        TableName tableName;
        Throwable ex;

        RegionChecker(Configuration conf, Stoppable stopper, TableName tableName) throws IOException {
            super("RegionChecker", stopper, 10);
            this.conf = conf;
            this.tableName = tableName;
            this.connection = ConnectionFactory.createConnection((Configuration)conf);
        }

        void verifyRegionsUsingMetaTableAccessor() throws Exception {
            List regionList = MetaTableAccessor.getTableRegions((Connection)this.connection, (TableName)this.tableName, (boolean)true);
            this.verifyTableRegions(regionList.stream().collect(Collectors.toCollection(() -> new TreeSet(RegionInfo.COMPARATOR))));
            regionList = MetaTableAccessor.getAllRegions((Connection)this.connection, (boolean)true);
            this.verifyTableRegions(regionList.stream().collect(Collectors.toCollection(() -> new TreeSet(RegionInfo.COMPARATOR))));
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        void verifyRegionsUsingHTable() throws IOException {
            Table table = null;
            try {
                table = this.connection.getTable(this.tableName);
                try (RegionLocator rl = this.connection.getRegionLocator(this.tableName);){
                    Pair keys = rl.getStartEndKeys();
                    this.verifyStartEndKeys((Pair<byte[][], byte[][]>)keys);
                    TreeSet<RegionInfo> regions = new TreeSet<RegionInfo>(RegionInfo.COMPARATOR);
                    for (HRegionLocation loc : rl.getAllRegionLocations()) {
                        regions.add((RegionInfo)loc.getRegionInfo());
                    }
                    this.verifyTableRegions(regions);
                }
            }
            catch (Throwable throwable) {
                IOUtils.closeQuietly(table);
                throw throwable;
            }
            IOUtils.closeQuietly((Closeable)table);
        }

        void verify() throws Exception {
            this.verifyRegionsUsingMetaTableAccessor();
            this.verifyRegionsUsingHTable();
        }

        void verifyTableRegions(Set<RegionInfo> regions) {
            TestEndToEndSplitTransaction.log("Verifying " + regions.size() + " regions: " + regions);
            byte[][] startKeys = new byte[regions.size()][];
            byte[][] endKeys = new byte[regions.size()][];
            int i = 0;
            for (RegionInfo region : regions) {
                startKeys[i] = region.getStartKey();
                endKeys[i] = region.getEndKey();
                ++i;
            }
            Pair keys = new Pair((Object)startKeys, (Object)endKeys);
            this.verifyStartEndKeys((Pair<byte[][], byte[][]>)keys);
        }

        void verifyStartEndKeys(Pair<byte[][], byte[][]> keys) {
            byte[][] startKeys = (byte[][])keys.getFirst();
            byte[][] endKeys = (byte[][])keys.getSecond();
            Assert.assertEquals((long)startKeys.length, (long)endKeys.length);
            Assert.assertTrue((String)"Found 0 regions for the table", (startKeys.length > 0 ? 1 : 0) != 0);
            Assert.assertArrayEquals((String)"Start key for the first region is not byte[0]", (byte[])HConstants.EMPTY_START_ROW, (byte[])startKeys[0]);
            byte[] prevEndKey = HConstants.EMPTY_START_ROW;
            for (int i = 0; i < startKeys.length; ++i) {
                Assert.assertArrayEquals((String)("Hole in hbase:meta is detected. prevEndKey=" + Bytes.toStringBinary((byte[])prevEndKey) + " ,regionStartKey=" + Bytes.toStringBinary((byte[])startKeys[i])), (byte[])prevEndKey, (byte[])startKeys[i]);
                prevEndKey = endKeys[i];
            }
            Assert.assertArrayEquals((String)"End key for the last region is not byte[0]", (byte[])HConstants.EMPTY_END_ROW, (byte[])endKeys[endKeys.length - 1]);
        }

        protected void chore() {
            try {
                this.verify();
            }
            catch (Throwable ex) {
                this.ex = ex;
                this.getStopper().stop("caught exception");
            }
        }
    }

    static class RegionSplitter
    extends Thread {
        final Connection connection;
        Throwable ex;
        Table table;
        TableName tableName;
        byte[] family;
        Admin admin;
        HRegionServer rs;

        RegionSplitter(Table table) throws IOException {
            this.table = table;
            this.tableName = table.getName();
            this.family = (byte[])table.getTableDescriptor().getFamiliesKeys().iterator().next();
            this.admin = TEST_UTIL.getAdmin();
            this.rs = TEST_UTIL.getMiniHBaseCluster().getRegionServer(0);
            this.connection = TEST_UTIL.getConnection();
        }

        @Override
        public void run() {
            try {
                Random random = new Random();
                for (int i = 0; i < 5; ++i) {
                    List regions = MetaTableAccessor.getTableRegions((Connection)this.connection, (TableName)this.tableName, (boolean)true);
                    if (regions.isEmpty()) continue;
                    int regionIndex = random.nextInt(regions.size());
                    RegionInfo region = (RegionInfo)Iterators.get(regions.iterator(), (int)regionIndex);
                    int start = 0;
                    int end = Integer.MAX_VALUE;
                    if (region.getStartKey().length > 0) {
                        start = Bytes.toInt((byte[])region.getStartKey());
                    }
                    if (region.getEndKey().length > 0) {
                        end = Bytes.toInt((byte[])region.getEndKey());
                    }
                    int mid = start + (end - start) / 2;
                    byte[] splitPoint = Bytes.toBytes((int)mid);
                    this.addData(start);
                    this.addData(mid);
                    TestEndToEndSplitTransaction.flushAndBlockUntilDone(this.admin, this.rs, region.getRegionName());
                    TestEndToEndSplitTransaction.compactAndBlockUntilDone(this.admin, this.rs, region.getRegionName());
                    TestEndToEndSplitTransaction.log("Initiating region split for:" + region.getRegionNameAsString());
                    try {
                        this.admin.splitRegion(region.getRegionName(), splitPoint);
                        TestEndToEndSplitTransaction.blockUntilRegionSplit(CONF, 50000L, region.getRegionName(), true);
                        continue;
                    }
                    catch (NotServingRegionException notServingRegionException) {
                        // empty catch block
                    }
                }
            }
            catch (Throwable ex) {
                this.ex = ex;
            }
        }

        void addData(int start) throws IOException {
            ArrayList<Put> puts = new ArrayList<Put>();
            for (int i = start; i < start + 100; ++i) {
                Put put = new Put(Bytes.toBytes((int)i));
                put.addColumn(this.family, this.family, Bytes.toBytes((int)i));
                puts.add(put);
            }
            this.table.put(puts);
        }
    }
}

