/*
 * Decompiled with CFR 0.152.
 */
package org.apache.james.mailbox;

import com.google.common.base.Strings;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.util.Iterator;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CountDownLatch;
import java.util.stream.Stream;
import javax.mail.Flags;
import javax.mail.internet.SharedInputStream;
import org.apache.james.core.Username;
import org.apache.james.core.quota.QuotaCountLimit;
import org.apache.james.core.quota.QuotaCountUsage;
import org.apache.james.core.quota.QuotaLimitValue;
import org.apache.james.core.quota.QuotaSizeLimit;
import org.apache.james.core.quota.QuotaSizeUsage;
import org.apache.james.core.quota.QuotaUsageValue;
import org.apache.james.events.EventBus;
import org.apache.james.events.EventListener;
import org.apache.james.events.RegistrationKey;
import org.apache.james.mailbox.MailboxManager;
import org.apache.james.mailbox.MailboxSession;
import org.apache.james.mailbox.MessageManager;
import org.apache.james.mailbox.MessageUid;
import org.apache.james.mailbox.MetadataWithMailboxId;
import org.apache.james.mailbox.ModSeq;
import org.apache.james.mailbox.SubscriptionManager;
import org.apache.james.mailbox.events.MailboxEvents;
import org.apache.james.mailbox.events.MailboxIdRegistrationKey;
import org.apache.james.mailbox.events.MessageMoveEvent;
import org.apache.james.mailbox.exception.AnnotationException;
import org.apache.james.mailbox.exception.HasEmptyMailboxNameInHierarchyException;
import org.apache.james.mailbox.exception.InboxAlreadyCreated;
import org.apache.james.mailbox.exception.InsufficientRightsException;
import org.apache.james.mailbox.exception.MailboxException;
import org.apache.james.mailbox.exception.MailboxExistsException;
import org.apache.james.mailbox.exception.MailboxNotFoundException;
import org.apache.james.mailbox.exception.TooLongMailboxNameException;
import org.apache.james.mailbox.extension.PreDeletionHook;
import org.apache.james.mailbox.mock.DataProvisioner;
import org.apache.james.mailbox.model.ComposedMessageId;
import org.apache.james.mailbox.model.FetchGroup;
import org.apache.james.mailbox.model.MailboxACL;
import org.apache.james.mailbox.model.MailboxAnnotation;
import org.apache.james.mailbox.model.MailboxAnnotationKey;
import org.apache.james.mailbox.model.MailboxCounters;
import org.apache.james.mailbox.model.MailboxId;
import org.apache.james.mailbox.model.MailboxMetaData;
import org.apache.james.mailbox.model.MailboxPath;
import org.apache.james.mailbox.model.MessageId;
import org.apache.james.mailbox.model.MessageRange;
import org.apache.james.mailbox.model.MessageResult;
import org.apache.james.mailbox.model.MultimailboxesSearchQuery;
import org.apache.james.mailbox.model.Quota;
import org.apache.james.mailbox.model.QuotaRoot;
import org.apache.james.mailbox.model.SearchQuery;
import org.apache.james.mailbox.model.search.MailboxQuery;
import org.apache.james.mailbox.util.EventCollector;
import org.apache.james.mime4j.dom.Message;
import org.apache.james.util.ClassLoaderUtils;
import org.apache.james.util.concurrency.ConcurrentTestRunner;
import org.assertj.core.api.AbstractListAssert;
import org.assertj.core.api.Assertions;
import org.assertj.core.api.IterableAssert;
import org.assertj.core.api.ListAssert;
import org.assertj.core.api.ObjectAssert;
import org.assertj.core.api.SoftAssertions;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Assumptions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Nested;
import org.junit.jupiter.api.Test;
import org.mockito.ArgumentCaptor;
import org.mockito.ArgumentMatchers;
import org.mockito.Mockito;
import org.mockito.verification.VerificationMode;
import org.reactivestreams.Publisher;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;

public abstract class MailboxManagerTest<T extends MailboxManager> {
    public static final Username USER_1 = Username.of((String)"USER_1");
    public static final Username USER_2 = Username.of((String)"USER_2");
    private static final int DEFAULT_MAXIMUM_LIMIT = 256;
    protected T mailboxManager;
    private SubscriptionManager subscriptionManager;
    private MailboxSession session;
    protected Message.Builder message;
    private PreDeletionHook preDeletionHook1;
    private PreDeletionHook preDeletionHook2;

    protected abstract T provideMailboxManager();

    protected abstract SubscriptionManager provideSubscriptionManager();

    protected abstract EventBus retrieveEventBus(T var1);

    protected Set<PreDeletionHook> preDeletionHooks() {
        return ImmutableSet.of((Object)this.preDeletionHook1, (Object)this.preDeletionHook2);
    }

    @BeforeEach
    void setUp() throws Exception {
        this.setupMockForPreDeletionHooks();
        this.mailboxManager = this.provideMailboxManager();
        this.subscriptionManager = this.provideSubscriptionManager();
        this.message = Message.Builder.of().setSubject("test").setBody("testmail", StandardCharsets.UTF_8);
    }

    private void setupMockForPreDeletionHooks() {
        this.preDeletionHook1 = (PreDeletionHook)Mockito.mock(PreDeletionHook.class);
        Mockito.when((Object)this.preDeletionHook1.notifyDelete((PreDeletionHook.DeleteOperation)ArgumentMatchers.any(PreDeletionHook.DeleteOperation.class))).thenReturn((Object)Mono.empty());
        this.preDeletionHook2 = (PreDeletionHook)Mockito.mock(PreDeletionHook.class);
        Mockito.when((Object)this.preDeletionHook2.notifyDelete((PreDeletionHook.DeleteOperation)ArgumentMatchers.any(PreDeletionHook.DeleteOperation.class))).thenReturn((Object)Mono.empty());
    }

    @AfterEach
    void tearDown() {
        this.mailboxManager.logout(this.session);
        this.mailboxManager.endProcessingRequest(this.session);
    }

    @Test
    protected void creatingConcurrentlyMailboxesWithSameParentShouldNotFail() throws Exception {
        MailboxSession session = this.mailboxManager.createSystemSession(USER_1);
        String mailboxName = "a.b.c.d.e.f.g.h.i.j.k.l.m.n.o.p.q.r.s.t.u.v.w.x.y.z";
        ConcurrentTestRunner.builder().operation((a, b) -> this.mailboxManager.createMailbox(MailboxPath.forUser((Username)USER_1, (String)(mailboxName + a)), session)).threadCount(10).runSuccessfullyWithin(Duration.ofMinutes(1L));
    }

    @Test
    void createMailboxShouldReturnRightId() throws Exception {
        this.session = this.mailboxManager.createSystemSession(USER_1);
        this.mailboxManager.startProcessingRequest(this.session);
        MailboxPath mailboxPath = MailboxPath.forUser((Username)USER_1, (String)"name.subfolder");
        Optional mailboxId = this.mailboxManager.createMailbox(mailboxPath, this.session);
        MessageManager retrievedMailbox = this.mailboxManager.getMailbox(mailboxPath, this.session);
        Assertions.assertThat((boolean)mailboxId.isPresent()).isTrue();
        Assertions.assertThat((Object)((MailboxId)mailboxId.get())).isEqualTo((Object)retrievedMailbox.getId());
    }

    @Nested
    class RightTests {
        private MailboxSession session2;

        RightTests() {
        }

        @BeforeEach
        void setUp() {
            Assumptions.assumeTrue((boolean)MailboxManagerTest.this.mailboxManager.hasCapability(MailboxManager.MailboxCapabilities.ACL));
            MailboxManagerTest.this.session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            this.session2 = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_2);
        }

        @Test
        void hasRightShouldThrowOnUnknownMailbox() {
            Assertions.assertThatThrownBy(() -> MailboxManagerTest.this.mailboxManager.hasRight(MailboxPath.forUser((Username)USER_1, (String)"notFound"), MailboxACL.Right.Administer, MailboxManagerTest.this.session)).isInstanceOf(MailboxNotFoundException.class);
        }

        @Test
        void listRightsShouldThrowOnUnknownMailbox() {
            Assertions.assertThatThrownBy(() -> MailboxManagerTest.this.mailboxManager.listRights(MailboxPath.forUser((Username)USER_1, (String)"notFound"), MailboxManagerTest.this.session)).isInstanceOf(MailboxNotFoundException.class);
        }

        @Test
        void myRightsShouldThrowOnUnknownMailbox() {
            Assertions.assertThatThrownBy(() -> MailboxManagerTest.this.mailboxManager.myRights(MailboxPath.forUser((Username)USER_1, (String)"notFound"), MailboxManagerTest.this.session)).isInstanceOf(MailboxNotFoundException.class);
        }

        @Test
        void listRightsForEntryShouldThrowOnUnknownMailbox() {
            Assertions.assertThatThrownBy(() -> MailboxManagerTest.this.mailboxManager.listRights(MailboxPath.forUser((Username)USER_1, (String)"notFound"), MailboxACL.EntryKey.createUserEntryKey((Username)USER_2), MailboxManagerTest.this.session)).isInstanceOf(MailboxNotFoundException.class);
        }

        @Test
        void setRightsShouldNotThrowOnUnknownMailbox() {
            Assertions.assertThatCode(() -> MailboxManagerTest.this.mailboxManager.setRights(MailboxPath.forUser((Username)USER_1, (String)"notFound"), MailboxACL.EMPTY, MailboxManagerTest.this.session)).doesNotThrowAnyException();
        }

        @Test
        void hasRightShouldThrowOnDeletedMailbox() throws Exception {
            MailboxId id = (MailboxId)MailboxManagerTest.this.mailboxManager.createMailbox(MailboxPath.forUser((Username)USER_1, (String)"deleted"), MailboxManagerTest.this.session).get();
            MailboxManagerTest.this.mailboxManager.deleteMailbox(id, MailboxManagerTest.this.session);
            Assertions.assertThatThrownBy(() -> MailboxManagerTest.this.mailboxManager.hasRight(id, MailboxACL.Right.Administer, MailboxManagerTest.this.session)).isInstanceOf(MailboxNotFoundException.class);
        }

        @Test
        void myRightsShouldThrowOnDeletedMailbox() throws Exception {
            MailboxId id = (MailboxId)MailboxManagerTest.this.mailboxManager.createMailbox(MailboxPath.forUser((Username)USER_1, (String)"deleted"), MailboxManagerTest.this.session).get();
            MailboxManagerTest.this.mailboxManager.deleteMailbox(id, MailboxManagerTest.this.session);
            Assertions.assertThatThrownBy(() -> Mono.from((Publisher)MailboxManagerTest.this.mailboxManager.myRights(id, MailboxManagerTest.this.session)).blockOptional()).hasCauseInstanceOf(MailboxNotFoundException.class);
        }

        @Test
        void setRightsShouldThrowOnDeletedMailbox() throws Exception {
            MailboxId id = (MailboxId)MailboxManagerTest.this.mailboxManager.createMailbox(MailboxPath.forUser((Username)USER_1, (String)"deleted"), MailboxManagerTest.this.session).get();
            MailboxManagerTest.this.mailboxManager.deleteMailbox(id, MailboxManagerTest.this.session);
            Assertions.assertThatThrownBy(() -> MailboxManagerTest.this.mailboxManager.setRights(id, MailboxACL.EMPTY, MailboxManagerTest.this.session)).isInstanceOf(MailboxNotFoundException.class);
        }

        @Test
        void setRightsByIdShouldThrowWhenNotOwner() throws Exception {
            MailboxId id = (MailboxId)MailboxManagerTest.this.mailboxManager.createMailbox(MailboxPath.forUser((Username)USER_2, (String)"mailbox"), this.session2).get();
            MailboxManagerTest.this.mailboxManager.setRights(id, MailboxACL.EMPTY.apply(MailboxACL.command().key(MailboxACL.EntryKey.createUserEntryKey((Username)USER_1)).rights(new MailboxACL.Rfc4314Rights(new MailboxACL.Right[]{MailboxACL.Right.Lookup})).asAddition()), this.session2);
            Assertions.assertThatThrownBy(() -> MailboxManagerTest.this.mailboxManager.setRights(id, MailboxACL.EMPTY.apply(MailboxACL.command().key(MailboxACL.EntryKey.createUserEntryKey((Username)USER_1)).rights(MailboxACL.FULL_RIGHTS).asAddition()), MailboxManagerTest.this.session)).isInstanceOf(InsufficientRightsException.class);
        }

        @Test
        void setRightsByPathShouldThrowWhenNotOwner() throws Exception {
            MailboxPath mailboxPath = MailboxPath.forUser((Username)USER_2, (String)"mailbox");
            MailboxManagerTest.this.mailboxManager.createMailbox(mailboxPath, this.session2).get();
            MailboxManagerTest.this.mailboxManager.setRights(mailboxPath, MailboxACL.EMPTY.apply(MailboxACL.command().key(MailboxACL.EntryKey.createUserEntryKey((Username)USER_1)).rights(new MailboxACL.Rfc4314Rights(new MailboxACL.Right[]{MailboxACL.Right.Lookup})).asAddition()), this.session2);
            Assertions.assertThatThrownBy(() -> MailboxManagerTest.this.mailboxManager.setRights(mailboxPath, MailboxACL.EMPTY.apply(MailboxACL.command().key(MailboxACL.EntryKey.createUserEntryKey((Username)USER_1)).rights(MailboxACL.FULL_RIGHTS).asAddition()), MailboxManagerTest.this.session)).isInstanceOf(InsufficientRightsException.class);
        }

        @Test
        void applyRightsCommandShouldThrowWhenNotOwner() throws Exception {
            MailboxPath mailboxPath = MailboxPath.forUser((Username)USER_2, (String)"mailbox");
            MailboxManagerTest.this.mailboxManager.createMailbox(mailboxPath, this.session2).get();
            MailboxManagerTest.this.mailboxManager.setRights(mailboxPath, MailboxACL.EMPTY.apply(MailboxACL.command().key(MailboxACL.EntryKey.createUserEntryKey((Username)USER_1)).rights(new MailboxACL.Rfc4314Rights(new MailboxACL.Right[]{MailboxACL.Right.Lookup})).asAddition()), this.session2);
            Assertions.assertThatThrownBy(() -> MailboxManagerTest.this.mailboxManager.applyRightsCommand(mailboxPath, MailboxACL.command().key(MailboxACL.EntryKey.createUserEntryKey((Username)USER_1)).rights(MailboxACL.FULL_RIGHTS).asAddition(), MailboxManagerTest.this.session)).isInstanceOf(InsufficientRightsException.class);
        }

        @Test
        void applyRightsCommandByIdShouldThrowWhenNotOwner() throws Exception {
            MailboxPath mailboxPath = MailboxPath.forUser((Username)USER_2, (String)"mailbox");
            MailboxId mailboxId = (MailboxId)MailboxManagerTest.this.mailboxManager.createMailbox(mailboxPath, this.session2).get();
            MailboxManagerTest.this.mailboxManager.setRights(mailboxPath, MailboxACL.EMPTY.apply(MailboxACL.command().key(MailboxACL.EntryKey.createUserEntryKey((Username)USER_1)).rights(new MailboxACL.Rfc4314Rights(new MailboxACL.Right[]{MailboxACL.Right.Lookup})).asAddition()), this.session2);
            Assertions.assertThatThrownBy(() -> MailboxManagerTest.this.mailboxManager.applyRightsCommand(mailboxId, MailboxACL.command().key(MailboxACL.EntryKey.createUserEntryKey((Username)USER_1)).rights(MailboxACL.FULL_RIGHTS).asAddition(), MailboxManagerTest.this.session)).isInstanceOf(InsufficientRightsException.class);
        }

        @Test
        void setRightsByIdShouldThrowWhenNoRights() throws Exception {
            MailboxPath mailboxPath = MailboxPath.forUser((Username)USER_2, (String)"mailbox");
            MailboxId mailboxId = (MailboxId)MailboxManagerTest.this.mailboxManager.createMailbox(mailboxPath, this.session2).get();
            Assertions.assertThatThrownBy(() -> MailboxManagerTest.this.mailboxManager.setRights(mailboxId, MailboxACL.EMPTY.apply(MailboxACL.command().key(MailboxACL.EntryKey.createUserEntryKey((Username)USER_1)).rights(MailboxACL.FULL_RIGHTS).asAddition()), MailboxManagerTest.this.session)).isInstanceOf(MailboxNotFoundException.class);
        }

        @Test
        void setRightsByPathShouldThrowWhenNoRights() throws Exception {
            MailboxPath mailboxPath = MailboxPath.forUser((Username)USER_2, (String)"mailbox");
            MailboxManagerTest.this.mailboxManager.createMailbox(mailboxPath, this.session2).get();
            Assertions.assertThatThrownBy(() -> MailboxManagerTest.this.mailboxManager.setRights(mailboxPath, MailboxACL.EMPTY.apply(MailboxACL.command().key(MailboxACL.EntryKey.createUserEntryKey((Username)USER_1)).rights(MailboxACL.FULL_RIGHTS).asAddition()), MailboxManagerTest.this.session)).isInstanceOf(MailboxNotFoundException.class);
        }

        @Test
        void applyRightsCommandShouldThrowWhenNoRights() throws Exception {
            MailboxPath mailboxPath = MailboxPath.forUser((Username)USER_2, (String)"mailbox");
            MailboxManagerTest.this.mailboxManager.createMailbox(mailboxPath, this.session2).get();
            Assertions.assertThatThrownBy(() -> MailboxManagerTest.this.mailboxManager.applyRightsCommand(mailboxPath, MailboxACL.command().key(MailboxACL.EntryKey.createUserEntryKey((Username)USER_1)).rights(MailboxACL.FULL_RIGHTS).asAddition(), MailboxManagerTest.this.session)).isInstanceOf(MailboxNotFoundException.class);
        }

        @Test
        void applyRightsCommandByIdShouldThrowWhenNoRights() throws Exception {
            MailboxPath mailboxPath = MailboxPath.forUser((Username)USER_2, (String)"mailbox");
            MailboxId mailboxId = (MailboxId)MailboxManagerTest.this.mailboxManager.createMailbox(mailboxPath, this.session2).get();
            Assertions.assertThatThrownBy(() -> MailboxManagerTest.this.mailboxManager.applyRightsCommand(mailboxId, MailboxACL.command().key(MailboxACL.EntryKey.createUserEntryKey((Username)USER_1)).rights(MailboxACL.FULL_RIGHTS).asAddition(), MailboxManagerTest.this.session)).isInstanceOf(MailboxNotFoundException.class);
        }
    }

    @Nested
    class MessageTests {
        private MessageManager inboxManager;

        MessageTests() {
        }

        @BeforeEach
        void setUp() throws Exception {
            MailboxManagerTest.this.session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxPath inbox = MailboxPath.inbox((MailboxSession)MailboxManagerTest.this.session);
            MailboxManagerTest.this.mailboxManager.createMailbox(inbox, MailboxManagerTest.this.session).get();
            this.inboxManager = MailboxManagerTest.this.mailboxManager.getMailbox(inbox, MailboxManagerTest.this.session);
        }

        @Test
        void listMessagesMetadataShouldReturnEmptyWhenNoMessages() {
            Assertions.assertThat((List)((List)Flux.from((Publisher)this.inboxManager.listMessagesMetadata(MessageRange.all(), MailboxManagerTest.this.session)).collectList().block())).isEmpty();
        }

        @Test
        void listMessagesMetadataShouldReturnAppendedMessage() throws Exception {
            Flags flags = new Flags(Flags.Flag.DELETED);
            ComposedMessageId composeId = this.inboxManager.appendMessage(MessageManager.AppendCommand.builder().withFlags(flags).build((SharedInputStream)ClassLoaderUtils.getSystemResourceAsSharedStream((String)"eml/twoAttachmentsApi.eml")), MailboxManagerTest.this.session).getId();
            ((ListAssert)Assertions.assertThat((List)((List)Flux.from((Publisher)this.inboxManager.listMessagesMetadata(MessageRange.all(), MailboxManagerTest.this.session)).collectList().block())).hasSize(1)).allSatisfy(ids -> SoftAssertions.assertSoftly(softly -> {
                softly.assertThat((Object)ids.getComposedMessageId().getMailboxId()).isEqualTo((Object)composeId.getMailboxId());
                softly.assertThat((Comparable)ids.getComposedMessageId().getUid()).isEqualTo((Object)composeId.getUid());
                softly.assertThat((Object)ids.getFlags()).isEqualTo((Object)flags);
            }));
        }

        @Test
        void listMessagesMetadataShouldReturnUpdatedMessage() throws Exception {
            Flags flags = new Flags(Flags.Flag.SEEN);
            ComposedMessageId composeId = this.inboxManager.appendMessage(MessageManager.AppendCommand.builder().withFlags(new Flags(Flags.Flag.DELETED)).build((SharedInputStream)ClassLoaderUtils.getSystemResourceAsSharedStream((String)"eml/twoAttachmentsApi.eml")), MailboxManagerTest.this.session).getId();
            this.inboxManager.setFlags(flags, MessageManager.FlagsUpdateMode.REPLACE, MessageRange.all(), MailboxManagerTest.this.session);
            ((ListAssert)Assertions.assertThat((List)((List)Flux.from((Publisher)this.inboxManager.listMessagesMetadata(MessageRange.all(), MailboxManagerTest.this.session)).collectList().block())).hasSize(1)).allSatisfy(ids -> SoftAssertions.assertSoftly(softly -> {
                softly.assertThat((Object)ids.getComposedMessageId().getMailboxId()).isEqualTo((Object)composeId.getMailboxId());
                softly.assertThat((Comparable)ids.getComposedMessageId().getUid()).isEqualTo((Object)composeId.getUid());
                softly.assertThat((Object)ids.getFlags()).isEqualTo((Object)flags);
            }));
        }

        @Test
        void listMessagesMetadataShouldNotReturnDeletedMessage() throws Exception {
            this.inboxManager.appendMessage(MessageManager.AppendCommand.builder().withFlags(new Flags(Flags.Flag.DELETED)).build((SharedInputStream)ClassLoaderUtils.getSystemResourceAsSharedStream((String)"eml/twoAttachmentsApi.eml")), MailboxManagerTest.this.session).getId();
            this.inboxManager.expunge(MessageRange.all(), MailboxManagerTest.this.session);
            Assertions.assertThat((List)((List)Flux.from((Publisher)this.inboxManager.listMessagesMetadata(MessageRange.all(), MailboxManagerTest.this.session)).collectList().block())).isEmpty();
        }
    }

    @Nested
    class HookTests {
        HookTests() {
        }

        @Nested
        class PreDeletion {
            private MailboxPath inbox;
            private MailboxId inboxId;
            private MailboxId anotherMailboxId;
            private MessageManager inboxManager;
            private MessageManager anotherMailboxManager;

            PreDeletion() {
            }

            @BeforeEach
            void setUp() throws Exception {
                MailboxManagerTest.this.session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
                this.inbox = MailboxPath.inbox((MailboxSession)MailboxManagerTest.this.session);
                MailboxPath anotherMailboxPath = MailboxPath.forUser((Username)USER_1, (String)"anotherMailbox");
                this.anotherMailboxId = (MailboxId)MailboxManagerTest.this.mailboxManager.createMailbox(anotherMailboxPath, MailboxManagerTest.this.session).get();
                this.inboxId = (MailboxId)MailboxManagerTest.this.mailboxManager.createMailbox(this.inbox, MailboxManagerTest.this.session).get();
                this.inboxManager = MailboxManagerTest.this.mailboxManager.getMailbox(this.inbox, MailboxManagerTest.this.session);
                this.anotherMailboxManager = MailboxManagerTest.this.mailboxManager.getMailbox(anotherMailboxPath, MailboxManagerTest.this.session);
            }

            @Test
            void expungeShouldCallAllPreDeletionHooks() throws Exception {
                ComposedMessageId composeId = this.inboxManager.appendMessage(MessageManager.AppendCommand.builder().withFlags(new Flags(Flags.Flag.DELETED)).build(MailboxManagerTest.this.message), MailboxManagerTest.this.session).getId();
                this.inboxManager.expunge(MessageRange.one((MessageUid)composeId.getUid()), MailboxManagerTest.this.session);
                ArgumentCaptor preDeleteCaptor1 = ArgumentCaptor.forClass(PreDeletionHook.DeleteOperation.class);
                ArgumentCaptor preDeleteCaptor2 = ArgumentCaptor.forClass(PreDeletionHook.DeleteOperation.class);
                ((PreDeletionHook)Mockito.verify((Object)MailboxManagerTest.this.preDeletionHook1, (VerificationMode)Mockito.times((int)1))).notifyDelete((PreDeletionHook.DeleteOperation)preDeleteCaptor1.capture());
                ((PreDeletionHook)Mockito.verify((Object)MailboxManagerTest.this.preDeletionHook2, (VerificationMode)Mockito.times((int)1))).notifyDelete((PreDeletionHook.DeleteOperation)preDeleteCaptor2.capture());
                ((ListAssert)((ListAssert)Assertions.assertThat((List)((PreDeletionHook.DeleteOperation)preDeleteCaptor1.getValue()).getDeletionMetadataList()).hasSize(1)).hasSameElementsAs((Iterable)((PreDeletionHook.DeleteOperation)preDeleteCaptor2.getValue()).getDeletionMetadataList())).allSatisfy(deleteMetadata -> SoftAssertions.assertSoftly(softy -> {
                    softy.assertThat((Object)deleteMetadata.getMailboxId()).isEqualTo((Object)this.inboxId);
                    softy.assertThat((Object)deleteMetadata.getMessageMetaData().getMessageId()).isEqualTo((Object)composeId.getMessageId());
                }));
            }

            @Test
            void deleteMailboxShouldCallAllPreDeletionHooks() throws Exception {
                ComposedMessageId composeId = this.inboxManager.appendMessage(MessageManager.AppendCommand.builder().withFlags(new Flags(Flags.Flag.DELETED)).build(MailboxManagerTest.this.message), MailboxManagerTest.this.session).getId();
                MailboxManagerTest.this.mailboxManager.deleteMailbox(this.inbox, MailboxManagerTest.this.session);
                ArgumentCaptor preDeleteCaptor1 = ArgumentCaptor.forClass(PreDeletionHook.DeleteOperation.class);
                ArgumentCaptor preDeleteCaptor2 = ArgumentCaptor.forClass(PreDeletionHook.DeleteOperation.class);
                ((PreDeletionHook)Mockito.verify((Object)MailboxManagerTest.this.preDeletionHook1, (VerificationMode)Mockito.times((int)1))).notifyDelete((PreDeletionHook.DeleteOperation)preDeleteCaptor1.capture());
                ((PreDeletionHook)Mockito.verify((Object)MailboxManagerTest.this.preDeletionHook2, (VerificationMode)Mockito.times((int)1))).notifyDelete((PreDeletionHook.DeleteOperation)preDeleteCaptor2.capture());
                ((ListAssert)((ListAssert)Assertions.assertThat((List)((PreDeletionHook.DeleteOperation)preDeleteCaptor1.getValue()).getDeletionMetadataList()).hasSize(1)).hasSameElementsAs((Iterable)((PreDeletionHook.DeleteOperation)preDeleteCaptor2.getValue()).getDeletionMetadataList())).allSatisfy(deleteMetadata -> SoftAssertions.assertSoftly(softy -> {
                    softy.assertThat((Object)deleteMetadata.getMailboxId()).isEqualTo((Object)this.inboxId);
                    softy.assertThat((Object)deleteMetadata.getMessageMetaData().getMessageId()).isEqualTo((Object)composeId.getMessageId());
                }));
            }

            @Test
            void deleteMailboxByIdShouldCallAllPreDeletionHooks() throws Exception {
                ComposedMessageId composeId = this.inboxManager.appendMessage(MessageManager.AppendCommand.builder().withFlags(new Flags(Flags.Flag.DELETED)).build(MailboxManagerTest.this.message), MailboxManagerTest.this.session).getId();
                MailboxManagerTest.this.mailboxManager.deleteMailbox(this.inboxId, MailboxManagerTest.this.session);
                ArgumentCaptor preDeleteCaptor1 = ArgumentCaptor.forClass(PreDeletionHook.DeleteOperation.class);
                ArgumentCaptor preDeleteCaptor2 = ArgumentCaptor.forClass(PreDeletionHook.DeleteOperation.class);
                ((PreDeletionHook)Mockito.verify((Object)MailboxManagerTest.this.preDeletionHook1, (VerificationMode)Mockito.times((int)1))).notifyDelete((PreDeletionHook.DeleteOperation)preDeleteCaptor1.capture());
                ((PreDeletionHook)Mockito.verify((Object)MailboxManagerTest.this.preDeletionHook2, (VerificationMode)Mockito.times((int)1))).notifyDelete((PreDeletionHook.DeleteOperation)preDeleteCaptor2.capture());
                ((ListAssert)((ListAssert)Assertions.assertThat((List)((PreDeletionHook.DeleteOperation)preDeleteCaptor1.getValue()).getDeletionMetadataList()).hasSize(1)).hasSameElementsAs((Iterable)((PreDeletionHook.DeleteOperation)preDeleteCaptor2.getValue()).getDeletionMetadataList())).allSatisfy(deleteMetadata -> SoftAssertions.assertSoftly(softy -> {
                    softy.assertThat((Object)deleteMetadata.getMailboxId()).isEqualTo((Object)this.inboxId);
                    softy.assertThat((Object)deleteMetadata.getMessageMetaData().getMessageId()).isEqualTo((Object)composeId.getMessageId());
                }));
            }

            @Test
            void expungeShouldCallAllPreDeletionHooksOnEachMessageDeletionCall() throws Exception {
                ComposedMessageId composeId1 = this.inboxManager.appendMessage(MessageManager.AppendCommand.builder().withFlags(new Flags(Flags.Flag.DELETED)).build(MailboxManagerTest.this.message), MailboxManagerTest.this.session).getId();
                ComposedMessageId composeId2 = this.inboxManager.appendMessage(MessageManager.AppendCommand.builder().withFlags(new Flags(Flags.Flag.DELETED)).build(MailboxManagerTest.this.message), MailboxManagerTest.this.session).getId();
                this.inboxManager.expunge(MessageRange.one((MessageUid)composeId1.getUid()), MailboxManagerTest.this.session);
                this.inboxManager.expunge(MessageRange.one((MessageUid)composeId2.getUid()), MailboxManagerTest.this.session);
                ArgumentCaptor preDeleteCaptor1 = ArgumentCaptor.forClass(PreDeletionHook.DeleteOperation.class);
                ArgumentCaptor preDeleteCaptor2 = ArgumentCaptor.forClass(PreDeletionHook.DeleteOperation.class);
                ((PreDeletionHook)Mockito.verify((Object)MailboxManagerTest.this.preDeletionHook1, (VerificationMode)Mockito.times((int)2))).notifyDelete((PreDeletionHook.DeleteOperation)preDeleteCaptor1.capture());
                ((PreDeletionHook)Mockito.verify((Object)MailboxManagerTest.this.preDeletionHook2, (VerificationMode)Mockito.times((int)2))).notifyDelete((PreDeletionHook.DeleteOperation)preDeleteCaptor2.capture());
                ((AbstractListAssert)((ListAssert)((ListAssert)Assertions.assertThat((List)preDeleteCaptor1.getAllValues()).hasSize(2)).hasSameElementsAs((Iterable)preDeleteCaptor2.getAllValues())).flatExtracting(PreDeletionHook.DeleteOperation::getDeletionMetadataList).allSatisfy(deleteMetadata -> Assertions.assertThat((Object)deleteMetadata.getMailboxId()).isEqualTo((Object)this.inboxId))).extracting(deleteMetadata -> deleteMetadata.getMessageMetaData().getMessageId()).containsOnly((Object[])new MessageId[]{composeId1.getMessageId(), composeId2.getMessageId()});
            }

            @Test
            void expungeShouldCallAllPreDeletionHooksOnlyOnMessagesMarkedAsDeleted() throws Exception {
                ComposedMessageId composeId1 = this.inboxManager.appendMessage(MessageManager.AppendCommand.builder().withFlags(new Flags(Flags.Flag.DELETED)).build(MailboxManagerTest.this.message), MailboxManagerTest.this.session).getId();
                this.inboxManager.appendMessage(MessageManager.AppendCommand.builder().build(MailboxManagerTest.this.message), MailboxManagerTest.this.session);
                this.inboxManager.expunge(MessageRange.all(), MailboxManagerTest.this.session);
                ArgumentCaptor preDeleteCaptor1 = ArgumentCaptor.forClass(PreDeletionHook.DeleteOperation.class);
                ArgumentCaptor preDeleteCaptor2 = ArgumentCaptor.forClass(PreDeletionHook.DeleteOperation.class);
                ((PreDeletionHook)Mockito.verify((Object)MailboxManagerTest.this.preDeletionHook1, (VerificationMode)Mockito.times((int)1))).notifyDelete((PreDeletionHook.DeleteOperation)preDeleteCaptor1.capture());
                ((PreDeletionHook)Mockito.verify((Object)MailboxManagerTest.this.preDeletionHook2, (VerificationMode)Mockito.times((int)1))).notifyDelete((PreDeletionHook.DeleteOperation)preDeleteCaptor2.capture());
                ((ListAssert)((ListAssert)Assertions.assertThat((List)((PreDeletionHook.DeleteOperation)preDeleteCaptor1.getValue()).getDeletionMetadataList()).hasSize(1)).hasSameElementsAs((Iterable)((PreDeletionHook.DeleteOperation)preDeleteCaptor2.getValue()).getDeletionMetadataList())).allSatisfy(deleteMetadata -> SoftAssertions.assertSoftly(softy -> {
                    softy.assertThat((Object)deleteMetadata.getMailboxId()).isEqualTo((Object)this.inboxId);
                    softy.assertThat((Object)deleteMetadata.getMessageMetaData().getMessageId()).isEqualTo((Object)composeId1.getMessageId());
                }));
            }

            @Test
            void expungeShouldNotCallPredeletionHooksWhenNoMessagesMarkedAsDeleted() throws Exception {
                this.inboxManager.appendMessage(MessageManager.AppendCommand.builder().build(MailboxManagerTest.this.message), MailboxManagerTest.this.session);
                this.inboxManager.expunge(MessageRange.all(), MailboxManagerTest.this.session);
                Mockito.verifyZeroInteractions((Object[])new Object[]{MailboxManagerTest.this.preDeletionHook1});
                Mockito.verifyZeroInteractions((Object[])new Object[]{MailboxManagerTest.this.preDeletionHook2});
            }

            @Test
            void expungeShouldCallAllPreDeletionHooksOnEachMessageDeletionOnDifferentMailboxes() throws Exception {
                ComposedMessageId composeId1 = this.inboxManager.appendMessage(MessageManager.AppendCommand.builder().withFlags(new Flags(Flags.Flag.DELETED)).build(MailboxManagerTest.this.message), MailboxManagerTest.this.session).getId();
                ComposedMessageId composeId2 = this.anotherMailboxManager.appendMessage(MessageManager.AppendCommand.builder().withFlags(new Flags(Flags.Flag.DELETED)).build(MailboxManagerTest.this.message), MailboxManagerTest.this.session).getId();
                this.inboxManager.expunge(MessageRange.one((MessageUid)composeId1.getUid()), MailboxManagerTest.this.session);
                this.anotherMailboxManager.expunge(MessageRange.one((MessageUid)composeId2.getUid()), MailboxManagerTest.this.session);
                ArgumentCaptor preDeleteCaptor1 = ArgumentCaptor.forClass(PreDeletionHook.DeleteOperation.class);
                ArgumentCaptor preDeleteCaptor2 = ArgumentCaptor.forClass(PreDeletionHook.DeleteOperation.class);
                ((PreDeletionHook)Mockito.verify((Object)MailboxManagerTest.this.preDeletionHook1, (VerificationMode)Mockito.times((int)2))).notifyDelete((PreDeletionHook.DeleteOperation)preDeleteCaptor1.capture());
                ((PreDeletionHook)Mockito.verify((Object)MailboxManagerTest.this.preDeletionHook2, (VerificationMode)Mockito.times((int)2))).notifyDelete((PreDeletionHook.DeleteOperation)preDeleteCaptor2.capture());
                ((ListAssert)Assertions.assertThat((List)preDeleteCaptor1.getAllValues()).hasSameElementsAs((Iterable)preDeleteCaptor2.getAllValues())).flatExtracting(PreDeletionHook.DeleteOperation::getDeletionMetadataList).extracting(deleteMetadata -> deleteMetadata.getMessageMetaData().getMessageId()).containsOnly((Object[])new MessageId[]{composeId1.getMessageId(), composeId2.getMessageId()});
                ((ListAssert)Assertions.assertThat((List)preDeleteCaptor1.getAllValues()).hasSameElementsAs((Iterable)preDeleteCaptor2.getAllValues())).flatExtracting(PreDeletionHook.DeleteOperation::getDeletionMetadataList).extracting(MetadataWithMailboxId::getMailboxId).containsOnly((Object[])new MailboxId[]{this.inboxId, this.anotherMailboxId});
            }

            @Test
            void expungeShouldNotBeExecutedWhenOneOfPreDeleteHooksFails() throws Exception {
                Mockito.when((Object)MailboxManagerTest.this.preDeletionHook1.notifyDelete((PreDeletionHook.DeleteOperation)ArgumentMatchers.any(PreDeletionHook.DeleteOperation.class))).thenThrow(new Throwable[]{new RuntimeException("throw at hook 1")});
                ComposedMessageId composeId1 = this.inboxManager.appendMessage(MessageManager.AppendCommand.builder().withFlags(new Flags(Flags.Flag.DELETED)).build(MailboxManagerTest.this.message), MailboxManagerTest.this.session).getId();
                Assertions.assertThatThrownBy(() -> this.inboxManager.expunge(MessageRange.one((MessageUid)composeId1.getUid()), MailboxManagerTest.this.session)).isInstanceOf(RuntimeException.class);
                ((ListAssert)Assertions.assertThat(ImmutableList.copyOf((Iterator)this.inboxManager.getMessages(MessageRange.one((MessageUid)composeId1.getUid()), FetchGroup.MINIMAL, MailboxManagerTest.this.session)).stream().map(MessageResult::getMessageId)).hasSize(1)).containsOnly((Object[])new MessageId[]{composeId1.getMessageId()});
            }

            @Test
            void expungeShouldBeExecutedAfterAllHooksFinish() throws Exception {
                CountDownLatch latchForHook1 = new CountDownLatch(1);
                Mockito.when((Object)MailboxManagerTest.this.preDeletionHook1.notifyDelete((PreDeletionHook.DeleteOperation)ArgumentMatchers.any(PreDeletionHook.DeleteOperation.class))).thenAnswer(invocation -> {
                    latchForHook1.countDown();
                    return Mono.empty();
                });
                CountDownLatch latchForHook2 = new CountDownLatch(1);
                Mockito.when((Object)MailboxManagerTest.this.preDeletionHook2.notifyDelete((PreDeletionHook.DeleteOperation)ArgumentMatchers.any(PreDeletionHook.DeleteOperation.class))).thenAnswer(invocation -> {
                    latchForHook2.countDown();
                    return Mono.empty();
                });
                ComposedMessageId composeId1 = this.inboxManager.appendMessage(MessageManager.AppendCommand.builder().build(MailboxManagerTest.this.message), MailboxManagerTest.this.session).getId();
                this.inboxManager.setFlags(new Flags(Flags.Flag.DELETED), MessageManager.FlagsUpdateMode.ADD, MessageRange.one((MessageUid)composeId1.getUid()), MailboxManagerTest.this.session);
                this.inboxManager.expunge(MessageRange.all(), MailboxManagerTest.this.session);
                latchForHook1.await();
                latchForHook2.await();
                Assertions.assertThat((Iterator)this.inboxManager.getMessages(MessageRange.one((MessageUid)composeId1.getUid()), FetchGroup.MINIMAL, MailboxManagerTest.this.session)).toIterable().isEmpty();
            }
        }
    }

    @Nested
    class SessionTests {
        SessionTests() {
        }

        @Test
        void createUser1SystemSessionShouldReturnValidSession() throws Exception {
            MailboxManagerTest.this.session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            Assertions.assertThat((Object)MailboxManagerTest.this.session.getUser()).isEqualTo((Object)USER_1);
        }

        @Test
        void closingSessionShouldWork() {
            MailboxManagerTest.this.session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxManagerTest.this.mailboxManager.startProcessingRequest(MailboxManagerTest.this.session);
            MailboxManagerTest.this.mailboxManager.logout(MailboxManagerTest.this.session);
            MailboxManagerTest.this.mailboxManager.endProcessingRequest(MailboxManagerTest.this.session);
            Assertions.assertThat((boolean)MailboxManagerTest.this.session.isOpen()).isFalse();
        }
    }

    @Nested
    public class BasicFeaturesTests {
        @Test
        void renameMailboxShouldReturnAllRenamedResultsIncludeChildren() throws MailboxException {
            MailboxSession session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxPath mailboxPath1 = MailboxPath.forUser((Username)USER_1, (String)"mbx1");
            MailboxPath mailboxPath2 = MailboxPath.forUser((Username)USER_1, (String)"mbx1.mbx2");
            MailboxPath mailboxPath3 = MailboxPath.forUser((Username)USER_1, (String)"mbx1.mbx2.mbx3");
            MailboxPath mailboxPath4 = MailboxPath.forUser((Username)USER_1, (String)"mbx1.mbx2.mbx3.mbx4");
            MailboxPath newMailboxPath = MailboxPath.forUser((Username)USER_1, (String)"mbx1.mbx9");
            MailboxManagerTest.this.mailboxManager.createMailbox(mailboxPath1, session);
            Optional mailboxId2 = MailboxManagerTest.this.mailboxManager.createMailbox(mailboxPath2, session);
            Optional mailboxId3 = MailboxManagerTest.this.mailboxManager.createMailbox(mailboxPath3, session);
            Optional mailboxId4 = MailboxManagerTest.this.mailboxManager.createMailbox(mailboxPath4, session);
            List mailboxRenamedResults = MailboxManagerTest.this.mailboxManager.renameMailbox(mailboxPath2, newMailboxPath, session);
            SoftAssertions.assertSoftly(softly -> {
                softly.assertThat(mailboxRenamedResults).hasSize(3);
                softly.assertThat(mailboxRenamedResults).contains((Object[])new MailboxManager.MailboxRenamedResult[]{new MailboxManager.MailboxRenamedResult((MailboxId)mailboxId2.get(), mailboxPath2, MailboxPath.forUser((Username)USER_1, (String)"mbx1.mbx9")), new MailboxManager.MailboxRenamedResult((MailboxId)mailboxId3.get(), mailboxPath3, MailboxPath.forUser((Username)USER_1, (String)"mbx1.mbx9.mbx3")), new MailboxManager.MailboxRenamedResult((MailboxId)mailboxId4.get(), mailboxPath4, MailboxPath.forUser((Username)USER_1, (String)"mbx1.mbx9.mbx3.mbx4"))});
            });
        }

        @Test
        void renameMailboxShouldReturnRenamedMailboxOnlyWhenNoChildren() throws MailboxException {
            MailboxSession session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxPath mailboxPath1 = MailboxPath.forUser((Username)USER_1, (String)"mbx1");
            MailboxPath mailboxPath2 = MailboxPath.forUser((Username)USER_1, (String)"mbx1.mbx2");
            MailboxPath originalPath = MailboxPath.forUser((Username)USER_1, (String)"mbx1.mbx2.mbx3");
            MailboxPath newMailboxPath = MailboxPath.forUser((Username)USER_1, (String)"mbx1.mbx2.mbx9");
            MailboxManagerTest.this.mailboxManager.createMailbox(mailboxPath1, session);
            MailboxManagerTest.this.mailboxManager.createMailbox(mailboxPath2, session);
            Optional mailboxId3 = MailboxManagerTest.this.mailboxManager.createMailbox(originalPath, session);
            List mailboxRenamedResults = MailboxManagerTest.this.mailboxManager.renameMailbox(originalPath, newMailboxPath, session);
            SoftAssertions.assertSoftly(softly -> {
                softly.assertThat(mailboxRenamedResults).hasSize(1);
                softly.assertThat(mailboxRenamedResults).contains((Object[])new MailboxManager.MailboxRenamedResult[]{new MailboxManager.MailboxRenamedResult((MailboxId)mailboxId3.get(), originalPath, newMailboxPath)});
            });
        }

        @Test
        void renameMailboxShouldRenamedChildMailboxesWithRenameOption() throws MailboxException {
            MailboxSession session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxPath originalPath = MailboxPath.forUser((Username)USER_1, (String)"mbx1");
            MailboxPath mailboxPath2 = MailboxPath.forUser((Username)USER_1, (String)"mbx1.mbx2");
            MailboxPath mailboxPath3 = MailboxPath.forUser((Username)USER_1, (String)"mbx1.mbx2.mbx3");
            MailboxPath newMailboxPath = MailboxPath.forUser((Username)USER_1, (String)"mbx9");
            MailboxManagerTest.this.mailboxManager.createMailbox(originalPath, session);
            MailboxManagerTest.this.mailboxManager.createMailbox(mailboxPath2, session);
            MailboxManagerTest.this.subscriptionManager.subscribe(session, originalPath.getName());
            MailboxManagerTest.this.subscriptionManager.subscribe(session, mailboxPath2.getName());
            MailboxManagerTest.this.mailboxManager.createMailbox(mailboxPath3, session);
            MailboxManagerTest.this.subscriptionManager.subscribe(session, mailboxPath3.getName());
            MailboxManagerTest.this.mailboxManager.renameMailbox(originalPath, newMailboxPath, MailboxManager.RenameOption.RENAME_SUBSCRIPTIONS, session);
            Assertions.assertThat((Iterable)MailboxManagerTest.this.subscriptionManager.subscriptions(session)).containsExactly((Object[])new String[]{newMailboxPath.getName(), "mbx9.mbx2", "mbx9.mbx2.mbx3"});
        }

        @Test
        void renameMailboxShouldRenameSubscriptionWhenCalledWithRenameSubscriptionOption() throws MailboxException {
            MailboxSession session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxPath originalPath = MailboxPath.forUser((Username)USER_1, (String)"mbx1");
            MailboxPath newMailboxPath = MailboxPath.forUser((Username)USER_1, (String)"mbx2");
            MailboxManagerTest.this.mailboxManager.createMailbox(originalPath, session);
            MailboxManagerTest.this.subscriptionManager.subscribe(session, originalPath.getName());
            MailboxManagerTest.this.mailboxManager.renameMailbox(originalPath, newMailboxPath, MailboxManager.RenameOption.RENAME_SUBSCRIPTIONS, session);
            Assertions.assertThat((Iterable)MailboxManagerTest.this.subscriptionManager.subscriptions(session)).containsExactly((Object[])new String[]{newMailboxPath.getName()});
        }

        @Test
        void renameMailboxShouldNotSubscribeUnsubscribedMailboxes() throws MailboxException {
            MailboxSession session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxPath originalPath = MailboxPath.forUser((Username)USER_1, (String)"mbx1");
            MailboxPath newMailboxPath = MailboxPath.forUser((Username)USER_1, (String)"mbx2");
            MailboxManagerTest.this.mailboxManager.createMailbox(originalPath, session);
            MailboxManagerTest.this.mailboxManager.renameMailbox(originalPath, newMailboxPath, MailboxManager.RenameOption.RENAME_SUBSCRIPTIONS, session);
            Assertions.assertThat((Iterable)MailboxManagerTest.this.subscriptionManager.subscriptions(session)).isEmpty();
        }

        @Test
        void renameMailboxShouldNotRenameSubscriptionWhenCalledWithoutRenameSubscriptionOption() throws MailboxException {
            MailboxSession session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxPath originalPath = MailboxPath.forUser((Username)USER_1, (String)"mbx1");
            MailboxPath newMailboxPath = MailboxPath.forUser((Username)USER_1, (String)"mbx2");
            MailboxManagerTest.this.mailboxManager.createMailbox(originalPath, session);
            MailboxManagerTest.this.subscriptionManager.subscribe(session, originalPath.getName());
            MailboxManagerTest.this.mailboxManager.renameMailbox(originalPath, newMailboxPath, MailboxManager.RenameOption.NONE, session);
            Assertions.assertThat((Iterable)MailboxManagerTest.this.subscriptionManager.subscriptions(session)).containsExactly((Object[])new String[]{originalPath.getName()});
        }

        @Test
        void renameMailboxByIdShouldRenamedMailboxesWithRenameOption() throws MailboxException {
            MailboxSession session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxPath originalPath = MailboxPath.forUser((Username)USER_1, (String)"mbx1");
            MailboxPath mailboxPath2 = MailboxPath.forUser((Username)USER_1, (String)"mbx1.mbx2");
            MailboxPath mailboxPath3 = MailboxPath.forUser((Username)USER_1, (String)"mbx1.mbx2.mbx3");
            MailboxPath newMailboxPath = MailboxPath.forUser((Username)USER_1, (String)"mbx9");
            Optional id = MailboxManagerTest.this.mailboxManager.createMailbox(originalPath, session);
            MailboxManagerTest.this.mailboxManager.createMailbox(mailboxPath2, session);
            MailboxManagerTest.this.subscriptionManager.subscribe(session, originalPath.getName());
            MailboxManagerTest.this.subscriptionManager.subscribe(session, mailboxPath2.getName());
            MailboxManagerTest.this.mailboxManager.createMailbox(mailboxPath3, session);
            MailboxManagerTest.this.subscriptionManager.subscribe(session, mailboxPath3.getName());
            MailboxManagerTest.this.mailboxManager.renameMailbox((MailboxId)id.get(), newMailboxPath, MailboxManager.RenameOption.RENAME_SUBSCRIPTIONS, session);
            Assertions.assertThat((Iterable)MailboxManagerTest.this.subscriptionManager.subscriptions(session)).containsExactly((Object[])new String[]{newMailboxPath.getName(), "mbx9.mbx2", "mbx9.mbx2.mbx3"});
        }

        @Test
        void renameMailboxByIdShouldRenameSubscriptionWhenCalledWithRenameSubscriptionOption() throws MailboxException {
            MailboxSession session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxPath originalPath = MailboxPath.forUser((Username)USER_1, (String)"mbx1");
            MailboxPath newMailboxPath = MailboxPath.forUser((Username)USER_1, (String)"mbx2");
            Optional id = MailboxManagerTest.this.mailboxManager.createMailbox(originalPath, session);
            MailboxManagerTest.this.subscriptionManager.subscribe(session, originalPath.getName());
            MailboxManagerTest.this.mailboxManager.renameMailbox((MailboxId)id.get(), newMailboxPath, MailboxManager.RenameOption.RENAME_SUBSCRIPTIONS, session);
            Assertions.assertThat((Iterable)MailboxManagerTest.this.subscriptionManager.subscriptions(session)).containsExactly((Object[])new String[]{newMailboxPath.getName()});
        }

        @Test
        void renameMailboxByIdShouldNotSubscribeUnsubscribedMailboxes() throws MailboxException {
            MailboxSession session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxPath originalPath = MailboxPath.forUser((Username)USER_1, (String)"mbx1");
            MailboxPath newMailboxPath = MailboxPath.forUser((Username)USER_1, (String)"mbx2");
            Optional id = MailboxManagerTest.this.mailboxManager.createMailbox(originalPath, session);
            MailboxManagerTest.this.mailboxManager.renameMailbox((MailboxId)id.get(), newMailboxPath, MailboxManager.RenameOption.RENAME_SUBSCRIPTIONS, session);
            Assertions.assertThat((Iterable)MailboxManagerTest.this.subscriptionManager.subscriptions(session)).isEmpty();
        }

        @Test
        void renameMailboxByIdShouldNotRenameSubscriptionWhenCalledWithoutRenameSubscriptionOption() throws MailboxException {
            MailboxSession session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxPath originalPath = MailboxPath.forUser((Username)USER_1, (String)"mbx1");
            MailboxPath newMailboxPath = MailboxPath.forUser((Username)USER_1, (String)"mbx2");
            Optional id = MailboxManagerTest.this.mailboxManager.createMailbox(originalPath, session);
            MailboxManagerTest.this.subscriptionManager.subscribe(session, originalPath.getName());
            MailboxManagerTest.this.mailboxManager.renameMailbox((MailboxId)id.get(), newMailboxPath, MailboxManager.RenameOption.NONE, session);
            Assertions.assertThat((Iterable)MailboxManagerTest.this.subscriptionManager.subscriptions(session)).containsExactly((Object[])new String[]{originalPath.getName()});
        }

        @Test
        void user1ShouldNotHaveAnInbox() throws Exception {
            MailboxManagerTest.this.session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxManagerTest.this.mailboxManager.startProcessingRequest(MailboxManagerTest.this.session);
            MailboxPath inbox = MailboxPath.inbox((MailboxSession)MailboxManagerTest.this.session);
            Assertions.assertThat((Boolean)((Boolean)Mono.from((Publisher)MailboxManagerTest.this.mailboxManager.mailboxExists(inbox, MailboxManagerTest.this.session)).block())).isFalse();
        }

        @Test
        void user1ShouldBeAbleToCreateInbox() throws Exception {
            MailboxManagerTest.this.session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxManagerTest.this.mailboxManager.startProcessingRequest(MailboxManagerTest.this.session);
            MailboxPath inbox = MailboxPath.inbox((MailboxSession)MailboxManagerTest.this.session);
            MailboxManagerTest.this.mailboxManager.createMailbox(inbox, MailboxManagerTest.this.session);
            Assertions.assertThat((Boolean)((Boolean)Mono.from((Publisher)MailboxManagerTest.this.mailboxManager.mailboxExists(inbox, MailboxManagerTest.this.session)).block())).isTrue();
        }

        @Test
        protected void renameMailboxShouldChangeTheMailboxPathOfAMailbox() throws Exception {
            MailboxSession session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxPath mailboxPath1 = MailboxPath.forUser((Username)USER_1, (String)"mbx1");
            MailboxPath mailboxPath2 = MailboxPath.forUser((Username)USER_1, (String)"mbx2");
            Optional mailboxId = MailboxManagerTest.this.mailboxManager.createMailbox(mailboxPath1, session);
            MailboxManagerTest.this.mailboxManager.renameMailbox(mailboxPath1, mailboxPath2, session);
            Assertions.assertThat((Object)MailboxManagerTest.this.mailboxManager.getMailbox((MailboxId)mailboxId.get(), session).getMailboxPath()).isEqualTo((Object)mailboxPath2);
        }

        @Test
        protected void renameMailboxShouldChangeTheMailboxPathOfTheChildMailbox() throws Exception {
            MailboxSession session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxPath mailboxPath1 = MailboxPath.forUser((Username)USER_1, (String)"mbx1");
            MailboxPath mailboxPath2 = MailboxPath.forUser((Username)USER_1, (String)"mbx2");
            MailboxManagerTest.this.mailboxManager.createMailbox(mailboxPath1, session);
            MailboxPath mailboxPath1Child = MailboxPath.forUser((Username)USER_1, (String)"mbx1.child");
            Optional mailboxChildId = MailboxManagerTest.this.mailboxManager.createMailbox(mailboxPath1Child, session);
            MailboxManagerTest.this.mailboxManager.renameMailbox(mailboxPath1, mailboxPath2, session);
            Assertions.assertThat((Object)MailboxManagerTest.this.mailboxManager.getMailbox((MailboxId)mailboxChildId.get(), session).getMailboxPath()).isEqualTo((Object)MailboxPath.forUser((Username)USER_1, (String)"mbx2.child"));
        }

        @Test
        protected void renameMailboxByIdShouldChangeTheMailboxPathOfAMailbox() throws Exception {
            MailboxSession session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxPath mailboxPath1 = MailboxPath.forUser((Username)USER_1, (String)"mbx1");
            MailboxPath mailboxPath2 = MailboxPath.forUser((Username)USER_1, (String)"mbx2");
            Optional mailboxId = MailboxManagerTest.this.mailboxManager.createMailbox(mailboxPath1, session);
            MailboxManagerTest.this.mailboxManager.renameMailbox((MailboxId)mailboxId.get(), mailboxPath2, session);
            Assertions.assertThat((Object)MailboxManagerTest.this.mailboxManager.getMailbox((MailboxId)mailboxId.get(), session).getMailboxPath()).isEqualTo((Object)mailboxPath2);
        }

        @Test
        void renameMailboxShouldThrowWhenMailboxPathsDoNotBelongToUser() throws Exception {
            MailboxSession sessionUser1 = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxSession sessionUser2 = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_2);
            MailboxPath mailboxPath1 = MailboxPath.forUser((Username)USER_1, (String)"mbx1");
            MailboxPath mailboxPath2 = MailboxPath.forUser((Username)USER_1, (String)"mbx2");
            MailboxManagerTest.this.mailboxManager.createMailbox(mailboxPath1, sessionUser1);
            Assertions.assertThatThrownBy(() -> MailboxManagerTest.this.mailboxManager.renameMailbox(mailboxPath1, mailboxPath2, sessionUser2)).isInstanceOf(MailboxNotFoundException.class);
        }

        @Test
        void renameMailboxByIdShouldThrowWhenMailboxPathsDoNotBelongToUser() throws Exception {
            MailboxSession sessionUser1 = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxSession sessionUser2 = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_2);
            MailboxPath mailboxPath1 = MailboxPath.forUser((Username)USER_1, (String)"mbx1");
            MailboxPath mailboxPath2 = MailboxPath.forUser((Username)USER_1, (String)"mbx2");
            Optional mailboxId = MailboxManagerTest.this.mailboxManager.createMailbox(mailboxPath1, sessionUser1);
            Assertions.assertThatThrownBy(() -> MailboxManagerTest.this.mailboxManager.renameMailbox((MailboxId)mailboxId.get(), mailboxPath2, sessionUser2)).isInstanceOf(MailboxNotFoundException.class);
        }

        @Test
        void renameMailboxShouldThrowWhenFromMailboxPathDoesNotBelongToUser() throws Exception {
            MailboxSession sessionUser1 = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxSession sessionUser2 = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_2);
            MailboxPath mailboxPath1 = MailboxPath.forUser((Username)USER_1, (String)"mbx1");
            MailboxPath mailboxPath2 = MailboxPath.forUser((Username)USER_2, (String)"mbx2");
            MailboxManagerTest.this.mailboxManager.createMailbox(mailboxPath1, sessionUser1);
            Assertions.assertThatThrownBy(() -> MailboxManagerTest.this.mailboxManager.renameMailbox(mailboxPath1, mailboxPath2, sessionUser2)).isInstanceOf(MailboxNotFoundException.class);
        }

        @Test
        void renameMailboxByIdShouldThrowWhenFromMailboxPathDoesNotBelongToUser() throws Exception {
            MailboxSession sessionUser1 = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxSession sessionUser2 = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_2);
            MailboxPath mailboxPath1 = MailboxPath.forUser((Username)USER_1, (String)"mbx1");
            MailboxPath mailboxPath2 = MailboxPath.forUser((Username)USER_2, (String)"mbx2");
            Optional mailboxId = MailboxManagerTest.this.mailboxManager.createMailbox(mailboxPath1, sessionUser1);
            Assertions.assertThatThrownBy(() -> MailboxManagerTest.this.mailboxManager.renameMailbox((MailboxId)mailboxId.get(), mailboxPath2, sessionUser2)).isInstanceOf(MailboxNotFoundException.class);
        }

        @Test
        void renameMailboxShouldThrowWhenToMailboxPathDoesNotBelongToUser() throws Exception {
            MailboxManagerTest.this.session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxPath mailboxPath1 = MailboxPath.forUser((Username)USER_1, (String)"mbx1");
            MailboxPath mailboxPath2 = MailboxPath.forUser((Username)USER_2, (String)"mbx2");
            MailboxManagerTest.this.mailboxManager.createMailbox(mailboxPath1, MailboxManagerTest.this.session);
            Assertions.assertThatThrownBy(() -> MailboxManagerTest.this.mailboxManager.renameMailbox(mailboxPath1, mailboxPath2, MailboxManagerTest.this.session)).isInstanceOf(MailboxNotFoundException.class);
        }

        @Test
        void renameMailboxByIdShouldThrowWhenToMailboxPathDoesNotBelongToUser() throws Exception {
            MailboxManagerTest.this.session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxPath mailboxPath1 = MailboxPath.forUser((Username)USER_1, (String)"mbx1");
            MailboxPath mailboxPath2 = MailboxPath.forUser((Username)USER_2, (String)"mbx2");
            Optional mailboxId = MailboxManagerTest.this.mailboxManager.createMailbox(mailboxPath1, MailboxManagerTest.this.session);
            Assertions.assertThatThrownBy(() -> MailboxManagerTest.this.mailboxManager.renameMailbox((MailboxId)mailboxId.get(), mailboxPath2, MailboxManagerTest.this.session)).isInstanceOf(MailboxNotFoundException.class);
        }

        @Test
        void user1ShouldNotBeAbleToCreateInboxTwice() throws Exception {
            MailboxManagerTest.this.session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxManagerTest.this.mailboxManager.startProcessingRequest(MailboxManagerTest.this.session);
            MailboxPath inbox = MailboxPath.inbox((MailboxSession)MailboxManagerTest.this.session);
            MailboxManagerTest.this.mailboxManager.createMailbox(inbox, MailboxManagerTest.this.session);
            Assertions.assertThatThrownBy(() -> MailboxManagerTest.this.mailboxManager.createMailbox(inbox, MailboxManagerTest.this.session)).isInstanceOf(MailboxExistsException.class);
        }

        @Test
        void user1ShouldNotHaveTestSubmailbox() throws Exception {
            MailboxManagerTest.this.session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxManagerTest.this.mailboxManager.startProcessingRequest(MailboxManagerTest.this.session);
            MailboxPath inbox = MailboxPath.inbox((MailboxSession)MailboxManagerTest.this.session);
            MailboxManagerTest.this.mailboxManager.createMailbox(inbox, MailboxManagerTest.this.session);
            Assertions.assertThat((Boolean)((Boolean)Mono.from((Publisher)MailboxManagerTest.this.mailboxManager.mailboxExists(new MailboxPath(inbox, "INBOX.Test"), MailboxManagerTest.this.session)).block())).isFalse();
        }

        @Test
        void user1ShouldBeAbleToCreateTestSubmailbox() throws Exception {
            MailboxManagerTest.this.session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxManagerTest.this.mailboxManager.startProcessingRequest(MailboxManagerTest.this.session);
            MailboxPath inbox = MailboxPath.inbox((MailboxSession)MailboxManagerTest.this.session);
            MailboxManagerTest.this.mailboxManager.createMailbox(inbox, MailboxManagerTest.this.session);
            MailboxPath inboxSubMailbox = new MailboxPath(inbox, "INBOX.Test");
            MailboxManagerTest.this.mailboxManager.createMailbox(inboxSubMailbox, MailboxManagerTest.this.session);
            Assertions.assertThat((Boolean)((Boolean)Mono.from((Publisher)MailboxManagerTest.this.mailboxManager.mailboxExists(inboxSubMailbox, MailboxManagerTest.this.session)).block())).isTrue();
        }

        @Test
        void user1ShouldBeAbleToDeleteInbox() throws Exception {
            MailboxManagerTest.this.session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxManagerTest.this.mailboxManager.startProcessingRequest(MailboxManagerTest.this.session);
            MailboxPath inbox = MailboxPath.inbox((MailboxSession)MailboxManagerTest.this.session);
            MailboxManagerTest.this.mailboxManager.createMailbox(inbox, MailboxManagerTest.this.session);
            MailboxPath inboxSubMailbox = new MailboxPath(inbox, "INBOX.Test");
            MailboxManagerTest.this.mailboxManager.createMailbox(inboxSubMailbox, MailboxManagerTest.this.session);
            MailboxManagerTest.this.mailboxManager.deleteMailbox(inbox, MailboxManagerTest.this.session);
            Assertions.assertThat((Boolean)((Boolean)Mono.from((Publisher)MailboxManagerTest.this.mailboxManager.mailboxExists(inbox, MailboxManagerTest.this.session)).block())).isFalse();
            Assertions.assertThat((Boolean)((Boolean)Mono.from((Publisher)MailboxManagerTest.this.mailboxManager.mailboxExists(inboxSubMailbox, MailboxManagerTest.this.session)).block())).isTrue();
        }

        @Test
        protected void user1ShouldBeAbleToDeleteInboxById() throws Exception {
            MailboxManagerTest.this.session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxManagerTest.this.mailboxManager.startProcessingRequest(MailboxManagerTest.this.session);
            MailboxPath inbox = MailboxPath.inbox((MailboxSession)MailboxManagerTest.this.session);
            MailboxId inboxId = (MailboxId)MailboxManagerTest.this.mailboxManager.createMailbox(inbox, MailboxManagerTest.this.session).get();
            MailboxPath inboxSubMailbox = new MailboxPath(inbox, "INBOX.Test");
            MailboxManagerTest.this.mailboxManager.createMailbox(inboxSubMailbox, MailboxManagerTest.this.session);
            MailboxManagerTest.this.mailboxManager.deleteMailbox(inboxId, MailboxManagerTest.this.session);
            Assertions.assertThat((Boolean)((Boolean)Mono.from((Publisher)MailboxManagerTest.this.mailboxManager.mailboxExists(inbox, MailboxManagerTest.this.session)).block())).isFalse();
            Assertions.assertThat((Boolean)((Boolean)Mono.from((Publisher)MailboxManagerTest.this.mailboxManager.mailboxExists(inboxSubMailbox, MailboxManagerTest.this.session)).block())).isTrue();
        }

        @Test
        void renamingMailboxByPathShouldThrowWhenFromNotFound() throws Exception {
            MailboxSession session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxPath originPath = MailboxPath.forUser((Username)USER_1, (String)"origin");
            MailboxPath destinationPath = MailboxPath.forUser((Username)USER_1, (String)"destination");
            Assertions.assertThatThrownBy(() -> MailboxManagerTest.this.mailboxManager.renameMailbox(originPath, destinationPath, session)).isInstanceOf(MailboxNotFoundException.class);
        }

        @Test
        void renamingMailboxByIdShouldThrowWhenFromNotFound() throws Exception {
            MailboxSession session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxPath notFound = MailboxPath.forUser((Username)USER_1, (String)"notFound");
            Assertions.assertThatThrownBy(() -> MailboxManagerTest.this.mailboxManager.deleteMailbox(notFound, session)).isInstanceOf(MailboxNotFoundException.class);
        }

        @Test
        void user2ShouldNotBeAbleToDeleteUser1Mailbox() throws Exception {
            MailboxSession sessionUser1 = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxSession sessionUser2 = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_2);
            MailboxPath inbox = MailboxPath.inbox((MailboxSession)sessionUser1);
            MailboxManagerTest.this.mailboxManager.createMailbox(inbox, sessionUser1);
            Assertions.assertThatThrownBy(() -> MailboxManagerTest.this.mailboxManager.deleteMailbox(inbox, sessionUser2)).isInstanceOf(MailboxNotFoundException.class);
        }

        @Test
        void user2ShouldNotBeAbleToDeleteUser1MailboxById() throws Exception {
            MailboxSession sessionUser1 = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxSession sessionUser2 = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_2);
            MailboxPath inbox = MailboxPath.inbox((MailboxSession)sessionUser1);
            MailboxId inboxId = (MailboxId)MailboxManagerTest.this.mailboxManager.createMailbox(inbox, sessionUser1).get();
            Assertions.assertThatThrownBy(() -> MailboxManagerTest.this.mailboxManager.deleteMailbox(inboxId, sessionUser2)).isInstanceOf(MailboxNotFoundException.class);
        }

        @Test
        void user1ShouldBeAbleToDeleteSubmailbox() throws Exception {
            MailboxManagerTest.this.session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxManagerTest.this.mailboxManager.startProcessingRequest(MailboxManagerTest.this.session);
            MailboxPath inbox = MailboxPath.inbox((MailboxSession)MailboxManagerTest.this.session);
            MailboxManagerTest.this.mailboxManager.createMailbox(inbox, MailboxManagerTest.this.session);
            MailboxPath inboxSubMailbox = new MailboxPath(inbox, "INBOX.Test");
            MailboxManagerTest.this.mailboxManager.createMailbox(inboxSubMailbox, MailboxManagerTest.this.session);
            MailboxManagerTest.this.mailboxManager.deleteMailbox(inboxSubMailbox, MailboxManagerTest.this.session);
            Assertions.assertThat((Boolean)((Boolean)Mono.from((Publisher)MailboxManagerTest.this.mailboxManager.mailboxExists(inbox, MailboxManagerTest.this.session)).block())).isTrue();
            Assertions.assertThat((Boolean)((Boolean)Mono.from((Publisher)MailboxManagerTest.this.mailboxManager.mailboxExists(inboxSubMailbox, MailboxManagerTest.this.session)).block())).isFalse();
        }

        @Test
        protected void user1ShouldBeAbleToDeleteSubmailboxByid() throws Exception {
            MailboxManagerTest.this.session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxManagerTest.this.mailboxManager.startProcessingRequest(MailboxManagerTest.this.session);
            MailboxPath inbox = MailboxPath.inbox((MailboxSession)MailboxManagerTest.this.session);
            MailboxManagerTest.this.mailboxManager.createMailbox(inbox, MailboxManagerTest.this.session);
            MailboxPath inboxSubMailbox = new MailboxPath(inbox, "INBOX.Test");
            MailboxId inboxSubMailboxId = (MailboxId)MailboxManagerTest.this.mailboxManager.createMailbox(inboxSubMailbox, MailboxManagerTest.this.session).get();
            MailboxManagerTest.this.mailboxManager.deleteMailbox(inboxSubMailboxId, MailboxManagerTest.this.session);
            Assertions.assertThat((Boolean)((Boolean)Mono.from((Publisher)MailboxManagerTest.this.mailboxManager.mailboxExists(inbox, MailboxManagerTest.this.session)).block())).isTrue();
            Assertions.assertThat((Boolean)((Boolean)Mono.from((Publisher)MailboxManagerTest.this.mailboxManager.mailboxExists(inboxSubMailbox, MailboxManagerTest.this.session)).block())).isFalse();
        }

        @Test
        void listShouldReturnMailboxes() throws Exception {
            MailboxManagerTest.this.session = MailboxManagerTest.this.mailboxManager.createSystemSession(Username.of((String)"manager"));
            MailboxManagerTest.this.mailboxManager.startProcessingRequest(MailboxManagerTest.this.session);
            DataProvisioner.feedMailboxManager(MailboxManagerTest.this.mailboxManager);
            Assertions.assertThat((List)MailboxManagerTest.this.mailboxManager.list(MailboxManagerTest.this.session)).hasSize(117);
        }

        @Test
        void listShouldReturnEmptyListWhenNoMailboxes() throws Exception {
            MailboxManagerTest.this.session = MailboxManagerTest.this.mailboxManager.createSystemSession(Username.of((String)"manager"));
            Assertions.assertThat((List)MailboxManagerTest.this.mailboxManager.list(MailboxManagerTest.this.session)).isEmpty();
        }

        @Test
        void user2ShouldBeAbleToCreateRootlessFolder() throws MailboxException {
            MailboxManagerTest.this.session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_2);
            MailboxPath trash = MailboxPath.forUser((Username)USER_2, (String)"Trash");
            MailboxManagerTest.this.mailboxManager.createMailbox(trash, MailboxManagerTest.this.session);
            Assertions.assertThat((Boolean)((Boolean)Mono.from((Publisher)MailboxManagerTest.this.mailboxManager.mailboxExists(trash, MailboxManagerTest.this.session)).block())).isTrue();
        }

        @Test
        void user2ShouldBeAbleToCreateNestedFoldersWithoutTheirParents() throws Exception {
            MailboxManagerTest.this.session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_2);
            MailboxPath nestedFolder = MailboxPath.forUser((Username)USER_2, (String)"INBOX.testfolder");
            MailboxManagerTest.this.mailboxManager.createMailbox(nestedFolder, MailboxManagerTest.this.session);
            Assertions.assertThat((Boolean)((Boolean)Mono.from((Publisher)MailboxManagerTest.this.mailboxManager.mailboxExists(nestedFolder, MailboxManagerTest.this.session)).block())).isTrue();
            MailboxManagerTest.this.mailboxManager.getMailbox(MailboxPath.inbox((MailboxSession)MailboxManagerTest.this.session), MailboxManagerTest.this.session).appendMessage(MessageManager.AppendCommand.from((Message.Builder)MailboxManagerTest.this.message), MailboxManagerTest.this.session);
        }

        @Test
        void moveMessagesShouldNotThrowWhenMovingAllMessagesOfAnEmptyMailbox() throws Exception {
            MailboxManagerTest.this.session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxPath inbox = MailboxPath.inbox((MailboxSession)MailboxManagerTest.this.session);
            MailboxManagerTest.this.mailboxManager.createMailbox(inbox, MailboxManagerTest.this.session);
            Assertions.assertThatCode(() -> MailboxManagerTest.this.mailboxManager.moveMessages(MessageRange.all(), inbox, inbox, MailboxManagerTest.this.session)).doesNotThrowAnyException();
        }

        @Test
        void moveMessagesShouldMoveAllMessagesFromOneMailboxToAnOtherOfASameUser() throws Exception {
            Assumptions.assumeTrue((boolean)MailboxManagerTest.this.mailboxManager.hasCapability(MailboxManager.MailboxCapabilities.ACL));
            Assumptions.assumeTrue((boolean)MailboxManagerTest.this.mailboxManager.hasCapability(MailboxManager.MailboxCapabilities.Move));
            MailboxManagerTest.this.session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxPath inbox = MailboxPath.inbox((MailboxSession)MailboxManagerTest.this.session);
            MailboxId inboxId = (MailboxId)MailboxManagerTest.this.mailboxManager.createMailbox(inbox, MailboxManagerTest.this.session).get();
            MailboxPath otherMailbox = MailboxPath.forUser((Username)USER_1, (String)"otherMailbox");
            MailboxId otherMailboxId = (MailboxId)MailboxManagerTest.this.mailboxManager.createMailbox(otherMailbox, MailboxManagerTest.this.session).get();
            MessageManager inboxMessageManager = MailboxManagerTest.this.mailboxManager.getMailbox(inbox, MailboxManagerTest.this.session);
            MessageId messageId1 = inboxMessageManager.appendMessage(MessageManager.AppendCommand.from((Message.Builder)MailboxManagerTest.this.message), MailboxManagerTest.this.session).getId().getMessageId();
            MessageId messageId2 = inboxMessageManager.appendMessage(MessageManager.AppendCommand.from((Message.Builder)MailboxManagerTest.this.message), MailboxManagerTest.this.session).getId().getMessageId();
            MailboxManagerTest.this.mailboxManager.moveMessages(MessageRange.all(), inbox, otherMailbox, MailboxManagerTest.this.session);
            MultimailboxesSearchQuery inboxQuery = MultimailboxesSearchQuery.from((SearchQuery)SearchQuery.matchAll()).inMailboxes(new MailboxId[]{inboxId}).build();
            MultimailboxesSearchQuery otherMailboxQuery = MultimailboxesSearchQuery.from((SearchQuery)SearchQuery.matchAll()).inMailboxes(new MailboxId[]{otherMailboxId}).build();
            Assertions.assertThat((List)((List)Flux.from((Publisher)MailboxManagerTest.this.mailboxManager.search(inboxQuery, MailboxManagerTest.this.session, 256L)).collectList().block())).isEmpty();
            Assertions.assertThat((List)((List)Flux.from((Publisher)MailboxManagerTest.this.mailboxManager.search(otherMailboxQuery, MailboxManagerTest.this.session, 256L)).collectList().block())).containsExactly((Object[])new MessageId[]{messageId1, messageId2});
        }

        @Test
        void moveMessagesShouldMoveOnlyOneMessageFromOneMailboxToAnOtherOfASameUser() throws Exception {
            Assumptions.assumeTrue((boolean)MailboxManagerTest.this.mailboxManager.hasCapability(MailboxManager.MailboxCapabilities.ACL));
            Assumptions.assumeTrue((boolean)MailboxManagerTest.this.mailboxManager.hasCapability(MailboxManager.MailboxCapabilities.Move));
            MailboxManagerTest.this.session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxPath inbox = MailboxPath.inbox((MailboxSession)MailboxManagerTest.this.session);
            MailboxId inboxId = (MailboxId)MailboxManagerTest.this.mailboxManager.createMailbox(inbox, MailboxManagerTest.this.session).get();
            MailboxPath otherMailbox = MailboxPath.forUser((Username)USER_1, (String)"otherMailbox");
            MailboxId otherMailboxId = (MailboxId)MailboxManagerTest.this.mailboxManager.createMailbox(otherMailbox, MailboxManagerTest.this.session).get();
            MessageManager inboxMessageManager = MailboxManagerTest.this.mailboxManager.getMailbox(inbox, MailboxManagerTest.this.session);
            ComposedMessageId composedMessageId1 = inboxMessageManager.appendMessage(MessageManager.AppendCommand.from((Message.Builder)MailboxManagerTest.this.message), MailboxManagerTest.this.session).getId();
            MessageId messageId2 = inboxMessageManager.appendMessage(MessageManager.AppendCommand.from((Message.Builder)MailboxManagerTest.this.message), MailboxManagerTest.this.session).getId().getMessageId();
            MailboxManagerTest.this.mailboxManager.moveMessages(MessageRange.one((MessageUid)composedMessageId1.getUid()), inbox, otherMailbox, MailboxManagerTest.this.session);
            MultimailboxesSearchQuery inboxQuery = MultimailboxesSearchQuery.from((SearchQuery)SearchQuery.matchAll()).inMailboxes(new MailboxId[]{inboxId}).build();
            MultimailboxesSearchQuery otherMailboxQuery = MultimailboxesSearchQuery.from((SearchQuery)SearchQuery.matchAll()).inMailboxes(new MailboxId[]{otherMailboxId}).build();
            Assertions.assertThat((List)((List)Flux.from((Publisher)MailboxManagerTest.this.mailboxManager.search(inboxQuery, MailboxManagerTest.this.session, 256L)).collectList().block())).containsExactly((Object[])new MessageId[]{messageId2});
            Assertions.assertThat((List)((List)Flux.from((Publisher)MailboxManagerTest.this.mailboxManager.search(otherMailboxQuery, MailboxManagerTest.this.session, 256L)).collectList().block())).containsExactly((Object[])new MessageId[]{composedMessageId1.getMessageId()});
        }

        @Test
        void moveMessagesShouldThrowWhenMovingMessageFromMailboxNotBelongingToSameUser() throws Exception {
            MailboxSession sessionUser1 = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxSession sessionUser2 = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_2);
            MailboxPath inbox1 = MailboxPath.inbox((MailboxSession)sessionUser1);
            MailboxManagerTest.this.mailboxManager.createMailbox(inbox1, sessionUser1);
            MailboxPath inbox2 = MailboxPath.inbox((MailboxSession)sessionUser2);
            MailboxManagerTest.this.mailboxManager.createMailbox(inbox2, sessionUser2);
            MessageManager inboxMessageManager = MailboxManagerTest.this.mailboxManager.getMailbox(inbox1, sessionUser1);
            inboxMessageManager.appendMessage(MessageManager.AppendCommand.from((Message.Builder)MailboxManagerTest.this.message), sessionUser1);
            Assertions.assertThatThrownBy(() -> MailboxManagerTest.this.mailboxManager.moveMessages(MessageRange.all(), inbox1, inbox2, sessionUser2)).isInstanceOf(MailboxNotFoundException.class);
        }

        @Test
        void moveMessagesShouldThrowWhenMovingMessageToMailboxNotBelongingToSameUser() throws Exception {
            MailboxSession sessionUser1 = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxSession sessionUser2 = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_2);
            MailboxPath inbox1 = MailboxPath.inbox((MailboxSession)sessionUser1);
            MailboxManagerTest.this.mailboxManager.createMailbox(inbox1, sessionUser1);
            MailboxPath inbox2 = MailboxPath.inbox((MailboxSession)sessionUser2);
            MailboxManagerTest.this.mailboxManager.createMailbox(inbox2, sessionUser2);
            MessageManager inboxMessageManager = MailboxManagerTest.this.mailboxManager.getMailbox(inbox1, sessionUser1);
            inboxMessageManager.appendMessage(MessageManager.AppendCommand.from((Message.Builder)MailboxManagerTest.this.message), sessionUser1);
            Assertions.assertThatThrownBy(() -> MailboxManagerTest.this.mailboxManager.moveMessages(MessageRange.all(), inbox1, inbox2, sessionUser1)).isInstanceOf(MailboxNotFoundException.class);
        }

        @Test
        void copyMessagesShouldNotThrowWhenMovingAllMessagesOfAnEmptyMailbox() throws Exception {
            MailboxManagerTest.this.session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxPath inbox = MailboxPath.inbox((MailboxSession)MailboxManagerTest.this.session);
            MailboxManagerTest.this.mailboxManager.createMailbox(inbox, MailboxManagerTest.this.session);
            Assertions.assertThatCode(() -> MailboxManagerTest.this.mailboxManager.copyMessages(MessageRange.all(), inbox, inbox, MailboxManagerTest.this.session)).doesNotThrowAnyException();
        }

        @Test
        void copyMessagesShouldCopyAllMessagesFromOneMailboxToAnOtherOfASameUser() throws Exception {
            Assumptions.assumeTrue((boolean)MailboxManagerTest.this.mailboxManager.hasCapability(MailboxManager.MailboxCapabilities.ACL));
            MailboxManagerTest.this.session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxPath inbox = MailboxPath.inbox((MailboxSession)MailboxManagerTest.this.session);
            MailboxId inboxId = (MailboxId)MailboxManagerTest.this.mailboxManager.createMailbox(inbox, MailboxManagerTest.this.session).get();
            MailboxPath otherMailbox = MailboxPath.forUser((Username)USER_1, (String)"otherMailbox");
            MailboxId otherMailboxId = (MailboxId)MailboxManagerTest.this.mailboxManager.createMailbox(otherMailbox, MailboxManagerTest.this.session).get();
            MessageManager inboxMessageManager = MailboxManagerTest.this.mailboxManager.getMailbox(inbox, MailboxManagerTest.this.session);
            MessageId messageId1 = inboxMessageManager.appendMessage(MessageManager.AppendCommand.from((Message.Builder)MailboxManagerTest.this.message), MailboxManagerTest.this.session).getId().getMessageId();
            MessageId messageId2 = inboxMessageManager.appendMessage(MessageManager.AppendCommand.from((Message.Builder)MailboxManagerTest.this.message), MailboxManagerTest.this.session).getId().getMessageId();
            MailboxManagerTest.this.mailboxManager.copyMessages(MessageRange.all(), inbox, otherMailbox, MailboxManagerTest.this.session);
            MultimailboxesSearchQuery inboxQuery = MultimailboxesSearchQuery.from((SearchQuery)SearchQuery.matchAll()).inMailboxes(new MailboxId[]{inboxId}).build();
            MultimailboxesSearchQuery otherMailboxQuery = MultimailboxesSearchQuery.from((SearchQuery)SearchQuery.matchAll()).inMailboxes(new MailboxId[]{otherMailboxId}).build();
            Assertions.assertThat((List)((List)Flux.from((Publisher)MailboxManagerTest.this.mailboxManager.search(inboxQuery, MailboxManagerTest.this.session, 256L)).collectList().block())).containsExactly((Object[])new MessageId[]{messageId1, messageId2});
            Assertions.assertThat((List)((List)Flux.from((Publisher)MailboxManagerTest.this.mailboxManager.search(otherMailboxQuery, MailboxManagerTest.this.session, 256L)).collectList().block())).containsExactly((Object[])new MessageId[]{messageId1, messageId2});
        }

        @Test
        void copyMessagesShouldCopyOnlyOneMessageFromOneMailboxToAnOtherOfASameUser() throws Exception {
            Assumptions.assumeTrue((boolean)MailboxManagerTest.this.mailboxManager.hasCapability(MailboxManager.MailboxCapabilities.ACL));
            MailboxManagerTest.this.session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxPath inbox = MailboxPath.inbox((MailboxSession)MailboxManagerTest.this.session);
            MailboxId inboxId = (MailboxId)MailboxManagerTest.this.mailboxManager.createMailbox(inbox, MailboxManagerTest.this.session).get();
            MailboxPath otherMailbox = MailboxPath.forUser((Username)USER_1, (String)"otherMailbox");
            MailboxId otherMailboxId = (MailboxId)MailboxManagerTest.this.mailboxManager.createMailbox(otherMailbox, MailboxManagerTest.this.session).get();
            MessageManager inboxMessageManager = MailboxManagerTest.this.mailboxManager.getMailbox(inbox, MailboxManagerTest.this.session);
            ComposedMessageId composedMessageId1 = inboxMessageManager.appendMessage(MessageManager.AppendCommand.from((Message.Builder)MailboxManagerTest.this.message), MailboxManagerTest.this.session).getId();
            MessageId messageId2 = inboxMessageManager.appendMessage(MessageManager.AppendCommand.from((Message.Builder)MailboxManagerTest.this.message), MailboxManagerTest.this.session).getId().getMessageId();
            MessageId messageId1 = composedMessageId1.getMessageId();
            MailboxManagerTest.this.mailboxManager.copyMessages(MessageRange.one((MessageUid)composedMessageId1.getUid()), inbox, otherMailbox, MailboxManagerTest.this.session);
            MultimailboxesSearchQuery inboxQuery = MultimailboxesSearchQuery.from((SearchQuery)SearchQuery.matchAll()).inMailboxes(new MailboxId[]{inboxId}).build();
            MultimailboxesSearchQuery otherMailboxQuery = MultimailboxesSearchQuery.from((SearchQuery)SearchQuery.matchAll()).inMailboxes(new MailboxId[]{otherMailboxId}).build();
            Assertions.assertThat((List)((List)Flux.from((Publisher)MailboxManagerTest.this.mailboxManager.search(inboxQuery, MailboxManagerTest.this.session, 256L)).collectList().block())).containsExactly((Object[])new MessageId[]{messageId1, messageId2});
            Assertions.assertThat((List)((List)Flux.from((Publisher)MailboxManagerTest.this.mailboxManager.search(otherMailboxQuery, MailboxManagerTest.this.session, 256L)).collectList().block())).containsExactly((Object[])new MessageId[]{messageId1});
        }

        @Test
        void copyMessagesShouldUpdateMailboxCounts() throws Exception {
            Assumptions.assumeTrue((boolean)MailboxManagerTest.this.mailboxManager.hasCapability(MailboxManager.MailboxCapabilities.ACL));
            MailboxManagerTest.this.session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxPath inbox = MailboxPath.inbox((MailboxSession)MailboxManagerTest.this.session);
            MailboxId inboxId = (MailboxId)MailboxManagerTest.this.mailboxManager.createMailbox(inbox, MailboxManagerTest.this.session).get();
            MailboxPath otherMailbox = MailboxPath.forUser((Username)USER_1, (String)"otherMailbox");
            MailboxId otherMailboxId = (MailboxId)MailboxManagerTest.this.mailboxManager.createMailbox(otherMailbox, MailboxManagerTest.this.session).get();
            MessageManager inboxMessageManager = MailboxManagerTest.this.mailboxManager.getMailbox(inbox, MailboxManagerTest.this.session);
            ComposedMessageId composedMessageId1 = inboxMessageManager.appendMessage(MessageManager.AppendCommand.from((Message.Builder)MailboxManagerTest.this.message), MailboxManagerTest.this.session).getId();
            MessageId messageId2 = inboxMessageManager.appendMessage(MessageManager.AppendCommand.from((Message.Builder)MailboxManagerTest.this.message), MailboxManagerTest.this.session).getId().getMessageId();
            MessageId messageId1 = composedMessageId1.getMessageId();
            MailboxManagerTest.this.mailboxManager.copyMessages(MessageRange.all(), inbox, otherMailbox, MailboxManagerTest.this.session);
            Assertions.assertThat((long)MailboxManagerTest.this.mailboxManager.getMailbox(inboxId, MailboxManagerTest.this.session).getMessageCount(MailboxManagerTest.this.session)).isEqualTo(2L);
            Assertions.assertThat((long)MailboxManagerTest.this.mailboxManager.getMailbox(otherMailboxId, MailboxManagerTest.this.session).getMessageCount(MailboxManagerTest.this.session)).isEqualTo(2L);
        }

        @Test
        void moveMessagesShouldUpdateMailboxCounts() throws Exception {
            Assumptions.assumeTrue((boolean)MailboxManagerTest.this.mailboxManager.hasCapability(MailboxManager.MailboxCapabilities.ACL));
            MailboxManagerTest.this.session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxPath inbox = MailboxPath.inbox((MailboxSession)MailboxManagerTest.this.session);
            MailboxId inboxId = (MailboxId)MailboxManagerTest.this.mailboxManager.createMailbox(inbox, MailboxManagerTest.this.session).get();
            MailboxPath otherMailbox = MailboxPath.forUser((Username)USER_1, (String)"otherMailbox");
            MailboxId otherMailboxId = (MailboxId)MailboxManagerTest.this.mailboxManager.createMailbox(otherMailbox, MailboxManagerTest.this.session).get();
            MessageManager inboxMessageManager = MailboxManagerTest.this.mailboxManager.getMailbox(inbox, MailboxManagerTest.this.session);
            ComposedMessageId composedMessageId1 = inboxMessageManager.appendMessage(MessageManager.AppendCommand.from((Message.Builder)MailboxManagerTest.this.message), MailboxManagerTest.this.session).getId();
            MessageId messageId2 = inboxMessageManager.appendMessage(MessageManager.AppendCommand.from((Message.Builder)MailboxManagerTest.this.message), MailboxManagerTest.this.session).getId().getMessageId();
            MessageId messageId1 = composedMessageId1.getMessageId();
            MailboxManagerTest.this.mailboxManager.moveMessages(MessageRange.all(), inbox, otherMailbox, MailboxManagerTest.this.session);
            Assertions.assertThat((long)MailboxManagerTest.this.mailboxManager.getMailbox(inboxId, MailboxManagerTest.this.session).getMessageCount(MailboxManagerTest.this.session)).isEqualTo(0L);
            Assertions.assertThat((long)MailboxManagerTest.this.mailboxManager.getMailbox(otherMailboxId, MailboxManagerTest.this.session).getMessageCount(MailboxManagerTest.this.session)).isEqualTo(2L);
        }

        @Test
        void copyMessagesShouldThrowWhenCopyingMessageFromMailboxNotBelongingToSameUser() throws Exception {
            MailboxSession sessionUser1 = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxSession sessionUser2 = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_2);
            MailboxPath inbox1 = MailboxPath.inbox((MailboxSession)sessionUser1);
            MailboxManagerTest.this.mailboxManager.createMailbox(inbox1, sessionUser1);
            MailboxPath inbox2 = MailboxPath.inbox((MailboxSession)sessionUser2);
            MailboxManagerTest.this.mailboxManager.createMailbox(inbox2, sessionUser2);
            MessageManager inboxMessageManager = MailboxManagerTest.this.mailboxManager.getMailbox(inbox1, sessionUser1);
            inboxMessageManager.appendMessage(MessageManager.AppendCommand.from((Message.Builder)MailboxManagerTest.this.message), sessionUser1);
            Assertions.assertThatThrownBy(() -> MailboxManagerTest.this.mailboxManager.copyMessages(MessageRange.all(), inbox1, inbox2, sessionUser2)).isInstanceOf(MailboxNotFoundException.class);
        }

        @Test
        void copyMessagesShouldThrowWhenCopyingMessageToMailboxNotBelongingToSameUser() throws Exception {
            MailboxSession sessionUser1 = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxSession sessionUser2 = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_2);
            MailboxPath inbox1 = MailboxPath.inbox((MailboxSession)sessionUser1);
            MailboxManagerTest.this.mailboxManager.createMailbox(inbox1, sessionUser1);
            MailboxPath inbox2 = MailboxPath.inbox((MailboxSession)sessionUser2);
            MailboxManagerTest.this.mailboxManager.createMailbox(inbox2, sessionUser2);
            MessageManager inboxMessageManager = MailboxManagerTest.this.mailboxManager.getMailbox(inbox1, sessionUser1);
            inboxMessageManager.appendMessage(MessageManager.AppendCommand.from((Message.Builder)MailboxManagerTest.this.message), sessionUser1);
            Assertions.assertThatThrownBy(() -> MailboxManagerTest.this.mailboxManager.copyMessages(MessageRange.all(), inbox1, inbox2, sessionUser1)).isInstanceOf(MailboxNotFoundException.class);
        }

        @Test
        void createMailboxShouldNotThrowWhenMailboxPathBelongsToUser() throws MailboxException {
            MailboxManagerTest.this.session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            Optional mailboxId = MailboxManagerTest.this.mailboxManager.createMailbox(MailboxPath.forUser((Username)USER_1, (String)"mailboxName"), MailboxManagerTest.this.session);
            Assertions.assertThat((Optional)mailboxId).isNotEmpty();
        }

        @Test
        void createMailboxShouldThrowWhenMailboxPathBelongsToAnotherUser() {
            MailboxManagerTest.this.session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            Assertions.assertThatThrownBy(() -> MailboxManagerTest.this.mailboxManager.createMailbox(MailboxPath.forUser((Username)USER_2, (String)"mailboxName"), MailboxManagerTest.this.session)).isInstanceOf(InsufficientRightsException.class);
        }

        @Test
        void getMailboxShouldThrowWhenMailboxDoesNotExist() {
            MailboxManagerTest.this.session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            Assertions.assertThatThrownBy(() -> MailboxManagerTest.this.mailboxManager.getMailbox(MailboxPath.forUser((Username)USER_1, (String)"mailboxName"), MailboxManagerTest.this.session)).isInstanceOf(MailboxNotFoundException.class);
        }

        @Test
        void getMailboxByPathShouldReturnMailboxWhenBelongingToUser() throws Exception {
            MailboxManagerTest.this.session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxPath mailboxPath = MailboxPath.forUser((Username)USER_1, (String)"mailboxName");
            Optional mailboxId = MailboxManagerTest.this.mailboxManager.createMailbox(mailboxPath, MailboxManagerTest.this.session);
            Assertions.assertThat((Object)MailboxManagerTest.this.mailboxManager.getMailbox(mailboxPath, MailboxManagerTest.this.session).getId()).isEqualTo(mailboxId.get());
        }

        @Test
        protected void getMailboxByIdShouldReturnMailboxWhenBelongingToUser() throws Exception {
            MailboxManagerTest.this.session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxPath mailboxPath = MailboxPath.forUser((Username)USER_1, (String)"mailboxName");
            Optional mailboxId = MailboxManagerTest.this.mailboxManager.createMailbox(mailboxPath, MailboxManagerTest.this.session);
            Assertions.assertThat((Object)MailboxManagerTest.this.mailboxManager.getMailbox((MailboxId)mailboxId.get(), MailboxManagerTest.this.session).getId()).isEqualTo(mailboxId.get());
        }

        @Test
        void getMailboxByPathShouldThrowWhenMailboxNotBelongingToUser() throws Exception {
            MailboxSession sessionUser1 = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxSession sessionUser2 = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_2);
            MailboxPath mailboxPath = MailboxPath.forUser((Username)USER_1, (String)"mailboxName");
            MailboxManagerTest.this.mailboxManager.createMailbox(mailboxPath, sessionUser1);
            Assertions.assertThatThrownBy(() -> MailboxManagerTest.this.mailboxManager.getMailbox(mailboxPath, sessionUser2)).isInstanceOf(MailboxNotFoundException.class);
        }

        @Test
        void getMailboxByIdShouldThrowWhenMailboxNotBelongingToUser() throws Exception {
            MailboxSession sessionUser1 = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxSession sessionUser2 = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_2);
            MailboxPath mailboxPath = MailboxPath.forUser((Username)USER_1, (String)"mailboxName");
            Optional mailboxId = MailboxManagerTest.this.mailboxManager.createMailbox(mailboxPath, sessionUser1);
            Assertions.assertThatThrownBy(() -> MailboxManagerTest.this.mailboxManager.getMailbox((MailboxId)mailboxId.get(), sessionUser2)).isInstanceOf(MailboxNotFoundException.class);
        }
    }

    @Nested
    class MetadataTests {
        MetadataTests() {
        }

        @Test
        void getMailboxCountersShouldReturnDefaultValueWhenNoReadRight() throws Exception {
            Assumptions.assumeTrue((boolean)MailboxManagerTest.this.mailboxManager.hasCapability(MailboxManager.MailboxCapabilities.ACL));
            MailboxSession session1 = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxSession session2 = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_2);
            MailboxPath inbox1 = MailboxPath.inbox((MailboxSession)session1);
            Optional mailboxIdInbox1 = MailboxManagerTest.this.mailboxManager.createMailbox(inbox1, session1);
            MailboxManagerTest.this.mailboxManager.setRights(inbox1, MailboxACL.EMPTY.apply(MailboxACL.command().forUser(USER_2).rights(new MailboxACL.Right[]{MailboxACL.Right.Lookup}).asAddition()), session1);
            MailboxManagerTest.this.mailboxManager.getMailbox(inbox1, session1).appendMessage(MessageManager.AppendCommand.from((Message.Builder)MailboxManagerTest.this.message), session1);
            MailboxCounters mailboxCounters = MailboxManagerTest.this.mailboxManager.getMailbox(inbox1, session2).getMailboxCounters(session2);
            Assertions.assertThat((Object)mailboxCounters).isEqualTo((Object)MailboxCounters.builder().mailboxId((MailboxId)mailboxIdInbox1.get()).count(0L).unseen(0L).build());
        }

        @Test
        void getMailboxCountersShouldReturnStoredValueWhenReadRight() throws Exception {
            Assumptions.assumeTrue((boolean)MailboxManagerTest.this.mailboxManager.hasCapability(MailboxManager.MailboxCapabilities.ACL));
            MailboxSession session1 = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxSession session2 = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_2);
            MailboxPath inbox1 = MailboxPath.inbox((MailboxSession)session1);
            Optional mailboxIdInbox1 = MailboxManagerTest.this.mailboxManager.createMailbox(inbox1, session1);
            MailboxManagerTest.this.mailboxManager.setRights(inbox1, MailboxACL.EMPTY.apply(MailboxACL.command().forUser(USER_2).rights(new MailboxACL.Right[]{MailboxACL.Right.Lookup, MailboxACL.Right.Read}).asAddition()), session1);
            MailboxManagerTest.this.mailboxManager.getMailbox(inbox1, session1).appendMessage(MessageManager.AppendCommand.builder().recent().build(MailboxManagerTest.this.message), session1);
            MailboxCounters mailboxCounters = MailboxManagerTest.this.mailboxManager.getMailbox(inbox1, session2).getMailboxCounters(session2);
            Assertions.assertThat((Object)mailboxCounters).isEqualTo((Object)MailboxCounters.builder().mailboxId((MailboxId)mailboxIdInbox1.get()).count(1L).unseen(1L).build());
        }

        @Test
        void getMetaDataShouldReturnDefaultValueWhenNoReadRight() throws Exception {
            Assumptions.assumeTrue((boolean)MailboxManagerTest.this.mailboxManager.hasCapability(MailboxManager.MailboxCapabilities.ACL));
            MailboxSession session1 = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxSession session2 = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_2);
            MailboxPath inbox1 = MailboxPath.inbox((MailboxSession)session1);
            MailboxManagerTest.this.mailboxManager.createMailbox(inbox1, session1);
            MailboxManagerTest.this.mailboxManager.setRights(inbox1, MailboxACL.EMPTY.apply(MailboxACL.command().forUser(USER_2).rights(new MailboxACL.Right[]{MailboxACL.Right.Lookup}).asAddition()), session1);
            MailboxManagerTest.this.mailboxManager.getMailbox(inbox1, session1).appendMessage(MessageManager.AppendCommand.builder().recent().build(MailboxManagerTest.this.message), session1);
            boolean resetRecent = false;
            MessageManager.MailboxMetaData metaData = MailboxManagerTest.this.mailboxManager.getMailbox(inbox1, session2).getMetaData(resetRecent, session2, MessageManager.MailboxMetaData.FetchGroup.UNSEEN_COUNT);
            SoftAssertions.assertSoftly(softly -> {
                softly.assertThat((Object)metaData).extracting(MessageManager.MailboxMetaData::getHighestModSeq).isEqualTo((Object)ModSeq.first());
                softly.assertThat((Object)metaData).extracting(MessageManager.MailboxMetaData::getUidNext).isEqualTo((Object)MessageUid.MIN_VALUE);
                softly.assertThat((Object)metaData).extracting(MessageManager.MailboxMetaData::getMessageCount).isEqualTo((Object)0L);
                softly.assertThat((Object)metaData).extracting(MessageManager.MailboxMetaData::getUnseenCount).isEqualTo((Object)0L);
                softly.assertThat((Object)metaData).extracting(MessageManager.MailboxMetaData::getRecent).isEqualTo((Object)ImmutableList.of());
                softly.assertThat((Object)metaData).extracting(MessageManager.MailboxMetaData::getPermanentFlags).isEqualTo((Object)new Flags());
            });
        }
    }

    @Nested
    class SearchTests {
        SearchTests() {
        }

        @Test
        void searchShouldNotReturnResultsFromOtherNamespaces() throws Exception {
            Assumptions.assumeTrue((boolean)MailboxManagerTest.this.mailboxManager.hasCapability(MailboxManager.MailboxCapabilities.Namespace));
            MailboxManagerTest.this.session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxManagerTest.this.mailboxManager.createMailbox(new MailboxPath("other_namespace", USER_1, "Other"), MailboxManagerTest.this.session);
            MailboxManagerTest.this.mailboxManager.createMailbox(MailboxPath.inbox((MailboxSession)MailboxManagerTest.this.session), MailboxManagerTest.this.session);
            List metaDatas = (List)MailboxManagerTest.this.mailboxManager.search(MailboxQuery.privateMailboxesBuilder((MailboxSession)MailboxManagerTest.this.session).matchesAllMailboxNames().build(), MailboxManagerTest.this.session).collectList().block();
            Assertions.assertThat((List)metaDatas).hasSize(1);
            Assertions.assertThat((Object)((MailboxMetaData)metaDatas.get(0)).getPath()).isEqualTo((Object)MailboxPath.inbox((MailboxSession)MailboxManagerTest.this.session));
        }

        @Test
        void searchShouldReturnACL() throws Exception {
            Assumptions.assumeTrue((boolean)MailboxManagerTest.this.mailboxManager.hasCapability(MailboxManager.MailboxCapabilities.Namespace));
            Assumptions.assumeTrue((boolean)MailboxManagerTest.this.mailboxManager.hasCapability(MailboxManager.MailboxCapabilities.ACL));
            MailboxManagerTest.this.session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            Optional inboxId = MailboxManagerTest.this.mailboxManager.createMailbox(MailboxPath.inbox((MailboxSession)MailboxManagerTest.this.session), MailboxManagerTest.this.session);
            MailboxACL acl = MailboxACL.EMPTY.apply(MailboxACL.command().forUser(USER_2).rights(new MailboxACL.Right[]{MailboxACL.Right.Read, MailboxACL.Right.Lookup}).asAddition());
            MailboxManagerTest.this.mailboxManager.setRights((MailboxId)inboxId.get(), acl, MailboxManagerTest.this.session);
            List metaDatas = (List)MailboxManagerTest.this.mailboxManager.search(MailboxQuery.privateMailboxesBuilder((MailboxSession)MailboxManagerTest.this.session).matchesAllMailboxNames().build(), MailboxManagerTest.this.session).collectList().block();
            ((ObjectAssert)((ListAssert)Assertions.assertThat((List)metaDatas).hasSize(1)).first()).extracting(MailboxMetaData::getResolvedAcls).isEqualTo((Object)acl.apply(MailboxACL.command().forOwner().rights(MailboxACL.Rfc4314Rights.allExcept((MailboxACL.Right[])new MailboxACL.Right[0])).asAddition()));
        }

        @Test
        void searchShouldNotReturnResultsFromOtherUsers() throws Exception {
            MailboxManagerTest.this.session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxSession session2 = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_2);
            MailboxManagerTest.this.mailboxManager.createMailbox(MailboxPath.forUser((Username)USER_2, (String)"Other"), session2);
            MailboxManagerTest.this.mailboxManager.createMailbox(MailboxPath.inbox((MailboxSession)MailboxManagerTest.this.session), MailboxManagerTest.this.session);
            List metaDatas = (List)MailboxManagerTest.this.mailboxManager.search(MailboxQuery.privateMailboxesBuilder((MailboxSession)MailboxManagerTest.this.session).matchesAllMailboxNames().build(), MailboxManagerTest.this.session).collectList().block();
            Assertions.assertThat((List)metaDatas).hasSize(1);
            Assertions.assertThat((Object)((MailboxMetaData)metaDatas.get(0)).getPath()).isEqualTo((Object)MailboxPath.inbox((MailboxSession)MailboxManagerTest.this.session));
        }

        @Test
        void searchShouldNotDuplicateMailboxWhenReportedAsUserMailboxesAndUserHasRightOnMailboxes() throws Exception {
            Assumptions.assumeTrue((boolean)MailboxManagerTest.this.mailboxManager.hasCapability(MailboxManager.MailboxCapabilities.ACL));
            MailboxSession session1 = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxPath inbox1 = MailboxPath.inbox((MailboxSession)session1);
            MailboxManagerTest.this.mailboxManager.createMailbox(inbox1, session1);
            MailboxManagerTest.this.mailboxManager.setRights(inbox1, MailboxACL.EMPTY.apply(MailboxACL.command().forUser(USER_1).rights(new MailboxACL.Right[]{MailboxACL.Right.Read, MailboxACL.Right.Lookup}).asAddition()), session1);
            MailboxQuery mailboxQuery = MailboxQuery.builder().matchesAllMailboxNames().build();
            ((AbstractListAssert)Assertions.assertThat((Stream)MailboxManagerTest.this.mailboxManager.search(mailboxQuery, session1).toStream()).extracting(MailboxMetaData::getPath).hasSize(1)).containsOnly((Object[])new MailboxPath[]{inbox1});
        }

        @Test
        void searchShouldIncludeDelegatedMailboxes() throws Exception {
            Assumptions.assumeTrue((boolean)MailboxManagerTest.this.mailboxManager.hasCapability(MailboxManager.MailboxCapabilities.ACL));
            MailboxSession session1 = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxSession session2 = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_2);
            MailboxPath inbox1 = MailboxPath.inbox((MailboxSession)session1);
            Optional mailboxIdInbox1 = MailboxManagerTest.this.mailboxManager.createMailbox(inbox1, session1);
            MailboxManagerTest.this.mailboxManager.setRights(inbox1, MailboxACL.EMPTY.apply(MailboxACL.command().forUser(USER_2).rights(new MailboxACL.Right[]{MailboxACL.Right.Lookup}).asAddition()), session1);
            MailboxQuery mailboxQuery = MailboxQuery.builder().matchesAllMailboxNames().build();
            Assertions.assertThat((Stream)MailboxManagerTest.this.mailboxManager.search(mailboxQuery, session2).toStream()).extracting(MailboxMetaData::getPath).containsOnly((Object[])new MailboxPath[]{inbox1});
        }

        @Test
        void searchShouldCombinePrivateAndDelegatedMailboxes() throws Exception {
            Assumptions.assumeTrue((boolean)MailboxManagerTest.this.mailboxManager.hasCapability(MailboxManager.MailboxCapabilities.ACL));
            MailboxSession session1 = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxSession session2 = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_2);
            MailboxPath inbox1 = MailboxPath.inbox((MailboxSession)session1);
            MailboxPath inbox2 = MailboxPath.inbox((MailboxSession)session2);
            MailboxManagerTest.this.mailboxManager.createMailbox(inbox1, session1);
            MailboxManagerTest.this.mailboxManager.createMailbox(inbox2, session2);
            MailboxManagerTest.this.mailboxManager.setRights(inbox1, MailboxACL.EMPTY.apply(MailboxACL.command().forUser(USER_2).rights(new MailboxACL.Right[]{MailboxACL.Right.Read, MailboxACL.Right.Lookup}).asAddition()), session1);
            MailboxQuery mailboxQuery = MailboxQuery.builder().matchesAllMailboxNames().build();
            Assertions.assertThat((Stream)MailboxManagerTest.this.mailboxManager.search(mailboxQuery, session2).toStream()).extracting(MailboxMetaData::getPath).containsOnly((Object[])new MailboxPath[]{inbox1, inbox2});
        }

        @Test
        void searchShouldAllowUserFiltering() throws Exception {
            Assumptions.assumeTrue((boolean)MailboxManagerTest.this.mailboxManager.hasCapability(MailboxManager.MailboxCapabilities.ACL));
            MailboxSession session1 = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxSession session2 = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_2);
            MailboxPath inbox1 = MailboxPath.inbox((MailboxSession)session1);
            MailboxPath inbox2 = MailboxPath.inbox((MailboxSession)session2);
            MailboxManagerTest.this.mailboxManager.createMailbox(inbox1, session1);
            MailboxManagerTest.this.mailboxManager.createMailbox(inbox2, session2);
            MailboxManagerTest.this.mailboxManager.setRights(inbox1, MailboxACL.EMPTY.apply(MailboxACL.command().forUser(USER_2).rights(new MailboxACL.Right[]{MailboxACL.Right.Read, MailboxACL.Right.Lookup}).asAddition()), session1);
            MailboxQuery mailboxQuery = MailboxQuery.builder().username(USER_1).matchesAllMailboxNames().build();
            Assertions.assertThat((Stream)MailboxManagerTest.this.mailboxManager.search(mailboxQuery, session2).toStream()).extracting(MailboxMetaData::getPath).containsOnly((Object[])new MailboxPath[]{inbox1});
        }

        @Test
        void searchShouldAllowNamespaceFiltering() throws Exception {
            Assumptions.assumeTrue((boolean)MailboxManagerTest.this.mailboxManager.hasCapability(MailboxManager.MailboxCapabilities.ACL));
            MailboxSession session1 = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxSession session2 = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_2);
            MailboxPath inbox1 = MailboxPath.inbox((MailboxSession)session1);
            String specificNamespace = "specificNamespace";
            MailboxPath mailboxPath1 = new MailboxPath(specificNamespace, USER_1, "mailbox");
            MailboxManagerTest.this.mailboxManager.createMailbox(inbox1, session1);
            MailboxManagerTest.this.mailboxManager.createMailbox(mailboxPath1, session1);
            MailboxManagerTest.this.mailboxManager.setRights(inbox1, MailboxACL.EMPTY.apply(MailboxACL.command().forUser(USER_2).rights(new MailboxACL.Right[]{MailboxACL.Right.Read, MailboxACL.Right.Lookup}).asAddition()), session1);
            MailboxManagerTest.this.mailboxManager.setRights(mailboxPath1, MailboxACL.EMPTY.apply(MailboxACL.command().forUser(USER_2).rights(new MailboxACL.Right[]{MailboxACL.Right.Read, MailboxACL.Right.Lookup}).asAddition()), session1);
            MailboxQuery mailboxQuery = MailboxQuery.builder().namespace(specificNamespace).matchesAllMailboxNames().build();
            Assertions.assertThat((Stream)MailboxManagerTest.this.mailboxManager.search(mailboxQuery, session2).toStream()).extracting(MailboxMetaData::getPath).containsOnly((Object[])new MailboxPath[]{mailboxPath1});
        }

        @Test
        void searchForMessageShouldReturnMessagesFromAllMyMailboxesIfNoMailboxesAreSpecified() throws Exception {
            Assumptions.assumeTrue((boolean)MailboxManagerTest.this.mailboxManager.getSupportedMessageCapabilities().contains(MailboxManager.MessageCapabilities.UniqueID));
            MailboxManagerTest.this.session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxPath cacahueteFolder = MailboxPath.forUser((Username)USER_1, (String)"CACAHUETE");
            MailboxId cacahueteMailboxId = (MailboxId)MailboxManagerTest.this.mailboxManager.createMailbox(cacahueteFolder, MailboxManagerTest.this.session).get();
            MessageManager cacahueteMessageManager = MailboxManagerTest.this.mailboxManager.getMailbox(cacahueteMailboxId, MailboxManagerTest.this.session);
            MessageId cacahueteMessageId = cacahueteMessageManager.appendMessage(MessageManager.AppendCommand.from((Message.Builder)MailboxManagerTest.this.message), MailboxManagerTest.this.session).getId().getMessageId();
            MailboxPath pirouetteFilder = MailboxPath.forUser((Username)USER_1, (String)"PIROUETTE");
            MailboxId pirouetteMailboxId = (MailboxId)MailboxManagerTest.this.mailboxManager.createMailbox(pirouetteFilder, MailboxManagerTest.this.session).get();
            MessageManager pirouetteMessageManager = MailboxManagerTest.this.mailboxManager.getMailbox(pirouetteMailboxId, MailboxManagerTest.this.session);
            MessageId pirouetteMessageId = pirouetteMessageManager.appendMessage(MessageManager.AppendCommand.from((Message.Builder)MailboxManagerTest.this.message), MailboxManagerTest.this.session).getId().getMessageId();
            MultimailboxesSearchQuery multiMailboxesQuery = MultimailboxesSearchQuery.from((SearchQuery)SearchQuery.matchAll()).build();
            Assertions.assertThat((List)((List)Flux.from((Publisher)MailboxManagerTest.this.mailboxManager.search(multiMailboxesQuery, MailboxManagerTest.this.session, 256L)).collectList().block())).containsOnly((Object[])new MessageId[]{cacahueteMessageId, pirouetteMessageId});
        }

        @Test
        void searchForMessageShouldReturnMessagesFromMyDelegatedMailboxes() throws Exception {
            Assumptions.assumeTrue((boolean)MailboxManagerTest.this.mailboxManager.hasCapability(MailboxManager.MailboxCapabilities.ACL));
            MailboxManagerTest.this.session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxSession sessionFromDelegater = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_2);
            MailboxPath delegatedMailboxPath = MailboxPath.forUser((Username)USER_2, (String)"SHARED");
            MailboxId delegatedMailboxId = (MailboxId)MailboxManagerTest.this.mailboxManager.createMailbox(delegatedMailboxPath, sessionFromDelegater).get();
            MessageManager delegatedMessageManager = MailboxManagerTest.this.mailboxManager.getMailbox(delegatedMailboxId, sessionFromDelegater);
            MessageId messageId = delegatedMessageManager.appendMessage(MessageManager.AppendCommand.from((Message.Builder)MailboxManagerTest.this.message), sessionFromDelegater).getId().getMessageId();
            MailboxManagerTest.this.mailboxManager.setRights(delegatedMailboxPath, MailboxACL.EMPTY.apply(MailboxACL.command().forUser(USER_1).rights(new MailboxACL.Right[]{MailboxACL.Right.Read, MailboxACL.Right.Lookup}).asAddition()), sessionFromDelegater);
            MultimailboxesSearchQuery multiMailboxesQuery = MultimailboxesSearchQuery.from((SearchQuery)SearchQuery.matchAll()).build();
            Assertions.assertThat((List)((List)Flux.from((Publisher)MailboxManagerTest.this.mailboxManager.search(multiMailboxesQuery, MailboxManagerTest.this.session, 256L)).collectList().block())).containsOnly((Object[])new MessageId[]{messageId});
        }

        @Test
        void searchForMessageShouldReturnMessagesFromMyDelegatedMailboxesWhenAccessibleNamespace() throws Exception {
            Assumptions.assumeTrue((boolean)MailboxManagerTest.this.mailboxManager.hasCapability(MailboxManager.MailboxCapabilities.ACL));
            MailboxManagerTest.this.session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxSession sessionFromDelegater = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_2);
            MailboxPath delegatedMailboxPath = MailboxPath.forUser((Username)USER_2, (String)"SHARED");
            MailboxId delegatedMailboxId = (MailboxId)MailboxManagerTest.this.mailboxManager.createMailbox(delegatedMailboxPath, sessionFromDelegater).get();
            MessageManager delegatedMessageManager = MailboxManagerTest.this.mailboxManager.getMailbox(delegatedMailboxId, sessionFromDelegater);
            MessageId messageId = delegatedMessageManager.appendMessage(MessageManager.AppendCommand.from((Message.Builder)MailboxManagerTest.this.message), sessionFromDelegater).getId().getMessageId();
            MailboxManagerTest.this.mailboxManager.setRights(delegatedMailboxPath, MailboxACL.EMPTY.apply(MailboxACL.command().forUser(USER_1).rights(new MailboxACL.Right[]{MailboxACL.Right.Read, MailboxACL.Right.Lookup}).asAddition()), sessionFromDelegater);
            MultimailboxesSearchQuery multiMailboxesQuery = MultimailboxesSearchQuery.from((SearchQuery)SearchQuery.matchAll()).inNamespace((MultimailboxesSearchQuery.Namespace)new MultimailboxesSearchQuery.AccessibleNamespace()).build();
            Assertions.assertThat((List)((List)Flux.from((Publisher)MailboxManagerTest.this.mailboxManager.search(multiMailboxesQuery, MailboxManagerTest.this.session, 256L)).collectList().block())).containsOnly((Object[])new MessageId[]{messageId});
        }

        @Test
        void searchForMessageShouldNotReturnMessagesFromMyDelegatedMailboxesICanNotRead() throws Exception {
            Assumptions.assumeTrue((boolean)MailboxManagerTest.this.mailboxManager.hasCapability(MailboxManager.MailboxCapabilities.ACL));
            MailboxManagerTest.this.session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxSession sessionFromDelegater = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_2);
            MailboxPath delegatedMailboxPath = MailboxPath.forUser((Username)USER_2, (String)"SHARED");
            MailboxId delegatedMailboxId = (MailboxId)MailboxManagerTest.this.mailboxManager.createMailbox(delegatedMailboxPath, sessionFromDelegater).get();
            MessageManager delegatedMessageManager = MailboxManagerTest.this.mailboxManager.getMailbox(delegatedMailboxId, sessionFromDelegater);
            delegatedMessageManager.appendMessage(MessageManager.AppendCommand.from((Message.Builder)MailboxManagerTest.this.message), sessionFromDelegater);
            MailboxManagerTest.this.mailboxManager.setRights(delegatedMailboxPath, MailboxACL.EMPTY.apply(MailboxACL.command().forUser(USER_1).rights(new MailboxACL.Right[]{MailboxACL.Right.Lookup}).asAddition()), sessionFromDelegater);
            MultimailboxesSearchQuery multiMailboxesQuery = MultimailboxesSearchQuery.from((SearchQuery)SearchQuery.matchAll()).build();
            Assertions.assertThat((List)((List)Flux.from((Publisher)MailboxManagerTest.this.mailboxManager.search(multiMailboxesQuery, MailboxManagerTest.this.session, 256L)).collectList().block())).isEmpty();
        }

        @Test
        void searchForMessageShouldOnlySearchInMailboxICanRead() throws Exception {
            Assumptions.assumeTrue((boolean)MailboxManagerTest.this.mailboxManager.hasCapability(MailboxManager.MailboxCapabilities.ACL));
            MailboxManagerTest.this.session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxSession sessionFromDelegater = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_2);
            MailboxPath otherMailboxPath = MailboxPath.forUser((Username)USER_2, (String)"OTHER_MAILBOX");
            MailboxId otherMailboxId = (MailboxId)MailboxManagerTest.this.mailboxManager.createMailbox(otherMailboxPath, sessionFromDelegater).get();
            MessageManager otherMailboxManager = MailboxManagerTest.this.mailboxManager.getMailbox(otherMailboxId, sessionFromDelegater);
            otherMailboxManager.appendMessage(MessageManager.AppendCommand.from((Message.Builder)MailboxManagerTest.this.message), sessionFromDelegater);
            MultimailboxesSearchQuery multiMailboxesQuery = MultimailboxesSearchQuery.from((SearchQuery)SearchQuery.matchAll()).build();
            Assertions.assertThat((List)((List)Flux.from((Publisher)MailboxManagerTest.this.mailboxManager.search(multiMailboxesQuery, MailboxManagerTest.this.session, 256L)).collectList().block())).isEmpty();
        }

        @Test
        void searchForMessageShouldIgnoreMailboxThatICanNotRead() throws Exception {
            Assumptions.assumeTrue((boolean)MailboxManagerTest.this.mailboxManager.hasCapability(MailboxManager.MailboxCapabilities.ACL));
            MailboxManagerTest.this.session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxSession sessionFromDelegater = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_2);
            MailboxPath otherMailboxPath = MailboxPath.forUser((Username)USER_2, (String)"SHARED");
            MailboxId otherMailboxId = (MailboxId)MailboxManagerTest.this.mailboxManager.createMailbox(otherMailboxPath, sessionFromDelegater).get();
            MessageManager otherMessageManager = MailboxManagerTest.this.mailboxManager.getMailbox(otherMailboxId, sessionFromDelegater);
            otherMessageManager.appendMessage(MessageManager.AppendCommand.from((Message.Builder)MailboxManagerTest.this.message), sessionFromDelegater);
            MultimailboxesSearchQuery multiMailboxesQuery = MultimailboxesSearchQuery.from((SearchQuery)SearchQuery.matchAll()).inMailboxes(new MailboxId[]{otherMailboxId}).build();
            Assertions.assertThat((List)((List)Flux.from((Publisher)MailboxManagerTest.this.mailboxManager.search(multiMailboxesQuery, MailboxManagerTest.this.session, 256L)).collectList().block())).isEmpty();
        }

        @Test
        void searchForMessageShouldCorrectlyExcludeMailbox() throws Exception {
            Assumptions.assumeTrue((boolean)MailboxManagerTest.this.mailboxManager.hasCapability(MailboxManager.MailboxCapabilities.ACL));
            MailboxManagerTest.this.session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxPath otherMailboxPath = MailboxPath.forUser((Username)USER_1, (String)"SHARED");
            MailboxId otherMailboxId = (MailboxId)MailboxManagerTest.this.mailboxManager.createMailbox(otherMailboxPath, MailboxManagerTest.this.session).get();
            MessageManager otherMessageManager = MailboxManagerTest.this.mailboxManager.getMailbox(otherMailboxId, MailboxManagerTest.this.session);
            otherMessageManager.appendMessage(MessageManager.AppendCommand.from((Message.Builder)MailboxManagerTest.this.message), MailboxManagerTest.this.session);
            MultimailboxesSearchQuery multiMailboxesQuery = MultimailboxesSearchQuery.from((SearchQuery)SearchQuery.matchAll()).notInMailboxes(new MailboxId[]{otherMailboxId}).build();
            Assertions.assertThat((List)((List)Flux.from((Publisher)MailboxManagerTest.this.mailboxManager.search(multiMailboxesQuery, MailboxManagerTest.this.session, 256L)).collectList().block())).isEmpty();
        }

        @Test
        void searchForMessageShouldPriorizeExclusionFromInclusion() throws Exception {
            Assumptions.assumeTrue((boolean)MailboxManagerTest.this.mailboxManager.hasCapability(MailboxManager.MailboxCapabilities.ACL));
            MailboxManagerTest.this.session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxPath otherMailboxPath = MailboxPath.forUser((Username)USER_1, (String)"SHARED");
            MailboxId otherMailboxId = (MailboxId)MailboxManagerTest.this.mailboxManager.createMailbox(otherMailboxPath, MailboxManagerTest.this.session).get();
            MessageManager otherMessageManager = MailboxManagerTest.this.mailboxManager.getMailbox(otherMailboxId, MailboxManagerTest.this.session);
            otherMessageManager.appendMessage(MessageManager.AppendCommand.from((Message.Builder)MailboxManagerTest.this.message), MailboxManagerTest.this.session);
            MultimailboxesSearchQuery multiMailboxesQuery = MultimailboxesSearchQuery.from((SearchQuery)SearchQuery.matchAll()).inMailboxes(new MailboxId[]{otherMailboxId}).notInMailboxes(new MailboxId[]{otherMailboxId}).build();
            Assertions.assertThat((List)((List)Flux.from((Publisher)MailboxManagerTest.this.mailboxManager.search(multiMailboxesQuery, MailboxManagerTest.this.session, 256L)).collectList().block())).isEmpty();
        }

        @Test
        void searchShouldRestrictResultsToTheSuppliedUserNamespace() throws Exception {
            Assumptions.assumeTrue((boolean)MailboxManagerTest.this.mailboxManager.hasCapability(MailboxManager.MailboxCapabilities.ACL));
            MailboxManagerTest.this.session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxSession sessionFromDelegater = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_2);
            MailboxPath delegatedMailboxPath = MailboxPath.forUser((Username)USER_2, (String)"SHARED");
            MailboxId delegatedMailboxId = (MailboxId)MailboxManagerTest.this.mailboxManager.createMailbox(delegatedMailboxPath, sessionFromDelegater).get();
            MessageManager delegatedMessageManager = MailboxManagerTest.this.mailboxManager.getMailbox(delegatedMailboxId, sessionFromDelegater);
            MessageId messageId = delegatedMessageManager.appendMessage(MessageManager.AppendCommand.from((Message.Builder)MailboxManagerTest.this.message), sessionFromDelegater).getId().getMessageId();
            MailboxManagerTest.this.mailboxManager.setRights(delegatedMailboxPath, MailboxACL.EMPTY.apply(MailboxACL.command().forUser(USER_1).rights(new MailboxACL.Right[]{MailboxACL.Right.Read, MailboxACL.Right.Lookup}).asAddition()), sessionFromDelegater);
            MultimailboxesSearchQuery multiMailboxesQuery = MultimailboxesSearchQuery.from((SearchQuery)SearchQuery.matchAll()).inNamespace((MultimailboxesSearchQuery.Namespace)new MultimailboxesSearchQuery.PersonalNamespace(MailboxManagerTest.this.session)).build();
            Assertions.assertThat((List)((List)Flux.from((Publisher)MailboxManagerTest.this.mailboxManager.search(multiMailboxesQuery, MailboxManagerTest.this.session, 256L)).collectList().block())).isEmpty();
        }

        @Test
        void searchForMessageShouldOnlySearchInGivenMailbox() throws Exception {
            Assumptions.assumeTrue((boolean)MailboxManagerTest.this.mailboxManager.hasCapability(MailboxManager.MailboxCapabilities.ACL));
            MailboxManagerTest.this.session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxPath searchedMailboxPath = MailboxPath.forUser((Username)USER_1, (String)"WANTED");
            MailboxId searchedMailboxId = (MailboxId)MailboxManagerTest.this.mailboxManager.createMailbox(searchedMailboxPath, MailboxManagerTest.this.session).get();
            MessageManager searchedMessageManager = MailboxManagerTest.this.mailboxManager.getMailbox(searchedMailboxId, MailboxManagerTest.this.session);
            MailboxPath otherMailboxPath = MailboxPath.forUser((Username)USER_1, (String)"SHARED");
            MailboxId otherMailboxId = (MailboxId)MailboxManagerTest.this.mailboxManager.createMailbox(otherMailboxPath, MailboxManagerTest.this.session).get();
            MessageManager otherMessageManager = MailboxManagerTest.this.mailboxManager.getMailbox(otherMailboxId, MailboxManagerTest.this.session);
            otherMessageManager.appendMessage(MessageManager.AppendCommand.from((Message.Builder)MailboxManagerTest.this.message), MailboxManagerTest.this.session);
            MessageId messageId = searchedMessageManager.appendMessage(MessageManager.AppendCommand.from((Message.Builder)MailboxManagerTest.this.message), MailboxManagerTest.this.session).getId().getMessageId();
            MultimailboxesSearchQuery multiMailboxesQuery = MultimailboxesSearchQuery.from((SearchQuery)SearchQuery.matchAll()).inMailboxes(new MailboxId[]{searchedMailboxId}).build();
            Assertions.assertThat((List)((List)Flux.from((Publisher)MailboxManagerTest.this.mailboxManager.search(multiMailboxesQuery, MailboxManagerTest.this.session, 256L)).collectList().block())).containsExactly((Object[])new MessageId[]{messageId});
        }

        @Test
        void searchShouldNotReturnNoMoreDelegatedMailboxes() throws Exception {
            Assumptions.assumeTrue((boolean)MailboxManagerTest.this.mailboxManager.hasCapability(MailboxManager.MailboxCapabilities.ACL));
            MailboxSession session1 = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxSession session2 = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_2);
            MailboxPath inbox1 = MailboxPath.inbox((MailboxSession)session1);
            Optional mailboxIdInbox1 = MailboxManagerTest.this.mailboxManager.createMailbox(inbox1, session1);
            MailboxManagerTest.this.mailboxManager.setRights(inbox1, MailboxACL.EMPTY.apply(MailboxACL.command().forUser(USER_2).rights(new MailboxACL.Right[]{MailboxACL.Right.Read, MailboxACL.Right.Lookup}).asAddition()), session1);
            MailboxManagerTest.this.mailboxManager.setRights(inbox1, MailboxACL.EMPTY.apply(MailboxACL.command().forUser(USER_2).rights(new MailboxACL.Right[]{MailboxACL.Right.Read, MailboxACL.Right.Lookup}).asRemoval()), session1);
            MailboxQuery mailboxQuery = MailboxQuery.builder().matchesAllMailboxNames().build();
            Assertions.assertThat((Stream)MailboxManagerTest.this.mailboxManager.search(mailboxQuery, session2).toStream()).isEmpty();
        }
    }

    @Nested
    class EventTests {
        private final QuotaRoot quotaRoot = QuotaRoot.quotaRoot((String)"#private&user_1", Optional.empty());
        private EventCollector listener;
        private MailboxPath inbox;
        private MailboxId inboxId;
        private MessageManager inboxManager;
        private MailboxPath newPath;

        EventTests() {
        }

        @BeforeEach
        void setUp() throws Exception {
            MailboxManagerTest.this.session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            this.inbox = MailboxPath.inbox((MailboxSession)MailboxManagerTest.this.session);
            this.newPath = MailboxPath.forUser((Username)USER_1, (String)"specialMailbox");
            this.listener = new EventCollector();
            this.inboxId = (MailboxId)MailboxManagerTest.this.mailboxManager.createMailbox(this.inbox, MailboxManagerTest.this.session).get();
            this.inboxManager = MailboxManagerTest.this.mailboxManager.getMailbox(this.inbox, MailboxManagerTest.this.session);
        }

        @Test
        void deleteMailboxShouldFireMailboxDeletionEvent() throws Exception {
            Assumptions.assumeTrue((boolean)MailboxManagerTest.this.mailboxManager.hasCapability(MailboxManager.MailboxCapabilities.Quota));
            Mono.from((Publisher)MailboxManagerTest.this.retrieveEventBus(MailboxManagerTest.this.mailboxManager).register((EventListener)this.listener, (RegistrationKey)new MailboxIdRegistrationKey(this.inboxId))).block();
            MailboxManagerTest.this.mailboxManager.deleteMailbox(this.inbox, MailboxManagerTest.this.session);
            ((ObjectAssert)((ObjectAssert)((ObjectAssert)((ObjectAssert)((IterableAssert)((IterableAssert)Assertions.assertThat(this.listener.getEvents()).filteredOn(event -> event instanceof MailboxEvents.MailboxDeletion)).hasSize(1)).extracting(event -> (MailboxEvents.MailboxDeletion)event).element(0)).satisfies(event -> Assertions.assertThat((Object)event.getMailboxId()).isEqualTo((Object)this.inboxId))).satisfies(event -> Assertions.assertThat((Object)event.getQuotaRoot()).isEqualTo((Object)this.quotaRoot))).satisfies(event -> Assertions.assertThat((Object)event.getDeletedMessageCount()).isEqualTo((Object)QuotaCountUsage.count((long)0L)))).satisfies(event -> Assertions.assertThat((Object)event.getTotalDeletedSize()).isEqualTo((Object)QuotaSizeUsage.size((long)0L)));
        }

        @Test
        void deleteMailboxByIdShouldFireMailboxDeletionEvent() throws Exception {
            Assumptions.assumeTrue((boolean)MailboxManagerTest.this.mailboxManager.hasCapability(MailboxManager.MailboxCapabilities.Quota));
            Mono.from((Publisher)MailboxManagerTest.this.retrieveEventBus(MailboxManagerTest.this.mailboxManager).register((EventListener)this.listener, (RegistrationKey)new MailboxIdRegistrationKey(this.inboxId))).block();
            MailboxManagerTest.this.mailboxManager.deleteMailbox(this.inboxId, MailboxManagerTest.this.session);
            ((ObjectAssert)((ObjectAssert)((ObjectAssert)((ObjectAssert)((IterableAssert)((IterableAssert)Assertions.assertThat(this.listener.getEvents()).filteredOn(event -> event instanceof MailboxEvents.MailboxDeletion)).hasSize(1)).extracting(event -> (MailboxEvents.MailboxDeletion)event).element(0)).satisfies(event -> Assertions.assertThat((Object)event.getMailboxId()).isEqualTo((Object)this.inboxId))).satisfies(event -> Assertions.assertThat((Object)event.getQuotaRoot()).isEqualTo((Object)this.quotaRoot))).satisfies(event -> Assertions.assertThat((Object)event.getDeletedMessageCount()).isEqualTo((Object)QuotaCountUsage.count((long)0L)))).satisfies(event -> Assertions.assertThat((Object)event.getTotalDeletedSize()).isEqualTo((Object)QuotaSizeUsage.size((long)0L)));
        }

        @Test
        void createMailboxShouldFireMailboxAddedEvent() throws Exception {
            MailboxManagerTest.this.retrieveEventBus(MailboxManagerTest.this.mailboxManager).register((EventListener.GroupEventListener)this.listener);
            Optional newId = MailboxManagerTest.this.mailboxManager.createMailbox(this.newPath, MailboxManagerTest.this.session);
            ((ObjectAssert)((IterableAssert)((IterableAssert)Assertions.assertThat(this.listener.getEvents()).filteredOn(event -> event instanceof MailboxEvents.MailboxAdded)).hasSize(1)).extracting(event -> (MailboxEvents.MailboxAdded)event).element(0)).satisfies(event -> Assertions.assertThat((Object)event.getMailboxId()).isEqualTo(newId.get()));
        }

        @Test
        void addingMessageShouldFireQuotaUpdateEvent() throws Exception {
            Assumptions.assumeTrue((boolean)MailboxManagerTest.this.mailboxManager.hasCapability(MailboxManager.MailboxCapabilities.Quota));
            MailboxManagerTest.this.retrieveEventBus(MailboxManagerTest.this.mailboxManager).register((EventListener.GroupEventListener)this.listener);
            this.inboxManager.appendMessage(MessageManager.AppendCommand.builder().build(MailboxManagerTest.this.message), MailboxManagerTest.this.session);
            ((ObjectAssert)((ObjectAssert)((ObjectAssert)((IterableAssert)((IterableAssert)Assertions.assertThat(this.listener.getEvents()).filteredOn(event -> event instanceof MailboxEvents.QuotaUsageUpdatedEvent)).hasSize(1)).extracting(event -> (MailboxEvents.QuotaUsageUpdatedEvent)event).element(0)).satisfies(event -> Assertions.assertThat((Object)event.getQuotaRoot()).isEqualTo((Object)this.quotaRoot))).satisfies(event -> Assertions.assertThat((Object)event.getSizeQuota()).isEqualTo((Object)Quota.builder().used((QuotaUsageValue)QuotaSizeUsage.size((long)85L)).computedLimit((QuotaLimitValue)QuotaSizeLimit.unlimited()).build()))).satisfies(event -> Assertions.assertThat((Object)event.getCountQuota()).isEqualTo((Object)Quota.builder().used((QuotaUsageValue)QuotaCountUsage.count((long)1L)).computedLimit((QuotaLimitValue)QuotaCountLimit.unlimited()).build()));
        }

        @Test
        void addingMessageShouldFireAddedEvent() throws Exception {
            Mono.from((Publisher)MailboxManagerTest.this.retrieveEventBus(MailboxManagerTest.this.mailboxManager).register((EventListener)this.listener, (RegistrationKey)new MailboxIdRegistrationKey(this.inboxId))).block();
            this.inboxManager.appendMessage(MessageManager.AppendCommand.builder().build(MailboxManagerTest.this.message), MailboxManagerTest.this.session);
            ((ObjectAssert)((ObjectAssert)((IterableAssert)((IterableAssert)Assertions.assertThat(this.listener.getEvents()).filteredOn(event -> event instanceof MailboxEvents.Added)).hasSize(1)).extracting(event -> (MailboxEvents.Added)event).element(0)).satisfies(event -> Assertions.assertThat((Object)event.getMailboxId()).isEqualTo((Object)this.inboxId))).satisfies(event -> Assertions.assertThat((Iterable)event.getUids()).hasSize(1));
        }

        @Test
        void expungeMessageShouldFireExpungedEvent() throws Exception {
            this.inboxManager.appendMessage(MessageManager.AppendCommand.builder().build(MailboxManagerTest.this.message), MailboxManagerTest.this.session);
            this.inboxManager.setFlags(new Flags(Flags.Flag.DELETED), MessageManager.FlagsUpdateMode.ADD, MessageRange.all(), MailboxManagerTest.this.session);
            Mono.from((Publisher)MailboxManagerTest.this.retrieveEventBus(MailboxManagerTest.this.mailboxManager).register((EventListener)this.listener, (RegistrationKey)new MailboxIdRegistrationKey(this.inboxId))).block();
            this.inboxManager.expunge(MessageRange.all(), MailboxManagerTest.this.session);
            ((ObjectAssert)((ObjectAssert)((IterableAssert)((IterableAssert)Assertions.assertThat(this.listener.getEvents()).filteredOn(event -> event instanceof MailboxEvents.Expunged)).hasSize(1)).extracting(event -> (MailboxEvents.Expunged)event).element(0)).satisfies(event -> Assertions.assertThat((Object)event.getMailboxId()).isEqualTo((Object)this.inboxId))).satisfies(event -> Assertions.assertThat((Iterable)event.getUids()).hasSize(1));
        }

        @Test
        void deleteMessageShouldFireExpungedEvent() throws Exception {
            ComposedMessageId messageId = this.inboxManager.appendMessage(MessageManager.AppendCommand.builder().build(MailboxManagerTest.this.message), MailboxManagerTest.this.session).getId();
            this.inboxManager.setFlags(new Flags(Flags.Flag.DELETED), MessageManager.FlagsUpdateMode.ADD, MessageRange.all(), MailboxManagerTest.this.session);
            Mono.from((Publisher)MailboxManagerTest.this.retrieveEventBus(MailboxManagerTest.this.mailboxManager).register((EventListener)this.listener, (RegistrationKey)new MailboxIdRegistrationKey(this.inboxId))).block();
            this.inboxManager.delete((List)ImmutableList.of((Object)messageId.getUid()), MailboxManagerTest.this.session);
            ((ObjectAssert)((ObjectAssert)((IterableAssert)((IterableAssert)Assertions.assertThat(this.listener.getEvents()).filteredOn(event -> event instanceof MailboxEvents.Expunged)).hasSize(1)).extracting(event -> (MailboxEvents.Expunged)event).element(0)).satisfies(event -> Assertions.assertThat((Object)event.getMailboxId()).isEqualTo((Object)this.inboxId))).satisfies(event -> Assertions.assertThat((Iterable)event.getUids()).hasSize(1));
        }

        @Test
        void setFlagsShouldFireFlagsUpdatedEvent() throws Exception {
            this.inboxManager.appendMessage(MessageManager.AppendCommand.builder().build(MailboxManagerTest.this.message), MailboxManagerTest.this.session);
            Mono.from((Publisher)MailboxManagerTest.this.retrieveEventBus(MailboxManagerTest.this.mailboxManager).register((EventListener)this.listener, (RegistrationKey)new MailboxIdRegistrationKey(this.inboxId))).block();
            this.inboxManager.setFlags(new Flags(Flags.Flag.FLAGGED), MessageManager.FlagsUpdateMode.ADD, MessageRange.all(), MailboxManagerTest.this.session);
            ((ObjectAssert)((ObjectAssert)((IterableAssert)((IterableAssert)Assertions.assertThat(this.listener.getEvents()).filteredOn(event -> event instanceof MailboxEvents.FlagsUpdated)).hasSize(1)).extracting(event -> (MailboxEvents.FlagsUpdated)event).element(0)).satisfies(event -> Assertions.assertThat((Object)event.getMailboxId()).isEqualTo((Object)this.inboxId))).satisfies(event -> Assertions.assertThat((Iterable)event.getUids()).hasSize(1));
        }

        @Test
        void moveShouldFireAddedEventInTargetMailbox() throws Exception {
            Assumptions.assumeTrue((boolean)MailboxManagerTest.this.mailboxManager.hasCapability(MailboxManager.MailboxCapabilities.Move));
            Optional targetMailboxId = MailboxManagerTest.this.mailboxManager.createMailbox(this.newPath, MailboxManagerTest.this.session);
            this.inboxManager.appendMessage(MessageManager.AppendCommand.builder().build(MailboxManagerTest.this.message), MailboxManagerTest.this.session);
            Mono.from((Publisher)MailboxManagerTest.this.retrieveEventBus(MailboxManagerTest.this.mailboxManager).register((EventListener)this.listener, (RegistrationKey)new MailboxIdRegistrationKey((MailboxId)targetMailboxId.get()))).block();
            MailboxManagerTest.this.mailboxManager.moveMessages(MessageRange.all(), this.inbox, this.newPath, MailboxManagerTest.this.session);
            ((ObjectAssert)((ObjectAssert)((IterableAssert)((IterableAssert)Assertions.assertThat(this.listener.getEvents()).filteredOn(event -> event instanceof MailboxEvents.Added)).hasSize(1)).extracting(event -> (MailboxEvents.Added)event).element(0)).satisfies(event -> Assertions.assertThat((Object)event.getMailboxId()).isEqualTo(targetMailboxId.get()))).satisfies(event -> Assertions.assertThat((Iterable)event.getUids()).hasSize(1));
        }

        @Test
        void moveShouldFireExpungedEventInOriginMailbox() throws Exception {
            Assumptions.assumeTrue((boolean)MailboxManagerTest.this.mailboxManager.hasCapability(MailboxManager.MailboxCapabilities.Move));
            MailboxManagerTest.this.mailboxManager.createMailbox(this.newPath, MailboxManagerTest.this.session);
            this.inboxManager.appendMessage(MessageManager.AppendCommand.builder().build(MailboxManagerTest.this.message), MailboxManagerTest.this.session);
            Mono.from((Publisher)MailboxManagerTest.this.retrieveEventBus(MailboxManagerTest.this.mailboxManager).register((EventListener)this.listener, (RegistrationKey)new MailboxIdRegistrationKey(this.inboxId))).block();
            MailboxManagerTest.this.mailboxManager.moveMessages(MessageRange.all(), this.inbox, this.newPath, MailboxManagerTest.this.session);
            ((ObjectAssert)((ObjectAssert)((IterableAssert)((IterableAssert)Assertions.assertThat(this.listener.getEvents()).filteredOn(event -> event instanceof MailboxEvents.Expunged)).hasSize(1)).extracting(event -> (MailboxEvents.Expunged)event).element(0)).satisfies(event -> Assertions.assertThat((Object)event.getMailboxId()).isEqualTo((Object)this.inboxId))).satisfies(event -> Assertions.assertThat((Iterable)event.getUids()).hasSize(1));
        }

        @Test
        void copyShouldFireAddedEventInTargetMailbox() throws Exception {
            Optional targetMailboxId = MailboxManagerTest.this.mailboxManager.createMailbox(this.newPath, MailboxManagerTest.this.session);
            this.inboxManager.appendMessage(MessageManager.AppendCommand.builder().build(MailboxManagerTest.this.message), MailboxManagerTest.this.session);
            Mono.from((Publisher)MailboxManagerTest.this.retrieveEventBus(MailboxManagerTest.this.mailboxManager).register((EventListener)this.listener, (RegistrationKey)new MailboxIdRegistrationKey((MailboxId)targetMailboxId.get()))).block();
            MailboxManagerTest.this.mailboxManager.copyMessages(MessageRange.all(), this.inbox, this.newPath, MailboxManagerTest.this.session);
            ((ObjectAssert)((ObjectAssert)((IterableAssert)((IterableAssert)Assertions.assertThat(this.listener.getEvents()).filteredOn(event -> event instanceof MailboxEvents.Added)).hasSize(1)).extracting(event -> (MailboxEvents.Added)event).element(0)).satisfies(event -> Assertions.assertThat((Object)event.getMailboxId()).isEqualTo(targetMailboxId.get()))).satisfies(event -> Assertions.assertThat((Iterable)event.getUids()).hasSize(1));
        }

        @Test
        void copyShouldFireMovedEventInTargetMailbox() throws Exception {
            Assumptions.assumeTrue((boolean)MailboxManagerTest.this.mailboxManager.getSupportedMessageCapabilities().contains(MailboxManager.MessageCapabilities.UniqueID));
            Optional targetMailboxId = MailboxManagerTest.this.mailboxManager.createMailbox(this.newPath, MailboxManagerTest.this.session);
            ComposedMessageId messageId = this.inboxManager.appendMessage(MessageManager.AppendCommand.builder().build(MailboxManagerTest.this.message), MailboxManagerTest.this.session).getId();
            Mono.from((Publisher)MailboxManagerTest.this.retrieveEventBus(MailboxManagerTest.this.mailboxManager).register((EventListener)this.listener, (RegistrationKey)new MailboxIdRegistrationKey((MailboxId)targetMailboxId.get()))).block();
            MailboxManagerTest.this.mailboxManager.copyMessages(MessageRange.all(), this.inbox, this.newPath, MailboxManagerTest.this.session);
            ((ObjectAssert)((IterableAssert)((IterableAssert)Assertions.assertThat(this.listener.getEvents()).filteredOn(event -> event instanceof MessageMoveEvent)).hasSize(1)).extracting(event -> (MessageMoveEvent)event).element(0)).satisfies(event -> Assertions.assertThat((Iterable)event.getMessageIds()).containsExactly((Object[])new MessageId[]{messageId.getMessageId()}));
        }

        @Test
        void copyShouldNotFireMovedEventInOriginMailbox() throws Exception {
            Assumptions.assumeTrue((boolean)MailboxManagerTest.this.mailboxManager.getSupportedMessageCapabilities().contains(MailboxManager.MessageCapabilities.UniqueID));
            MailboxManagerTest.this.mailboxManager.createMailbox(this.newPath, MailboxManagerTest.this.session);
            this.inboxManager.appendMessage(MessageManager.AppendCommand.builder().build(MailboxManagerTest.this.message), MailboxManagerTest.this.session);
            Mono.from((Publisher)MailboxManagerTest.this.retrieveEventBus(MailboxManagerTest.this.mailboxManager).register((EventListener)this.listener, (RegistrationKey)new MailboxIdRegistrationKey(this.inboxId))).block();
            MailboxManagerTest.this.mailboxManager.copyMessages(MessageRange.all(), this.inbox, this.newPath, MailboxManagerTest.this.session);
            ((IterableAssert)Assertions.assertThat(this.listener.getEvents()).filteredOn(event -> event instanceof MessageMoveEvent)).isEmpty();
        }

        @Test
        void moveShouldFireMovedEventInTargetMailbox() throws Exception {
            Assumptions.assumeTrue((boolean)MailboxManagerTest.this.mailboxManager.getSupportedMessageCapabilities().contains(MailboxManager.MessageCapabilities.UniqueID));
            Optional targetMailboxId = MailboxManagerTest.this.mailboxManager.createMailbox(this.newPath, MailboxManagerTest.this.session);
            ComposedMessageId messageId = this.inboxManager.appendMessage(MessageManager.AppendCommand.builder().build(MailboxManagerTest.this.message), MailboxManagerTest.this.session).getId();
            Mono.from((Publisher)MailboxManagerTest.this.retrieveEventBus(MailboxManagerTest.this.mailboxManager).register((EventListener)this.listener, (RegistrationKey)new MailboxIdRegistrationKey((MailboxId)targetMailboxId.get()))).block();
            MailboxManagerTest.this.mailboxManager.moveMessages(MessageRange.all(), this.inbox, this.newPath, MailboxManagerTest.this.session);
            ((ObjectAssert)((IterableAssert)((IterableAssert)Assertions.assertThat(this.listener.getEvents()).filteredOn(event -> event instanceof MessageMoveEvent)).hasSize(1)).extracting(event -> (MessageMoveEvent)event).element(0)).satisfies(event -> Assertions.assertThat((Iterable)event.getMessageIds()).containsExactly((Object[])new MessageId[]{messageId.getMessageId()}));
        }

        @Test
        void moveShouldFireMovedEventInOriginMailbox() throws Exception {
            Assumptions.assumeTrue((boolean)MailboxManagerTest.this.mailboxManager.getSupportedMessageCapabilities().contains(MailboxManager.MessageCapabilities.UniqueID));
            MailboxManagerTest.this.mailboxManager.createMailbox(this.newPath, MailboxManagerTest.this.session);
            ComposedMessageId messageId = this.inboxManager.appendMessage(MessageManager.AppendCommand.builder().build(MailboxManagerTest.this.message), MailboxManagerTest.this.session).getId();
            Mono.from((Publisher)MailboxManagerTest.this.retrieveEventBus(MailboxManagerTest.this.mailboxManager).register((EventListener)this.listener, (RegistrationKey)new MailboxIdRegistrationKey(this.inboxId))).block();
            MailboxManagerTest.this.mailboxManager.moveMessages(MessageRange.all(), this.inbox, this.newPath, MailboxManagerTest.this.session);
            ((ObjectAssert)((IterableAssert)((IterableAssert)Assertions.assertThat(this.listener.getEvents()).filteredOn(event -> event instanceof MessageMoveEvent)).hasSize(1)).extracting(event -> (MessageMoveEvent)event).element(0)).satisfies(event -> Assertions.assertThat((Iterable)event.getMessageIds()).containsExactly((Object[])new MessageId[]{messageId.getMessageId()}));
        }

        @Test
        void copyShouldNotFireExpungedEventInOriginMailbox() throws Exception {
            MailboxManagerTest.this.mailboxManager.createMailbox(this.newPath, MailboxManagerTest.this.session);
            this.inboxManager.appendMessage(MessageManager.AppendCommand.builder().build(MailboxManagerTest.this.message), MailboxManagerTest.this.session);
            MailboxManagerTest.this.retrieveEventBus(MailboxManagerTest.this.mailboxManager).register((EventListener.GroupEventListener)this.listener);
            MailboxManagerTest.this.mailboxManager.copyMessages(MessageRange.all(), this.inbox, this.newPath, MailboxManagerTest.this.session);
            ((IterableAssert)Assertions.assertThat(this.listener.getEvents()).filteredOn(event -> event instanceof MailboxEvents.Expunged)).isEmpty();
        }
    }

    @Nested
    class AnnotationTests {
        private final MailboxAnnotationKey privateKey = new MailboxAnnotationKey("/private/comment");
        private final MailboxAnnotationKey privateChildKey = new MailboxAnnotationKey("/private/comment/user");
        private final MailboxAnnotationKey privateGrandchildKey = new MailboxAnnotationKey("/private/comment/user/name");
        private final MailboxAnnotationKey sharedKey = new MailboxAnnotationKey("/shared/comment");
        private final MailboxAnnotation privateAnnotation = MailboxAnnotation.newInstance((MailboxAnnotationKey)this.privateKey, (String)"My private comment");
        private final MailboxAnnotation privateChildAnnotation = MailboxAnnotation.newInstance((MailboxAnnotationKey)this.privateChildKey, (String)"My private comment");
        private final MailboxAnnotation privateGrandchildAnnotation = MailboxAnnotation.newInstance((MailboxAnnotationKey)this.privateGrandchildKey, (String)"My private comment");
        private final MailboxAnnotation privateAnnotationUpdate = MailboxAnnotation.newInstance((MailboxAnnotationKey)this.privateKey, (String)"My updated private comment");
        private final MailboxAnnotation sharedAnnotation = MailboxAnnotation.newInstance((MailboxAnnotationKey)this.sharedKey, (String)"My shared comment");
        private final List<MailboxAnnotation> annotations = ImmutableList.of((Object)this.privateAnnotation, (Object)this.sharedAnnotation);

        AnnotationTests() {
        }

        @Test
        void updateAnnotationsShouldUpdateStoredAnnotation() throws Exception {
            Assumptions.assumeTrue((boolean)MailboxManagerTest.this.mailboxManager.hasCapability(MailboxManager.MailboxCapabilities.Annotation));
            MailboxManagerTest.this.session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_2);
            MailboxPath inbox = MailboxPath.inbox((MailboxSession)MailboxManagerTest.this.session);
            MailboxManagerTest.this.mailboxManager.createMailbox(inbox, MailboxManagerTest.this.session);
            MailboxManagerTest.this.mailboxManager.updateAnnotations(inbox, MailboxManagerTest.this.session, (List)ImmutableList.of((Object)this.privateAnnotation));
            MailboxManagerTest.this.mailboxManager.updateAnnotations(inbox, MailboxManagerTest.this.session, (List)ImmutableList.of((Object)this.privateAnnotationUpdate));
            Assertions.assertThat((List)MailboxManagerTest.this.mailboxManager.getAllAnnotations(inbox, MailboxManagerTest.this.session)).containsOnly((Object[])new MailboxAnnotation[]{this.privateAnnotationUpdate});
        }

        @Test
        void updateAnnotationsShouldDeleteAnnotationWithNilValue() throws Exception {
            Assumptions.assumeTrue((boolean)MailboxManagerTest.this.mailboxManager.hasCapability(MailboxManager.MailboxCapabilities.Annotation));
            MailboxManagerTest.this.session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_2);
            MailboxPath inbox = MailboxPath.inbox((MailboxSession)MailboxManagerTest.this.session);
            MailboxManagerTest.this.mailboxManager.createMailbox(inbox, MailboxManagerTest.this.session);
            MailboxManagerTest.this.mailboxManager.updateAnnotations(inbox, MailboxManagerTest.this.session, (List)ImmutableList.of((Object)this.privateAnnotation));
            MailboxManagerTest.this.mailboxManager.updateAnnotations(inbox, MailboxManagerTest.this.session, (List)ImmutableList.of((Object)MailboxAnnotation.nil((MailboxAnnotationKey)this.privateKey)));
            Assertions.assertThat((List)MailboxManagerTest.this.mailboxManager.getAllAnnotations(inbox, MailboxManagerTest.this.session)).isEmpty();
        }

        @Test
        void updateAnnotationsShouldThrowExceptionIfMailboxDoesNotExist() throws Exception {
            Assumptions.assumeTrue((boolean)MailboxManagerTest.this.mailboxManager.hasCapability(MailboxManager.MailboxCapabilities.Annotation));
            MailboxManagerTest.this.session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_2);
            MailboxPath inbox = MailboxPath.inbox((MailboxSession)MailboxManagerTest.this.session);
            Assertions.assertThatThrownBy(() -> MailboxManagerTest.this.mailboxManager.updateAnnotations(inbox, MailboxManagerTest.this.session, (List)ImmutableList.of((Object)this.privateAnnotation))).isInstanceOf(MailboxNotFoundException.class);
        }

        @Test
        void getAnnotationsShouldReturnEmptyForNonStoredAnnotation() throws Exception {
            Assumptions.assumeTrue((boolean)MailboxManagerTest.this.mailboxManager.hasCapability(MailboxManager.MailboxCapabilities.Annotation));
            MailboxManagerTest.this.session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_2);
            MailboxPath inbox = MailboxPath.inbox((MailboxSession)MailboxManagerTest.this.session);
            MailboxManagerTest.this.mailboxManager.createMailbox(inbox, MailboxManagerTest.this.session);
            Assertions.assertThat((List)MailboxManagerTest.this.mailboxManager.getAllAnnotations(inbox, MailboxManagerTest.this.session)).isEmpty();
        }

        @Test
        void getAllAnnotationsShouldRetrieveStoredAnnotations() throws Exception {
            Assumptions.assumeTrue((boolean)MailboxManagerTest.this.mailboxManager.hasCapability(MailboxManager.MailboxCapabilities.Annotation));
            MailboxManagerTest.this.session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_2);
            MailboxPath inbox = MailboxPath.inbox((MailboxSession)MailboxManagerTest.this.session);
            MailboxManagerTest.this.mailboxManager.createMailbox(inbox, MailboxManagerTest.this.session);
            MailboxManagerTest.this.mailboxManager.updateAnnotations(inbox, MailboxManagerTest.this.session, this.annotations);
            Assertions.assertThat((List)MailboxManagerTest.this.mailboxManager.getAllAnnotations(inbox, MailboxManagerTest.this.session)).isEqualTo(this.annotations);
        }

        @Test
        void getAllAnnotationsShouldThrowExceptionIfMailboxDoesNotExist() throws Exception {
            Assumptions.assumeTrue((boolean)MailboxManagerTest.this.mailboxManager.hasCapability(MailboxManager.MailboxCapabilities.Annotation));
            MailboxManagerTest.this.session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_2);
            MailboxPath inbox = MailboxPath.inbox((MailboxSession)MailboxManagerTest.this.session);
            Assertions.assertThatThrownBy(() -> MailboxManagerTest.this.mailboxManager.getAllAnnotations(inbox, MailboxManagerTest.this.session)).isInstanceOf(MailboxNotFoundException.class);
        }

        @Test
        void getAnnotationsByKeysShouldRetrieveStoresAnnotationsByKeys() throws Exception {
            Assumptions.assumeTrue((boolean)MailboxManagerTest.this.mailboxManager.hasCapability(MailboxManager.MailboxCapabilities.Annotation));
            MailboxManagerTest.this.session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_2);
            MailboxPath inbox = MailboxPath.inbox((MailboxSession)MailboxManagerTest.this.session);
            MailboxManagerTest.this.mailboxManager.createMailbox(inbox, MailboxManagerTest.this.session);
            MailboxManagerTest.this.mailboxManager.updateAnnotations(inbox, MailboxManagerTest.this.session, this.annotations);
            Assertions.assertThat((List)MailboxManagerTest.this.mailboxManager.getAnnotationsByKeys(inbox, MailboxManagerTest.this.session, (Set)ImmutableSet.of((Object)this.privateKey))).containsOnly((Object[])new MailboxAnnotation[]{this.privateAnnotation});
        }

        @Test
        void getAnnotationsByKeysShouldThrowExceptionIfMailboxDoesNotExist() throws Exception {
            Assumptions.assumeTrue((boolean)MailboxManagerTest.this.mailboxManager.hasCapability(MailboxManager.MailboxCapabilities.Annotation));
            MailboxManagerTest.this.session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_2);
            MailboxPath inbox = MailboxPath.inbox((MailboxSession)MailboxManagerTest.this.session);
            Assertions.assertThatThrownBy(() -> MailboxManagerTest.this.mailboxManager.getAnnotationsByKeys(inbox, MailboxManagerTest.this.session, (Set)ImmutableSet.of((Object)this.privateKey))).isInstanceOf(MailboxNotFoundException.class);
        }

        @Test
        void getAnnotationsByKeysWithOneDepthShouldRetriveAnnotationsWithOneDepth() throws Exception {
            Assumptions.assumeTrue((boolean)MailboxManagerTest.this.mailboxManager.hasCapability(MailboxManager.MailboxCapabilities.Annotation));
            MailboxManagerTest.this.session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_2);
            MailboxPath inbox = MailboxPath.inbox((MailboxSession)MailboxManagerTest.this.session);
            MailboxManagerTest.this.mailboxManager.createMailbox(inbox, MailboxManagerTest.this.session);
            MailboxManagerTest.this.mailboxManager.updateAnnotations(inbox, MailboxManagerTest.this.session, (List)ImmutableList.of((Object)this.privateAnnotation, (Object)this.privateChildAnnotation, (Object)this.privateGrandchildAnnotation));
            Assertions.assertThat((List)MailboxManagerTest.this.mailboxManager.getAnnotationsByKeysWithOneDepth(inbox, MailboxManagerTest.this.session, (Set)ImmutableSet.of((Object)this.privateKey))).contains((Object[])new MailboxAnnotation[]{this.privateAnnotation, this.privateChildAnnotation});
        }

        @Test
        void getAnnotationsByKeysWithAllDepthShouldThrowExceptionWhenMailboxDoesNotExist() throws Exception {
            Assumptions.assumeTrue((boolean)MailboxManagerTest.this.mailboxManager.hasCapability(MailboxManager.MailboxCapabilities.Annotation));
            MailboxManagerTest.this.session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_2);
            MailboxPath inbox = MailboxPath.inbox((MailboxSession)MailboxManagerTest.this.session);
            Assertions.assertThatThrownBy(() -> MailboxManagerTest.this.mailboxManager.getAnnotationsByKeysWithAllDepth(inbox, MailboxManagerTest.this.session, (Set)ImmutableSet.of((Object)this.privateKey))).isInstanceOf(MailboxNotFoundException.class);
        }

        @Test
        void getAnnotationsByKeysWithAllDepthShouldRetriveAnnotationsWithAllDepth() throws Exception {
            Assumptions.assumeTrue((boolean)MailboxManagerTest.this.mailboxManager.hasCapability(MailboxManager.MailboxCapabilities.Annotation));
            MailboxManagerTest.this.session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_2);
            MailboxPath inbox = MailboxPath.inbox((MailboxSession)MailboxManagerTest.this.session);
            MailboxManagerTest.this.mailboxManager.createMailbox(inbox, MailboxManagerTest.this.session);
            MailboxManagerTest.this.mailboxManager.updateAnnotations(inbox, MailboxManagerTest.this.session, (List)ImmutableList.of((Object)this.privateAnnotation, (Object)this.privateChildAnnotation, (Object)this.privateGrandchildAnnotation));
            Assertions.assertThat((List)MailboxManagerTest.this.mailboxManager.getAnnotationsByKeysWithAllDepth(inbox, MailboxManagerTest.this.session, (Set)ImmutableSet.of((Object)this.privateKey))).contains((Object[])new MailboxAnnotation[]{this.privateAnnotation, this.privateChildAnnotation, this.privateGrandchildAnnotation});
        }

        @Test
        void updateAnnotationsShouldThrowExceptionIfAnnotationDataIsOverLimitation() throws Exception {
            Assumptions.assumeTrue((boolean)MailboxManagerTest.this.mailboxManager.hasCapability(MailboxManager.MailboxCapabilities.Annotation));
            MailboxManagerTest.this.session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_2);
            MailboxPath inbox = MailboxPath.inbox((MailboxSession)MailboxManagerTest.this.session);
            MailboxManagerTest.this.mailboxManager.createMailbox(inbox, MailboxManagerTest.this.session);
            Assertions.assertThatThrownBy(() -> MailboxManagerTest.this.mailboxManager.updateAnnotations(inbox, MailboxManagerTest.this.session, (List)ImmutableList.of((Object)MailboxAnnotation.newInstance((MailboxAnnotationKey)this.privateKey, (String)"The limitation of data is less than 30")))).isInstanceOf(AnnotationException.class);
        }

        @Test
        void shouldUpdateAnnotationWhenRequestCreatesNewAndMailboxIsNotOverLimit() throws Exception {
            Assumptions.assumeTrue((boolean)MailboxManagerTest.this.mailboxManager.hasCapability(MailboxManager.MailboxCapabilities.Annotation));
            MailboxManagerTest.this.session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_2);
            MailboxPath inbox = MailboxPath.inbox((MailboxSession)MailboxManagerTest.this.session);
            MailboxManagerTest.this.mailboxManager.createMailbox(inbox, MailboxManagerTest.this.session);
            ImmutableList.Builder builder = ImmutableList.builder();
            builder.add((Object)MailboxAnnotation.newInstance((MailboxAnnotationKey)new MailboxAnnotationKey("/private/comment1"), (String)"AnyValue"));
            builder.add((Object)MailboxAnnotation.newInstance((MailboxAnnotationKey)new MailboxAnnotationKey("/private/comment2"), (String)"AnyValue"));
            builder.add((Object)MailboxAnnotation.newInstance((MailboxAnnotationKey)new MailboxAnnotationKey("/private/comment3"), (String)"AnyValue"));
            MailboxManagerTest.this.mailboxManager.updateAnnotations(inbox, MailboxManagerTest.this.session, (List)builder.build());
        }

        @Test
        void updateAnnotationsShouldThrowExceptionIfRequestCreateNewButMailboxIsOverLimit() throws Exception {
            Assumptions.assumeTrue((boolean)MailboxManagerTest.this.mailboxManager.hasCapability(MailboxManager.MailboxCapabilities.Annotation));
            MailboxManagerTest.this.session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_2);
            MailboxPath inbox = MailboxPath.inbox((MailboxSession)MailboxManagerTest.this.session);
            MailboxManagerTest.this.mailboxManager.createMailbox(inbox, MailboxManagerTest.this.session);
            ImmutableList.Builder builder = ImmutableList.builder();
            builder.add((Object)MailboxAnnotation.newInstance((MailboxAnnotationKey)new MailboxAnnotationKey("/private/comment1"), (String)"AnyValue"));
            builder.add((Object)MailboxAnnotation.newInstance((MailboxAnnotationKey)new MailboxAnnotationKey("/private/comment2"), (String)"AnyValue"));
            builder.add((Object)MailboxAnnotation.newInstance((MailboxAnnotationKey)new MailboxAnnotationKey("/private/comment3"), (String)"AnyValue"));
            builder.add((Object)MailboxAnnotation.newInstance((MailboxAnnotationKey)new MailboxAnnotationKey("/private/comment4"), (String)"AnyValue"));
            Assertions.assertThatThrownBy(() -> MailboxManagerTest.this.mailboxManager.updateAnnotations(inbox, MailboxManagerTest.this.session, (List)builder.build())).isInstanceOf(MailboxException.class);
        }
    }

    @Nested
    public class MailboxNameLimitTests {
        @Test
        void creatingMailboxShouldNotFailWhenLimitNameLength() throws Exception {
            MailboxSession session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            String mailboxName = Strings.repeat((String)"a", (int)200);
            Assertions.assertThatCode(() -> MailboxManagerTest.this.mailboxManager.createMailbox(MailboxPath.forUser((Username)USER_1, (String)mailboxName), session)).doesNotThrowAnyException();
        }

        @Test
        void renamingMailboxShouldNotFailWhenLimitNameLength() throws Exception {
            MailboxSession session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            String mailboxName = Strings.repeat((String)"a", (int)200);
            MailboxPath originPath = MailboxPath.forUser((Username)USER_1, (String)"origin");
            MailboxManagerTest.this.mailboxManager.createMailbox(originPath, session);
            Assertions.assertThatCode(() -> MailboxManagerTest.this.mailboxManager.renameMailbox(originPath, MailboxPath.forUser((Username)USER_1, (String)mailboxName), session)).doesNotThrowAnyException();
        }

        @Test
        protected void renamingMailboxByIdShouldNotFailWhenLimitNameLength() throws Exception {
            MailboxSession session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            String mailboxName = Strings.repeat((String)"a", (int)200);
            MailboxPath originPath = MailboxPath.forUser((Username)USER_1, (String)"origin");
            MailboxId mailboxId = (MailboxId)MailboxManagerTest.this.mailboxManager.createMailbox(originPath, session).get();
            Assertions.assertThatCode(() -> MailboxManagerTest.this.mailboxManager.renameMailbox(mailboxId, MailboxPath.forUser((Username)USER_1, (String)mailboxName), session)).doesNotThrowAnyException();
        }

        @Test
        void creatingMailboxShouldThrowWhenOverLimitNameLength() throws Exception {
            MailboxSession session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            String mailboxName = Strings.repeat((String)"a", (int)201);
            Assertions.assertThatThrownBy(() -> MailboxManagerTest.this.mailboxManager.createMailbox(MailboxPath.forUser((Username)USER_1, (String)mailboxName), session)).isInstanceOf(TooLongMailboxNameException.class);
        }

        @Test
        void renamingMailboxShouldThrowWhenOverLimitNameLength() throws Exception {
            MailboxSession session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            String mailboxName = Strings.repeat((String)"a", (int)201);
            MailboxPath originPath = MailboxPath.forUser((Username)USER_1, (String)"origin");
            MailboxManagerTest.this.mailboxManager.createMailbox(originPath, session);
            Assertions.assertThatThrownBy(() -> MailboxManagerTest.this.mailboxManager.renameMailbox(originPath, MailboxPath.forUser((Username)USER_1, (String)mailboxName), session)).isInstanceOf(TooLongMailboxNameException.class);
        }

        @Test
        void renamingMailboxByIdShouldThrowWhenOverLimitNameLength() throws Exception {
            MailboxSession session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            String mailboxName = Strings.repeat((String)"a", (int)201);
            MailboxPath originPath = MailboxPath.forUser((Username)USER_1, (String)"origin");
            MailboxId mailboxId = (MailboxId)MailboxManagerTest.this.mailboxManager.createMailbox(originPath, session).get();
            Assertions.assertThatThrownBy(() -> MailboxManagerTest.this.mailboxManager.renameMailbox(mailboxId, MailboxPath.forUser((Username)USER_1, (String)mailboxName), session)).isInstanceOf(TooLongMailboxNameException.class);
        }

        @Test
        void creatingMailboxShouldNotThrowWhenNameWithoutEmptyHierarchicalLevel() throws Exception {
            MailboxSession session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            String mailboxName = "a.b.c";
            Assertions.assertThatCode(() -> MailboxManagerTest.this.mailboxManager.createMailbox(MailboxPath.forUser((Username)USER_1, (String)mailboxName), session)).doesNotThrowAnyException();
        }

        @Test
        void creatingMailboxShouldNotThrowWhenNameWithASingleToBeNormalizedTrailingDelimiter() throws Exception {
            MailboxSession session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            String mailboxName = "a.b.";
            Assertions.assertThatCode(() -> MailboxManagerTest.this.mailboxManager.createMailbox(MailboxPath.forUser((Username)USER_1, (String)mailboxName), session)).doesNotThrowAnyException();
        }

        @Test
        void creatingMailboxShouldThrowWhenNameWithMoreThanOneTrailingDelimiter() throws Exception {
            MailboxSession session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            String mailboxName = "a..";
            Assertions.assertThatThrownBy(() -> MailboxManagerTest.this.mailboxManager.createMailbox(MailboxPath.forUser((Username)USER_1, (String)mailboxName), session)).isInstanceOf(HasEmptyMailboxNameInHierarchyException.class);
        }

        @Test
        void creatingMailboxShouldThrowWhenNameWithHeadingDelimiter() throws Exception {
            MailboxSession session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            String mailboxName = ".a";
            Assertions.assertThatThrownBy(() -> MailboxManagerTest.this.mailboxManager.createMailbox(MailboxPath.forUser((Username)USER_1, (String)mailboxName), session)).isInstanceOf(HasEmptyMailboxNameInHierarchyException.class);
        }

        @Test
        void creatingMailboxShouldThrowWhenNameWithEmptyHierarchicalLevel() throws Exception {
            MailboxSession session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            String mailboxName = "a..b";
            Assertions.assertThatThrownBy(() -> MailboxManagerTest.this.mailboxManager.createMailbox(MailboxPath.forUser((Username)USER_1, (String)mailboxName), session)).isInstanceOf(HasEmptyMailboxNameInHierarchyException.class);
        }

        @Test
        void renamingMailboxShouldNotThrowWhenNameWithoutEmptyHierarchicalLevel() throws Exception {
            MailboxSession session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            String mailboxName = "a.b.c";
            MailboxPath originPath = MailboxPath.forUser((Username)USER_1, (String)"origin");
            MailboxManagerTest.this.mailboxManager.createMailbox(originPath, session);
            Assertions.assertThatCode(() -> MailboxManagerTest.this.mailboxManager.renameMailbox(originPath, MailboxPath.forUser((Username)USER_1, (String)mailboxName), session)).doesNotThrowAnyException();
        }

        @Test
        protected void renamingMailboxByIdShouldNotThrowWhenNameWithoutEmptyHierarchicalLevel() throws Exception {
            MailboxSession session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            String mailboxName = "a.b.c";
            MailboxPath originPath = MailboxPath.forUser((Username)USER_1, (String)"origin");
            MailboxId mailboxId = (MailboxId)MailboxManagerTest.this.mailboxManager.createMailbox(originPath, session).get();
            Assertions.assertThatCode(() -> MailboxManagerTest.this.mailboxManager.renameMailbox(mailboxId, MailboxPath.forUser((Username)USER_1, (String)mailboxName), session)).doesNotThrowAnyException();
        }

        @Test
        void renamingMailboxShouldNotThrowWhenNameWithASingleToBeNormalizedTrailingDelimiter() throws Exception {
            MailboxSession session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            String mailboxName = "a.b.";
            MailboxPath originPath = MailboxPath.forUser((Username)USER_1, (String)"origin");
            MailboxManagerTest.this.mailboxManager.createMailbox(originPath, session);
            Assertions.assertThatCode(() -> MailboxManagerTest.this.mailboxManager.renameMailbox(originPath, MailboxPath.forUser((Username)USER_1, (String)mailboxName), session)).doesNotThrowAnyException();
        }

        @Test
        protected void renamingMailboxByIdShouldNotThrowWhenNameWithASingleToBeNormalizedTrailingDelimiter() throws Exception {
            MailboxSession session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            String mailboxName = "a.b.";
            MailboxPath originPath = MailboxPath.forUser((Username)USER_1, (String)"origin");
            MailboxId mailboxId = (MailboxId)MailboxManagerTest.this.mailboxManager.createMailbox(originPath, session).get();
            Assertions.assertThatCode(() -> MailboxManagerTest.this.mailboxManager.renameMailbox(mailboxId, MailboxPath.forUser((Username)USER_1, (String)mailboxName), session)).doesNotThrowAnyException();
        }

        @Test
        void renamingMailboxShouldThrowWhenNameWithMoreThanOneTrailingDelimiter() throws Exception {
            MailboxSession session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            String mailboxName = "a..";
            MailboxPath originPath = MailboxPath.forUser((Username)USER_1, (String)"origin");
            MailboxManagerTest.this.mailboxManager.createMailbox(originPath, session);
            Assertions.assertThatThrownBy(() -> MailboxManagerTest.this.mailboxManager.renameMailbox(originPath, MailboxPath.forUser((Username)USER_1, (String)mailboxName), session)).isInstanceOf(HasEmptyMailboxNameInHierarchyException.class);
        }

        @Test
        void renamingMailboxByIdShouldThrowWhenNameWithMoreThanOneTrailingDelimiter() throws Exception {
            MailboxSession session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            String mailboxName = "a..";
            MailboxPath originPath = MailboxPath.forUser((Username)USER_1, (String)"origin");
            MailboxId mailboxId = (MailboxId)MailboxManagerTest.this.mailboxManager.createMailbox(originPath, session).get();
            Assertions.assertThatThrownBy(() -> MailboxManagerTest.this.mailboxManager.renameMailbox(mailboxId, MailboxPath.forUser((Username)USER_1, (String)mailboxName), session)).isInstanceOf(HasEmptyMailboxNameInHierarchyException.class);
        }

        @Test
        void renamingMailboxShouldThrowWhenNameWithHeadingDelimiter() throws Exception {
            MailboxSession session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            String mailboxName = ".a";
            MailboxPath originPath = MailboxPath.forUser((Username)USER_1, (String)"origin");
            MailboxManagerTest.this.mailboxManager.createMailbox(originPath, session);
            Assertions.assertThatThrownBy(() -> MailboxManagerTest.this.mailboxManager.renameMailbox(originPath, MailboxPath.forUser((Username)USER_1, (String)mailboxName), session)).isInstanceOf(HasEmptyMailboxNameInHierarchyException.class);
        }

        @Test
        void renamingMailboxByIdShouldThrowWhenNameWithHeadingDelimiter() throws Exception {
            MailboxSession session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            String mailboxName = ".a";
            MailboxPath originPath = MailboxPath.forUser((Username)USER_1, (String)"origin");
            MailboxId mailboxId = (MailboxId)MailboxManagerTest.this.mailboxManager.createMailbox(originPath, session).get();
            Assertions.assertThatThrownBy(() -> MailboxManagerTest.this.mailboxManager.renameMailbox(mailboxId, MailboxPath.forUser((Username)USER_1, (String)mailboxName), session)).isInstanceOf(HasEmptyMailboxNameInHierarchyException.class);
        }

        @Test
        void renamingMailboxShouldThrowWhenNameWithEmptyHierarchicalLevel() throws Exception {
            MailboxSession session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            String mailboxName = "a..b";
            MailboxPath originPath = MailboxPath.forUser((Username)USER_1, (String)"origin");
            MailboxManagerTest.this.mailboxManager.createMailbox(originPath, session);
            Assertions.assertThatThrownBy(() -> MailboxManagerTest.this.mailboxManager.renameMailbox(originPath, MailboxPath.forUser((Username)USER_1, (String)mailboxName), session)).isInstanceOf(HasEmptyMailboxNameInHierarchyException.class);
        }

        @Test
        void renamingMailboxByIdShouldThrowWhenNameWithEmptyHierarchicalLevel() throws Exception {
            MailboxSession session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            String mailboxName = "a..b";
            MailboxPath originPath = MailboxPath.forUser((Username)USER_1, (String)"origin");
            MailboxId mailboxId = (MailboxId)MailboxManagerTest.this.mailboxManager.createMailbox(originPath, session).get();
            Assertions.assertThatThrownBy(() -> MailboxManagerTest.this.mailboxManager.renameMailbox(mailboxId, MailboxPath.forUser((Username)USER_1, (String)mailboxName), session)).isInstanceOf(HasEmptyMailboxNameInHierarchyException.class);
        }
    }

    @Nested
    class MailboxCreationTests {
        MailboxCreationTests() {
        }

        @Test
        void hasInboxShouldBeFalseWhenINBOXIsNotCreated() throws Exception {
            MailboxManagerTest.this.session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxManagerTest.this.mailboxManager.startProcessingRequest(MailboxManagerTest.this.session);
            Assertions.assertThat((Boolean)((Boolean)Mono.from((Publisher)MailboxManagerTest.this.mailboxManager.hasInbox(MailboxManagerTest.this.session)).block())).isFalse();
        }

        @Test
        void hasInboxShouldBeTrueWhenINBOXIsCreated() throws Exception {
            MailboxManagerTest.this.session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxManagerTest.this.mailboxManager.startProcessingRequest(MailboxManagerTest.this.session);
            MailboxPath mailboxPath = MailboxPath.inbox((MailboxSession)MailboxManagerTest.this.session);
            Optional mailboxId = MailboxManagerTest.this.mailboxManager.createMailbox(mailboxPath, MailboxManagerTest.this.session);
            MessageManager retrievedMailbox = MailboxManagerTest.this.mailboxManager.getMailbox(mailboxPath, MailboxManagerTest.this.session);
            Assertions.assertThat((Boolean)((Boolean)Mono.from((Publisher)MailboxManagerTest.this.mailboxManager.hasInbox(MailboxManagerTest.this.session)).block())).isTrue();
            Assertions.assertThat((Object)((MailboxId)mailboxId.get())).isEqualTo((Object)retrievedMailbox.getId());
        }

        @Test
        void creatingMixedCaseINBOXShouldCreateItAsINBOX() throws Exception {
            MailboxManagerTest.this.session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxManagerTest.this.mailboxManager.startProcessingRequest(MailboxManagerTest.this.session);
            Optional mailboxId = MailboxManagerTest.this.mailboxManager.createMailbox(MailboxPath.forUser((Username)USER_1, (String)"iNbOx"), MailboxManagerTest.this.session);
            MessageManager retrievedMailbox = MailboxManagerTest.this.mailboxManager.getMailbox(MailboxPath.inbox((MailboxSession)MailboxManagerTest.this.session), MailboxManagerTest.this.session);
            Assertions.assertThat((Boolean)((Boolean)Mono.from((Publisher)MailboxManagerTest.this.mailboxManager.hasInbox(MailboxManagerTest.this.session)).block())).isTrue();
            Assertions.assertThat((Object)((MailboxId)mailboxId.get())).isEqualTo((Object)retrievedMailbox.getId());
        }

        @Test
        void creatingMixedCaseINBOXShouldNotBeRetrievableAsIt() throws Exception {
            MailboxManagerTest.this.session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxManagerTest.this.mailboxManager.startProcessingRequest(MailboxManagerTest.this.session);
            MailboxPath mailboxPath = MailboxPath.forUser((Username)USER_1, (String)"iNbOx");
            Optional mailboxId = MailboxManagerTest.this.mailboxManager.createMailbox(mailboxPath, MailboxManagerTest.this.session);
            Assertions.assertThat((Optional)mailboxId).isPresent();
            Assertions.assertThatThrownBy(() -> MailboxManagerTest.this.mailboxManager.getMailbox(mailboxPath, MailboxManagerTest.this.session)).isInstanceOf(MailboxNotFoundException.class);
        }

        @Test
        void creatingMixedCaseINBOXWhenItHasAlreadyBeenCreatedShouldThrow() throws Exception {
            MailboxManagerTest.this.session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxManagerTest.this.mailboxManager.startProcessingRequest(MailboxManagerTest.this.session);
            MailboxManagerTest.this.mailboxManager.createMailbox(MailboxPath.inbox((MailboxSession)MailboxManagerTest.this.session), MailboxManagerTest.this.session);
            Assertions.assertThatThrownBy(() -> MailboxManagerTest.this.mailboxManager.createMailbox(MailboxPath.forUser((Username)USER_1, (String)"iNbOx"), MailboxManagerTest.this.session)).isInstanceOf(InboxAlreadyCreated.class);
        }

        @Test
        void creatingMixedCaseINBOXShouldCreateItAsINBOXUponChildMailboxCreation() throws Exception {
            MailboxManagerTest.this.session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxManagerTest.this.mailboxManager.startProcessingRequest(MailboxManagerTest.this.session);
            Optional mailboxId = MailboxManagerTest.this.mailboxManager.createMailbox(MailboxPath.forUser((Username)USER_1, (String)"iNbOx.submailbox"), MailboxManagerTest.this.session);
            MessageManager retrievedMailbox = MailboxManagerTest.this.mailboxManager.getMailbox(MailboxPath.inbox((MailboxSession)MailboxManagerTest.this.session), MailboxManagerTest.this.session);
            Assertions.assertThat((Boolean)((Boolean)Mono.from((Publisher)MailboxManagerTest.this.mailboxManager.hasInbox(MailboxManagerTest.this.session)).block())).isTrue();
        }

        @Test
        void creatingMixedCaseINBOXChildShouldNormalizeChildPath() throws Exception {
            MailboxManagerTest.this.session = MailboxManagerTest.this.mailboxManager.createSystemSession(USER_1);
            MailboxManagerTest.this.mailboxManager.startProcessingRequest(MailboxManagerTest.this.session);
            MailboxPath childPath = MailboxPath.forUser((Username)USER_1, (String)"iNbOx.submailbox");
            Optional mailboxId = MailboxManagerTest.this.mailboxManager.createMailbox(childPath, MailboxManagerTest.this.session);
            MessageManager retrievedMailbox = MailboxManagerTest.this.mailboxManager.getMailbox(childPath, MailboxManagerTest.this.session);
            Assertions.assertThat((Boolean)((Boolean)Mono.from((Publisher)MailboxManagerTest.this.mailboxManager.hasInbox(MailboxManagerTest.this.session)).block())).isTrue();
            Assertions.assertThat((Object)((MailboxId)mailboxId.get())).isEqualTo((Object)retrievedMailbox.getId());
        }
    }
}

