/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.streams.processor.internals;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import org.apache.kafka.clients.consumer.Consumer;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.apache.kafka.clients.consumer.ConsumerRecords;
import org.apache.kafka.common.PartitionInfo;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.errors.TimeoutException;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.streams.errors.StreamsException;
import org.apache.kafka.streams.processor.internals.ChangelogReader;
import org.apache.kafka.streams.processor.internals.StateRestorer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class StoreChangelogReader
implements ChangelogReader {
    private static final Logger log = LoggerFactory.getLogger(StoreChangelogReader.class);
    private final Consumer<byte[], byte[]> consumer;
    private final String logPrefix;
    private final Time time;
    private final long partitionValidationTimeoutMs;
    private final Map<String, List<PartitionInfo>> partitionInfo = new HashMap<String, List<PartitionInfo>>();
    private final Map<TopicPartition, StateRestorer> stateRestorers = new HashMap<TopicPartition, StateRestorer>();
    private final Map<TopicPartition, StateRestorer> needsRestoring = new HashMap<TopicPartition, StateRestorer>();
    private final Map<TopicPartition, StateRestorer> needsInitializing = new HashMap<TopicPartition, StateRestorer>();
    private final Map<TopicPartition, Long> endOffsets = new HashMap<TopicPartition, Long>();
    private boolean initialized = false;

    public StoreChangelogReader(String threadId, Consumer<byte[], byte[]> consumer, Time time, long partitionValidationTimeoutMs) {
        this.time = time;
        this.consumer = consumer;
        this.partitionValidationTimeoutMs = partitionValidationTimeoutMs;
        this.logPrefix = String.format("stream-thread [%s]", threadId);
    }

    public StoreChangelogReader(Consumer<byte[], byte[]> consumer, Time time, long partitionValidationTimeoutMs) {
        this("", consumer, time, partitionValidationTimeoutMs);
    }

    @Override
    public void validatePartitionExists(TopicPartition topicPartition, String storeName) {
        long start = this.time.milliseconds();
        if (this.partitionInfo.isEmpty()) {
            try {
                this.partitionInfo.putAll(this.consumer.listTopics());
            }
            catch (TimeoutException e) {
                log.warn("{} Could not list topics so will fall back to partition by partition fetching", (Object)this.logPrefix);
            }
        }
        long endTime = this.time.milliseconds() + this.partitionValidationTimeoutMs;
        while (!this.hasPartition(topicPartition) && this.time.milliseconds() < endTime) {
            try {
                List partitions = this.consumer.partitionsFor(topicPartition.topic());
                if (partitions == null) continue;
                this.partitionInfo.put(topicPartition.topic(), partitions);
            }
            catch (TimeoutException e) {
                throw new StreamsException(String.format("Could not fetch partition info for topic: %s before expiration of the configured request timeout", topicPartition.topic()));
            }
        }
        if (!this.hasPartition(topicPartition)) {
            throw new StreamsException(String.format("Store %s's change log (%s) does not contain partition %s", storeName, topicPartition.topic(), topicPartition.partition()));
        }
        log.debug("{} Took {}ms to validate that partition {} exists", new Object[]{this.logPrefix, this.time.milliseconds() - start, topicPartition});
    }

    @Override
    public void register(StateRestorer restorer) {
        this.stateRestorers.put(restorer.partition(), restorer);
        this.needsInitializing.put(restorer.partition(), restorer);
    }

    @Override
    public Collection<TopicPartition> restore() {
        if (!this.needsInitializing.isEmpty()) {
            this.initialize();
        }
        if (this.needsRestoring.isEmpty()) {
            this.consumer.assign(Collections.emptyList());
            return this.completed();
        }
        HashSet<TopicPartition> partitions = new HashSet<TopicPartition>(this.needsRestoring.keySet());
        ConsumerRecords allRecords = this.consumer.poll(10L);
        for (TopicPartition partition : partitions) {
            this.restorePartition((ConsumerRecords<byte[], byte[]>)allRecords, partition);
        }
        if (this.needsRestoring.isEmpty()) {
            this.consumer.assign(Collections.emptyList());
        }
        return this.completed();
    }

    private void initialize() {
        HashMap<TopicPartition, StateRestorer> newTasksNeedingRestoration = new HashMap<TopicPartition, StateRestorer>();
        if (!this.consumer.subscription().isEmpty()) {
            throw new IllegalStateException(String.format("Restore consumer should have not subscribed to any partitions (%s) beforehand", this.consumer.subscription()));
        }
        this.endOffsets.putAll(this.consumer.endOffsets(this.needsInitializing.keySet()));
        for (Map.Entry<TopicPartition, Long> entry : this.endOffsets.entrySet()) {
            TopicPartition topicPartition = entry.getKey();
            Long offset = entry.getValue();
            StateRestorer restorer = this.needsInitializing.get(topicPartition);
            if (restorer == null) continue;
            if (restorer.checkpoint() >= offset) {
                restorer.setRestoredOffset(restorer.checkpoint());
                continue;
            }
            if (restorer.offsetLimit() == 0L || this.endOffsets.get(topicPartition) == 0L) {
                restorer.setRestoredOffset(0L);
                continue;
            }
            newTasksNeedingRestoration.put(topicPartition, restorer);
        }
        log.debug("{} Starting restoring state stores from changelog topics {}", (Object)this.logPrefix, newTasksNeedingRestoration.keySet());
        HashSet assignment = new HashSet(this.consumer.assignment());
        assignment.addAll(newTasksNeedingRestoration.keySet());
        this.consumer.assign(assignment);
        ArrayList<StateRestorer> needsPositionUpdate = new ArrayList<StateRestorer>();
        for (StateRestorer restorer : newTasksNeedingRestoration.values()) {
            if (restorer.checkpoint() != -1L) {
                this.consumer.seek(restorer.partition(), restorer.checkpoint());
                this.logRestoreOffsets(restorer.partition(), restorer.checkpoint(), this.endOffsets.get(restorer.partition()));
                restorer.setStartingOffset(this.consumer.position(restorer.partition()));
                continue;
            }
            this.consumer.seekToBeginning(Collections.singletonList(restorer.partition()));
            needsPositionUpdate.add(restorer);
        }
        for (StateRestorer restorer : needsPositionUpdate) {
            long position = this.consumer.position(restorer.partition());
            restorer.setStartingOffset(position);
            this.logRestoreOffsets(restorer.partition(), position, this.endOffsets.get(restorer.partition()));
        }
        this.needsRestoring.putAll(newTasksNeedingRestoration);
        this.needsInitializing.clear();
    }

    private void logRestoreOffsets(TopicPartition partition, long startingOffset, Long endOffset) {
        log.debug("{} Restoring partition {} from offset {} to endOffset {}", new Object[]{this.logPrefix, partition, startingOffset, endOffset});
    }

    private Collection<TopicPartition> completed() {
        HashSet<TopicPartition> completed = new HashSet<TopicPartition>(this.stateRestorers.keySet());
        completed.removeAll(this.needsRestoring.keySet());
        log.debug("{} completed partitions {}", (Object)this.logPrefix, completed);
        return completed;
    }

    @Override
    public Map<TopicPartition, Long> restoredOffsets() {
        HashMap<TopicPartition, Long> restoredOffsets = new HashMap<TopicPartition, Long>();
        for (Map.Entry<TopicPartition, StateRestorer> entry : this.stateRestorers.entrySet()) {
            StateRestorer restorer = entry.getValue();
            if (!restorer.isPersistent()) continue;
            restoredOffsets.put(entry.getKey(), restorer.restoredOffset());
        }
        return restoredOffsets;
    }

    @Override
    public void reset() {
        this.partitionInfo.clear();
        this.stateRestorers.clear();
        this.needsRestoring.clear();
        this.endOffsets.clear();
        this.needsInitializing.clear();
    }

    private void restorePartition(ConsumerRecords<byte[], byte[]> allRecords, TopicPartition topicPartition) {
        StateRestorer restorer = this.stateRestorers.get(topicPartition);
        Long endOffset = this.endOffsets.get(topicPartition);
        long pos = this.processNext(allRecords.records(topicPartition), restorer, endOffset);
        restorer.setRestoredOffset(pos);
        if (restorer.hasCompleted(pos, endOffset)) {
            if (pos > endOffset + 1L) {
                throw new IllegalStateException(String.format("Log end offset of %s should not change while restoring: old end offset %d, current offset %d", topicPartition, endOffset, pos));
            }
            log.debug("{} Completed restoring state from changelog {} with {} records ranging from offset {} to {}", new Object[]{this.logPrefix, topicPartition, restorer.restoredNumRecords(), restorer.startingOffset(), restorer.restoredOffset()});
            this.needsRestoring.remove(topicPartition);
        }
    }

    private long processNext(List<ConsumerRecord<byte[], byte[]>> records, StateRestorer restorer, Long endOffset) {
        for (ConsumerRecord<byte[], byte[]> record : records) {
            long offset = record.offset();
            if (restorer.hasCompleted(offset, endOffset)) {
                return offset;
            }
            if (record.key() == null) continue;
            restorer.restore((byte[])record.key(), (byte[])record.value());
        }
        return this.consumer.position(restorer.partition());
    }

    private boolean hasPartition(TopicPartition topicPartition) {
        List<PartitionInfo> partitions = this.partitionInfo.get(topicPartition.topic());
        if (partitions == null) {
            return false;
        }
        for (PartitionInfo partition : partitions) {
            if (partition.partition() != topicPartition.partition()) continue;
            return true;
        }
        return false;
    }
}

