/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jetty.websocket.client.common;

import java.nio.ByteBuffer;
import java.util.List;
import org.eclipse.jetty.io.ByteBufferPool;
import org.eclipse.jetty.util.BufferUtil;
import org.eclipse.jetty.util.log.Log;
import org.eclipse.jetty.util.log.Logger;
import org.eclipse.jetty.websocket.api.MessageTooLargeException;
import org.eclipse.jetty.websocket.api.ProtocolException;
import org.eclipse.jetty.websocket.api.WebSocketBehavior;
import org.eclipse.jetty.websocket.api.WebSocketException;
import org.eclipse.jetty.websocket.api.WebSocketPolicy;
import org.eclipse.jetty.websocket.api.extensions.Extension;
import org.eclipse.jetty.websocket.api.extensions.Frame;
import org.eclipse.jetty.websocket.api.extensions.IncomingFrames;
import org.eclipse.jetty.websocket.client.common.CloseInfo;
import org.eclipse.jetty.websocket.client.common.OpCode;
import org.eclipse.jetty.websocket.client.common.WebSocketFrame;
import org.eclipse.jetty.websocket.client.common.frames.BinaryFrame;
import org.eclipse.jetty.websocket.client.common.frames.CloseFrame;
import org.eclipse.jetty.websocket.client.common.frames.ContinuationFrame;
import org.eclipse.jetty.websocket.client.common.frames.PingFrame;
import org.eclipse.jetty.websocket.client.common.frames.PongFrame;
import org.eclipse.jetty.websocket.client.common.frames.TextFrame;
import org.eclipse.jetty.websocket.client.common.io.payload.DeMaskProcessor;
import org.eclipse.jetty.websocket.client.common.io.payload.PayloadProcessor;

public class Parser {
    private static final Logger LOG = Log.getLogger(Parser.class);
    private final WebSocketPolicy policy;
    private final ByteBufferPool bufferPool;
    private State state = State.START;
    private int cursor = 0;
    private WebSocketFrame frame;
    private Frame priorDataFrame;
    private byte lastDataOpcode;
    private ByteBuffer payload;
    private int payloadLength;
    private PayloadProcessor maskProcessor = new DeMaskProcessor();
    private byte flagsInUse = 0;
    private IncomingFrames incomingFramesHandler;

    public Parser(WebSocketPolicy wspolicy, ByteBufferPool bufferPool) {
        this.bufferPool = bufferPool;
        this.policy = wspolicy;
    }

    private void assertSanePayloadLength(long len) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Payload Length: {} - {}", new Object[]{len, this});
        }
        if (len > Integer.MAX_VALUE) {
            throw new MessageTooLargeException("[int-sane!] cannot handle payload lengths larger than 2147483647");
        }
        switch (this.frame.getOpCode()) {
            case 8: {
                if (len == 1L) {
                    throw new ProtocolException("Invalid close frame payload length, [" + this.payloadLength + "]");
                }
            }
            case 9: 
            case 10: {
                if (len <= 125L) break;
                throw new ProtocolException("Invalid control frame payload length, [" + this.payloadLength + "] cannot exceed [" + 125 + "]");
            }
            case 1: {
                this.policy.assertValidTextMessageSize((int)len);
                break;
            }
            case 2: {
                this.policy.assertValidBinaryMessageSize((int)len);
            }
        }
    }

    public void configureFromExtensions(List<? extends Extension> exts) {
        this.flagsInUse = 0;
        for (Extension extension : exts) {
            if (extension.isRsv1User()) {
                this.flagsInUse = (byte)(this.flagsInUse | 0x40);
            }
            if (extension.isRsv2User()) {
                this.flagsInUse = (byte)(this.flagsInUse | 0x20);
            }
            if (!extension.isRsv3User()) continue;
            this.flagsInUse = (byte)(this.flagsInUse | 0x10);
        }
    }

    public IncomingFrames getIncomingFramesHandler() {
        return this.incomingFramesHandler;
    }

    public WebSocketPolicy getPolicy() {
        return this.policy;
    }

    public boolean isRsv1InUse() {
        return (this.flagsInUse & 0x40) != 0;
    }

    public boolean isRsv2InUse() {
        return (this.flagsInUse & 0x20) != 0;
    }

    public boolean isRsv3InUse() {
        return (this.flagsInUse & 0x10) != 0;
    }

    protected void notifyFrame(Frame f) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("{} Notify {}", new Object[]{this.policy.getBehavior(), this.incomingFramesHandler});
        }
        if (this.policy.getBehavior() == WebSocketBehavior.SERVER) {
            if (!f.isMasked()) {
                throw new ProtocolException("Client MUST mask all frames (RFC-6455: Section 5.1)");
            }
        } else if (this.policy.getBehavior() == WebSocketBehavior.CLIENT && f.isMasked()) {
            throw new ProtocolException("Server MUST NOT mask any frames (RFC-6455: Section 5.1)");
        }
        if (this.incomingFramesHandler == null) {
            return;
        }
        try {
            this.incomingFramesHandler.incomingFrame(f);
        }
        catch (WebSocketException e) {
            this.notifyWebSocketException(e);
        }
        catch (Throwable t) {
            LOG.warn(t);
            this.notifyWebSocketException(new WebSocketException(t));
        }
    }

    protected void notifyWebSocketException(WebSocketException e) {
        LOG.warn((Throwable)e);
        if (this.incomingFramesHandler == null) {
            return;
        }
        this.incomingFramesHandler.incomingError((Throwable)e);
    }

    public synchronized void parse(ByteBuffer buffer) {
        if (buffer.remaining() <= 0) {
            return;
        }
        try {
            while (this.parseFrame(buffer)) {
                LOG.debug("{} Parsed Frame: {}", new Object[]{this.policy.getBehavior(), this.frame});
                this.notifyFrame(this.frame);
                if (this.frame.isDataFrame() && this.frame.isFin()) {
                    this.priorDataFrame = null;
                    continue;
                }
                this.priorDataFrame = this.frame;
            }
        }
        catch (WebSocketException e) {
            buffer.position(buffer.limit());
            this.payload = null;
            this.notifyWebSocketException(e);
        }
        catch (Throwable t) {
            buffer.position(buffer.limit());
            this.payload = null;
            this.notifyWebSocketException(new WebSocketException(t));
        }
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private boolean parseFrame(ByteBuffer buffer) {
        if (buffer.remaining() <= 0) {
            return false;
        }
        LOG.debug("{} Parsing {} bytes", new Object[]{this.policy.getBehavior(), buffer.remaining()});
        while (buffer.hasRemaining()) {
            switch (this.state) {
                case START: {
                    byte b;
                    if (this.frame != null && this.frame.isFin()) {
                        this.frame.reset();
                    }
                    boolean fin = ((b = buffer.get()) & 0x80) != 0;
                    byte opc = (byte)(b & 0xF);
                    byte opcode = opc;
                    if (!OpCode.isKnown(opcode)) {
                        throw new ProtocolException("Unknown opcode: " + opc);
                    }
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("OpCode {}, fin={} rsv={}{}{}", new Object[]{OpCode.name(opcode), fin, Character.valueOf(this.isRsv1InUse() ? (char)'1' : '.'), Character.valueOf(this.isRsv2InUse() ? (char)'1' : '.'), Character.valueOf(this.isRsv3InUse() ? (char)'1' : '.')});
                    }
                    switch (opcode) {
                        case 1: {
                            this.frame = new TextFrame();
                            this.lastDataOpcode = opcode;
                            if (this.priorDataFrame == null || this.priorDataFrame.isFin()) break;
                            throw new ProtocolException("Unexpected " + OpCode.name(opcode) + " frame, was expecting CONTINUATION");
                        }
                        case 2: {
                            this.frame = new BinaryFrame();
                            this.lastDataOpcode = opcode;
                            if (this.priorDataFrame == null || this.priorDataFrame.isFin()) break;
                            throw new ProtocolException("Unexpected " + OpCode.name(opcode) + " frame, was expecting CONTINUATION");
                        }
                        case 0: {
                            this.frame = new ContinuationFrame();
                            this.lastDataOpcode = opcode;
                            if (this.priorDataFrame == null) {
                                throw new ProtocolException("CONTINUATION frame without prior !FIN");
                            }
                            opcode = this.lastDataOpcode;
                            break;
                        }
                        case 8: {
                            this.frame = new CloseFrame();
                            if (fin) break;
                            throw new ProtocolException("Fragmented Close Frame [" + OpCode.name(opcode) + "]");
                        }
                        case 9: {
                            this.frame = new PingFrame();
                            if (fin) break;
                            throw new ProtocolException("Fragmented Ping Frame [" + OpCode.name(opcode) + "]");
                        }
                        case 10: {
                            this.frame = new PongFrame();
                            if (fin) break;
                            throw new ProtocolException("Fragmented Pong Frame [" + OpCode.name(opcode) + "]");
                        }
                    }
                    this.frame.setFin(fin);
                    if ((b & 0x70) != 0) {
                        if ((b & 0x40) != 0) {
                            if (!this.isRsv1InUse()) throw new ProtocolException("RSV1 not allowed to be set");
                            this.frame.setRsv1(true);
                        }
                        if ((b & 0x20) != 0) {
                            if (!this.isRsv2InUse()) throw new ProtocolException("RSV2 not allowed to be set");
                            this.frame.setRsv2(true);
                        }
                        if ((b & 0x10) != 0) {
                            if (!this.isRsv3InUse()) throw new ProtocolException("RSV3 not allowed to be set");
                            this.frame.setRsv3(true);
                        }
                    } else if (LOG.isDebugEnabled()) {
                        LOG.debug("OpCode {}, fin={} rsv=000", new Object[]{OpCode.name(opcode), fin});
                    }
                    this.state = State.PAYLOAD_LEN;
                    break;
                }
                case PAYLOAD_LEN: {
                    byte b = buffer.get();
                    this.frame.setMasked((b & 0x80) != 0);
                    this.payloadLength = (byte)(0x7F & b);
                    if (this.payloadLength == 127) {
                        this.payloadLength = 0;
                        this.state = State.PAYLOAD_LEN_BYTES;
                        this.cursor = 8;
                        break;
                    }
                    if (this.payloadLength == 126) {
                        this.payloadLength = 0;
                        this.state = State.PAYLOAD_LEN_BYTES;
                        this.cursor = 2;
                        break;
                    }
                    this.assertSanePayloadLength(this.payloadLength);
                    if (this.frame.isMasked()) {
                        this.state = State.MASK;
                        break;
                    }
                    if (this.payloadLength == 0) {
                        this.state = State.START;
                        return true;
                    }
                    this.maskProcessor.reset(this.frame);
                    this.state = State.PAYLOAD;
                    break;
                }
                case PAYLOAD_LEN_BYTES: {
                    byte b = buffer.get();
                    --this.cursor;
                    this.payloadLength |= (b & 0xFF) << 8 * this.cursor;
                    if (this.cursor != 0) break;
                    this.assertSanePayloadLength(this.payloadLength);
                    if (this.frame.isMasked()) {
                        this.state = State.MASK;
                        break;
                    }
                    if (this.payloadLength == 0) {
                        this.state = State.START;
                        return true;
                    }
                    this.maskProcessor.reset(this.frame);
                    this.state = State.PAYLOAD;
                    break;
                }
                case MASK: {
                    byte[] m = new byte[4];
                    this.frame.setMask(m);
                    if (buffer.remaining() >= 4) {
                        buffer.get(m, 0, 4);
                        if (this.payloadLength == 0) {
                            this.state = State.START;
                            return true;
                        }
                        this.maskProcessor.reset(this.frame);
                        this.state = State.PAYLOAD;
                        break;
                    }
                    this.state = State.MASK_BYTES;
                    this.cursor = 4;
                    break;
                }
                case MASK_BYTES: {
                    byte b;
                    this.frame.getMask()[4 - this.cursor] = b = buffer.get();
                    --this.cursor;
                    if (this.cursor != 0) break;
                    if (this.payloadLength == 0) {
                        this.state = State.START;
                        return true;
                    }
                    this.maskProcessor.reset(this.frame);
                    this.state = State.PAYLOAD;
                    break;
                }
                case PAYLOAD: {
                    if (!this.parsePayload(buffer)) break;
                    if (this.frame.getOpCode() == 8) {
                        new CloseInfo(this.frame);
                    }
                    this.state = State.START;
                    return true;
                }
            }
        }
        return false;
    }

    private boolean parsePayload(ByteBuffer buffer) {
        if (this.payloadLength == 0) {
            return true;
        }
        if (buffer.hasRemaining()) {
            if (this.payload == null) {
                this.frame.assertValid();
                this.payload = this.bufferPool.acquire(this.payloadLength, false);
                BufferUtil.clearToFill((ByteBuffer)this.payload);
            }
            ByteBuffer window = buffer.slice();
            int bytesExpected = this.payloadLength - this.payload.position();
            int bytesAvailable = buffer.remaining();
            int windowBytes = Math.min(bytesAvailable, bytesExpected);
            window.limit(window.position() + windowBytes);
            if (LOG.isDebugEnabled()) {
                LOG.debug("Window: {}", new Object[]{BufferUtil.toDetailString((ByteBuffer)window)});
            }
            this.maskProcessor.process(window);
            int len = BufferUtil.put((ByteBuffer)window, (ByteBuffer)this.payload);
            buffer.position(buffer.position() + len);
            if (this.payload.position() >= this.payloadLength) {
                BufferUtil.flipToFlush((ByteBuffer)this.payload, (int)0);
                this.frame.setPayload(this.payload);
                this.payload = null;
                return true;
            }
        }
        return false;
    }

    public void setIncomingFramesHandler(IncomingFrames incoming) {
        this.incomingFramesHandler = incoming;
    }

    public String toString() {
        StringBuilder builder = new StringBuilder();
        builder.append("Parser@").append(Integer.toHexString(this.hashCode()));
        builder.append("[");
        if (this.incomingFramesHandler == null) {
            builder.append("NO_HANDLER");
        } else {
            builder.append(this.incomingFramesHandler.getClass().getSimpleName());
        }
        builder.append(",s=").append((Object)this.state);
        builder.append(",c=").append(this.cursor);
        builder.append(",len=").append(this.payloadLength);
        builder.append(",f=").append(this.frame);
        builder.append(",p=").append(this.policy);
        builder.append("]");
        return builder.toString();
    }

    private static enum State {
        START,
        PAYLOAD_LEN,
        PAYLOAD_LEN_BYTES,
        MASK,
        MASK_BYTES,
        PAYLOAD;

    }
}

