/*
 * Copyright 2010-2025 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package org.jetbrains.kotlin.gradle.dsl.abi

import org.gradle.api.Task
import org.gradle.api.file.DirectoryProperty
import org.gradle.api.tasks.TaskProvider
import org.jetbrains.kotlin.gradle.dsl.KotlinGradlePluginDsl
import org.jetbrains.kotlin.gradle.tasks.abi.KotlinLegacyAbiCheckTask
import org.jetbrains.kotlin.gradle.tasks.abi.KotlinLegacyAbiDumpTask

/**
 * Use this extension to migrate ABI dumps that are stored in the legacy format
 * (created by the [Binary compatibility validator plugin](https://github.com/Kotlin/binary-compatibility-validator))
 * to the new ABI dump format.
 *
 * @since 2.1.20
 */
@KotlinGradlePluginDsl
@ExperimentalAbiValidation
interface AbiValidationLegacyDumpExtension {
    /**
     * The directory containing reference dumps that the dump generated from the current code is compared with by the [legacyCheckTaskProvider] task.
     */
    val referenceDumpDir: DirectoryProperty

    /**
     * A provider for the task that generates actual dumps from the current code.
     *
     * These dumps are saved in the [KotlinLegacyAbiDumpTask.dumpDir] directory and used when comparing with dumps from the [referenceDumpDir] by the [legacyCheckTaskProvider] task.
     */
    val legacyDumpTaskProvider: TaskProvider<KotlinLegacyAbiDumpTask>

    /**
     *  A provider for the task that compares actual dumps from the current code generated by the [legacyDumpTaskProvider] with dumps from [referenceDumpDir].
     *
     * This task fails if any differences are found between the files.
     */
    val legacyCheckTaskProvider: TaskProvider<KotlinLegacyAbiCheckTask>

    /**
     * Overwrite dumps in the [referenceDumpDir] directory with the actual dumps from [KotlinLegacyAbiDumpTask.dumpDir] for the current code.
     */
    val legacyUpdateTaskProvider: TaskProvider<Task>
}
