package com.rabbit.blade.comm.exeption;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import android.text.TextUtils;

import com.rabbit.blade.comm.util.LogUtil;

import java.util.HashMap;
import java.util.Map;

/**
 * <b>Description: 异常映射关系缓存对象</b>
 * Created by <a href="mailto:fjd@xdja.com">fanjiandong</a> on 2017/5/8 19:20.
 */

public class OkMapper {
    private final Map<String, Map<String, String>> errorMapper;

    private String mark = "OkMapper";

    public OkMapper() {
        this.errorMapper = new HashMap<>(4);
    }

    public OkMapper(String mark) {
        this();
        this.mark = mark;
    }

    /**
     * 注册一个异常映射关系
     *
     * @param cls       异常类型
     * @param errorCode 错误码
     * @param userMsg   错误提示信息
     */
    public OkMapper registSingleMapper(@NonNull Class<? extends OkException> cls,
                                       @NonNull String errorCode,
                                       @NonNull String userMsg) {
        Map<String, String> msgMapper = this.errorMapper.get(cls.getName());
        if (msgMapper == null) {
            msgMapper = new HashMap<>();
            this.registMultipleMapper(cls, msgMapper);
        }
        msgMapper.put(errorCode, userMsg);
        return this;
    }

    /**
     * 注销一个异常映射关系
     *
     * @param cls       异常类型
     * @param errorCode 错误码
     */
    public OkMapper unRegistSingleMapper(@NonNull Class<? extends OkException> cls,
                                         @NonNull String errorCode) {
        Map<String, String> msgMapper = this.errorMapper.get(cls.getName());
        if (msgMapper != null && msgMapper.containsKey(errorCode)) {
            msgMapper.remove(errorCode);
        }
        return this;
    }

    /**
     * 注册一组异常映射关系
     *
     * @param exeCls 异常类型
     * @param mapper 成组的异常映射关系
     */
    public OkMapper registMultipleMapper(@NonNull Class<? extends OkException> exeCls,
                                         @Nullable Map<String, String> mapper) {
        Map<String, String> msgMapper = this.errorMapper.get(exeCls.getName());
        if (msgMapper != null && mapper != null) {
            msgMapper.putAll(mapper);
        } else {
            this.errorMapper.put(exeCls.getName(), mapper);
        }
        return this;
    }

    /**
     * 注销一组异常映射关系
     *
     * @param exeCls 异常类型
     */
    public OkMapper unRegistMultipleMapper(@NonNull Class<? extends OkException> exeCls) {
        this.errorMapper.remove(exeCls.getName());
        return this;
    }

    /**
     * 注销所有异常映射关系
     */
    public OkMapper reset() {
        this.errorMapper.clear();
        return this;
    }

    /**
     * 匹配异常对应的用户信息
     *
     * @param ex 异常对象
     * @return 匹配到的用户信息 string
     */
    @Nullable
    public String map(@Nullable OkException ex) {
        if (ex == null) {
            LogUtil.getUtils().d(mark + " : " + "异常对象为空");
            return null;
        }
        Map<String, String> exceptionMapper = this.errorMapper.get(ex.getClass().getName());
        if (exceptionMapper == null) {
            LogUtil.getUtils().d(mark + " : " + "该异常对应的用户信息匹配器为空");
            return null;
        }
        String okCode = ex.getOkCode();
        if (TextUtils.isEmpty(okCode)) {
            LogUtil.getUtils().d(mark + " : " + "该异常错误码为空");
            return null;
        }
        String result = exceptionMapper.get(okCode);
        if (result == null) {
            LogUtil.getUtils().d(mark + " : " + "未找到该异常信息对应的用户信息匹配");
            return null;
        }

        return result;
    }
}
