package com.rabbit.blade.data.net;

import android.content.Context;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.text.TextUtils;

import com.rabbit.blade.comm.annotation.DiConfig;
import com.rabbit.blade.comm.annotation.qua.AroundQua;
import com.rabbit.blade.comm.util.LogUtil;
import com.rabbit.blade.data.cache.ConfigCache;

import java.io.IOException;
import java.io.InputStream;
import java.security.KeyManagementException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.security.UnrecoverableKeyException;
import java.security.cert.CertificateException;
import java.util.concurrent.TimeUnit;

import javax.inject.Inject;
import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLSession;
import javax.net.ssl.TrustManagerFactory;

import okhttp3.OkHttpClient;

/**
 * <b>Description : https的okhttp实现</b>
 * <p>Created by <a href="mailto:fanjiandong@outlook.com">fanjiandong</a> on 2017/3/2 9:07.</p>
 */


public class OkHttpsBuilder {
    /**
     * 默认的超时时间
     */
    public static final int CONN_TIME_OUT_UNIT = 30 * 1000;
    public static final int READ_TIME_OUT_UNIT = 15 * 1000;
    public static final int WRITE_TIME_OUT_UNIT = 15 * 1000;


    private final Context context;
    private final ConfigCache configCache;
    private final HostnameVerifier hostnameVerifier;

    @Inject
    public OkHttpsBuilder(@AroundQua(DiConfig.QUA_APPLICATION) Context context,
                          ConfigCache configCache) {
        this.context = context;
        this.configCache = configCache;
        this.hostnameVerifier = new HostnameVerifier() {
            @Override
            public boolean verify(String hostname, SSLSession session) {
                String hostnameLocal = OkHttpsBuilder.this.configCache.getConfig("hostname");
                return !TextUtils.isEmpty(hostnameLocal) && hostnameLocal.equals(hostname);
            }
        };
    }

    public OkHttpClient.Builder build() {
        SSLContext sslContext = null;
        try {
            sslContext = getSSLContext();
        } catch (IOException
                | KeyStoreException
                | NoSuchAlgorithmException
                | UnrecoverableKeyException
                | CertificateException
                | KeyManagementException e) {
            LogUtil.getUtils().w(e.getMessage());
        }

        if (sslContext == null) {
            return null;
        }

        OkHttpClient.Builder builder = new OkHttpClient.Builder();
        builder.connectTimeout(CONN_TIME_OUT_UNIT, TimeUnit.MILLISECONDS)
                .readTimeout(READ_TIME_OUT_UNIT, TimeUnit.MILLISECONDS)
                .writeTimeout(WRITE_TIME_OUT_UNIT, TimeUnit.MILLISECONDS)
                .sslSocketFactory(sslContext.getSocketFactory())
                .hostnameVerifier(this.hostnameVerifier);
        return builder;
    }


    /**
     * 获取SSL连接上下文
     *
     * @return sslContext
     */
    @Nullable
    private SSLContext getSSLContext()
            throws NoSuchAlgorithmException, KeyStoreException, UnrecoverableKeyException, KeyManagementException, IOException, CertificateException {

        SSLContext sslContext = SSLContext.getInstance("TLS");

        KeyStore keyStore = readKeyStore();

        TrustManagerFactory trustManagerFactory = getTrustManagerFactory(keyStore);

        KeyManagerFactory keyManagerFactory = getKeyManagerFactory(keyStore);

        sslContext.init(
                keyManagerFactory.getKeyManagers(),
                trustManagerFactory.getTrustManagers(),
                new SecureRandom()
        );

        return sslContext;
    }

    /**
     * 读取本地证书
     *
     * @return 获取到的证书
     */
    @Nullable
    private KeyStore readKeyStore()
            throws IOException, KeyStoreException, NoSuchAlgorithmException, CertificateException {
        KeyStore keyStore = KeyStore.getInstance(KeyStore.getDefaultType());
        if (configCache != null) {
            String certPath = configCache.getConfig("certpath");
            String certPwd = configCache.getConfig("certpwd");
            if (!TextUtils.isEmpty(certPath) && !TextUtils.isEmpty(certPwd)) {
                InputStream inputStream = this.context.getAssets().open(certPath);
                keyStore.load(inputStream, certPwd.toCharArray());
            }
        }
        return keyStore;
    }

    private KeyManagerFactory getKeyManagerFactory(@Nullable KeyStore keyStore)
            throws NoSuchAlgorithmException, KeyStoreException, UnrecoverableKeyException {
        KeyManagerFactory keyManagerFactory =
                KeyManagerFactory.getInstance(KeyManagerFactory.getDefaultAlgorithm());
        String certPwd = configCache.getConfig("certpwd");
        if (!TextUtils.isEmpty(certPwd)) {
            keyManagerFactory.init(keyStore, certPwd.toCharArray());
        }
        return keyManagerFactory;
    }

    @NonNull
    private TrustManagerFactory getTrustManagerFactory(
            @Nullable KeyStore keyStore) throws NoSuchAlgorithmException, KeyStoreException {
        TrustManagerFactory trustManagerFactory =
                TrustManagerFactory.getInstance(TrustManagerFactory.getDefaultAlgorithm());
        trustManagerFactory.init(keyStore);
        return trustManagerFactory;
    }
}
