package com.rabbit.blade.data.persistent;


import android.content.Context;
import android.content.SharedPreferences;
import android.content.SharedPreferences.Editor;
import android.preference.PreferenceManager;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.text.TextUtils;

import com.rabbit.blade.comm.annotation.DiConfig;
import com.rabbit.blade.comm.annotation.qua.AroundQua;

import javax.inject.Inject;


/**
 * <b>Description : 持久化文件操作类</b>
 * <p>Created by <a href="mailto:fanjiandong@outlook.com">fanjiandong</a> on 2017/3/15 18:17.</p>
 */

public class PreferencesEditor {

    private final SharedPreferences sp;

    /**
     * Instantiates a new Preferences editor.
     *
     * @param context the context
     */
    @Inject
    public PreferencesEditor(@AroundQua(DiConfig.QUA_APPLICATION) Context context) {
        sp = PreferenceManager.getDefaultSharedPreferences(context);
    }

    /**
     * Instantiates a new Preferences editor.
     *
     * @param context  the context
     * @param prefName the pref name
     */
    public PreferencesEditor(Context context, @Nullable String prefName) {
        if (!TextUtils.isEmpty(prefName)) {
            sp = context.getSharedPreferences(prefName, Context.MODE_PRIVATE);
        } else {
            sp = PreferenceManager.getDefaultSharedPreferences(context);
        }
    }


    /**
     * Sets string value.
     *
     * @param key   the key
     * @param value the value
     * @return the string value
     */
    public boolean setStringValue(@NonNull String key, @Nullable String value) {
        Editor editor = sp.edit();
        editor.putString(key, value);
        return editor.commit();
    }


    /**
     * Sets boolean value.
     *
     * @param key   the key
     * @param value the value
     * @return the boolean value
     */
    public boolean setBooleanValue(@NonNull String key, boolean value) {
        Editor editor = sp.edit();
        editor.putBoolean(key, value);
        return editor.commit();
    }


    /**
     * Sets float value.
     *
     * @param key   the key
     * @param value the value
     * @return the float value
     */
    public boolean setFloatValue(@NonNull String key, float value) {
        Editor editor = sp.edit();
        editor.putFloat(key, value);
        return editor.commit();
    }


    /**
     * Sets int value.
     *
     * @param key   the key
     * @param value the value
     * @return the int value
     */
    public boolean setIntValue(@NonNull String key, int value) {
        Editor editor = sp.edit();
        editor.putInt(key, value);
        return editor.commit();
    }


    /**
     * Sets long value.
     *
     * @param key   the key
     * @param value the value
     * @return the long value
     */
    public boolean setLongValue(@NonNull String key, long value) {
        Editor editor = sp.edit();
        editor.putLong(key, value);
        return editor.commit();
    }

    /**
     * Gets string value.
     *
     * @param key          the key
     * @param defaultValue the default value
     * @return the string value
     */
    @Nullable
    public String getStringValue(@NonNull String key, @Nullable String defaultValue) {
        return sp.getString(key, defaultValue);
    }

    /**
     * Gets boolean value.
     *
     * @param key          the key
     * @param defaultValue the default value
     * @return the boolean value
     */
    public Boolean getBooleanValue(@NonNull String key, boolean defaultValue) {
        return sp.getBoolean(key, defaultValue);
    }

    /**
     * Gets int value.
     *
     * @param key          the key
     * @param defaultValue the default value
     * @return the int value
     */
    public int getIntValue(@NonNull String key, int defaultValue) {
        return sp.getInt(key, defaultValue);
    }

    /**
     * Gets long value.
     *
     * @param key          the key
     * @param defaultValue the default value
     * @return the long value
     */
    public long getLongValue(@NonNull String key, long defaultValue) {
        return sp.getLong(key, defaultValue);
    }

    /**
     * Gets float value.
     *
     * @param key          the key
     * @param defaultValue the default value
     * @return the float value
     */
    public float getFloatValue(@NonNull String key, float defaultValue) {
        return sp.getFloat(key, defaultValue);
    }

    /**
     * Contains key boolean.
     *
     * @param key the key
     * @return the boolean
     */
    public boolean containsKey(@NonNull String key) {
        return sp.contains(key);
    }

    /**
     * Remove key.
     *
     * @param key the key
     */
    public void removeKey(@NonNull String key) {
        Editor editor = sp.edit();
        editor.remove(key);
        editor.apply();
    }
}
