package com.rabbit.blade.data.persistent;

import android.content.Context;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.text.TextUtils;

import com.rabbit.blade.comm.annotation.DiConfig;
import com.rabbit.blade.comm.annotation.qua.AroundQua;
import com.rabbit.blade.comm.util.LogUtil;

import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import javax.inject.Inject;

/**
 * <b>Description : 配置文件操作类</b>
 * <p>Created by <a href="mailto:fanjiandong@outlook.com">fanjiandong</a> on 2017/3/15 17:10.</p>
 */

public class PropertyReader {
    /**
     * 上下文句柄
     */
    private final Context context;
    /**
     * 配置文件对象
     */
    private Properties properties;

    @Inject
    public PropertyReader(@NonNull @AroundQua(DiConfig.QUA_APPLICATION) Context context) {
        this.context = context;
    }

    /**
     * 从某个文件中加载配置
     *
     * @param fileName 文件名称
     */
    public void load(@NonNull String fileName) throws IOException {

        if (TextUtils.isEmpty(fileName)) {
            return;
        }
        InputStream is = null;
        try {
            is = this.context.getAssets().open(fileName);
            this.properties = new Properties();
            this.properties.load(is);
        } finally {
            if (is != null) {
                try {
                    is.close();
                } catch (IOException e) {
                    LogUtil.getUtils().w(e.getMessage());
                }
            }
        }
    }

    /**
     * 获取一个配置的值
     *
     * @param name key
     * @return value
     */
    public String get(@NonNull String name) {
        if (properties != null && !TextUtils.isEmpty(name)) {
            return properties.getProperty(name);
        }
        return null;
    }

    /**
     * 获取一个配置的值
     *
     * @param name         key
     * @param defaultValue 默认值
     * @return value
     */
    public String get(@NonNull String name, @Nullable String defaultValue) {
        if (properties != null && !TextUtils.isEmpty(name)) {
            return properties.getProperty(name, defaultValue);
        }
        return null;
    }

    /**
     * 获取所有的配置信息
     *
     * @return 配置信息集合
     */
    public Map<String, String> getAll() {
        if (properties != null) {
            Set<String> keys = properties.stringPropertyNames();
            if (keys != null && !keys.isEmpty()) {
                Map<String, String> kv = new HashMap<>();
                for (String key : keys) {
                    String value = properties.getProperty(key);
                    kv.put(key, value);
                }
                return kv;
            }
        }
        return null;
    }
}
