package com.rabbit.blade.domain.usecase;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.rabbit.blade.domain.event.AgentMessage;

import rx.Subscriber;
import rx.Subscription;
import rx.functions.Action0;
import rx.functions.Action1;
import rx.functions.Action2;

/**
 * <b>Description : 基本业务用例接口定义</b>
 * <p>Created by <a href="mailto:fanjiandong@outlook.com">fanjiandong</a> on 2017/3/15 16:01.</p>
 *
 * @param <Result> the type parameter
 */
public interface Interactor<Result> {

    /**
     * 用例执行入口
     *
     * @param useCaseSubscriber 订阅者
     * @return the subscription
     */
    @Nullable
    Subscription execute(@NonNull Subscriber<? super Result> useCaseSubscriber);

    /**
     * 用例执行入口
     *
     * @param useCaseSubscriber 订阅者
     * @param isConnectable     是否延迟执行
     * @return the subscription
     */

    @Nullable
    Subscription execute(@NonNull
                                 Subscriber<? super Result> useCaseSubscriber, boolean isConnectable);

    /**
     * 用例执行入口
     *
     * @param action1 订阅者
     * @return the subscription
     */
    @Nullable
    Subscription execute(@NonNull Action1<? super Result> action1);

    /**
     * 用例执行入口
     *
     * @param action1       订阅者
     * @param isConnectable 是否延迟执行
     * @return the subscription
     */

    @Nullable
    Subscription execute(@NonNull Action1<? super Result> action1, boolean isConnectable);

    /**
     * 执行可延迟的用例
     *
     * @return the subscription
     */
    @Nullable
    Subscription connect();

    /**
     * 注销接收代理消息
     *
     * @param <I> the type parameter
     * @return the subscription
     */
    <I extends Interactor<Result>> I unSubscribeMessage();

    /**
     * 订阅用例执行过程中的消息（常用于更新进度，获取中间执行结果等操作）
     *
     * @param <I>                 the type parameter
     * @param agentMessageAction1 消息订阅者
     * @param isSerial            是否串行
     * @return the subscription
     */

    <I extends Interactor<Result>> I subscribeMessage(
            @Nullable Action1<AgentMessage<?>> agentMessageAction1, boolean isSerial);

    /**
     * Subscribe message with subscription.
     *
     * @param agentMessageAction1 消息订阅者
     * @param isSerial            是否串行
     * @return the subscription
     */
    @Nullable
    Subscription subscribeMessageWith(
            @Nullable Action1<AgentMessage<?>> agentMessageAction1, boolean isSerial);

    /**
     * 释放相关资源(取消所有的订阅，包括对结果的订阅和对代理消息的订阅)
     */
    <I extends Interactor<Result>> I cancel();

    /**
     * 标识资源是否被释放
     *
     * @return 是否被释放
     */
    boolean isCanceled();

    /**
     * 用例执行之前调用该动作
     *
     * @param <I>           the type parameter
     * @param onStartAction 要执行的操作
     * @return 当前用例对象
     */

    <I extends Interactor<Result>> I doOnStart(Action0 onStartAction);

    /**
     * {@link rx.Observable#doOnNext}
     *
     * @param <I>           the type parameter
     * @param onNextAction1 the on next action 1
     * @return the
     */

    <I extends Interactor<Result>> I doOnNext(Action1<? super Result> onNextAction1);

    /**
     * {@link rx.Observable#doOnError}
     *
     * @param <I>            the type parameter
     * @param onErrorAction1 the on error action 1
     * @return the
     */

    <I extends Interactor<Result>> I doOnError(Action1<Throwable> onErrorAction1);

    /**
     * {@link rx.Observable#doOnCompleted}
     *
     * @param <I>               the type parameter
     * @param onCompletedAction the on completed action
     * @return the
     */

    <I extends Interactor<Result>> I doOnCompleted(Action0 onCompletedAction);

    /**
     * {@link rx.Observable#doOnSubscribe}
     *
     * @param <I>             the type parameter
     * @param subscribeAction the subscribe action
     * @return the
     */

    <I extends Interactor<Result>> I doOnSubscribe(Action0 subscribeAction);

    /**
     * {@link rx.Observable#doOnUnsubscribe}
     *
     * @param <I>               the type parameter
     * @param unsubscribeAction the unsubscribe action
     * @return the
     */

    <I extends Interactor<Result>> I doOnUnsubscribe(Action0 unsubscribeAction);

    /**
     * {@link rx.Observable#doOnTerminate}
     *
     * @param <I>               the type parameter
     * @param onTerminateAction the on terminate action
     * @return the
     */

    <I extends Interactor<Result>> I doOnTerminate(Action0 onTerminateAction);

    /**
     * {@link rx.Observable#doAfterTerminate}
     *
     * @param <I>                  the type parameter
     * @param afterTerminateAction the after terminate action
     * @return the
     */

    <I extends Interactor<Result>> I doAfterTerminate(Action0 afterTerminateAction);

    /**
     * 为当前用例前置一个业务用例
     * （是将当前用例和待前置的用例进行组合拼接，待前置用例在当前用例之前执行，执行完成后继续执行当前用例）
     *
     * @param <I>           当前用例类型定义
     * @param <I1>          待前置用例的类型定义
     * @param preInteractor 待前置的用例
     * @param preposeAction 前置用例执行结果与向当前用例的过渡规则
     * @return 组合后的新用例
     */
    <I extends Interactor<Result>, I1 extends Interactor<?>>
    I preposeInteractor(
            @Nullable I1 preInteractor, @Nullable Action2<Object, I> preposeAction);

    /**
     * 为当前用例后置一个业务用例
     * （是将当前用例和待后置的用例进行组合拼接，先执行当前用例，然后执行后置用例）
     *
     * @param <Result1>         后置用例的返回值类型定义
     * @param <I1>              后置用例类型定义
     * @param composeInteractor 待后置的用例
     * @param composeAction     当前用例执行结果与后置用例的过渡规则
     * @return 组合后的新用例 1
     */
    <Result1, I1 extends Interactor<Result1>>
    I1 composeInteractor(
            @NonNull I1 composeInteractor, @NonNull Action2<Result, I1> composeAction);


    /**
     * 为当前用例传入参数
     *
     * @param <I>         用例类型定义
     * @param domainParam 参数对象
     * @return 用例对象
     */

    <I extends Interactor<Result>> I params(@Nullable DomainParam domainParam);

    /**
     * 为当前用例传入参数
     *
     * @param <I>     用例类型定义
     * @param <Param> 参数1类型定义
     * @param param   参数1
     * @return 用例对象
     */

    <I extends Interactor<Result>, Param> I params(@Nullable Param param);

    /**
     * 为当前用例传入参数
     *
     * @param <I>      用例类型定义
     * @param <Param>  参数1类型定义
     * @param <Param1> 参数2类型定义
     * @param param    参数1
     * @param param1   参数2
     * @return 用例对象
     */

    <I extends Interactor<Result>, Param, Param1> I params(
            @Nullable Param param, @Nullable Param1 param1);

    /**
     * 为当前用例传入参数
     *
     * @param <I>      用例类型定义
     * @param <Param>  参数1类型定义
     * @param <Param1> 参数2类型定义
     * @param <Param2> 参数3类型定义
     * @param param    参数1
     * @param param1   参数2
     * @param param2   参数3
     * @return 用例对象
     */

    <I extends Interactor<Result>, Param, Param1, Param2> I params(
            @Nullable Param param,
            @Nullable Param1 param1, @Nullable Param2 param2);

    /**
     * 为当前用例传入参数
     *
     * @param <I>      用例类型定义
     * @param <Param>  参数1类型定义
     * @param <Param1> 参数2类型定义
     * @param <Param2> 参数3类型定义
     * @param <Param3> 参数4类型定义
     * @param param    参数1
     * @param param1   参数2
     * @param param2   参数3
     * @param param3   参数4
     * @return 用例对象
     */

    <I extends Interactor<Result>, Param, Param1, Param2, Param3> I params(
            @Nullable Param param,
            @Nullable Param1 param1,
            @Nullable Param2 param2, @Nullable Param3 param3);

    /**
     * 为当前用例传入参数
     *
     * @param <I>      用例类型定义
     * @param <Param>  参数1类型定义
     * @param <Param1> 参数2类型定义
     * @param <Param2> 参数3类型定义
     * @param <Param3> 参数4类型定义
     * @param <Param4> 参数5类型定义
     * @param param    参数1
     * @param param1   参数2
     * @param param2   参数3
     * @param param3   参数4
     * @param param4   参数5
     * @return 用例对象
     */

    <I extends Interactor<Result>, Param, Param1, Param2, Param3, Param4> I params(
            @Nullable Param param,
            @Nullable Param1 param1,
            @Nullable Param2 param2,
            @Nullable Param3 param3, @Nullable Param4 param4);

    /**
     * 获取参数1
     *
     * @param <Param> 参数1类型定义
     * @return 获取结果 param
     */
    <Param> Param getParam();

    /**
     * 获取参数2
     *
     * @param <Param1> 参数2类型定义
     * @return 获取结果 param 1
     */
    <Param1> Param1 getParam1();

    /**
     * 获取参数3
     *
     * @param <Param2> 参数3类型定义
     * @return 获取结果 param 2
     */
    <Param2> Param2 getParam2();

    /**
     * 获取参数4
     *
     * @param <Param3> 参数4类型定义
     * @return 获取结果 param 3
     */
    <Param3> Param3 getParam3();

    /**
     * 获取参数5
     *
     * @param <Param4> 参数5类型定义
     * @return 获取结果 param 4
     */
    <Param4> Param4 getParam4();
}
