package com.rabbit.blade.domain.usecase;


import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.rabbit.blade.domain.event.AgentMessage;
import com.rabbit.blade.domain.event.DomainEventProvider;
import com.rabbit.blade.domain.event.MessageEvent;
import com.rabbit.blade.domain.excutor.PostExecutionThread;
import com.rabbit.blade.domain.excutor.ThreadExecutor;

import java.util.HashSet;
import java.util.Set;

import rx.Observable;
import rx.Subscriber;
import rx.Subscription;
import rx.functions.Action0;
import rx.functions.Action1;
import rx.functions.Action2;
import rx.functions.Func1;
import rx.observables.ConnectableObservable;
import rx.schedulers.Schedulers;
import rx.subjects.PublishSubject;

import static com.rabbit.blade.domain.usecase.DomainParam.NAME_PARAM;
import static com.rabbit.blade.domain.usecase.DomainParam.NAME_PARAM1;
import static com.rabbit.blade.domain.usecase.DomainParam.NAME_PARAM2;
import static com.rabbit.blade.domain.usecase.DomainParam.NAME_PARAM3;
import static com.rabbit.blade.domain.usecase.DomainParam.NAME_PARAM4;

/**
 * <b>Description : 业务用例基类</b>
 * <p>Created by <a href="mailto:fanjiandong@outlook.com">fanjiandong</a> on 2017/3/15 16:01.</p>
 *
 * @param <Result> the type parameter
 */
public abstract class UseCase<Result> implements Interactor<Result> {
    @NonNull
    private final ThreadExecutor executor;
    @NonNull
    private final PostExecutionThread mainThread;
    @Nullable
    private PublishSubject<AgentMessage<?>> agentMessagePublishSubject;

    @Nullable
    private Action1<? super Result> onNextAction1;
    @Nullable
    private Action1<Throwable> onErrorAction1;
    @Nullable
    private Action0 onCompletedAction;
    @Nullable
    private Action0 subscribeAction;
    @Nullable
    private Action0 unsubscribeAction;
    @Nullable
    private Action0 onTerminateAction;
    @Nullable
    private Action0 afterTerminateAction;
    @Nullable
    private Action0 onStartAction;

    @Nullable
    private Interactor<?> preInteractor;
    @Nullable
    private Action2<Object, Interactor<Result>> preposeAction;
    @Nullable
    private DomainParam domainParam;

    private Observable<Result> source;
    private ConnectableObservable<Result> connSource;
    private Set<Subscription> msgSubscriptions;
    private Set<Subscription> useCaseSubscriptions;

    private boolean isCanceled = false;
    private boolean isRegistedAgentMessage = false;

    private long eventCategory = Long.MAX_VALUE;


    /**
     * Instantiates a new Use case.
     *
     * @param threadExecutor      the thread executor
     * @param postExecutionThread the post execution thread
     */
    public UseCase(final @NonNull ThreadExecutor threadExecutor, final @NonNull
            PostExecutionThread postExecutionThread) {
        this.executor = threadExecutor;
        this.mainThread = postExecutionThread;
    }

    /**
     * 构建业务处理事件流
     *
     * @return 目标事件流 observable
     */
    public abstract Observable<Result> buildUseCaseObservable();

    /**
     * 业务执行入口
     *
     * @param useCaseSubscriber 监听处理类
     */
    @Override
    @Nullable
    public final Subscription execute(
            @NonNull final Subscriber<? super Result> useCaseSubscriber, boolean isConnectable) {
        return executeWith(useCaseSubscriber, null, isConnectable);
    }

    @Override
    @Nullable
    public final Subscription execute(@NonNull final Subscriber<? super Result> useCaseSubscriber) {
        return execute(useCaseSubscriber, false);
    }

    /**
     * 业务执行入口
     *
     * @param action1 监听处理类（只监听onNext方法的返回值）
     */
    @Override
    @Nullable
    public final Subscription execute(
            @NonNull final Action1<? super Result> action1, boolean isConnectable) {
        return executeWith(null, action1, isConnectable);
    }

    @Override
    public final Subscription execute(@NonNull final Action1<? super Result> action1) {
        return execute(action1, false);
    }

    @Override
    @Nullable
    public final Subscription connect() {
        if (this.connSource != null && !this.isCanceled) {
            if (this.onStartAction != null) {
                this.onStartAction.call();
            }
            Subscription subscription = this.connSource.connect();
            return pushUseCaseSubscription(subscription);
        }
        return null;
    }

    @SuppressWarnings("unchecked")
    @Override
    public final <I extends Interactor<Result>> I subscribeMessage(
            @Nullable final Action1<AgentMessage<?>> agentMessageAction1, boolean isSerial) {
        subscribeMessageWith(agentMessageAction1, isSerial);
        return ((I) this);
    }

    @Override
    @Nullable
    public final Subscription subscribeMessageWith(
            @Nullable Action1<AgentMessage<?>> agentMessageAction1, boolean isSerial) {
        if (agentMessageAction1 == null || this.isCanceled) return null;

        if (!this.isRegistedAgentMessage) {
            this.isRegistedAgentMessage = true;
            DomainEventProvider.instance().regist(this);
        }

        if (this.agentMessagePublishSubject == null) {
            this.agentMessagePublishSubject = PublishSubject.create();
        }
        Subscription subscription;
        if (isSerial) {
            subscription = this.agentMessagePublishSubject.subscribe(agentMessageAction1);
        } else {
            subscription = this.agentMessagePublishSubject
                    //在主线程监听
                    .observeOn(mainThread.getScheduler())
                    .subscribe(agentMessageAction1);
        }
        return pushMsgSubscription(subscription);
    }


    @SuppressWarnings("unchecked")
    @Override
    public final <I extends Interactor<Result>> I cancel() {
        this.unSubscribeMessage();

        if (!this.isCanceled) {
            if (this.useCaseSubscriptions != null && !this.useCaseSubscriptions.isEmpty()) {
                for (Subscription subscription : this.useCaseSubscriptions) {
                    if (subscription != null && !subscription.isUnsubscribed()) {
                        subscription.unsubscribe();
                    }
                }
            }
            this.isCanceled = true;
        }

        return ((I) this);
    }

    @Override
    public final boolean isCanceled() {
        return this.isCanceled;
    }

    @SuppressWarnings("unchecked")
    @Override
    public final <I extends Interactor<Result>> I unSubscribeMessage() {
        if (this.isRegistedAgentMessage) {
            if (this.msgSubscriptions != null && !this.msgSubscriptions.isEmpty()) {
                for (Subscription subscription : this.msgSubscriptions) {
                    if (subscription != null && !subscription.isUnsubscribed()) {
                        subscription.unsubscribe();
                    }
                }
            }

            DomainEventProvider.instance().unRegist(this);
            this.isRegistedAgentMessage = false;
        }
        return ((I) this);
    }

    /**
     * Dispatch message event.
     *
     * @param messageEvent the message event
     */

    public void dispatchMessageEvent(@Nullable final MessageEvent<?> messageEvent) {

        if (messageEvent == null || this.isCanceled || !this.isRegistedAgentMessage) return;

        Long category = messageEvent.getCategory();
        if (category == null) category = Long.MAX_VALUE;
        if (category != this.eventCategory) return;

        AgentMessage<?> agentMessage = messageEvent.getMaterialContent();

        if (this.agentMessagePublishSubject != null) {
            this.agentMessagePublishSubject.onNext(agentMessage);
        }
    }

    @SuppressWarnings("unchecked")
    @Override
    public final <I extends Interactor<Result>> I doOnNext(final Action1<? super Result> onNextAction1) {
        this.onNextAction1 = onNextAction1;
        return ((I) this);
    }

    @SuppressWarnings("unchecked")
    @Override
    public final <I extends Interactor<Result>> I doOnError(final Action1<Throwable> onErrorAction1) {
        this.onErrorAction1 = onErrorAction1;
        return ((I) this);
    }

    @SuppressWarnings("unchecked")
    @Override
    public final <I extends Interactor<Result>> I doOnCompleted(final Action0 onCompletedAction) {
        this.onCompletedAction = onCompletedAction;
        return ((I) this);
    }

    @SuppressWarnings("unchecked")
    @Override
    public final <I extends Interactor<Result>> I doOnSubscribe(final Action0 subscribeAction) {
        this.subscribeAction = subscribeAction;
        return ((I) this);
    }

    @SuppressWarnings("unchecked")
    @Override
    public final <I extends Interactor<Result>> I doOnUnsubscribe(final Action0 unsubscribeAction) {
        this.unsubscribeAction = unsubscribeAction;
        return ((I) this);
    }

    @SuppressWarnings("unchecked")
    @Override
    public final <I extends Interactor<Result>> I doOnTerminate(final Action0 onTerminateAction) {
        this.onTerminateAction = onTerminateAction;
        return ((I) this);
    }

    @SuppressWarnings("unchecked")
    @Override
    public final <I extends Interactor<Result>> I doAfterTerminate(final Action0 afterTerminateAction) {
        this.afterTerminateAction = afterTerminateAction;
        return ((I) this);
    }

    @SuppressWarnings("unchecked")
    @Override
    public final <I extends Interactor<Result>> I doOnStart(Action0 onStartAction) {
        this.onStartAction = onStartAction;
        return ((I) this);
    }

    @SuppressWarnings("unchecked")
    @Override
    public final <I extends Interactor<Result>, I1 extends Interactor<?>>
    I preposeInteractor(
            @Nullable final I1 preInteractor, @Nullable final Action2<Object, I> preposeAction) {
        this.preInteractor = preInteractor;
        this.preposeAction = (Action2<Object, Interactor<Result>>) preposeAction;
        return ((I) this);
    }

    @SuppressWarnings("unchecked")
    @Override
    public final <Result1, I1 extends Interactor<Result1>> I1 composeInteractor(
            @NonNull final I1 composeInteractor,
            @NonNull final Action2<Result, I1> composeAction) {
        composeInteractor.preposeInteractor(this, ((Action2<Object, I1>) composeAction));
        return composeInteractor;
    }

    /**
     * Sets event category.
     *
     * @param eventCategory the event category
     */
    public final void setEventCategory(long eventCategory) {
        this.eventCategory = eventCategory;
    }

    @SuppressWarnings("unchecked")
    @Override
    public final <I extends Interactor<Result>> I params(@Nullable final DomainParam domainParam) {
        this.domainParam = domainParam;
        return ((I) this);
    }

    @SuppressWarnings("unchecked")
    @Override
    public final <I extends Interactor<Result>, Param> I params(@Nullable final Param param) {
        if (this.domainParam == null) {
            this.domainParam = new DomainParam();
        }
        this.domainParam.param(NAME_PARAM, param);
        return ((I) this);
    }

    @SuppressWarnings("unchecked")
    @Override
    public final <I extends Interactor<Result>, Param, Param1> I params(
            @Nullable final Param param, @Nullable final Param1 param1) {
        this.params(param);
        if (this.domainParam != null) {
            this.domainParam.param(NAME_PARAM1, param1);
        }
        return ((I) this);
    }

    @SuppressWarnings("unchecked")
    @Override
    public final <I extends Interactor<Result>, Param, Param1, Param2> I params(
            @Nullable final Param param,
            @Nullable final Param1 param1, @Nullable final Param2 param2) {
        this.params(param, param1);
        if (this.domainParam != null) {
            this.domainParam.param(NAME_PARAM2, param2);
        }
        return ((I) this);
    }

    @SuppressWarnings("unchecked")
    @Override
    public final <I extends Interactor<Result>, Param, Param1, Param2, Param3> I params(
            @Nullable final Param param,
            @Nullable final Param1 param1,
            @Nullable final Param2 param2, @Nullable final Param3 param3) {
        this.params(param, param1, param2);
        if (this.domainParam != null) {
            this.domainParam.param(NAME_PARAM3, param3);
        }
        return ((I) this);
    }

    @SuppressWarnings("unchecked")
    @Override
    public final <I extends Interactor<Result>, Param, Param1, Param2, Param3, Param4> I params(
            @Nullable final Param param,
            @Nullable final Param1 param1,
            @Nullable final Param2 param2,
            @Nullable final Param3 param3, @Nullable final Param4 param4) {
        this.params(param, param1, param2, param3);
        if (this.domainParam != null) {
            this.domainParam.param(NAME_PARAM4, param4);
        }
        return ((I) this);
    }

    @Nullable
    public final DomainParam getDomainParam() {
        return this.domainParam;
    }

    @Nullable
    public final <Param> Param getParam() {
        if (this.domainParam != null) {
            return this.domainParam.getParam(NAME_PARAM);
        }
        return null;
    }

    @Nullable
    public final <Param1> Param1 getParam1() {
        if (this.domainParam != null) {
            return this.domainParam.getParam(NAME_PARAM1);
        }
        return null;
    }

    @Nullable
    public final <Param2> Param2 getParam2() {
        if (this.domainParam != null) {
            return this.domainParam.getParam(NAME_PARAM2);
        }
        return null;
    }

    @Nullable
    public final <Param3> Param3 getParam3() {
        if (this.domainParam != null) {
            return this.domainParam.getParam(NAME_PARAM3);
        }
        return null;
    }

    @Nullable
    public final <Param4> Param4 getParam4() {
        if (this.domainParam != null) {
            return this.domainParam.getParam(NAME_PARAM4);
        }
        return null;
    }

    private Subscription executeWith(
            @Nullable Subscriber<? super Result> useCaseSubscriber,
            @Nullable Action1<? super Result> action1, boolean isConnectable) {
        if ((useCaseSubscriber == null && action1 == null) || this.isCanceled) {
            return null;
        }

        this.source = buildObservable()
                //异步执行业务
                .subscribeOn(Schedulers.from(executor))
                //在主线程监听
                .observeOn(mainThread.getScheduler());

        Subscription subscription = null;
        if (isConnectable) {
            if (this.source != null) {
                this.connSource = this.source.publish();
            }
            if (this.connSource != null) {
                if (useCaseSubscriber != null) {
                    subscription = this.connSource.subscribe(useCaseSubscriber);
                } else {
                    subscription = this.connSource.subscribe(action1);
                }
            }
        } else {
            if (this.onStartAction != null) {
                this.onStartAction.call();
            }
            if (useCaseSubscriber != null) {
                subscription = this.source.subscribe(useCaseSubscriber);
            } else {
                subscription = this.source.subscribe(action1);
            }
        }
        return pushUseCaseSubscription(subscription);
    }

    private Observable<Result> buildObservable() {

        Observable<?> preObservable = buildPreObservable();

        if (preObservable != null) {
            return preObservable.flatMap(new Func1<Object, Observable<Result>>() {
                @Override
                public Observable<Result> call(Object o) {
                    if (UseCase.this.preposeAction != null) {
                        UseCase.this.preposeAction.call(o, UseCase.this);
                    }
                    return faceObservable();
                }
            });
        } else {
            return faceObservable();
        }
    }

    @Nullable
    private Observable<?> buildPreObservable() {
        Observable<?> preObservable = null;
        if (this.preInteractor != null) {
            preObservable = ((UseCase<?>) this.preInteractor).buildObservable();
        }
        return preObservable;
    }

    private Observable<Result> faceObservable() {

        Observable<Result> buo = this.buildUseCaseObservable();
        if (this.onNextAction1 != null) {
            buo = buo.doOnNext(this.onNextAction1);
        }

        if (this.onErrorAction1 != null) {
            buo = buo.doOnError(this.onErrorAction1);
        }

        if (this.onCompletedAction != null) {
            buo = buo.doOnCompleted(this.onCompletedAction);
        }

        if (this.subscribeAction != null) {
            buo = buo.doOnSubscribe(subscribeAction);
        }

        if (this.unsubscribeAction != null) {
            buo = buo.doOnUnsubscribe(this.unsubscribeAction);
        }

        if (this.onTerminateAction != null) {
            buo = buo.doOnTerminate(this.onTerminateAction);
        }

        if (this.afterTerminateAction != null) {
            buo = buo.doAfterTerminate(this.afterTerminateAction);
        }

        return buo;
    }

    private Subscription pushUseCaseSubscription(Subscription subscription) {
        return pushSubscription(subscription, this.useCaseSubscriptions);
    }

    private Subscription pushMsgSubscription(Subscription subscription) {
        return pushSubscription(subscription, this.msgSubscriptions);
    }

    private Subscription pushSubscription(Subscription subscription, Set<Subscription> collections) {
        if (subscription == null) {
            return null;
        }
        if (collections == null) {
            collections = new HashSet<>(1);
        }
        collections.add(subscription);
        return subscription;
    }
}
