package com.rabbit.blade.domain.operator;


import com.rabbit.blade.comm.exeption.CheckException;

import rx.Observable;
import rx.Producer;
import rx.Subscriber;
import rx.exceptions.Exceptions;
import rx.exceptions.OnErrorThrowable;
import rx.functions.Func1;
import rx.plugins.RxJavaHooks;

/**
 * <b>Description : 自定义的参数验证操作符</b>
 * <p>Created by <a href="mailto:fjd@xdja.com">fanjiandong</a> on 2016/12/27 21:37.</p>
 */

public class OperatorVerify<T, E extends RuntimeException> implements Observable.Operator<T, T> {

    private final Func1<T, VerifyResult<E>> verifyResultFunc1;

    public OperatorVerify(Func1<T, VerifyResult<E>> verifyResultFunc1) {
        this.verifyResultFunc1 = verifyResultFunc1;
    }

    @Override
    public Subscriber<? super T> call(Subscriber<? super T> subscriber) {
        VerifySubscriber<T, E> parent = new VerifySubscriber<>(subscriber, verifyResultFunc1);
        subscriber.add(parent);
        return parent;
    }

    static final class VerifySubscriber<T, E1 extends RuntimeException> extends Subscriber<T> {

        final Subscriber<? super T> actual;

        private final Func1<T, VerifyResult<E1>> verifyResultFunc1;

        boolean done;

        protected VerifySubscriber(Subscriber<? super T> actual, Func1<T, VerifyResult<E1>> verifyResultFunc1) {
            this.actual = actual;
            this.verifyResultFunc1 = verifyResultFunc1;
        }

        @Override
        public void onNext(T t) {
            VerifyResult<E1> result;
            try {
                result = verifyResultFunc1.call(t);
                if (result != null) {
                    if (!result.isVerify()) {
                        E1 exception = result.getException();
                        if (exception != null) {
                            Exceptions.throwIfFatal(exception);
                            unsubscribe();
                            onError(OnErrorThrowable.addValueAsLastCause(exception, t));
                        } else {
                            CheckException ce = new CheckException("未知的校验异常",CheckException.CODE_PARAMS_NOT_VALIDATE);
                            Exceptions.throwIfFatal(ce);
                            unsubscribe();
                            onError(OnErrorThrowable.addValueAsLastCause(ce, t));
                        }
                        return;
                    }
                }
            } catch (Throwable ex) {
                Exceptions.throwIfFatal(ex);
                unsubscribe();
                onError(OnErrorThrowable.addValueAsLastCause(ex, t));
                return;
            }

            actual.onNext(t);
        }

        @Override
        public void onError(Throwable e) {
            if (done) {
                RxJavaHooks.onError(e);
                return;
            }
            done = true;

            actual.onError(e);
        }


        @Override
        public void onCompleted() {
            if (done) {
                return;
            }
            actual.onCompleted();
        }

        @Override
        public void setProducer(Producer p) {
            actual.setProducer(p);
        }
    }
}
