package com.rabbit.blade.domain.usecase;


import com.rabbit.blade.domain.event.AgentMessage;

import java.lang.annotation.Documented;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import rx.Subscriber;
import rx.Subscription;
import rx.functions.Action0;
import rx.functions.Action1;

/**
 * <b>Description : 业务流程接口定义</b>
 * <p>Created by <a href="mailto:fjd@xdja.com">fanjiandong</a> on 2017/3/15 16:01.</p>
 */
public interface DomainService {

    /********************************** Perform DomainService ************************************/

    /**
     * 流程执行入口
     *
     * @param <T>               the type parameter
     * @param useCaseSubscriber 订阅者
     * @return the subscription
     */
    @Nullable
    <T> Subscription execute(@NonNull Subscriber<T> useCaseSubscriber);

    /**
     * 流程执行入口
     *
     * @param <T>               the type parameter
     * @param useCaseSubscriber 订阅者
     * @param isConnectable     是否延迟执行
     * @return the subscription
     */
    @Nullable
    <T> Subscription execute(@NonNull Subscriber<T> useCaseSubscriber, boolean isConnectable);

    /**
     * 流程执行入口
     *
     * @param <T>     the type parameter
     * @param action1 订阅者
     * @return the subscription
     */
    @Nullable
    <T> Subscription execute(@NonNull Action1<T> action1);

    /**
     * 流程执行入口
     *
     * @param <T>           the type parameter
     * @param action1       订阅者
     * @param isConnectable 是否延迟执行
     * @return the subscription
     */
    @Nullable
    <T> Subscription execute(@NonNull Action1<T> action1, boolean isConnectable);

    /**
     * 执行可延迟的流程
     *
     * @return the subscription
     */
    @Nullable
    Subscription connect();

    /********************************** Subscribe Message ************************************/

    /**
     * 订阅流程执行过程中的消息（常用于更新进度，获取中间执行结果等操作）
     *
     * @param agentMessageAction1 消息订阅者
     * @param isSerial            是否串行
     * @return the subscription
     */

    DomainService subscribeMessage(@Nullable Action1<AgentMessage<?>> agentMessageAction1, boolean isSerial);

    /**
     * Subscribe message with subscription.
     *
     * @param agentMessageAction1 the agent message action 1
     * @param isSerial            是否串行
     * @return the subscription
     */
    @Nullable
    Subscription subscribeMessageWith(@Nullable Action1<AgentMessage<?>> agentMessageAction1, boolean isSerial);

    /**
     * 注销接收代理消息
     *
     * @return the subscription
     */

    DomainService unSubscribeMessage();

    /********************************** Release DomainService ************************************/

    /**
     * 释放相关资源(取消所有的订阅，包括对结果的订阅和对代理消息的订阅)
     *
     * @return the service
     */

    DomainService cancel();

    /**
     * 标识资源是否被释放
     *
     * @return 是否被释放 boolean
     */
    boolean isCanceled();

    /********************************** DomainService Hook ************************************/

    /**
     * 流程执行之前调用该动作
     *
     * @param onStartAction 要执行的操作
     * @return 当前流程对象 service
     */

    DomainService doOnStart(Action0 onStartAction);

    /**
     * {@link rx.Observable#doOnNext}
     *
     * @param <T>           the type parameter
     * @param onNextAction1 the on nextNode action 1
     * @return the service
     */
    <T> DomainService doOnNext(Action1<T> onNextAction1);

    /**
     * {@link rx.Observable#doOnError}
     *
     * @param onErrorAction1 the on error action 1
     * @return the service
     */

    DomainService doOnError(Action1<Throwable> onErrorAction1);

    /**
     * {@link rx.Observable#doOnCompleted}
     *
     * @param onCompletedAction the on completed action
     * @return the service
     */

    DomainService doOnCompleted(Action0 onCompletedAction);

    /**
     * {@link rx.Observable#doOnSubscribe}
     *
     * @param subscribeAction the subscribe action
     * @return the service
     */

    DomainService doOnSubscribe(Action0 subscribeAction);

    /**
     * {@link rx.Observable#doOnUnsubscribe}
     *
     * @param unsubscribeAction the unsubscribe action
     * @return the service
     */

    DomainService doOnUnsubscribe(Action0 unsubscribeAction);

    /**
     * {@link rx.Observable#doOnTerminate}
     *
     * @param onTerminateAction the on terminate action
     * @return the service
     */

    DomainService doOnTerminate(Action0 onTerminateAction);

    /**
     * {@link rx.Observable#doAfterTerminate}
     *
     * @param afterTerminateAction the after terminate action
     * @return the service
     */

    DomainService doAfterTerminate(Action0 afterTerminateAction);

    /********************************** Fill Params ************************************/

    /**
     * 为当前流程传入参数
     *
     * @param domainParam 参数对象
     * @return 流程对象 service
     */

    DomainService params(@Nullable DomainParam domainParam);

    /**
     * 为当前流程传入参数
     *
     * @param <Param> 参数1类型定义
     * @param param   参数1
     * @return 流程对象 service
     */
    <Param> DomainService params(@Nullable Param param);

    /**
     * 为当前流程传入参数
     *
     * @param <Param>  参数1类型定义
     * @param <Param1> 参数2类型定义
     * @param param    参数1
     * @param param1   参数2
     * @return 流程对象 service
     */

    <Param, Param1> DomainService params(
            @Nullable Param param, @Nullable Param1 param1);

    /**
     * 为当前流程传入参数
     *
     * @param <Param>  参数1类型定义
     * @param <Param1> 参数2类型定义
     * @param <Param2> 参数3类型定义
     * @param param    参数1
     * @param param1   参数2
     * @param param2   参数3
     * @return 流程对象 service
     */

    <Param, Param1, Param2> DomainService params(
            @Nullable Param param,
            @Nullable Param1 param1, @Nullable Param2 param2);

    /**
     * 为当前流程传入参数
     *
     * @param <Param>  参数1类型定义
     * @param <Param1> 参数2类型定义
     * @param <Param2> 参数3类型定义
     * @param <Param3> 参数4类型定义
     * @param param    参数1
     * @param param1   参数2
     * @param param2   参数3
     * @param param3   参数4
     * @return 流程对象 service
     */

    <Param, Param1, Param2, Param3> DomainService params(
            @Nullable Param param,
            @Nullable Param1 param1,
            @Nullable Param2 param2, @Nullable Param3 param3);

    /**
     * 为当前流程传入参数
     *
     * @param <Param>  参数1类型定义
     * @param <Param1> 参数2类型定义
     * @param <Param2> 参数3类型定义
     * @param <Param3> 参数4类型定义
     * @param <Param4> 参数5类型定义
     * @param param    参数1
     * @param param1   参数2
     * @param param2   参数3
     * @param param3   参数4
     * @param param4   参数5
     * @return 流程对象 service
     */

    <Param, Param1, Param2, Param3, Param4> DomainService params(
            @Nullable Param param,
            @Nullable Param1 param1,
            @Nullable Param2 param2,
            @Nullable Param3 param3, @Nullable Param4 param4);

    /**
     * 获取参数1
     *
     * @param <Param> 参数1类型定义
     * @return 获取结果 param
     */
    <Param> Param getParam();

    /**
     * 获取参数2
     *
     * @param <Param1> 参数2类型定义
     * @return 获取结果 param 1
     */
    <Param1> Param1 getParam1();

    /**
     * 获取参数3
     *
     * @param <Param2> 参数3类型定义
     * @return 获取结果 param 2
     */
    <Param2> Param2 getParam2();

    /**
     * 获取参数4
     *
     * @param <Param3> 参数4类型定义
     * @return 获取结果 param 3
     */
    <Param3> Param3 getParam3();

    /**
     * 获取参数5
     *
     * @param <Param4> 参数5类型定义
     * @return 获取结果 param 4
     */
    <Param4> Param4 getParam4();

    /********************************** Node Next ************************************/

    /**
     * 切换到指定的业务流程节点
     *
     * @param index 节点标识
     * @return the service
     */
    DomainService nextNode(int index);

    /**
     * 重置业务流程的节点位置
     *
     * @return the service
     */
    DomainService resetNode();

    /**
     * The interface Node.
     */
    @Documented
    @Retention(RetentionPolicy.RUNTIME)
    @interface Node {
        /**
         * Index int.
         *
         * @return the int
         */
        int index() default 0;
    }
}
