package com.rabbit.blade.domain.usecase;


import android.util.SparseArray;

import com.rabbit.blade.comm.event.DefaultBusProvider;
import com.rabbit.blade.comm.util.LogUtil;
import com.rabbit.blade.domain.event.AgentMessage;
import com.rabbit.blade.domain.event.MessageEvent;
import com.rabbit.blade.domain.excutor.PostExecutionThread;
import com.rabbit.blade.domain.excutor.ThreadExecutor;

import java.lang.reflect.Method;
import java.util.HashSet;
import java.util.Set;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import rx.Observable;
import rx.Subscriber;
import rx.Subscription;
import rx.functions.Action0;
import rx.functions.Action1;
import rx.observables.ConnectableObservable;
import rx.schedulers.Schedulers;
import rx.subjects.PublishSubject;

import static com.rabbit.blade.domain.usecase.DomainParam.NAME_PARAM;
import static com.rabbit.blade.domain.usecase.DomainParam.NAME_PARAM1;
import static com.rabbit.blade.domain.usecase.DomainParam.NAME_PARAM2;
import static com.rabbit.blade.domain.usecase.DomainParam.NAME_PARAM3;
import static com.rabbit.blade.domain.usecase.DomainParam.NAME_PARAM4;

/**
 * <b>Description : 业务用例基类</b>
 * <p>Created by <a href="mailto:fjd@xdja.com">fanjiandong</a> on 2017/3/15 16:01.</p>
 */

public class Flow implements DomainService {
    @NonNull
    private final ThreadExecutor executor;
    @NonNull
    private final PostExecutionThread mainThread;

    @Nullable
    private PublishSubject<AgentMessage<?>> agentMessagePublishSubject;

    @Nullable
    private Action1<?> onNextAction1;
    @Nullable
    private Action1<Throwable> onErrorAction1;
    @Nullable
    private Action0 onCompletedAction;
    @Nullable
    private Action0 subscribeAction;
    @Nullable
    private Action0 unsubscribeAction;
    @Nullable
    private Action0 onTerminateAction;
    @Nullable
    private Action0 afterTerminateAction;
    @Nullable
    private Action0 onStartAction;

    @Nullable
    private DomainParam domainParam;

    private Observable<?> source;
    private ConnectableObservable<?> connectableSource;

    private Set<Subscription> msgSubscriptions;
    private Set<Subscription> useCaseSubscriptions;

    private SparseArray<Method> observableMethods;

    private boolean isCanceled = false;
    private boolean isRegistedAgentMessage = false;

    private long eventCategory = Long.MAX_VALUE;

    private int nodeIndex = 0;


    /**
     * Instantiates a new Use case.
     *
     * @param threadExecutor      the thread executor
     * @param postExecutionThread the post execution thread
     */
    public Flow(@NonNull ThreadExecutor threadExecutor,
                @NonNull PostExecutionThread postExecutionThread) {
        this.executor = threadExecutor;
        this.mainThread = postExecutionThread;
    }


    @Override
    @Nullable
    public final <T> Subscription execute(
            @NonNull final Subscriber<T> useCaseSubscriber, boolean isConnectable) {
        return executeWith(useCaseSubscriber, null, isConnectable);
    }

    @Override
    @Nullable
    public final <T> Subscription execute(
            @NonNull final Subscriber<T> useCaseSubscriber) {
        return execute(useCaseSubscriber, false);
    }

    @Override
    @Nullable
    public final <T> Subscription execute(
            @NonNull final Action1<T> action1, boolean isConnectable) {
        return executeWith(null, action1, isConnectable);
    }

    @Override
    public final <T> Subscription execute(@NonNull final Action1<T> action1) {
        return execute(action1, false);
    }

    @Override
    @Nullable
    public final Subscription connect() {
        if (this.connectableSource != null && !this.isCanceled) {
            if (this.onStartAction != null) {
                this.onStartAction.call();
            }
            Subscription subscription = this.connectableSource.connect();
            cacheUseCaseSubscription(subscription);
            return subscription;
        }
        return null;
    }

    @Override
    public final DomainService subscribeMessage(
            @Nullable final Action1<AgentMessage<?>> agentMessageAction1, boolean isSerial) {
        subscribeMessageWith(agentMessageAction1, isSerial);
        return this;
    }

    @Override
    @Nullable
    public final Subscription subscribeMessageWith(
            @Nullable Action1<AgentMessage<?>> agentMessageAction1, boolean isSerial) {
        if (agentMessageAction1 == null || this.isCanceled) return null;

        if (!this.isRegistedAgentMessage) {
            this.isRegistedAgentMessage = true;
            DefaultBusProvider.instance().register(this);
        }

        if (this.agentMessagePublishSubject == null) {
            this.agentMessagePublishSubject = PublishSubject.create();
        }
        Subscription subscription;
        if (isSerial) {
            subscription = this.agentMessagePublishSubject.subscribe(agentMessageAction1);
        } else {
            subscription = this.agentMessagePublishSubject
                    //在主线程监听
                    .observeOn(mainThread.getScheduler())
                    .subscribe(agentMessageAction1);
        }
        cacheMsgSubscription(subscription);
        return subscription;
    }

    @Override
    public final DomainService unSubscribeMessage() {
        if (this.isRegistedAgentMessage) {
            if (this.msgSubscriptions != null && !this.msgSubscriptions.isEmpty()) {
                for (Subscription subscription : this.msgSubscriptions) {
                    if (subscription != null && !subscription.isUnsubscribed()) {
                        subscription.unsubscribe();
                    }
                }
            }

            DefaultBusProvider.instance().unregister(this);
            this.isRegistedAgentMessage = false;
        }
        return this;
    }

    @Override
    public final DomainService cancel() {
        if (!this.isCanceled) {
            this.unSubscribeMessage();
            if (this.useCaseSubscriptions != null && !this.useCaseSubscriptions.isEmpty()) {
                for (Subscription subscription : this.useCaseSubscriptions) {
                    if (subscription != null && !subscription.isUnsubscribed()) {
                        subscription.unsubscribe();
                    }
                }
            }
            this.isCanceled = true;
        }

        return this;
    }

    @Override
    public final boolean isCanceled() {
        return this.isCanceled;
    }

    /**
     * Dispatch message event.
     *
     * @param messageEvent the message event
     */

    public void dispatchMessageEvent(@Nullable final MessageEvent<?> messageEvent) {

        if (messageEvent == null || this.isCanceled || !this.isRegistedAgentMessage) return;

        Long category = messageEvent.getCategory();
        if (category == null) category = Long.MAX_VALUE;
        if (category != this.eventCategory) return;

        AgentMessage<?> agentMessage = messageEvent.getMaterialContent();

        if (this.agentMessagePublishSubject != null) {
            this.agentMessagePublishSubject.onNext(agentMessage);
        }
    }


    @Override
    public final <T> DomainService doOnNext(final Action1<T> onNextAction1) {
        this.onNextAction1 = onNextAction1;
        return this;
    }

    @Override
    public final DomainService doOnError(final Action1<Throwable> onErrorAction1) {
        this.onErrorAction1 = onErrorAction1;
        return this;
    }

    @Override
    public final DomainService doOnCompleted(final Action0 onCompletedAction) {
        this.onCompletedAction = onCompletedAction;
        return this;
    }

    @Override
    public final DomainService doOnSubscribe(final Action0 subscribeAction) {
        this.subscribeAction = subscribeAction;
        return this;
    }

    @Override
    public final DomainService doOnUnsubscribe(final Action0 unsubscribeAction) {
        this.unsubscribeAction = unsubscribeAction;
        return this;
    }

    @Override
    public final DomainService doOnTerminate(final Action0 onTerminateAction) {
        this.onTerminateAction = onTerminateAction;
        return this;
    }

    @Override
    public final DomainService doAfterTerminate(final Action0 afterTerminateAction) {
        this.afterTerminateAction = afterTerminateAction;
        return this;
    }

    @Override
    public final DomainService doOnStart(Action0 onStartAction) {
        this.onStartAction = onStartAction;
        return this;
    }

    /**
     * Next node domain service.
     *
     * @param index the index
     * @return the domain service
     */
    @Override
    public DomainService nextNode(int index) {
        release();
        this.nodeIndex = index;
        return this;
    }

    /**
     * Reset node domain service.
     *
     * @return the domain service
     */
    @Override
    public DomainService resetNode() {
        release();
        this.nodeIndex = 0;
        return this;
    }

    /**
     * Sets event category.
     *
     * @param eventCategory the event category
     */

    public final void setEventCategory(long eventCategory) {
        this.eventCategory = eventCategory;
    }

    @Override
    public final DomainService params(@Nullable final DomainParam domainParam) {
        this.domainParam = domainParam;
        return this;
    }

    @Override
    public final <Param> DomainService params(@Nullable final Param param) {
        if (this.domainParam == null) {
            this.domainParam = new DomainParam();
        }
        this.domainParam.param(NAME_PARAM, param);
        return this;
    }

    @Override
    public final <Param, Param1> DomainService params(
            @Nullable final Param param, @Nullable final Param1 param1) {
        this.params(param);
        if (this.domainParam != null) {
            this.domainParam.param(NAME_PARAM1, param1);
        }
        return this;
    }

    @Override
    public final <Param, Param1, Param2> DomainService params(
            @Nullable final Param param,
            @Nullable final Param1 param1, @Nullable final Param2 param2) {
        this.params(param, param1);
        if (this.domainParam != null) {
            this.domainParam.param(NAME_PARAM2, param2);
        }
        return this;
    }

    @Override
    public final <Param, Param1, Param2, Param3> DomainService params(
            @Nullable final Param param,
            @Nullable final Param1 param1,
            @Nullable final Param2 param2, @Nullable final Param3 param3) {
        this.params(param, param1, param2);
        if (this.domainParam != null) {
            this.domainParam.param(NAME_PARAM3, param3);
        }
        return this;
    }

    @Override
    public final <Param, Param1, Param2, Param3, Param4> DomainService params(
            @Nullable final Param param,
            @Nullable final Param1 param1,
            @Nullable final Param2 param2,
            @Nullable final Param3 param3, @Nullable final Param4 param4) {
        this.params(param, param1, param2, param3);
        if (this.domainParam != null) {
            this.domainParam.param(NAME_PARAM4, param4);
        }
        return this;
    }

    /**
     * Gets domain param.
     *
     * @return the domain param
     */
    @Nullable
    public final DomainParam getDomainParam() {
        return this.domainParam;
    }

    @Nullable
    public final <Param> Param getParam() {
        if (this.domainParam != null) {
            return this.domainParam.getParam(NAME_PARAM);
        }
        return null;
    }

    @Nullable
    public final <Param1> Param1 getParam1() {
        if (this.domainParam != null) {
            return this.domainParam.getParam(NAME_PARAM1);
        }
        return null;
    }

    @Nullable
    public final <Param2> Param2 getParam2() {
        if (this.domainParam != null) {
            return this.domainParam.getParam(NAME_PARAM2);
        }
        return null;
    }

    @Nullable
    public final <Param3> Param3 getParam3() {
        if (this.domainParam != null) {
            return this.domainParam.getParam(NAME_PARAM3);
        }
        return null;
    }

    @Nullable
    public final <Param4> Param4 getParam4() {
        if (this.domainParam != null) {
            return this.domainParam.getParam(NAME_PARAM4);
        }
        return null;
    }

    @SuppressWarnings("unchecked")
    private <T> Subscription executeWith(
            @Nullable Subscriber<T> useCaseSubscriber,
            @Nullable Action1<T> action1,
            boolean isConnectable) {

        if ((useCaseSubscriber == null && action1 == null) || this.isCanceled) {
            return null;
        }

        this.source = faceObservable()
                //异步执行业务
                .subscribeOn(Schedulers.from(executor))
                //在主线程监听
                .observeOn(mainThread.getScheduler());

        Subscription subscription = null;
        if (isConnectable) {
            if (this.source != null) {
                this.connectableSource = this.source.publish();
            }
            if (this.connectableSource != null) {
                if (useCaseSubscriber != null) {
                    subscription = ((ConnectableObservable<T>) this.connectableSource).subscribe(useCaseSubscriber);
                } else {
                    subscription = ((ConnectableObservable<T>) this.connectableSource).subscribe(action1);
                }
            }
        } else {
            if (this.onStartAction != null) {
                this.onStartAction.call();
            }
            if (useCaseSubscriber != null) {
                subscription = ((Observable<T>) this.source).subscribe(useCaseSubscriber);
            } else {
                subscription = ((Observable<T>) this.source).subscribe(action1);
            }
        }
        cacheUseCaseSubscription(subscription);
        return subscription;
    }

    @SuppressWarnings("unchecked")
    private <T> Observable<T> faceObservable() {

        Observable<T> buo = this.buildUseCaseObservable();
        if (buo != null) {


            if (this.onNextAction1 != null) {
                buo = buo.doOnNext(((Action1<T>) this.onNextAction1));
            }

            if (this.onErrorAction1 != null) {
                buo = buo.doOnError(this.onErrorAction1);
            }

            if (this.onCompletedAction != null) {
                buo = buo.doOnCompleted(this.onCompletedAction);
            }

            if (this.subscribeAction != null) {
                buo = buo.doOnSubscribe(subscribeAction);
            }

            if (this.unsubscribeAction != null) {
                buo = buo.doOnUnsubscribe(this.unsubscribeAction);
            }

            if (this.onTerminateAction != null) {
                buo = buo.doOnTerminate(this.onTerminateAction);
            }

            if (this.afterTerminateAction != null) {
                buo = buo.doAfterTerminate(this.afterTerminateAction);
            }
        }

        return buo;
    }

    /**
     * 构建业务处理事件流
     *
     * @return 目标事件流 observable
     */
    @SuppressWarnings("unchecked")
    private <T> Observable<T> buildUseCaseObservable() {
        if (this.observableMethods == null) {
            filterNodeMethods();
        }

        if (this.observableMethods.size() == 0) {
            return null;
        }

        Method method = this.observableMethods.get(this.nodeIndex);
        if (method != null) {
            try {
                method.setAccessible(true);
                return ((Observable<T>) method.invoke(this));
            } catch (Exception e) {
                LogUtil.getUtils().w(e.getMessage());
            }
        }
        return null;
    }

    private void release() {
        this.cancel();

        this.agentMessagePublishSubject = null;

        this.onNextAction1 = null;

        this.onErrorAction1 = null;
        this.onCompletedAction = null;
        this.subscribeAction = null;
        this.unsubscribeAction = null;
        this.onTerminateAction = null;
        this.afterTerminateAction = null;
        this.onStartAction = null;

        this.domainParam = null;

        this.source = null;
        this.connectableSource = null;
        this.msgSubscriptions = null;
        this.useCaseSubscriptions = null;

        this.isCanceled = false;
        this.isRegistedAgentMessage = false;

        this.eventCategory = Long.MAX_VALUE;
    }

    private void filterNodeMethods() {
        this.observableMethods = new SparseArray<>(1);

        Class<? extends Flow> processorClass = this.getClass();
        Method[] declaredMethods = processorClass.getDeclaredMethods();
        if (declaredMethods == null || declaredMethods.length <= 0) return;

        for (Method declaredMethod : declaredMethods) {
            if (declaredMethod == null) continue;

            Node nodeAnn = declaredMethod.getAnnotation(Node.class);
            if (nodeAnn == null) continue;

            Class<?> returnType = declaredMethod.getReturnType();
            Class<?>[] parameterTypes = declaredMethod.getParameterTypes();
            if (returnType.getName().equals(Observable.class.getName()) && parameterTypes.length == 0) {
                int index = nodeAnn.index();
                observableMethods.append(index, declaredMethod);
            }
        }
    }

    private void cacheUseCaseSubscription(Subscription subscription) {
        if (subscription == null) return;
        if (this.useCaseSubscriptions == null) {
            this.useCaseSubscriptions = new HashSet<>(1);
        }
        this.useCaseSubscriptions.add(subscription);
    }

    private void cacheMsgSubscription(Subscription subscription) {
        if (subscription == null) return;
        if (this.msgSubscriptions == null) {
            this.msgSubscriptions = new HashSet<>(1);
        }
        this.msgSubscriptions.add(subscription);
    }
}
