package com.rabbit.blade.presenter;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.blankj.utilcode.utils.Utils;
import com.rabbit.blade.comm.BuildConfig;
import com.rabbit.blade.comm.exeption.OkException;
import com.rabbit.blade.presenter.assist.AssistProvider;
import com.rabbit.blade.presenter.assist.Cake;
import com.rabbit.blade.presenter.assist.FlyHunter;
import com.rabbit.blade.presenter.assist.Loader;
import com.rabbit.blade.presenter.assist.ToastCake;
import com.rabbit.blade.presenter.di.components.ActivityComponent;
import com.rabbit.blade.presenter.di.components.ApplicationComponent;
import com.rabbit.blade.presenter.di.components.DaggerApplicationComponent;
import com.rabbit.blade.presenter.di.components.FragmentComponent;
import com.rabbit.blade.presenter.di.modules.ActivityModule;
import com.rabbit.blade.presenter.di.modules.ApplicationModule;
import com.rabbit.blade.presenter.di.modules.EventModule;
import com.rabbit.blade.presenter.di.modules.ExecutorModule;
import com.rabbit.blade.presenter.di.modules.FragmentModule;
import com.rabbit.blade.presenter.di.modules.NetworkModule;
import com.rabbit.blade.presenter.di.modules.UtilModule;
import com.rabbit.blade.presenter.main.ActivityStack;
import com.rabbit.blade.presenter.main.SmartActivity;
import com.rabbit.blade.presenter.main.SmartFragment;

/**
 * <b>Description : Application基类</b>
 * <p>Created by <a href="mailto:fanjiandong@outlook.com">fanjiandong</a> on 2017/3/20 14:06.</p>
 */

public abstract class AndroidApplication extends ForkedApplication implements FlyHunter, AssistProvider {

    public static final String COMTAG_APPLICATION_COMPONENT = "applicationComponent";

    /**
     * Gets application component.
     *
     * @return the application component
     */
    public ApplicationComponent getApplicationComponent() {
        ApplicationComponent applicationComponent = getComponent(COMTAG_APPLICATION_COMPONENT);
        if (applicationComponent == null) {
            applicationComponent = createApplicationComponent();
            registComponent(COMTAG_APPLICATION_COMPONENT, applicationComponent);
        }
        return applicationComponent;
    }

    @Override
    public void onCreate() {
        Utils.init(this);
        super.onCreate();
    }

    private ApplicationComponent createApplicationComponent() {
        return DaggerApplicationComponent
                .builder()
                .applicationModule(new ApplicationModule(this))
                .utilModule(new UtilModule())
                .networkModule(new NetworkModule(BuildConfig.DEBUG))
                .executorModule(new ExecutorModule())
                .eventModule(new EventModule())
                .build();
    }

    /**
     * Create activity component activity component.
     *
     * @param activity the activity
     * @return the activity component
     */
    public ActivityComponent createActivityComponent(@NonNull SmartActivity activity) {
        return this.getApplicationComponent()
                .activityComponentBuilder()
                .activityModule(new ActivityModule(activity))
                .build();
    }

    /**
     * Create fragment component fragment component.
     *
     * @param fragment          the fragment
     * @param activityComponent the activity component
     * @return the fragment component
     */
    public FragmentComponent createFragmentComponent(@NonNull SmartFragment fragment,
                                                     @NonNull ActivityComponent activityComponent) {
        return activityComponent.fragmentComponentBuilder()
                .fragmentModule(new FragmentModule(fragment))
                .build();
    }

    @Override
    public boolean handleOkException(@Nullable String okCode, @Nullable String userMsg, @Nullable
            OkException ex, @Nullable String mark) {
        return false;
    }

    @Override
    public void defaultOkException(@Nullable String code, @Nullable String userMsg, @Nullable
            OkException ex, @Nullable String mark) {

    }

    @Override
    public boolean handlerThrowable(@Nullable Throwable throwable, @Nullable String mark) {
        return false;
    }

    @Override
    public void defaultThrowable(@Nullable Throwable throwable, @Nullable String mark) {

    }

    @Override
    public Cake provideCake() {
        return new ToastCake(this);
    }

    @Override
    public FlyHunter provideFlyHunter() {
        return this;
    }

    @Override
    public Loader provideLoader() {
        SmartActivity topActivity = ActivityStack.getInstance().getTopActivity();
        if (topActivity != null) {
            ActivityComponent activityComponent = topActivity.getActivityComponent();
            if (activityComponent != null) {
                return activityComponent.activityLoader();
            }
        }
        return null;
    }
}
