package com.rabbit.blade.presenter;

import android.app.Application;
import android.content.ComponentCallbacks;
import android.content.res.Configuration;
import android.support.annotation.Nullable;

import java.util.ArrayList;
import java.util.List;

/**
 * <b>Description : 可以管理和分发多个模拟Application对象的Application</b>
 * <p>Created by <a href="mailto:fanjiandong@outlook.com">fanjiandong</a> on 2017/3/20 14:09.</p>
 */

public abstract class ForkedApplication extends ComponentApplication {

    private List<BoundedApplication> boundedApplications;

    /**
     * 根据类型获取已经存在的模拟Application
     *
     * @param appCls 模拟Application类型
     * @param <T>    模拟Application类型定义
     * @return 模拟Application对象
     */
    @Nullable
    public <T extends BoundedApplication> T getBoundedApp(Class<T> appCls) {

        if (boundedApplications != null && !boundedApplications.isEmpty()) {
            for (BoundedApplication cycle : boundedApplications) {
                if (cycle != null && cycle.getClass().getName().equals(appCls.getName())) {
                    return appCls.cast(cycle);
                }
            }
        }
        return null;
    }


    private List<BoundedApplication> initBoundedApps(List<BoundedAppProvider> boundedAppProviders) {
        if (boundedAppProviders != null && !boundedAppProviders.isEmpty()) {
            List<BoundedApplication> boundedApplications = new ArrayList<>();
            for (BoundedAppProvider provider : boundedAppProviders)
                boundedApplications.addAll(provider.provideBoundedApplications());
            return boundedApplications;
        }
        return null;
    }

    /**
     * 构建BoundedApp图表
     *
     * @return BoundedApp图表
     */
    @Nullable
    public abstract List<BoundedAppProvider> buildBoundedAppGraph();

    /**
     * ApplicationLifeCycle初始化完成
     */
    public abstract void afterAppsBounded();


    /**
     * 用于执行ApplicationLifeCycle的create方法，该方法可以根据不同情况灵活调用
     *
     * @throws IllegalStateException 该方法只能被调用一次，否则将报错
     */
    public void onApplicationCreate() {
        if (this.boundedApplications != null && !this.boundedApplications.isEmpty()) {
            for (BoundedApplication cycle : boundedApplications)
                cycle.onCreate(this);
        }
    }


    @Override
    public void onCreate() {
        super.onCreate();
        //初始化模块接口和Application生命周期对象
        this.boundedApplications = initBoundedApps(buildBoundedAppGraph());
        //初始化
        this.afterAppsBounded();
        //开始Application生命周期
        onApplicationCreate();
    }

    @Override
    public void onTerminate() {
        super.onTerminate();
        if (this.boundedApplications != null && !this.boundedApplications.isEmpty()) {
            for (BoundedApplication cycle : boundedApplications) cycle.onTerminate();
        }
    }

    @Override
    public void onConfigurationChanged(Configuration newConfig) {
        super.onConfigurationChanged(newConfig);
        if (this.boundedApplications != null && !this.boundedApplications.isEmpty()) {
            for (BoundedApplication cycle : boundedApplications)
                cycle.onConfigurationChanged(newConfig);
        }
    }

    @Override
    public void onLowMemory() {
        super.onLowMemory();
        if (this.boundedApplications != null && !this.boundedApplications.isEmpty()) {
            for (BoundedApplication cycle : boundedApplications) cycle.onLowMemory();
        }
    }

    @Override
    public void onTrimMemory(int level) {
        super.onTrimMemory(level);
        if (this.boundedApplications != null && !this.boundedApplications.isEmpty()) {
            for (BoundedApplication cycle : boundedApplications) cycle.onTrimMemory(level);
        }
    }

    @Override
    public void registerComponentCallbacks(ComponentCallbacks callback) {
        super.registerComponentCallbacks(callback);
        if (this.boundedApplications != null && !this.boundedApplications.isEmpty()) {
            for (BoundedApplication cycle : boundedApplications)
                cycle.registerComponentCallbacks(callback);
        }
    }

    @Override
    public void unregisterComponentCallbacks(ComponentCallbacks callback) {
        super.unregisterComponentCallbacks(callback);
        if (this.boundedApplications != null && !this.boundedApplications.isEmpty()) {
            for (BoundedApplication cycle : boundedApplications)
                cycle.unregisterComponentCallbacks(callback);
        }
    }

    @Override
    public void registerActivityLifecycleCallbacks(Application.ActivityLifecycleCallbacks callback) {
        super.registerActivityLifecycleCallbacks(callback);
        if (this.boundedApplications != null && !this.boundedApplications.isEmpty()) {
            for (BoundedApplication cycle : boundedApplications)
                cycle.registerActivityLifecycleCallbacks(callback);
        }
    }

    @Override
    public void unregisterActivityLifecycleCallbacks(Application.ActivityLifecycleCallbacks callback) {
        super.unregisterActivityLifecycleCallbacks(callback);
        if (this.boundedApplications != null && !this.boundedApplications.isEmpty()) {
            for (BoundedApplication cycle : boundedApplications)
                cycle.unregisterActivityLifecycleCallbacks(callback);
        }
    }

    @Override
    public void registerOnProvideAssistDataListener(Application.OnProvideAssistDataListener callback) {
        super.registerOnProvideAssistDataListener(callback);
        if (this.boundedApplications != null && !this.boundedApplications.isEmpty()) {
            for (BoundedApplication cycle : boundedApplications)
                cycle.registerOnProvideAssistDataListener(callback);
        }
    }

    @Override
    public void unregisterOnProvideAssistDataListener(Application.OnProvideAssistDataListener callback) {
        super.unregisterOnProvideAssistDataListener(callback);
        if (this.boundedApplications != null && !this.boundedApplications.isEmpty()) {
            for (BoundedApplication cycle : boundedApplications)
                cycle.unregisterOnProvideAssistDataListener(callback);
        }
    }
}
