package com.rabbit.blade.presenter.assist;

import android.content.DialogInterface;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.rabbit.blade.comm.exeption.OkException;
import com.rabbit.blade.comm.exeption.OkMatcher;

import rx.Subscriber;
import rx.functions.Action1;


/**
 * <b>Description : 自加载进度显示的Observer</b>
 * <p>Created by <a href="mailto:fanjiandong@outlook.com">fanjiandong</a> on 2017/3/20 13:51.</p>
 */

public class SchSubscriber<T> extends OkSubscriber<T> {

    private static final String PRE_LOADING_MSG = "正在";
    private static final String DEFAULT_LOADING_MSG = "加载";
    private static final String AFTER_LOADING_MSG = "...";

    private String loadingMessage = DEFAULT_LOADING_MSG;
    private boolean isLoading = true;
    private final Loader loader;
    private boolean cancelable = true;
    private DialogInterface.OnCancelListener onCancelListener;

    public SchSubscriber(
            @Nullable PlanFly planFly, @Nullable Loader loader, @Nullable String mark) {
        super(planFly, mark);
        this.loader = loader;
    }

    public SchSubscriber(@Nullable PlanFly planFly, @Nullable Loader loader) {
        this(planFly, loader, null);
    }

    /**
     * 设置是否显示加载进度
     *
     * @param loading true表示显示，否则为不显示
     * @return Observer对象
     */
    public SchSubscriber<T> setLoading(boolean loading) {
        isLoading = loading;
        return this;
    }

    /**
     * see {@link #registerLoadingMsg(String, boolean)}
     *
     * @param loadingMsg the loading msg
     * @return the sch subscriber
     */
    public SchSubscriber<T> registerLoadingMsg(@NonNull String loadingMsg) {
        return registerLoadingMsg(loadingMsg, true);
    }

    /**
     * 注册loading显示的信息，loading从{@link #onStart()}开始，到{@link #onCompleted()}
     * 或{@link #onError(Throwable)}结束。
     *
     * @param loadingMsg loading的信息，默认为{@link #DEFAULT_LOADING_MSG}
     * @param isAutoAdd  是否自动填充loading的前后字符串，默认为{@link #PRE_LOADING_MSG},{@link #AFTER_LOADING_MSG}
     * @return observer对象
     */
    public SchSubscriber<T> registerLoadingMsg(@NonNull String loadingMsg, boolean isAutoAdd) {
        if (isAutoAdd) {
            loadingMessage = PRE_LOADING_MSG + loadingMsg + AFTER_LOADING_MSG;
        } else {
            loadingMessage = loadingMsg;
        }
        return this;
    }

    /**
     * 设置是否可以取消loading
     *
     * @param isCancelable true表示可以取消，否则为不可以取消
     * @return observer对象
     */
    public SchSubscriber<T> setCancelable(boolean isCancelable) {
        this.cancelable = isCancelable;
        return this;
    }

    /**
     * 设置loading取消监听
     *
     * @param cancelListener 监听对象
     * @return observer对象
     */
    public SchSubscriber<T> setCancelListener(DialogInterface.OnCancelListener cancelListener) {
        this.onCancelListener = cancelListener;
        return this;
    }

    public <T1 extends OkException> SchSubscriber<T> registUserMsg(@NonNull Class<T1> cls,
                                                                   @NonNull String errorCode,
                                                                   @NonNull String userMsg,
                                                                   boolean isClearPreMsg) {
        if (getPlanFly() != null) {
            OkMatcher matcher = getPlanFly().getMatcher();
            if (matcher != null) {
                if (isClearPreMsg) {
                    matcher.getCustomMapper().reset();
                }
                matcher.getCustomMapper().registSingleMapper(cls, errorCode, userMsg);
            }
        }

        return this;
    }

    /**
     * <li>此处需要注意，onStart()方法的调用线程为实例化Subscriber的线程，并非一定在Ui线程，
     * 和onNext()，onError()以及onComplete()不一样。如果要在此处处理界面相关业务，需要自己切换线程。</li>
     * <li>另外需要注意：onStart()并非一定是在Observable被执行的时候调用，如果为ConnectableObservable
     * 在Observable被订阅是就会执行,但是真是的执行是在调用了connect()方法之后。</li>
     */
    @Override
    public void onStart() {
        super.onStart();
        showLoading();
    }

    @Override
    public void onError(Throwable e) {
        super.onError(e);
        dismissDialog();
    }

    @Override
    public void onCompleted() {
        super.onCompleted();
        dismissDialog();
    }

    private void showLoading() {
        if (loadingMessage.equals(DEFAULT_LOADING_MSG)) {
            loadingMessage = PRE_LOADING_MSG + loadingMessage + AFTER_LOADING_MSG;
        }
        showLoadingDialog();
    }

    /**
     * Show loading dialog.
     */
    private void showLoadingDialog() {
        if (!isLoading) {
            return;
        }
        if (this.loader != null) {
            this.loader.loadingMessage(loadingMessage, cancelable, this.onCancelListener);
        }
    }

    /**
     * Dismiss dialog.
     */
    private void dismissDialog() {
        if (!isLoading) {
            return;
        }
        if (this.loader != null)
            this.loader.cancel();
    }

    public static class Builder<T> {
        /**
         * loading信息
         */
        private String loadingMessage = DEFAULT_LOADING_MSG;
        /**
         * 是否展示loading
         */
        private boolean isLoading = true;
        /**
         * 是否可以取消
         */
        private boolean cancelable = true;
        /**
         * loading对象
         */
        private Loader loader;
        /**
         * 取消监听
         */
        private DialogInterface.OnCancelListener onCancelListener;
        /**
         * 业务标识
         */
        private String mark;
        /**
         * 异常处理类
         */
        private PlanFly planFly;
        /**
         * 是否自动完善loading信息（如：loadingMessage为"登录"，若自动完成则loading显示的信息为"正在登录..."，否则显示为"登录"）
         */
        private boolean isAutoCompleteLoadingMessage = true;
        /**
         * 事件订阅对象
         */
        private Subscriber<T> subscriber;

        private Object[] userMessageArray;

        public Builder<T> planFly(@Nullable PlanFly planFly) {
            this.planFly = planFly;
            return this;
        }

        public Builder<T> loadingMessage(@NonNull String loadingMessage) {
            this.loadingMessage = loadingMessage;
            return this;
        }

        public Builder<T> showLoading(boolean isLoading) {
            this.isLoading = isLoading;
            return this;
        }

        public Builder<T> cancelable(boolean cancelable) {
            this.cancelable = cancelable;
            return this;
        }

        public Builder<T> cancelListener(DialogInterface.OnCancelListener cancelListener){
            this.onCancelListener = cancelListener;
            return this;
        }

        public Builder<T> loader(@Nullable Loader loader) {
            this.loader = loader;
            return this;
        }

        public Builder<T> mark(@Nullable String mark) {
            this.mark = mark;
            return this;
        }

        public Builder<T> isAutoCompleteLoadingMessage(boolean isAutoCompleteLoadingMessage) {
            this.isAutoCompleteLoadingMessage = isAutoCompleteLoadingMessage;
            return this;
        }

        public Builder<T> subscriber(Subscriber<T> subscriber) {
            this.subscriber = subscriber;
            return this;
        }

        public Builder<T> action(final Action1<T> action0) {
            this.subscriber = new Subscriber<T>() {
                @Override
                public void onCompleted() {

                }

                @Override
                public void onError(Throwable e) {

                }

                @Override
                public void onNext(T t) {
                    action0.call(t);
                }
            };
            return this;
        }

        public <T1 extends OkException> Builder<T> registUserMessage(@NonNull Class<T1> cls,
                                                                     @NonNull String errorCode,
                                                                     @NonNull String userMsg,
                                                                     boolean isClearPreMsg) {
            this.userMessageArray = new Object[4];
            this.userMessageArray[0] = cls;
            this.userMessageArray[1] = errorCode;
            this.userMessageArray[2] = userMsg;
            this.userMessageArray[3] = isClearPreMsg;
            return this;
        }


        public SchSubscriber<T> build() {
            SchSubscriber<T> schSubscriber = new SchSubscriber<T>(this.planFly, this.loader, mark) {
                @Override
                public void onNext(T t) {
                    super.onNext(t);
                    if (subscriber != null) {
                        subscriber.onNext(t);
                    }
                }

                @Override
                public void onStart() {
                    super.onStart();
                    if (subscriber != null) {
                        subscriber.onStart();
                    }
                }

                @Override
                public void onError(Throwable e) {
                    super.onError(e);
                    if (subscriber != null) {
                        subscriber.onError(e);
                    }
                }

                @Override
                public void onCompleted() {
                    super.onCompleted();
                    if (subscriber != null) {
                        subscriber.onCompleted();
                    }
                }
            };
            schSubscriber.setLoading(this.isLoading);
            if (this.isLoading) {
                schSubscriber.registerLoadingMsg(this.loadingMessage, this.isAutoCompleteLoadingMessage);
            }
            schSubscriber.setCancelable(this.cancelable);
            if (this.cancelable && this.onCancelListener != null) {
                schSubscriber.setCancelListener(this.onCancelListener);
            }
            if (this.userMessageArray != null) {
                //noinspection unchecked
                schSubscriber.registUserMsg(((Class<? extends OkException>) this.userMessageArray[0]),
                        ((String) this.userMessageArray[1]),
                        ((String) this.userMessageArray[2]),
                        ((Boolean) this.userMessageArray[3]));
            }
            return schSubscriber;
        }

    }
}
