package com.rabbit.blade.presenter.gui.mvp.view;

import android.app.Activity;
import android.os.Bundle;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.v4.app.Fragment;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.AbsListView;

import com.rabbit.blade.presenter.gui.mvp.presenter.Presenter;


/**
 * <b>Description : MVP范式中View层接口定义</b>
 * <p>Created by <a href="mailto:fanjiandong@outlook.com">fanjiandong</a> on 2015/5/23 15:10.</p>
 */
public interface Vu {
    /**
     * 设置与view绑定的Presenter接口实现
     *
     * @param presenter 与view绑定的Presenter接口实现
     */
    void bindPresenter(@Nullable Presenter presenter);

    /**
     * MVP框架模式中，用Activity做Presenter时，View层接口定义推荐继承该接口
     */
    interface ActivityVu extends Vu {
        /**
         * 在Activity的onCreate()方法中初始化View
         *
         * @param savedInstanceState {@link Activity#onCreate(Bundle)}
         */
        void initView(@Nullable Bundle savedInstanceState);

        /**
         * 释放View相关资源
         */
        void releaseView();

        /**
         * 绑定与View相关的Activity
         *
         * @param activity 与View绑定的Activity
         */
        void bindActivity(@NonNull Activity activity);
    }

    /**
     * MVP框架模式中，用Fragment做Presenter时，View层接口定义推荐继承该接口
     */
    interface FragmentVu extends Vu {
        /**
         * 初始化View
         */

        View initView(
                @NonNull LayoutInflater inflater,
                @Nullable ViewGroup container, @Nullable Bundle savedInstanceState);

        /**
         * 释放View相关资源
         */

        void releaseView();

        /**
         * 设置和view绑定的Fragment
         *
         * @param fragment 目标Fragment
         */
        void bindFragment(@NonNull Fragment fragment);
    }

    /**
     * MVP框架模式中，用Adapter做Presenter时，View层接口定义推荐继承该接口
     */
    interface AdapterVu<D> extends Vu {
        /**
         * 适配器相关的View被新创建出来
         *
         * @param position 索引
         */
        void onViewCreated(int position);

        /**
         * 适配器相关的View被重用
         *
         * @param position 索引
         */
        void onViewReused(int position);

        /**
         * VIew被动态更新
         *
         * @param position 索引
         */
        void onViewUpdate(int position);

        /**
         * 绑定数据源到View上
         *
         * @param position   索引
         * @param dataSource 数据源
         */
        void displayDataSource(int position, @Nullable D dataSource);

        /**
         * 绑定数据列表到当前View
         *
         * @param adapterView 带绑定的数据列表对象
         */
        void bindListView(@Nullable AbsListView adapterView);

        /**
         * 初始化View
         */
        View initView(@NonNull LayoutInflater inflater, @Nullable ViewGroup container);

    }

}
