package com.xdja.aircert.sdk.data;

import android.text.TextUtils;

import com.alibaba.fastjson.JSON;
import com.xdja.aircert.sdk.AirCertSdk;
import com.xdja.aircert.sdk.bean.CertApplyBean;
import com.xdja.aircert.sdk.bean.CertDetailInfoBean;
import com.xdja.aircert.sdk.bean.CommonResponse;
import com.xdja.aircert.sdk.bean.RevokedCertBean;
import com.xdja.aircert.sdk.bean.VerifyInfoBean;
import com.xdja.aircert.sdk.bean.WriteFailBean;
import com.xdja.aircert.sdk.cert.PKCS10Utils;
import com.xdja.aircert.sdk.config.AirCertConstant;
import com.xdja.aircert.sdk.config.AirErrorCode;
import com.xdja.aircert.sdk.device.CertHelper;
import com.xdja.aircert.sdk.device.DeviceCache;
import com.xdja.aircert.sdk.device.DeviceWrapper;
import com.xdja.aircert.sdk.device.NetVHSMHandle;
import com.xdja.aircert.sdk.device.OperateCertHelper;
import com.xdja.aircert.sdk.net.HttpManager;
import com.xdja.aircert.sdk.net.HttpUrl;
import com.xdja.aircert.sdk.net.Params;
import com.xdja.aircert.sdk.util.ResponseUtil;
import com.xdja.aircert.sdk.util.SdkLog;
import com.xdja.cryptodev.CryptoDevManager;
import com.xdja.cryptodev.CryptoDevType;
import com.xdja.cryptodev.driver.chipmanager.ChipManagerDriver;

import org.json.JSONObject;

import java.io.IOException;

import okhttp3.Call;
import okhttp3.Request;
import okhttp3.Response;

/**
 * @author: lyl
 * desc:证书仓库
 * 2020/11/6
 */
class CertRepository {
    static void applyCert(String phone, final HttpCallBack<CertDetailInfoBean> callBack) {
        try {
            //校验手机号
            JSONObject object = new JSONObject();
            object.put("phone", phone);
            Request requestBody = HttpManager.initRequest(HttpUrl.IDENTITY_PHONE, object.toString());
            Response response = HttpManager.getInstance().getHttpClient().newCall(requestBody).execute();
            if (!response.isSuccessful()) {
                error(callBack, AirErrorCode.ERROR_CODE_IDENTITY, "phone net identity error");
                return;
            }
            CommonResponse commonResponse = ResponseUtil.parseResponse(response, CommonResponse.class);
            if (commonResponse == null || commonResponse.getResult() != 0) {
                error(callBack, AirErrorCode.ERROR_CODE_IDENTITY, "phone identity error");
                return;
            }
            //身份认证
            VerifyInfoBean bean = Params.getVerifyInfoBean();
            bean.getChipInfo().setPhone(phone);
            Request request = HttpManager.initRequest(HttpUrl.IDENTITY_AUTH, JSON.toJSONString(bean));
            Response identityResponse = HttpManager.getInstance().getHttpClient().newCall(request).execute();
            CommonResponse identityCommonResponse = checkResponse(identityResponse, callBack, AirErrorCode.ERROR_CODE_IDENTITY, "identity error");
            if (identityCommonResponse == null) {
                return;
            }
            if (AirCertSdk.getsInstance().getAirCertConfig().getCardType() == AirCertConstant.DEV_TYPE_VHSM) {
                boolean initSuccess = true;
                if (!NetVHSMHandle.isNetVHSMCanUse()) {
                    /*当前为VHSM,并不可用,初始化*/
                    initSuccess = NetVHSMHandle.initVHSM(AirCertSdk.getsInstance().getContext(), AirCertSdk.getsInstance().getAirCertConfig().getVhsmIpPort());
                }
                if (initSuccess) {
                    CryptoDevManager.getInstance().onDriverDeviceAdded(
                            new ChipManagerDriver(), CryptoDevType.DEV_TYPE_Soft,
                            NetVHSMHandle.getNetVhsmCardId(AirCertSdk.getsInstance().getContext()));
                    DeviceWrapper.getDevInfo();
                } else {
                    /*vhsm初始化失败*/
                    error(callBack, AirErrorCode.ERROR_CODE_IDENTITY, "vhsm init error");
                    return;
                }
            }
            /*生成P10*/
            String result = identityCommonResponse.getMessage();
            String p10RequestBC = PKCS10Utils.getInstance().getP10RequestBC(DeviceCache.getsInstance().getCacheCryptoDevInfo().getType(),
                    AirCertSdk.getsInstance().getAirCertConfig().getContainerNum(), result, true);
            if (TextUtils.isEmpty(p10RequestBC) || p10RequestBC.startsWith("errorCode")) {
                error(callBack, AirErrorCode.ERROR_CODE_IDENTITY, "p10 error");
                return;
            }
            /*申请证书*/
            CertApplyBean certApplyBean = Params.getCertApplyBean(p10RequestBC, phone);
            Request certApplyRequest = HttpManager.initRequest(HttpUrl.APPLY, JSON.toJSONString(certApplyBean));
            Response certApplyResponse = HttpManager.getInstance().getHttpClient().newCall(certApplyRequest).execute();
            CommonResponse certApplyCommonResponse = checkResponse(certApplyResponse, callBack, AirErrorCode.ERROR_CODE_IDENTITY, " cert apply error");
            if (certApplyCommonResponse == null) {
                return;
            }
            /*  int resultCode = writeCert(certApplyCommonResponse);*/
            int resultCode = -1;
            if (resultCode != 0) {
                //写入卡失败
                error(callBack, AirErrorCode.ERROR_CODE_IDENTITY, "operate cert error");
                //上报写证书失败
                WriteFailBean writeFailBean = Params.getWriteFailBean(certApplyCommonResponse.getCert());
                Request writeFailRequest = HttpManager.initRequest(HttpUrl.WRITE_STATUS, JSON.toJSONString(writeFailBean));
                Response writeFailResponse = HttpManager.getInstance().getHttpClient().newCall(writeFailRequest).execute();
                return;
            }
            CertDetailInfoBean certDetailInfoBean = CertHelper.readCertInfo();
            success(callBack, certDetailInfoBean);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * 查询证书状态
     *
     * @param callBack
     */
    static void queryCertState(HttpCallBack<Integer> callBack) {
        try {
            JSONObject jsonObject = new JSONObject();
            jsonObject.put("cardType", DeviceWrapper.convertCardType(DeviceWrapper.getCurrentDevInfo().getType().getType()));
            CertDetailInfoBean bean = CertHelper.readCertInfo();
            jsonObject.put("sn", bean == null ? "" : bean.getSn());
            Request requestBody = HttpManager.initRequest(HttpUrl.CERT_STATUS, jsonObject.toString());
            Response response = HttpManager.getInstance().getHttpClient().newCall(requestBody).execute();
            CommonResponse commonResponse = ResponseUtil.parseResponse(response, CommonResponse.class);
            if (commonResponse == null) {
                error(callBack, AirErrorCode.ERROR_CODE_NET, "querycert net error");
                return;
            }
            if (commonResponse.getErrCode() != 0) {
                if (commonResponse.getErrCode() == AirErrorCode.ERROR_CODE_SERVER_CERT_EMPTY) {
                    //查询无证书
                    CertHelper.deleteCert();
                    success(callBack, AirErrorCode.ERROR_CODE_CERT_EMPTY);
                } else {
                    error(callBack, AirErrorCode.ERROR_CODE_CERT_ERROR, "cert states error");
                }
                return;
            }
            if (commonResponse.getResult() == AirErrorCode.ERROR_CODE_CERT_EMPTY) {
                CertHelper.deleteCert();
                success(callBack, AirErrorCode.ERROR_CODE_CERT_EMPTY);
            } else if (commonResponse.getResult() == AirErrorCode.ERROR_CODE_CERT_NORMAL) {
                if (bean == null) {
                    success(callBack, AirErrorCode.ERROR_CODE_CERT_LOCAL_UN_EXIST);
                } else {
                    success(callBack, AirErrorCode.ERROR_CODE_CERT_NORMAL);
                }
            } else {
                error(callBack, AirErrorCode.ERROR_CODE_CERT_ERROR, "cert states error");
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * 撤销证书
     *
     * @param callBack
     */
    static void revoke(HttpCallBack<Boolean> callBack) {
        try {
            RevokedCertBean bean = Params.getRevokedCertBean();
            Request requestBody = HttpManager.initRequest(HttpUrl.REVOKE, JSON.toJSONString(bean));
            Response response = HttpManager.getInstance().getHttpClient().newCall(requestBody).execute();
            CommonResponse commonResponse = ResponseUtil.parseResponse(response, CommonResponse.class);
            if (commonResponse == null) {
                error(callBack, AirErrorCode.ERROR_CODE_NET, "revoke net error");
                return;
            }
            if (commonResponse.getResult() == AirErrorCode.ERROR_CODE_REVOKE_SUCCESS) {
                OperateCertHelper.clearContainer(DeviceWrapper.getCurrentDevInfo().getType(), AirCertSdk.getsInstance().getAirCertConfig().getContainerNum());
                success(callBack, true);
                return;
            }
            error(callBack, AirErrorCode.ERROR_CODE_REVOKE_ERROR, "revoke error");
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    /**
     * 写证书
     */
    private static int writeCert(CommonResponse certInfoBean) {
        OperateCertHelper helper = new OperateCertHelper();
        return helper.importCert(DeviceCache.getsInstance().getCacheCryptoDevInfo().getType(), AirCertSdk.getsInstance().getAirCertConfig().getContainerNum(), certInfoBean.getCert());
    }

    private static CommonResponse checkResponse(Response response, HttpCallBack<CertDetailInfoBean> callBack, int errorCode, String errorMsg) {
        CommonResponse commonResponse = ResponseUtil.parseResponse(response, CommonResponse.class);
        if (!response.isSuccessful() || commonResponse == null || !commonResponse.isSuccess()) {
            String msg = (commonResponse != null && !TextUtils.isEmpty(commonResponse.getMessage())) ? commonResponse.getMessage() : errorMsg;
            SdkLog.loge("checkResponse----" + errorCode + "----" + errorMsg);
            error(callBack, errorCode, msg);
            return null;
        }
        return commonResponse;
    }

    /**
     * 回调错误信息
     *
     * @param callBack
     * @param code
     * @param msg
     */
    private static void error(final HttpCallBack callBack, final int code, final String msg) {
        if (callBack == null) {
            return;
        }
        HttpManager.getInstance().executorUi(new Runnable() {
            @Override
            public void run() {
                SdkLog.loge("error----" + code + "----" + msg);
                callBack.onError(code, msg);
            }
        });
    }

    /**
     * 成功数据回调
     *
     * @param callBack
     * @param data
     * @param <T>
     */
    private static <T> void success(final HttpCallBack<T> callBack, final T data) {
        if (callBack == null) {
            return;
        }
        HttpManager.getInstance().executorUi(new Runnable() {
            @Override
            public void run() {
                callBack.onSuccess(data);
            }
        });
    }
}
