package com.xdja.aircert.sdk.device;

import android.text.TextUtils;

import com.xdja.aircert.sdk.AirCertSdk;
import com.xdja.aircert.sdk.bean.CertDetailInfoBean;
import com.xdja.aircert.sdk.config.AirCertConstant;
import com.xdja.aircert.sdk.util.FidUtil;
import com.xdja.aircert.sdk.util.SdkLog;
import com.xdja.cryptodev.CryptoDevInfo;
import com.xdja.cryptodev.CryptoDevManager;
import com.xdja.cryptodev.CryptoDevType;
import com.xdja.cryptodev.devapi.CryptoInstance;
import com.xdja.cryptodev.driver.chipmanager.ChipManagerDriver;
import com.xdja.cryptodev.driver.xdja.XdjaDevExtraInfo;
import com.xdja.multichip.param.JniApiParam;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.text.SimpleDateFormat;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * @author wanghao
 * @date 2017/9/6
 */

public class DeviceWrapper {
    /**
     * 获取当前设备卡Id
     */
    public static CryptoDevInfo getCurrentDevInfo() {
        if (AirCertSdk.getsInstance().getAirCertConfig().getCardType() == AirCertConstant.DEV_TYPE_VHSM) {
            return getVhsmCryptoDev();
        } else {
            Map<Integer, CryptoDevInfo> devInfoMap = getDevInfo();
            if (devInfoMap == null || devInfoMap.size() == 0) {
                return null;
            }
            return devInfoMap.get(getDeviceCryType(AirCertSdk.getsInstance().getAirCertConfig().getCardType()).getType());
        }
    }

    /**
     * VHSM设备信息
     *
     * @return
     */
    public static CryptoDevInfo getVhsmCryptoDev() {
        CryptoDevInfo vhsmInfo = new CryptoDevInfo();
        vhsmInfo.setCardId(NetVHSMHandle.getNetVhsmCardId(AirCertSdk.getsInstance().getContext()));
        vhsmInfo.setDriver(ChipManagerDriver.DRIVER_NAME);
        vhsmInfo.setType(CryptoDevType.DEV_TYPE_Soft);

        XdjaDevExtraInfo extraInfo = new XdjaDevExtraInfo();
        vhsmInfo.setPrivateData((Object) extraInfo.setType(JniApiParam.TYPE_VHSM_NET));
        SdkLog.loge("vhsmId" + vhsmInfo.getCardId());
        return vhsmInfo;
    }

    /**
     * 转换卡类型
     *
     * @param certType
     * @return
     */
    public static CryptoDevType getDeviceCryType(@AirCertConstant.DEV_TYPE int certType) {
        switch (certType) {
            case AirCertConstant.DEV_TYPE_CHIP:
                return CryptoDevType.DEV_TYPE_Chip;
            case AirCertConstant.DEV_TYPE_TF_CARD:
                return CryptoDevType.DEV_TYPE_SdCard;
            case AirCertConstant.DEV_TYPE_SIM_CARD:
                return CryptoDevType.DEV_TYPE_SimKey;
            case AirCertConstant.DEV_TYPE_VHSM:
                return CryptoDevType.DEV_TYPE_Soft;
            default:
                return CryptoDevType.DEV_TYPE_Soft;
        }
    }

    /**
     * 获取所有卡设备
     *
     * @return
     */
    public static Map<Integer, CryptoDevInfo> getDevInfo() {
        scanningDevice();
        Map<Integer, CryptoDevInfo> infoMap = new HashMap<>();
        List<CryptoDevInfo> devList = getDeviceList();
        if (devList == null || devList.size() <= 0) {
            return infoMap;
        }
        for (CryptoDevInfo devInfo : devList) {
            switch (devInfo.getType()) {
                case DEV_TYPE_SdCard:
                    infoMap.put(CryptoDevType.DEV_TYPE_SdCard.getType(), devInfo);
                    break;
                case DEV_TYPE_Chip:
                    infoMap.put(CryptoDevType.DEV_TYPE_Chip.getType(), devInfo);
                    break;
                case DEV_TYPE_SimKey:
                    infoMap.put(CryptoDevType.DEV_TYPE_SimKey.getType(), devInfo);
                    break;
                case DEV_TYPE_Soft:
                    infoMap.put(CryptoDevType.DEV_TYPE_Soft.getType(), devInfo);
                    break;
                default:
                    break;

            }
        }
        return infoMap;
    }

    /**
     * 扫描设备
     */
    private static void scanningDevice() {
        boolean isScanningDevice = CryptoDevManager.getInstance().isScanning();
        while (isScanningDevice) {
            try {
                Thread.sleep(1000);
            } catch (InterruptedException e) {
                e.printStackTrace();
            }
            isScanningDevice = CryptoDevManager.getInstance().isScanning();
        }
    }

    /**
     * 获取加密列表
     *
     * @return
     */
    private static List<CryptoDevInfo> getDeviceList() {
        List<CryptoDevInfo> devList = CryptoDevManager.getInstance().getCryptoDevInfos();
        if (devList.size() <= 0) {
            return null;
        }
        return devList;
    }

    /**
     * 读取证书信息
     *
     * @param type
     * @param containerNo
     * @return
     */
    public CertDetailInfoBean readCertFromContainer(CryptoDevType type, int containerNo) {
        String certType = AirCertSdk.getsInstance().getAirCertConfig().getCertType();
        byte[] fid = null;
        byte[][] certFid = FidUtil.getCertFid(containerNo);

        if (certType.equals(AirCertConstant.CERT_TYPE_SIGNATURE)) {
            fid = certFid[0];
            return readCertFromFid(type, fid);
        } else if (certType.equals(AirCertConstant.CERT_TYPE_ENCRYPTION)) {
            fid = certFid[1];
            return readCertFromFid(type, fid);
        } else {
            CertDetailInfoBean info;
            info = readCertFromFid(type, certFid[0]);
            if (info == null) {
                info = readCertFromFid(type, certFid[1]);
            }
            return info;
        }

    }

    private CertDetailInfoBean readCertFromFid(CryptoDevType type, byte[] fid) {
        byte[] certBuf = new byte[2048];
        int[] certLen = new int[1];
        byte[] bytes = readCertBytes(type, fid, certBuf, certLen);
        if (bytes == null) {
            return null;
        }

        return readBytesToCert(fid, certBuf, certLen);
    }

    /**
     * 读取指定设备0号容器内的签名证书
     *
     * @param type 要读取的设备类型
     * @param fid  读取的位置
     * @return
     */
    public byte[] readCertBytes(CryptoDevType type, byte[] fid, byte[] certBuf, int[] certLen) {
        CryptoInstance cryptoInstance = getCryptoInstance(type);
        if (cryptoInstance == null) {
            return null;
        }
        int result = cryptoInstance.readCert(fid, certBuf, certLen);
        cryptoInstance.close();

        if (result != 0) {
            return null;
        }
        return certBuf;
    }

    /**
     * 将证书byte[]解析为可读的字符串
     * author jff
     *
     * @param fid     设备内证书id
     * @param certBuf 待解析的证书byte[]
     * @return 证书实体
     */
    private CertDetailInfoBean readBytesToCert(byte[] fid, byte[] certBuf, int[] certLen) {
        if (fid.length == 0 || certBuf.length == 0 || certLen.length == 0) {
            return null;
        }
        try {
            SimpleDateFormat formatter = new SimpleDateFormat("yyyy.MM.dd");
            ByteArrayInputStream bin = new ByteArrayInputStream(certBuf);
            CertificateFactory cf = CertificateFactory.getInstance("X.509");
            X509Certificate cert = (X509Certificate) cf.generateCertificate(bin);
            CertDetailInfoBean certDetailInfo = new CertDetailInfoBean();

            certDetailInfo.setIssuerOrg(cert.getIssuerX500Principal().getName());
            certDetailInfo.setIssuerCn(cert.getIssuerDN().getName());
            certDetailInfo.setSubjectOrg(cert.getSubjectX500Principal().getName());
            certDetailInfo.setSubjectCn(cert.getSubjectDN().getName());


            certDetailInfo.setAlg(cert.getSigAlgName());

            certDetailInfo.setVer(cert.getVersion());

            System.arraycopy(fid, 0, certDetailInfo.getCertId(), 0, 2);
            certDetailInfo.setIndate(formatter.format(cert.getNotBefore()) + " - " + formatter.format(cert.getNotAfter()));
            certDetailInfo.setNotBefore(cert.getNotBefore());
            certDetailInfo.setNotAfter(cert.getNotAfter());

            //sn应该是16进制的
            certDetailInfo.setSn(cert.getSerialNumber().toString(16));
            certDetailInfo.setCertLen(certLen);
            bin.close();

            return certDetailInfo;
        } catch (CertificateException e) {
            e.printStackTrace();
        } catch (IOException e) {
            e.printStackTrace();
        }
        return null;
    }


    /**
     * 根据实际情况获取CryptoInstance实例
     * author jff
     *
     * @return
     */
    public static CryptoInstance getCryptoInstance(CryptoDevType type) {
        scanningDevice();
        CryptoInstance cryptoDev;
        try {
            cryptoDev = CryptoDevManager.getInstance().createInstance(ChipManagerDriver.DRIVER_NAME, type);
        } catch (Exception e) {
            SdkLog.loge("220 DeviceWrapper getCryptoInstance : " + e.getMessage());
            e.printStackTrace();
            return null;
        }
        if (cryptoDev == null) {
            return null;
        }
        int result = cryptoDev.open();
        if (result != 0) {
            SdkLog.loge("getCryptoInstance  error  " + result);
            return null;
        }
        return cryptoDev;
    }

    /**
     * 由于服务端芯片类型对应的int值和客户端不一致，所以需要转换。
     * type 对应的是CryptoDevType的类型，返回的是接口文档对应的类型
     *
     * @return
     */
    public static int convertCardType(int type) {
        switch (type) {
            case 0:
                //tf
                return 3;
            case 1:
                //芯片
                return 2;
            case 3:
                return 6;
            case 5://
                return 7;
            default:
                return -1;
        }
    }

    /**
     * @param type
     * @return -1 设备获取失败
     * 0 验证成功
     */
    public static int verifyPin(CryptoDevType type, int containerNum, String pinInput) {
        CryptoInstance cryptoInstance = getCryptoInstance(type);
        if (cryptoInstance == null) {
            return -1;
        }
        String pin;
        if (pinInput.isEmpty()) {
            pin = AirCertConstant.PIN;
        } else {
            pin = pinInput;
        }
        int result = cryptoInstance.verifyPIN(pin, CertHelper.getRole(containerNum));
        return result;
    }

    /**
     * 获取所有设备信息
     *
     * @return
     */
    public static List<CryptoDevInfo> getCryptoDevInfos() {
        scanningDevice();
        return CryptoDevManager.getInstance().getCryptoDevInfos();
    }
}
