package com.xdja.safeclient.certcreation.device;

import android.text.TextUtils;

import com.aircert.util.ModuleLog;
import com.xdja.cryptodev.CryptoDevManager;
import com.xdja.cryptodev.CryptoDevType;
import com.xdja.cryptodev.devapi.CryptoInstance;
import com.xdja.cryptodev.devapi.RSAPubkey;
import com.xdja.safeclient.certcreation.AppConfig;
import com.xdja.safeclient.certcreation.config.CertRule;
import com.xdja.safeclient.certcreation.config.CertType;
import com.xdja.safeclient.certcreation.config.XDJAAlgParams;
import com.xdja.safeclient.certcreation.util.FidUtil;

import org.bouncycastle.asn1.ASN1ObjectIdentifier;
import org.bouncycastle.asn1.DERBitString;
import org.bouncycastle.asn1.DEROutputStream;
import org.bouncycastle.asn1.pkcs.CertificationRequest;
import org.bouncycastle.asn1.pkcs.CertificationRequestInfo;
import org.bouncycastle.asn1.x500.X500Name;
import org.bouncycastle.asn1.x509.AlgorithmIdentifier;
import org.bouncycastle.asn1.x509.SubjectPublicKeyInfo;
import org.bouncycastle.openssl.jcajce.JcaPEMKeyConverter;
import org.bouncycastle.operator.DefaultSignatureAlgorithmIdentifierFinder;
import org.bouncycastle.pkcs.PKCS10CertificationRequest;
import org.bouncycastle.util.BigIntegers;
import org.bouncycastle.util.encoders.Base64;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.math.BigInteger;
import java.security.InvalidKeyException;
import java.security.KeyFactory;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.PublicKey;
import java.security.SignatureException;
import java.security.spec.RSAPublicKeySpec;

/**
 * Created by jff on 2018/11/14.
 */

public class PKCS10ForRSAUtils extends PKCS10Utils {


    private static PKCS10ForRSAUtils pkcs10ForRSAUtils = new PKCS10ForRSAUtils();


    public static PKCS10ForRSAUtils getInstance() {
        if (pkcs10ForRSAUtils == null) {
            pkcs10ForRSAUtils = new PKCS10ForRSAUtils();
        }
        return pkcs10ForRSAUtils;
    }

    public PKCS10ForRSAUtils() {

    }

    /**
     * 生成RSA单证请求的P10
     *
     * @param dn 主题
     * @return 返回p10请求的字符串
     * @throws SignatureException
     * @throws NoSuchProviderException
     * @throws NoSuchAlgorithmException
     * @throws InvalidKeyException
     */
    @Override
    public String getP10RequestBC(CryptoDevType type, int containerNum, String dn, boolean isApplyCert) {

        if (TextUtils.isEmpty(dn)) {
            return "";
        }

        if (isApplyCert) {
            int keyResult = generateRSAKeys(type, containerNum);

            //TODO JFF 2018.7.20 需要跟zjc确认错误码处理
            if (keyResult != 0 && keyResult != -1) {
                return "errorCode" + CryptoDevManager.getInstance().getErrorText(keyResult);
            }
        }


        CryptoInstance cryptoInstance = DeviceWrapper.getCryptoInstance(type);
        //        ModuleLog.e("RSA cryptoInstance = " + cryptoInstance);
        if (cryptoInstance == null) {
            return "";
        }

        byte[][] keyPair = FidUtil.getKeyPairFid(type, containerNum);


        byte[] pubFid = keyPair[0];
        byte[] priFid = keyPair[1];
        RSAPubkey pubkey = new RSAPubkey();
        int result = cryptoInstance.readRSAPubKey(pubFid, pubkey);
        //        ModuleLog.e("读取RSA公钥 result = " + result + "  " + CryptoDevManager.getInstance().getErrorText(result));
        if (result != 0) {
            return CryptoDevManager.getInstance().getErrorText(result);
        }
        try {
            String m;
            if (RSA_ALG_BITS == XDJAAlgParams.RSA_KEY_LEN_1024) {
                byte[] realM = new byte[128];
                System.arraycopy(pubkey.getM(), 0, realM, 0, realM.length);
                m = Base64.toBase64String(realM);
            } else {
                m = Base64.toBase64String(pubkey.getM());
            }
            byte[] publicKeyByte = Base64.decode(m);

            BigInteger bigIntModulus = BigIntegers.fromUnsignedByteArray(publicKeyByte);
            RSAPublicKeySpec keySpec = new RSAPublicKeySpec(bigIntModulus, BigInteger.valueOf(pubkey.getE()));
            KeyFactory keyFactory = KeyFactory.getInstance("RSA", "BC");
            PublicKey publicKey = keyFactory.generatePublic(keySpec);


            ASN1ObjectIdentifier rsaOID = new ASN1ObjectIdentifier(XDJAAlgParams.OID_RSA);
            //            SubjectPublicKeyInfo spi = new SubjectPublicKeyInfo(new AlgorithmIdentifier(rsaOID), pub);
            SubjectPublicKeyInfo spi = SubjectPublicKeyInfo.getInstance(publicKey.getEncoded());
            final X500Name x500Name = new X500Name(dn);
            CertificationRequestInfo cf1 = new CertificationRequestInfo(x500Name, spi, null);
            ByteArrayOutputStream outStream = new ByteArrayOutputStream();
            DEROutputStream derOutStream = new DEROutputStream(outStream);
            try {
                derOutStream.writeObject(cf1);
            } catch (IOException e) {
                e.printStackTrace();
            }
            byte[] certInfoBytes = outStream.toByteArray();
            //对CertificationRequestInfo进行签名
            byte[] signData = new byte[256];
            int[] signDataLen = new int[1];


            int encodeRequestInfo = cryptoInstance.RSASign(
                    RSA_ALG_BITS, priFid, 1, certInfoBytes, certInfoBytes.length, signData, signDataLen);

            if (encodeRequestInfo != 0) {
                return null;
            }

            DERBitString dbs = new DERBitString(signData);
            String signAlg = "SHA1WITHRSA";
            AlgorithmIdentifier signIdentifier = new DefaultSignatureAlgorithmIdentifierFinder().find(signAlg);
            CertificationRequest cr = new CertificationRequest(cf1, signIdentifier, dbs);
            PKCS10CertificationRequest request = new PKCS10CertificationRequest(cr);

            byte[] buffer = request.getEncoded();

            String p10 = new String(Base64.encode(buffer));
            ModuleLog.e("RSA_P10请求字符串 ： " + p10);


            if (!TextUtils.isEmpty(p10)) {
                PublicKey key = getPublicKeyFromP10(p10);
                if (key != null) {
                    byte[] pub = key.getEncoded();
                    String pubKey = new String(Base64.encode(pub));
                    ModuleLog.e("RSA公钥:" + pubKey);
                }

            }

            return p10;
        } catch (Exception e) {
            e.printStackTrace();
        }
        return "";
    }


    /**
     * RA-宫冠鹏给的RA从P10中解析公钥的方法
     *
     * @param p10 base64格式的p10
     * @return
     * @throws Exception
     */
    private static PublicKey getPublicKeyFromP10(String p10) throws Exception {
        final String P10_HEAD = "-----BEGIN CERTIFICATE REQUEST-----";
        final String P10_TAIL = "-----END CERTIFICATE REQUEST-----";
        p10 = p10.replace(P10_TAIL, "").replace(P10_HEAD, "");
        p10 = p10.replace("\r", "").replace("\n", "");
        p10 = p10.replace("\\r", "").replace("\\n", "");
        PKCS10CertificationRequest re = new PKCS10CertificationRequest(Base64.decode(p10));
        JcaPEMKeyConverter converter = new JcaPEMKeyConverter().setProvider("BC");
        PublicKey publicKey = converter.getPublicKey(re.getSubjectPublicKeyInfo());
        return publicKey;
    }


    /**
     * 生成RSA签名公私钥对
     * author jff
     * todo jff 生成签名公私钥对
     *
     * @return
     */
    private int generateRSAKeys(CryptoDevType type, int containerNum) {
        int genRsaKeysPairResult = -1;
        CryptoInstance cryptoInstance = DeviceWrapper.getCryptoInstance(type);
        if (cryptoInstance == null) {
            return genRsaKeysPairResult;
        }
        AppConfig module = AppConfig.getInstance();

        //此处添加验证PIN，是因为生成公私钥操作返回-15（没有权限）
        int pinResult = DeviceWrapper.verifyPin(type, containerNum, "");
        if (pinResult == 0) {
            genRsaKeysPairResult = genRSAKeyPair(cryptoInstance, type, containerNum);
        } else {
            return pinResult;
        }

        if (genRsaKeysPairResult == 2030 || genRsaKeysPairResult == 2041) {
            if (cryptoInstance.createContainer(module.getRole(type, containerNum), module.getDefaultConfig().getPin(), containerNum, "") == 0) {
                genRsaKeysPairResult = genRSAKeyPair(cryptoInstance, type, containerNum);
            } else {
                return genRsaKeysPairResult;
            }
        }
        cryptoInstance.close();
        return genRsaKeysPairResult;
    }


    private int genRSAKeyPair(CryptoInstance cryptoInstance, CryptoDevType type, int containerNum) {
        int genRsaKeysPairResult = -1;
        if (AppConfig.getInstance().getCertConfig().getCertRule(type, containerNum).equals(CertRule.SINGLE)) {
            if (AppConfig.getInstance().getCertConfig().getCertType(type, containerNum).equals(CertType.SIGNING)
                    || TextUtils.isEmpty(AppConfig.getInstance().getCertConfig().getCertType(type, containerNum))) {
                genRsaKeysPairResult = cryptoInstance.genSignRSAKeyPair(containerNum, RSA_ALG_BITS);
            } else {
                genRsaKeysPairResult = cryptoInstance.genExchangeRSAKeyPair(containerNum, RSA_ALG_BITS);
            }
        } else {
            genRsaKeysPairResult = cryptoInstance.genSignRSAKeyPair(containerNum, RSA_ALG_BITS);
        }
        return genRsaKeysPairResult;
    }


}
