package com.xdja.jwt.location;

import android.app.Service;
import android.content.Context;
import android.content.Intent;
import android.os.Handler;
import android.os.IBinder;
import android.os.Message;
import android.text.TextUtils;
import android.util.Log;

import com.founder.location.FounderLocationClient;
import com.founder.location.FounderLocationClientOption;
import com.founder.location.FounderLocationListener;
import com.founder.location.constant.Constants;
import com.founder.location.entity.FounderLocation;
import com.xdja.jwt.location.data.XdjaLocation;
import com.xdja.jwt.location.thread.Platform;
import com.xdja.jwt.location.utils.ServiceUtil;
import com.xdja.jwt.location.utils.SharePrefUtil;
import com.xdja.publicclass.SocketManager;

/**
 * DESCRIBE:位置信息上报服务(方正方案)
 * Created by jarlen on 2016/11/8.
 */

public class LocationUploadService extends Service implements FounderLocationListener {

    private final static String TAG = LocationUploadService.class.getName();

    private FounderLocationClient locationClient;
    private FounderLocationClientOption option;

    /**
     * GPS定位信息
     */
    private XdjaLocation gpsLoaction;

    /* 标记已上报次数 */
    private int count = 0;

    private static final int LOOP_MESSAGE_TYPE_NORMAL = 0;
    private static final int LOOP_MESSAGE_TYPE_START = 1;
    private static final int LOOP_MESSAGE_TYPE_STOP = -1;

    /* 控制循环上报 */
    private boolean isLoopStop = false;

    /* 定位上报服务配置参数 */
    private LocationConfig config;

    private SocketManager socketManagerCMCC;
    private SocketManager socketManagerUNICOME;

    private Handler loopHandler = new Handler() {

        @Override
        public void handleMessage(Message msg) {
            switch (msg.what) {
                case LOOP_MESSAGE_TYPE_NORMAL:
                case LOOP_MESSAGE_TYPE_START:
                    upload();
                    break;
                case LOOP_MESSAGE_TYPE_STOP:
                    stopSelf();
                    break;

                default:

                    break;
            }
        }
    };


    @Override
    public IBinder onBind(Intent intent) {
        return null;
    }

    @Override
    public void onCreate() {
        super.onCreate();
    }

    @Override
    public int onStartCommand(Intent intent, int flags, int startId) {

        if (null == intent || !intent.hasExtra("tag")) {
            return super.onStartCommand(intent, flags, startId);
        }

        String tag = intent.getStringExtra("tag");
        if ("start".equals(tag)) {
            //启动上报启动
            if (locationClient == null) {
                if (getLocationConfig()) {
                    isLoopStop = false;
                    initLocationClient();
                    loopHandler.sendEmptyMessage(LOOP_MESSAGE_TYPE_START);
                } else {
                    Log.e(TAG, "start  --->  user id 获取失败");
                }
            }
        } else if ("stop".equals(tag)) {
            //停止上报服务
            isLoopStop = true;
            loopHandler.sendEmptyMessage(LOOP_MESSAGE_TYPE_STOP);
        }
        return super.onStartCommand(intent, flags, startId);
    }

    /**
     * 定位信息上报
     */
    private void upload() {
        Platform.get().execute(new Runnable() {
            @Override
            public void run() {
                if (isLoopStop) {
                    return;
                }
                try {
                    String reqString = "";

                    if (gpsLoaction != null && gpsLoaction.getLongitude() > 0 && gpsLoaction.getLatitude() > 0) {
                        reqString = ServiceUtil.uploadGPSMsg(getApplicationContext(), gpsLoaction);
                    }

                    if (!TextUtils.isEmpty(reqString)) {
                        SocketManager cs;
                        String retMsg;
                        if ("中国移动".equals(ServiceUtil.getProvider(getApplicationContext()))) {
                            if (socketManagerCMCC == null) {
                                socketManagerCMCC = new SocketManager(0, config.getCMCCIp(), config.getCMCCPort());
                            }
                            cs = socketManagerCMCC;
                        } else {
                            if (socketManagerUNICOME == null) {
                                socketManagerUNICOME = new SocketManager(0, config.getUNICOMEIp(), config.getCMCCPort());
                            }
                            cs = socketManagerUNICOME;
                        }
                        retMsg = cs.execute(reqString);
                        if (retMsg.substring(0, 2).equals("00")) {
                            String[] uploadInfo = retMsg.split("\\|");
                            String timeString = uploadInfo[1];
                            config.setUploadLoopTime(Integer.parseInt(timeString) * 1000);
                            count++;
                            Intent intent = new Intent();
                            intent.setPackage(getPackageName());
                            intent.setAction(LocationConstant.ACTION_LOCATION_UPLOAD);
                            intent.putExtra(LocationConstant.KEY_UPLOAD_COUNT, count);
                            intent.putExtra(LocationConstant.KEY_UPLOAD_TIME, uploadInfo[2]);
                            intent.putExtra(LocationConstant.KEY_LOCATION_DATA, gpsLoaction);
                            sendBroadcast(intent);
                        } else {
                            Log.e(TAG, "上报失败 ---> " + retMsg);
                        }
                        cs = null;
                        ServiceUtil.writeToSDCard(ServiceUtil.GPS_LOG_PATH, "sendMsg  =" + reqString + System.getProperty("line.separator") + "regMsg =" + retMsg, true);
                    } else {
                        Log.e(TAG, "未获取到位置信息");
                        ServiceUtil.writeToSDCard(ServiceUtil.GPS_LOG_PATH, "未获取到位置信息", true);
                    }

                } catch (Exception e) {
                    e.printStackTrace();
                    Log.e(TAG, "Exception ---> " + e.toString());
                }

                if (!isLoopStop) {
                    loopHandler.sendEmptyMessageDelayed(LOOP_MESSAGE_TYPE_NORMAL, config.getUploadLoopTime());
                }
            }
        });
    }

    /**
     * 初始化并启动定位服务
     */
    private void initLocationClient() {
        locationClient = new FounderLocationClient(getApplicationContext());

        option = new FounderLocationClientOption();
        option.setLBSServerUrl("http://" + config.getLbsServerIp() + ":" + config.getLbsServerPort() + "/lbsServer/");
        option.setRectify(false);
        option.setScanSpan(3000);
        option.setShare(false);
        option.setSource(Constants.GB);

        locationClient.setLocOption(option);
        locationClient.registerListener(this);
        locationClient.start();
    }

    @Override
    public void onReceiveLocation(FounderLocation founderLocation) {
        /*
        *定位结果描述：GPS定位结果
        *public static final int TypeGpsLocation = 0;
        *定位结果描述：wifi定位结果
        *public static final int TypeWifiLocation = 1;
        *定位结果描述：基站定位结果
        *public static final int TypeBSLocation = 2;
        *定位结果描述：网络连接失败
        *public static final int TypeNetWorkException = 10;
        *定位结果描述：无效定位结果
        *public static final int TypeNone = 11;
        *定位结果描述：server定位失败，没有对应的位置信息
        *public static final int TypeServerError = 12;
        *
        * founderLocation.getSatelliteNum()
        * GPS定位方式下，定位星数；其他定位方式下，该值为0，无意义
        *
        * founderLocation.getLocationtype()
        * 定位方式，分别为：
        * 1.GPS定位， LOCATION_FROM_GPS;
        * 2.WIFI定位，LOCATION_FROM_WIFI;
        * 3.GSM定位，LOCATION_FROM_GSM;
        *
        * founderLocation.getDescribe()
        * 定位结果码
        *
        * founderLocation.getDescribeContents()
        * 定位结果描述
        *
        *  方正定位很奇葩！经纬度是反的，维护的兄弟注意！！！
        *
        * */

        if (founderLocation == null) {
            gpsLoaction = null;
            return;
        }

        /* 定位方式 */
        String locationtype = "";

        switch (founderLocation.getDescribe()) {
            case FounderLocation.TypeNone:
            case FounderLocation.TypeNetWorkException:
                gpsLoaction = null;
                return;
            case FounderLocation.TypeBSLocation:
            case FounderLocation.TypeServerError:
                locationtype = "基站定位";
                break;
            case FounderLocation.TypeGpsLocation:
                locationtype = "GPS定位";
                break;
            case FounderLocation.TypeWifiLocation:
                locationtype = "WIFI定位";
                break;
            default:
                locationtype = "";
                return;
        }

         /* 经度 */
        double lon = founderLocation.getLongitude();

        /* 纬度 */
        double lat = founderLocation.getLatitude();

        /* 当前定位时间 */
        long locTime = founderLocation.getLocTime();

        /* 当前速度 */
        float speed = founderLocation.getSpeed();

        /* 海拔 */
        double altitude = founderLocation.getElev();

        /* 地址 */
        String addrDesc = founderLocation.getAddress();

        /* 方向 */
        float bearing = founderLocation.getDir();

        if (gpsLoaction == null) {
            gpsLoaction = new XdjaLocation();
        }

        gpsLoaction.setLatitude(lat);
        gpsLoaction.setLongitude(lon);
        gpsLoaction.setTime(locTime);
        gpsLoaction.setSpeed(speed);
        gpsLoaction.setAltitude(altitude);
        gpsLoaction.setAddrDesc(addrDesc);
        gpsLoaction.setBearing(bearing);
        gpsLoaction.setLocationType(locationtype);

        ServiceUtil.writeToSDCard(ServiceUtil.LOCATION_LOG_PATH, "Describe: " + founderLocation.getDescribe() + " 定位方式:  " + locationtype + " lon: " + lon + " lat: " + lat + System.getProperty("line.separator"), true);
    }


    @Override
    public void onDestroy() {
        super.onDestroy();
        recyleLocation();
        isLoopStop = true;
        gpsLoaction = null;
        socketManagerCMCC = null;
        socketManagerUNICOME = null;
    }

    /**
     * 回收定位服务资源
     */
    private void recyleLocation() {
        if (locationClient != null) {
            locationClient.stop();
            locationClient.unRegisterListener(this);
            locationClient = null;
            option = null;
        }
    }

    /**
     * 启动定位上报服务(对外)
     */
    public static void startService() {
        Intent starter = new Intent(LocationConstant.getContext(), LocationUploadService.class);
        starter.putExtra("tag", "start");
        LocationConstant.getContext().startService(starter);
    }

    /**
     * 停止定位上报服务(对外)
     */
    public static void stopService() {
        Intent stoper = new Intent(LocationConstant.getContext(), LocationUploadService.class);
        stoper.putExtra("tag", "stop");
        LocationConstant.getContext().startService(stoper);
    }

    /**
     * 初始化定位上报服务配置参数
     *
     * @param context
     * @param config
     */
    public static void initLocationConfig(Context context, LocationConfig config) {

        if (context == null) {
            throw new IllegalArgumentException("context can't be NULL");
        }
        LocationConstant.initContext(context);

        if (config != null) {
            SharePrefUtil spUtil = SharePrefUtil.getSpUtil(context);

            if (!TextUtils.isEmpty(config.getUserId())) {
                spUtil.saveString("user_id", config.getUserId());
            } else {
                Log.e(TAG, "initLocationConfig ---> " + "没有用户ID是不行的^_^");
                return;
            }

            spUtil.saveInt("loop_upload_time", config.getUploadLoopTime());

            if (!TextUtils.isEmpty(config.getLbsServerIp())) {
                spUtil.saveString("lbs_server_ip", config.getLbsServerIp());
            }

            if (!TextUtils.isEmpty(config.getLbsServerPort())) {
                spUtil.saveString("lbs_server_port", config.getLbsServerPort());
            }

            if (!TextUtils.isEmpty(config.getCMCCIp())) {
                spUtil.saveString("cmcc_ip", config.getCMCCIp());
            }

            if (!TextUtils.isEmpty(config.getUNICOMEIp())) {
                spUtil.saveString("unicome_ip", config.getUNICOMEIp());
            }

            if (!TextUtils.isEmpty(config.getCMCCPort())) {
                spUtil.saveString("cmcc_port", config.getCMCCPort());
            }

            if (!TextUtils.isEmpty(config.getUNICOMEPort())) {
                spUtil.saveString("unicome_port", config.getUNICOMEPort());
            }
        }

    }

    /**
     * 获取定位上报参数
     */
    private boolean getLocationConfig() {
        SharePrefUtil spUtil = SharePrefUtil.getSpUtil(getApplicationContext());

        if (TextUtils.isEmpty(spUtil.getString("user_id", ""))) {
            return false;
        }

        this.config = new LocationConfig();
        config.setUserId(spUtil.getString("user_id", ""));
        config.setUploadLoopTime(spUtil.getInt("loop_upload_time", config.getUploadLoopTime()));
        config.setLbsServerIp(spUtil.getString("lbs_server_ip", config.getLbsServerIp()));
        config.setLbsServerPort(spUtil.getString("lbs_server_port", config.getLbsServerPort()));
        config.setCMCCIp(spUtil.getString("cmcc_ip", config.getCMCCIp()));
        config.setCMCCPort(spUtil.getString("cmcc_port", config.getCMCCPort()));
        config.setUNICOMEIp(spUtil.getString("cmcc_port", config.getUNICOMEIp()));
        config.setUNICOMEPort(spUtil.getString("unicome_port", config.getUNICOMEPort()));
        return true;
    }
}
