package com.xdja.gprsconverter.convert;

import com.xdja.gprsconverter.protocol.ChangePwdRequest;
import com.xdja.gprsconverter.protocol.ChangePwdResponse;
import com.xdja.gprsconverter.protocol.EntryResponse;
import com.xdja.gprsconverter.protocol.EntryRequest;
import com.xdja.gprsconverter.protocol.LoginRequest;
import com.xdja.gprsconverter.protocol.LoginResponse;
import com.xdja.gprsconverter.protocol.LogoutRequest;
import com.xdja.gprsconverter.protocol.LogoutResponse;
import com.xdja.gprsconverter.protocol.QueryRequest;
import com.xdja.gprsconverter.protocol.QueryResponse;

/**
 * 转换工厂，所有协议相互转换入口
 * <p>
 * Created by Guojie on 2017/8/28.
 */

public class ConvertFactory<T> {
    /**
     * 无关类型转换方法
     *
     * @param obj 请求协议
     * @return
     */
    public static String generateXML(Object obj) {
        if (obj instanceof QueryRequest) {
            return generateQueryXML((QueryRequest) obj);
        } else if (obj instanceof EntryRequest) {
            return generateEntryXML((EntryRequest) obj);
        } else if (obj instanceof LoginRequest) {
            return generateLoginXML((LoginRequest) obj);
        } else if (obj instanceof LogoutResponse) {
            return generateLogoutXML((LogoutRequest) obj);
        } else if (obj instanceof ChangePwdRequest) {
            return generateChangePwdXML((ChangePwdRequest) obj);
        } else if (obj instanceof String) {
            return (String) obj;
        }
        return null;
    }

    /**
     * 根据泛型自动解析
     *
     * @param response
     * @param tClass
     * @return
     */
    public T parserResponse(String response, Class<T> tClass) {
        T t = null;
        try {
            t = tClass.newInstance();
            if (t instanceof LoginResponse) {
                t = (T) parserLoginResponse(response);
            } else if (t instanceof LogoutResponse) {
                t = (T) parserLogoutResponse(response);
            } else if (t instanceof QueryResponse) {
                t = (T) parserQueryResponse(response);
            } else if (t instanceof EntryRequest) {
                t = (T) parserEnterResponse(response);
            } else if (t instanceof ChangePwdResponse) {
                t = (T) parserChangePwdResponse(response);
            } else if (t instanceof String) {
                t = (T) response;
            }
        } catch (InstantiationException e) {
            e.printStackTrace();
        } catch (IllegalAccessException e) {
            e.printStackTrace();
        }
        return t;
    }

    /**
     * 将封装的查询协议bean转换为xml
     *
     * @param queryRequest 查询协议bean
     * @return 转换结果xml
     */
    public static String generateQueryXML(QueryRequest queryRequest) {
        return new QueryProtocolConvert().convertBean2XML(queryRequest);
    }

    /**
     * 将封装的录入协议bean转换为xml
     *
     * @param entryRequest 录入协议bean
     * @return 转换结果xml
     */
    public static String generateEntryXML(EntryRequest entryRequest) {
        return new EntryProtocolConvert().convertBean2XML(entryRequest);
    }

    /**
     * 生成修改密码协议xml
     *
     * @param changePwdRequest 修改密码协议bean
     * @return 转换结果xml
     */
    public static String generateChangePwdXML(ChangePwdRequest changePwdRequest) {
        return new ChangePwdProtocolConvert().convertBean2XML(changePwdRequest);
    }

    /**
     * 生成登录协议xml
     *
     * @param loginRequest 登录协议bean
     * @return 转换结果xml
     */
    public static String generateLoginXML(LoginRequest loginRequest) {
        return new LoginProtocolConvert().convertBean2XML(loginRequest);
    }

    /**
     * 生成登出协议xml
     *
     * @param logoutRequest 登出协议bean
     * @return 转换结果xml
     */
    public static String generateLogoutXML(LogoutRequest logoutRequest) {
        return new LogoutProtocolConvert().convertBean2XML(logoutRequest);
    }

    /**
     * 将后台返回的查询结果转化为查询结果bean
     *
     * @param queryResponse 查询结果xml
     * @return 查询结果bean
     */
    public static QueryResponse parserQueryResponse(String queryResponse) {
        return new QueryProtocolConvert().convertXML2Bean(queryResponse);
    }

    /**
     * 将后台返回的录入结果转化为录入结果bean
     *
     * @param queryResponse 录入结果xml
     * @return 录入结果bean
     */
    public static EntryResponse parserEnterResponse(String queryResponse) {
        return new EntryProtocolConvert().convertXML2Bean(queryResponse);
    }

    /**
     * 解析修改密码结果，是否直接返回true或者false好点？
     *
     * @param changePwdResponse 后台返回结果
     * @return 修改结果bean
     */
    public static ChangePwdResponse parserChangePwdResponse(String changePwdResponse) {
        return new ChangePwdProtocolConvert().convertXML2Bean(changePwdResponse);
    }

    /**
     * 解析登录返回结果
     *
     * @param loginResponse 解析登录返回结果
     * @return 用户信息bean
     */
    public static LoginResponse parserLoginResponse(String loginResponse) {
        return new LoginProtocolConvert().convertXML2Bean(loginResponse);
    }

    /**
     * 解析登出结果，是否直接返回true或者false好点？
     *
     * @param logoutResponse 后台返回结果
     * @return 登出结果bean
     */
    public static LogoutResponse parserLogoutResponse(String logoutResponse) {
        return new LogoutProtocolConvert().convertXML2Bean(logoutResponse);
    }
}
