package com.xdja.combox.inputbutton;

import java.util.HashMap;

import com.xdja.combox.utils.DensityUtil;

import android.content.Context;
import android.graphics.drawable.Drawable;
import android.text.Editable;
import android.text.InputFilter;
import android.text.TextWatcher;
import android.text.method.DigitsKeyListener;
import android.util.AttributeSet;
import android.util.Log;
import android.view.Gravity;
import android.view.View;
import android.widget.LinearLayout;
import android.widget.RelativeLayout;
import android.widget.TextView;

/**
 * 由文本显示框和文本输入框组成的复合控件
 * 
 * @author fjd
 * 
 */
public class XDEditTextView extends RelativeLayout {
	
	private Context mContext = null;							// 窗体句柄
	private HashMap<String, Integer> attrsMap = null; 			// 属性字典集合
	private XDEditText mEditText = null; 						// 复合控件中的输入框
	private TextView mTextView = null; 							// 复合控件中的文本显示框
	private int layoutType = 1; 								// 子控件布局方式代码（0：外漏；1：内嵌）
	private int displayType = 1; 								// 控件显示风格
	private int layoutHeight = -2;								// 控件高度
	private int layoutWidth = -1;								// 控件宽度
	private int textviewWidth = -2;								// 文本显示框的宽度
//	private int textview = 0x00000001;						// 控件ID号
//	private int edittext = 0x00000002;						// 控件ID号
	private LinearLayout ll = null;
	public XDEditTextView(Context context,int displayType) {
		this(context, null);
		this.setDisplayType(displayType);
	}

	public XDEditTextView(Context context, AttributeSet attrs) {
		super(context, attrs);
		mContext = context;
		// 初始化内部两个子控件
		mEditText = new XDEditText(context);
//		mEditText.setId(edittext);
		mTextView = new TextView(context);
//		mTextView.setId(textview);
		mTextView.setGravity(Gravity.CENTER);
		mTextView.setSingleLine(false);
		initialAttrMap(context);
		analysisAttrs(attrsMap, attrs);
		initialControl(context);
	}
	
	/**
	 * 初始化属性字典
	 * 
	 * @param context 窗体句柄
	 */
	private void initialAttrMap(Context context) {
		// 初始化属性字典
		attrsMap = new HashMap<String, Integer>();
		attrsMap.put("displaytype", 0); 					// 显示风格
		attrsMap.put("edittext_background", 1); 			// 文本输入框背景
		attrsMap.put("textview_background", 2); 			// 文本显示框背景
		attrsMap.put("textview_textcolor", 3); 				// 文本显示框字体颜色
		attrsMap.put("edittext_textcolor", 4); 				// 文本输入框字体颜色
		attrsMap.put("edittext_textsize", 5); 				// 文本输入框字体大小
		attrsMap.put("textview_textsize", 6); 				// 文本显示框字体大小
		attrsMap.put("textview_text", 7); 					// 文本显示框文本
		attrsMap.put("edittext_text", 8); 					// 文本输入框文本
		attrsMap.put("textview_visiable", 9); 				// 文本显示框是否可见
		attrsMap.put("layouttype", 10); 					// 布局方式（0：外露，1：内嵌）
		attrsMap.put("edittext_enable", 12); 				// 文本输入框是否可用
		attrsMap.put("edittext_digits", 13); 				// 文本输入框输入内容过滤器
		attrsMap.put("edittext_maxlength", 14);				// 文本输入框最大输入长度
		attrsMap.put("textview_width", 15); 				// 文本显示框的宽度
		attrsMap.put("layout_widht", 16);					// 控件宽度
		attrsMap.put("layout_height", 17); 					// 空间高度
	}
	
	/**
	 * 解析xml布局传递的属性参数
	 * 
	 * @param attrsMap
	 *            属性字典
	 * @param attrs
	 */
	private void analysisAttrs(HashMap<String, Integer> attrsMap,
			AttributeSet attrs) {
		int attrId = -1;
		if (attrs != null) {
			int attrsCount = attrs.getAttributeCount();
			for (int i = 0; i < attrsCount; i++) {
				if (attrsMap.containsKey(attrs.getAttributeName(i))) {
					attrId = attrsMap.get(attrs.getAttributeName(i));
					switch (attrId) {
					case 0:
						setBackgroundByType(attrs.getAttributeValue(i));
						break;
					case 1:
						setEditTextBackground(attrs.getAttributeResourceValue(
								i, -1));
						break;
					case 2:
						setTextViewBackground(attrs.getAttributeResourceValue(
								i, -1));
						break;
					case 3:
						setTextViewTextColor(attrs.getAttributeResourceValue(i,
								-1));
						break;
					case 4:
						setEditTextTextColor(attrs.getAttributeResourceValue(i,
								-1));
						break;
					case 5:
						setEditText_textSize(formatFloatVlaue(attrs
								.getAttributeValue(i)));
						break;
					case 6:
						setTextView_textSize(formatFloatVlaue(attrs
								.getAttributeValue(i)));
						break;
					case 7:
						setTextView_text(attrs.getAttributeValue(i));
						break;
					case 8:
						setEditText_text(attrs.getAttributeValue(i));
						break;
					case 9:
						setTextView_visiable(attrs.getAttributeBooleanValue(i,
								true));
						break;
					case 10:
						this.layoutType = attrs.getAttributeIntValue(i, 1);
						break;
					case 12:
						setEditText_enable(attrs.getAttributeBooleanValue(i,
								true));
						break;
					case 13:
						setEditText_digits(attrs.getAttributeValue(i));
						break;
					case 14:
						setEditText_maxLength(attrs
								.getAttributeIntValue(i, 100));
						break;
					case 15:
						this.textviewWidth = formatIntVlaue(attrs
								.getAttributeValue(i));
						setTextView_width(textviewWidth);
						break;
					case 16:
						this.layoutWidth = formatIntVlaue(attrs
								.getAttributeValue(i));
						break;
					case 17:
						this.layoutHeight = formatIntVlaue(attrs
								.getAttributeValue(i));
						break;
					default:
						break;
					}
				}
			}
		}
	}
	/**
	 * 设置控件的宽和高
	 * @param width 宽度
	 * @param height 高度
	 */
	public void setControlSize(int width,int height){
		this.layoutHeight = height;
		this.layoutWidth = width;
		initialControl(this.mContext);
	}
	
	/**
	 * 初始化控件
	 */
	private void initialControl(Context context) {
		this.setLayoutParams(new LayoutParams(layoutWidth,layoutHeight));
		this.setPadding(0, 0, 0, 0);
		this.removeAllViews();
		setGravity(Gravity.CENTER_VERTICAL);
		/*
		 * 外露布局
		 */
		if (this.layoutType == 0) {
			if(ll != null && ll.getChildCount()>0){
				ll.removeAllViews();
			}
			ll = new LinearLayout(context);
			ll.setOrientation(0);
			ll.setLayoutParams(new RelativeLayout.LayoutParams(
					LayoutParams.FILL_PARENT, this.layoutHeight));
			LinearLayout.LayoutParams textViewLayoutParams = new LinearLayout.LayoutParams(
					this.textviewWidth,this.layoutHeight);
			ll.addView(mTextView, textViewLayoutParams);
			LinearLayout.LayoutParams editTextLayoutParams = new LinearLayout.LayoutParams(
					LayoutParams.FILL_PARENT, this.layoutHeight);
			editTextLayoutParams.weight = 1;
			mEditText.setPadding(DensityUtil.dp2px(mContext, 5), 0, 0, 0);
			ll.addView(mEditText, editTextLayoutParams);
			this.addView(ll);
			/*
			 * 内嵌布局
			 */
		} else if (this.layoutType == 1) {
			RelativeLayout.LayoutParams editTextLayoutParams = new RelativeLayout.LayoutParams(
					LayoutParams.FILL_PARENT, this.layoutHeight);
//			mEditText.setId(edittext);
			this.addView(mEditText, editTextLayoutParams);
			RelativeLayout.LayoutParams textViewLayoutParams = new RelativeLayout.LayoutParams(
					this.textviewWidth, this.layoutHeight);
			textViewLayoutParams.addRule(RelativeLayout.ALIGN_LEFT, mEditText.getId());
			textViewLayoutParams.addRule(RelativeLayout.CENTER_VERTICAL,
					this.getId());
			this.addView(mTextView, textViewLayoutParams);
			mEditText.setPadding(mTextView.getWidth()+DensityUtil.dp2px(mContext, 5), 0, 0, 0);
		}
	}
	@Override
	protected void onLayout(boolean changed, int l, int t, int r, int b) {
		// TODO Auto-generated method stub
		super.onLayout(changed, l, t, r, b);
		if(layoutType == 1){
			mEditText.setPadding(mTextView.getWidth()+DensityUtil.dp2px(mContext, 5), 0, 0, 0);
		}
	}
	/**
	 * 根据风格代码设置不同的风格
	 * 
	 * @param typeStr
	 */
	private void setBackgroundByType(String typeStr) {
		if (typeStr != null) {
			if (typeStr.equals("0")) {
				this.displayType = 0;
			} else if (typeStr.equals("1")) {
				this.displayType = 1;
			}
			mEditText.setDisplayType(this.displayType);
		}
	}

	/**
	 * 根据传递的资源号来设置输入框的前景色
	 * 
	 * @param resourceId
	 */
	private void setEditTextTextColor(int resourceId) {
		if (resourceId > 0) {
			setEditText_textColor(getContext().getResources().getColor(
					resourceId));
		}
	}

	/**
	 * 根据传递的资源号来设置文本框的前景色
	 * 
	 * @param resourceId
	 */
	private void setTextViewTextColor(int resourceId) {
		boolean isFormat = true;
		if (resourceId > 0) {
			try {
				setTextView_textColor(getContext().getResources().getColor(
						resourceId));
			} catch (Exception ex) {
				isFormat = false;
			}
		}
		if (!isFormat) {
			try {
				setTextView_textColor(resourceId);
			} catch (Exception e) {
				Log.e("错误的颜色值", resourceId + "为错误的颜色值");
			}
		}
	}

	/**
	 * 根据传递的资源号来设置文本框的背景
	 * 
	 * @param resourceId
	 */
	private void setTextViewBackground(int resourceId) {
		if (resourceId > 0) {
			setTextView_background(resourceId);
		}
	}

	/**
	 * 根据传递的资源号来设置输入框的背景
	 * 
	 * @param resourceId
	 */
	private void setEditTextBackground(int resourceId) {
		if (resourceId > 0) {
			setEditText_background(resourceId);
		}
	}

	/**
	 * 格式化xml布局文件中传入的字体大小设置的值
	 * 
	 * @param val
	 *            传入的字符串
	 * @return 格式化得到的float类型值
	 */
	private float formatFloatVlaue(String val) {
		boolean isFormat = true;
		float value = (float) 14.0;
		if (val.endsWith("sp")) {
			val = val.substring(0, val.length() - 2);
		}
		try {
			value = Float.parseFloat(val);
		} catch (NumberFormatException e) {
			isFormat = false;
		}
		if (!isFormat) {
			try {
				value = (float) Integer.parseInt(val);
			} catch (NumberFormatException ex) {
				Log.e("NumberFormatException", val + "不能被转化为float类型");
			}
		}
		return value;
	}
	/**
	 * 格式化xml布局文件中传入的宽度大小值为整数
	 * 
	 * @param val
	 *            传入的字符串
	 * @return 得到的整数结果
	 */
	private int formatIntVlaue(String val) {
		int value = -1;
		if (val.equals("wrap_content")) {
			value = -2;
		} else if (val.equals("fill_parent")) {
			value = -1;
		} else if (val.endsWith("dp")||val.endsWith("dip")) {
			if(val.endsWith("dp")){
				val = val.substring(0, val.length() - 2);
			}else if(val.endsWith("dip")){
				val = val.substring(0, val.length() - 5);
			}
			try {
				value = Integer.parseInt(val);
				value = DensityUtil.dp2px(mContext, (float) value);
				Log.v("pxValue", value + "");
			} catch (NumberFormatException ex) {
				Log.e("NumberFormatException", val + "无法转换为整数");
			}
		} else {
			if (val.endsWith("px")) {
				val = val.substring(0, val.length() - 2);
			}
			try {
				value = Integer.parseInt(val);
				Log.v("pxValue", value + "");
			} catch (NumberFormatException ex) {
				Log.e("NumberFormatException", val + "无法转换为整数");
				Log.v("pxValue", value + "");
			}
		}
		return value;
	}

	/**
	 * 设置是否显示文本显示框
	 * 
	 * @param visiable
	 *            true为显示，false为不显示
	 */
	public void setTextView_visiable(boolean visiable) {
		if (visiable) {
			mTextView.setVisibility(View.VISIBLE);
		} else {
			mTextView.setVisibility(View.GONE);
		}
	}

	/**
	 * 获取复合控件整体显示风格
	 * 
	 * @return 风格代码
	 */
	public int getDisplayType() {
		return displayType;
	}

	/**
	 * 设置复合控件整体显示风格
	 * 
	 * @param type
	 *            风格代码
	 */
	public void setDisplayType(int displayType) {
		this.displayType = displayType;
		mEditText.setDisplayType(displayType);
	}

	/**
	 * 获取布局方式代码（0：外漏，1：内嵌）
	 * 
	 * @return 布局方式代码
	 */
	public int getLayoutType() {
		return this.layoutType;
	}

	/**
	 * 设置布局方式
	 * 
	 * @param layoutType
	 *            布局方式代码
	 */
	public void setLayoutType(int layoutType) {
		this.layoutType = layoutType;
		initialControl(this.mContext);
	}

	/**
	 * 获取复合控件中文本输入框内的文本
	 * 
	 * @return 文本值
	 */
	public Editable getEditText_text() {
		return this.mEditText.getText();
	}

	/**
	 * 设置复合控件中文本输入框的文本
	 * 
	 * @param mEditText_text
	 *            目标文本
	 */
	public void setEditText_text(String mEditText_text) {
		this.mEditText.setText(mEditText_text);
	}

	/**
	 * 获取复合控件中文本显示框内的文本
	 * 
	 * @return 文本值
	 */
	public CharSequence getTextView_text() {
		return this.mTextView.getText();
	}

	/**
	 * 设置复合控件中文本显示框的文本
	 * 
	 * @param mTextView_text
	 *            目标文本
	 */
	public void setTextView_text(String mTextView_text) {
		this.mTextView.setText(mTextView_text);
	}

	/**
	 * 获取复合控件中文本显示框的文本大小
	 * 
	 * @return 获取到的文本大小值
	 */
	public float getTextView_textSize() {
		return this.mTextView.getTextSize();
	}

	/**
	 * 设置符合控件中文本显示框的文本大小
	 * 
	 * @param mTextView_textSize
	 *            目标文本大小
	 */
	public void setTextView_textSize(float mTextView_textSize) {
		this.mTextView.setTextSize(mTextView_textSize);
	}

	/**
	 * 获取复合控件中文本输入框的文本大小
	 * 
	 * @return 获取到的文本大小值
	 */
	public float getEditText_textSize() {
		return this.mEditText.getTextSize();
	}

	/**
	 * 设置复合控件中文本输入框的文本大小
	 * 
	 * @param mEditText_textSize
	 *            目标文本大小
	 */
	public void setEditText_textSize(float mEditText_textSize) {
		this.mEditText.setTextSize(mEditText_textSize);
	}

	/**
	 * 设置复合控件中文本输入框的文本颜色
	 * 
	 * @param mEditText_textColor
	 *            颜色值
	 */
	public void setEditText_textColor(int mEditText_textColor) {
		this.mEditText.setTextColor(mEditText_textColor);
	}

	/**
	 * 设置复合控件中文本显示框的文本颜色
	 * 
	 * @param mTextView_textColor
	 *            颜色值
	 */
	public void setTextView_textColor(int mTextView_textColor) {
		this.mTextView.setTextColor(mTextView_textColor);
	}

	/**
	 * 获取复合控件中文本显示框的背景
	 * 
	 * @return 文本显示框的背景
	 */
	public Drawable getTextView_background() {
		return this.mTextView.getBackground();
	}

	/**
	 * 设置复合控件中文本显示框的背景
	 * 
	 * @param resid
	 *            背景资源ID
	 */
	public void setTextView_background(int resid) {
		this.mTextView.setBackgroundResource(resid);
	}

	/**
	 * 获取复合控件中文本输入框的背景
	 * 
	 * @return 文本输入框的背景
	 */
	public Drawable getEditText_background() {
		return this.mEditText.getBackground();
	}

	/**
	 * 设置复合控件中文本输入框的背景
	 * 
	 * @param resid
	 *            背景资源ID
	 */
	public void setEditText_background(int resid) {
		this.mEditText.setBackgroundResource(resid);
	}

	/**
	 * 设置复合控件中的文本输入框是否可用
	 * 
	 * @param enable
	 *            true为可用false为不可用，默认为true
	 */
	public void setEditText_enable(boolean enable) {
		this.mEditText.setEnabled(enable);
		if (!enable) {
			this.mEditText.setFocusable(false);
			this.mEditText.setFocusableInTouchMode(false);
		} else {
			this.mEditText.setFocusable(true);
			this.mEditText.setFocusableInTouchMode(true);
			this.mEditText.requestFocus();
		}
	}

	/**
	 * 设置复合控件中的文本输入范围
	 * 
	 * @param digits
	 *            输入范围
	 */
	public void setEditText_digits(String digits) {
		mEditText.setKeyListener(DigitsKeyListener.getInstance(digits));
	}

	/**
	 * 设置最大输入长度
	 * 
	 * @param maxLength
	 */
	public void setEditText_maxLength(int maxLength) {
		InputFilter[] filter = { new InputFilter.LengthFilter(maxLength) };
		this.mEditText.setFilters(filter);
	}

	
	/**
	 * 设置文本显示框的宽度
	 * 
	 * @param textviewWidth
	 *            宽度值
	 */
	public void setTextView_width(int textviewWidth) {
		this.textviewWidth = textviewWidth;
		initialControl(mContext);
	}
	/**
	 * 获取复合控件中的文本输入框
	 * @return 获取到的EditText对象
	 */
	public XDEditText getEditText(){
		return this.mEditText;
	}
	
	/*
	 * 为控件注册事件
	 */
	public void addEditText_TextChangedListener(TextWatcher watcher) {
		mEditText.addTextChangedListener(watcher);
	}

	public void setEditText_OnClickListener(View.OnClickListener l) {
		mEditText.setOnClickListener(l);
	}

	public void setEditText_OnTouchListener(View.OnTouchListener l) {
		mEditText.setOnTouchListener(l);
	}

	public void setEditText_OnLongClickListener(View.OnLongClickListener l) {
		mEditText.setOnLongClickListener(l);
	}

	public void setEditText_OnCreateContextMenuListener(
			View.OnCreateContextMenuListener l) {
		mEditText.setOnCreateContextMenuListener(l);
	}

	public void setEditText_OnFocusChangeListener(View.OnFocusChangeListener l) {
		mEditText.setOnFocusChangeListener(l);
	}

	public void setEditText_OnKeyListener(View.OnKeyListener l) {
		mEditText.setOnKeyListener(l);
	}

	public void setEditText_PaddingRight(int right) {
		if(this.layoutType == 1){
			this.mEditText.setPadding(this.mTextView.getWidth()+DensityUtil.dp2px(mContext, 5), 0, right, 0);
		}else if(this.layoutType == 0){
			this.mEditText.setPadding(DensityUtil.dp2px(mContext, 5), 0, right, 0);
		}
	}

}
