package com.xdja.nutnet;

import com.xdja.Exception.CErrorInfo;
import com.xdja.gprsconverter.convert.ConvertFactory;
import com.xdja.publicclass.SocketManager;

import io.reactivex.Observable;
import io.reactivex.ObservableEmitter;
import io.reactivex.ObservableOnSubscribe;
import io.reactivex.Observer;
import io.reactivex.android.schedulers.AndroidSchedulers;
import io.reactivex.annotations.NonNull;
import io.reactivex.disposables.Disposable;
import io.reactivex.schedulers.Schedulers;

/**
 * Created by Guojie on 2017/9/5.
 */

public class SocketRequest {
    private static final String TAG = SocketRequest.class.getSimpleName();
    public static final int SAFE_CHANNEL = 1;
    public static final int UNSAFE_CHANNEL = 0;
    private Disposable mDisposable;
    private SocketManager socketManager;
    private int isSafe;

    public <RequestType, ResponseType> Observable<ResponseType> sendRequest(@NonNull final RequestType requestType, @NonNull final Class<ResponseType> classOfT) {
        Observable<ResponseType> observable = Observable.create(new ObservableOnSubscribe<ResponseType>() {
            @Override
            public void subscribe(@NonNull ObservableEmitter<ResponseType> e) throws Exception {
                String requestStr = ConvertFactory.generateXML(requestType);
                NutNetLog.i(TAG, "requestStr=" + requestStr);
                NutNetClient.Config config = NutNetClient.getInstance().getConfig();
                isSafe = config.isSafe() ? SAFE_CHANNEL : UNSAFE_CHANNEL;
                SocketManager.setTimeout(config.getConnectTimeout() * 1000);
                socketManager = new SocketManager(isSafe, config.getHost(), config.getPort());
                String response = socketManager.execute(requestStr);
                NutNetLog.i(TAG, "response=" + response);
                CErrorInfo.paserErrorMsg(response);
                if (CErrorInfo.isError && !CErrorInfo.message.contains("修改密码成功")) {
                    if (CErrorInfo.code.equals("0x13014034") || CErrorInfo.message.contains("登录超时")) {
                        //超时处理？
                        if (config.getTimeoutListener() != null) {
                            config.getTimeoutListener().onTimeout();
                        }
                    }
                    Throwable throwable = new Throwable(response);
                    if (!e.isDisposed())
                        e.onError(throwable);
                } else {
                    if (!e.isDisposed()) {
                        e.onNext(new ConvertFactory().parserResponse(response, classOfT));
                        e.onComplete();
                    }
                }
            }
        });
        return observable;
    }

    public <RequestType, ResponseType> void sendRequest(@NonNull final RequestType requestType, @NonNull final Class<ResponseType> classOfT, @NonNull final Callback<ResponseType> callback) {
        sendRequest(requestType, classOfT)
                .subscribeOn(Schedulers.io())
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(new Observer<ResponseType>() {
                    @Override
                    public void onSubscribe(@NonNull Disposable d) {
                        NutNetLog.i(TAG, "onSubscribe");
                        mDisposable = d;
                        callback.onStart();
                    }

                    @Override
                    public void onNext(@NonNull ResponseType t) {
                        NutNetLog.i(TAG, "onNext");
                        callback.onSuccess(t);
                    }

                    @Override
                    public void onError(@NonNull Throwable e) {
                        NutNetLog.i(TAG, "onError");
                        CErrorInfo.paserErrorMsg(e.getMessage());
                        callback.onFial(CErrorInfo.code, CErrorInfo.message);
                    }

                    @Override
                    public void onComplete() {
                        NutNetLog.i(TAG, "onComplete");
                    }
                });
    }

    public void cancel() {
        if (mDisposable != null) {
            mDisposable.dispose();
        }
        if (socketManager != null) {
            socketManager.disConnect();
        }

    }

}
