package com.xdja.reckon.api;

import android.content.Context;

import com.xdja.reckon.device.NetWorkUtil;

import java.io.File;
import java.io.IOException;
import java.util.concurrent.TimeUnit;

import okhttp3.Cache;
import okhttp3.Interceptor;
import okhttp3.OkHttpClient;
import okhttp3.Response;
import okhttp3.logging.HttpLoggingInterceptor;
import retrofit2.Retrofit;
import retrofit2.adapter.rxjava2.RxJava2CallAdapterFactory;
import retrofit2.converter.gson.GsonConverterFactory;

/**
 * Created by Guojie on 16/9/6.
 */

public class APIManager {
    private Context mContext;
    public static APIManager apiManager;
    private File httpCacheDirectory;
    private int cacheSize = 10 * 1024 * 1024; // 10 MiB
    private Cache cache;
    private Object monitor = new Object();
    private Interceptor REWRITE_CACHE_CONTROL_INTERCEPTOR;
    private HttpLoggingInterceptor httpLoggingInterceptor;
    private OkHttpClient client;
    /**
     * SDK API
     */
    private DeviceApi deviceApi;
    private UserApi userApi;
    private AppInfoApi appInfoApi;

    public APIManager(Context context) {
        this.mContext = context;
        initOkHttpCilent();
    }

    private void initOkHttpCilent() {
        httpCacheDirectory = new File(mContext.getCacheDir(), "reckonCache");
        cache = new Cache(httpCacheDirectory, cacheSize);
        REWRITE_CACHE_CONTROL_INTERCEPTOR = new Interceptor() {
            @Override
            public Response intercept(Chain chain) throws IOException {
                Response originalResponse = chain.proceed(chain.request());
                if (NetWorkUtil.isNetWorkAvailable(mContext)) {
                    int maxAge = 60; // 在线缓存在1分钟内可读取
                    return originalResponse.newBuilder()
                            .removeHeader("Pragma")
                            .removeHeader("Cache-Control")
                            .header("Cache-Control", "public, max-age=" + maxAge)
                            .build();
                } else {
                    int maxStale = 60 * 60 * 24 * 28; // 离线时缓存保存4周
                    return originalResponse.newBuilder()
                            .removeHeader("Pragma")
                            .removeHeader("Cache-Control")
                            .header("Cache-Control", "public, only-if-cached, max-stale=" + maxStale)
                            .build();
                }
            }
        };
        httpLoggingInterceptor = new HttpLoggingInterceptor();
        httpLoggingInterceptor.setLevel(HttpLoggingInterceptor.Level.BODY);
        client = new OkHttpClient.Builder()
                .addNetworkInterceptor(REWRITE_CACHE_CONTROL_INTERCEPTOR)
                .addInterceptor(httpLoggingInterceptor)
//                .cache(cache)
                .connectTimeout(5, TimeUnit.SECONDS)
                .readTimeout(30, TimeUnit.SECONDS)
                .writeTimeout(30, TimeUnit.SECONDS)
                .build();
    }

    public static APIManager getInstance(Context context) {
        if (apiManager == null) {
            synchronized (APIManager.class) {
                if (apiManager == null) {
                    apiManager = new APIManager(context);
                }
            }
        }
        return apiManager;
    }

    /**
     * 获取设备相关API
     *
     * @return
     */
    public DeviceApi getDeviceApiService() {
        if (deviceApi == null) {
            synchronized (monitor) {
                if (deviceApi == null) {
                    deviceApi = new Retrofit.Builder()
                            .baseUrl(Constants.getServerAddress(mContext))
                            .addCallAdapterFactory(RxJava2CallAdapterFactory.create())
                            .client(client)
                            .addConverterFactory(GsonConverterFactory.create())
                            .build().create(DeviceApi.class);
                }
            }
        }
        return deviceApi;
    }

    /**
     * 用户相关API
     *
     * @return
     */
    public UserApi getUserApiService() {
        if (userApi == null) {
            synchronized (monitor) {
                if (userApi == null) {
                    userApi = new Retrofit.Builder()
                            .baseUrl(Constants.getServerAddress(mContext))
                            .addCallAdapterFactory(RxJava2CallAdapterFactory.create())
                            .client(client)
                            .addConverterFactory(GsonConverterFactory.create())
                            .build().create(UserApi.class);
                }
            }
        }
        return userApi;
    }

    /**
     *
     */
    public AppInfoApi getAppInfoApiService() {
        if (appInfoApi == null) {
            synchronized (monitor) {
                if (appInfoApi == null) {
                    appInfoApi = new Retrofit.Builder()
                            .baseUrl(Constants.getServerAddress(mContext))
                            .addCallAdapterFactory(RxJava2CallAdapterFactory.create())
                            .client(client)
                            .addConverterFactory(GsonConverterFactory.create())
                            .build().create(AppInfoApi.class);
                }
            }
        }
        return appInfoApi;
    }
}
