package com.xdja.reckon.service;

import android.app.Service;
import android.content.Context;
import android.content.Intent;
import android.os.Binder;
import android.os.IBinder;
import android.text.TextUtils;

import com.xdja.reckon.api.APIManager;
import com.xdja.reckon.function.AppManager;
import com.xdja.reckon.function.DeviceManager;
import com.xdja.reckon.function.PersonManager;
import com.xdja.reckon.function.StateListener;
import com.xdja.reckon.model.AppCountPlus;
import com.xdja.reckon.model.AppInfo;
import com.xdja.reckon.model.AppResUsed;
import com.xdja.reckon.model.AppState;
import com.xdja.reckon.model.DeviceInfo;
import com.xdja.reckon.model.DeviceState;
import com.xdja.reckon.model.Response;
import com.xdja.reckon.share.SharePreConstants;
import com.xdja.reckon.share.SharePrefUtil;
import com.xdja.reckon.common.Log;

import java.util.concurrent.TimeUnit;

import io.reactivex.Observable;
import io.reactivex.ObservableEmitter;
import io.reactivex.ObservableOnSubscribe;
import io.reactivex.Observer;
import io.reactivex.android.schedulers.AndroidSchedulers;
import io.reactivex.disposables.CompositeDisposable;
import io.reactivex.disposables.Disposable;
import io.reactivex.functions.Consumer;
import io.reactivex.schedulers.Schedulers;

/**
 * 统计信息上报服务
 * <p>
 * Created by Guojie on 16/9/7.
 */

public class ReckonService extends Service {
    private static final String TAG = ReckonService.class.getSimpleName();
    private CompositeDisposable mCompositeDisposable;
    private ReportBinder reportBinder;
    private Context context;
    private boolean isResport = false;//控制上报应用运行状态线程是否继续执行
    /**
     * 设备ID
     */
    private String deviceId;
    /**
     * 应用ID
     */
    private String appId;
    /**
     * 用户ID
     */
    private String personID;
    /**
     * 上报次数统计
     */
    private int count;
    /**
     * 上报退出成功监听
     */
    private RepotrStateListener mRepotrStateListener;
    private StateListener mStateListener;

    @Override
    public void onCreate() {
        super.onCreate();
        context = this;
        mCompositeDisposable=new CompositeDisposable();
    }

    @Override
    public IBinder onBind(Intent intent) {
        Log.i(TAG, "onBind");
        return reportBinder = new ReportBinder();
    }

    /**
     * 控制上报服务运行状态
     */
    public class ReportBinder extends Binder {
        public void startReport(String personID, RepotrStateListener repotrStateListener) {
            mRepotrStateListener = repotrStateListener;
            PersonManager.savePersonID(context, personID);
            ReckonService.this.personID = personID;
            //初始化SDK,初始化SDK,获取应用ID和设备ID
            if (DeviceManager.isExistDeviceID(context) && DeviceManager.isSameDevice(context)) {
                //若存在设备ID,设备信息一致则上报一次设备状态,否则调接口获取设备ID
                reportDeviceState();
                if (AppManager.isExistAppID(context)) {
                    //若存在应用ID,上报一次运行次数,并开启统计服务,否则调节口获取应用ID
                    reportAppUseCount();
                    startReportState();
                } else {
                    getRemoteAppID();
                }
            } else {
                getRemoteDeviceID();
            }
        }

        /**
         * 停止统计上报服务
         */
        public void stopReport() {
            reportAppResUsed();
            reportAppState(AppState.STATE_EXIT);
            isResport = false;
            Log.i(TAG, "stopReport");
        }

        /**
         * 添加状态监听
         */
        public void addSateListener(StateListener stateListener) {
            if (stateListener != null) {
                mStateListener = stateListener;
            }
        }
    }

    /**
     * 开启统计服务线程
     */
    private void startReportState() {
        addState("开启统计服务成功");
        mRepotrStateListener.onSuccess();
        isResport = true;
        reportAppState(AppState.STATE_OPEN);
        Disposable reportDisposable = Observable.interval(5, TimeUnit.MINUTES)
                .subscribe(new Consumer<Long>() {
                    @Override
                    public void accept(Long aLong) throws Exception {
                        if (isResport) {
                            reportAppState(AppState.STATE_RUN);
                            reportAppResUsed();
                        }
                    }
                });
        mCompositeDisposable.add(reportDisposable);
    }

    /**
     * 上报设备信息,从服务器获取deviceID
     *
     * @return
     */
    private String getRemoteDeviceID() {
        DeviceInfo deviceInfo = DeviceInfo.buildEntity(context);
        APIManager.getInstance(context)
                .getDeviceApiService()
                .getDeviceID(deviceInfo)
                .subscribeOn(Schedulers.io())
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(new Observer<Response>() {
                    @Override
                    public void onSubscribe(Disposable d) {
                        mCompositeDisposable.add(d);
                    }

                    @Override
                    public void onNext(Response response) {
                        if (Response.STATE_SUCCESS.equals(response.getState())) {
                            deviceId = response.getData();
                            DeviceManager.saveDeviceID(context, deviceId);
                            DeviceManager.saveDeviceInfo(context);
                            reportDeviceState();
                            getRemoteAppID();
                            addState("获取设备ID成功:" + deviceId);
                        } else {
                            mRepotrStateListener.onError();
                            addState("获取设备ID错误:" + response.getError());
                        }
                    }

                    @Override
                    public void onError(Throwable e) {
                        mRepotrStateListener.onError();
                        addState("获取设备ID错误:" + e.getMessage());
                    }

                    @Override
                    public void onComplete() {
                        Log.i(TAG, "onCompleted");
                    }
                });

        return deviceId;
    }

    /**
     * 上报应用信息,从服务器获取appID
     *
     * @return
     */
    private String getRemoteAppID() {
        AppInfo appInfo = AppInfo.buildEntity(context);
        APIManager.getInstance(context)
                .getAppInfoApiService()
                .getAppId(appInfo)
                .subscribeOn(Schedulers.io())
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(
                        new Observer<Response>() {
                            @Override
                            public void onError(Throwable e) {
                                mRepotrStateListener.onError();
                                addState("获取应用ID错误:" + e.getMessage());
                            }

                            @Override
                            public void onComplete() {
                                Log.i(TAG, "onCompleted");
                            }

                            @Override
                            public void onSubscribe(Disposable d) {
                                mCompositeDisposable.add(d);
                            }

                            @Override
                            public void onNext(Response response) {
                                if (Response.STATE_SUCCESS.equals(response.getState())) {
                                    appId = response.getData();
                                    AppManager.saveAppID(context, appId);
                                    addState("获取应用ID成功:" + appId);
                                    reportAppUseCount();
                                    startReportState();
                                } else {
                                    mRepotrStateListener.onError();
                                    addState("获取应用ID错误:" + response.getError());
                                }
                            }
                        });
        return appId;
    }

    /**
     * 上报App运行状态
     *
     * @param state 状态信息
     */
    private void reportAppState(final String state) {
        AppState appState = new AppState();
        appState.setPersonId(SharePrefUtil.getString(context, SharePreConstants.PERSON_ID, null));
        appState.setDeviceId(SharePrefUtil.getString(context, SharePreConstants.DEVICE_ID, null));
        appState.setAppId(SharePrefUtil.getString(context, SharePreConstants.APP_ID, null));
        appState.setAppState(state);
        APIManager.getInstance(context)
                .getAppInfoApiService()
                .uploadAppInfo(appState)
                .subscribeOn(Schedulers.io())
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(
                        new Observer<Response>() {
                            @Override
                            public void onError(Throwable e) {
                                if (AppState.STATE_EXIT.equals(state)) {
                                    mRepotrStateListener.onError();
                                }
                                addState("上报应用运行状态失败:" + e.getMessage());
                            }

                            @Override
                            public void onComplete() {
                                Log.i(TAG, "onCompleted");
                            }

                            @Override
                            public void onSubscribe(Disposable d) {
                                mCompositeDisposable.add(d);
                            }

                            @Override
                            public void onNext(Response response) {
                                Log.i(TAG, response.toString());
                                if ("1".equals(response.getState())) {
                                    ++count;
                                    addState("上报应用运行状态成功,运行状态:" + state + ",第" + count + "次上传");
                                    if (null != mRepotrStateListener && AppState.STATE_EXIT.equals(state)) {
                                        mRepotrStateListener.onStopReport();
                                    }
                                } else {
                                    addState("上报应用运行状态失败:" + response.getError());
                                }
                            }
                        });
    }

    /**
     * 上报设备状态
     *
     * @return
     */
    private String reportDeviceState() {
        DeviceState deviceState = DeviceState.buildEntity(context);
        APIManager.getInstance(context)
                .getDeviceApiService()
                .uploadDeviceState(deviceState)
                .subscribeOn(Schedulers.io())
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(
                        new Observer<Response>() {

                            @Override
                            public void onError(Throwable e) {
                                addState("上报设备状态失败:" + e.getMessage());
                            }

                            @Override
                            public void onComplete() {
                                Log.i(TAG, "onCompleted");
                            }

                            @Override
                            public void onSubscribe(Disposable d) {
                                mCompositeDisposable.add(d);
                            }

                            @Override
                            public void onNext(Response response) {
                                if (Response.STATE_SUCCESS.equals(response.getState())) {
                                    addState("上报设备状态成功");
                                } else {
                                    addState("上报设备状态失败:" + response.getError());
                                }
                            }
                        });
        return appId;
    }

    /**
     * 上报App占用资源
     *
     * @return
     */
    private String reportAppResUsed() {
        AppResUsed appResUsed = AppResUsed.buildEntity(context);
        APIManager.getInstance(context)
                .getAppInfoApiService()
                .uploadAppResUsed(appResUsed)
                .subscribeOn(Schedulers.io())
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(
                        new Observer<Response>() {
                            @Override
                            public void onError(Throwable e) {
                                addState("上报应用占用资源信息失败:" + e.getMessage());
                            }

                            @Override
                            public void onComplete() {
                                Log.i(TAG, "onCompleted");
                            }

                            @Override
                            public void onSubscribe(Disposable d) {
                                mCompositeDisposable.add(d);
                            }

                            @Override
                            public void onNext(Response response) {
                                if (Response.STATE_SUCCESS.equals(response.getState())) {
                                    addState("上报应用占用资源信息成功");
                                } else {
                                    addState("上报应用占用资源信息失败:" + response.getError());
                                }
                            }
                        });
        return appId;
    }

    /**
     * 上报应用使用次数
     *
     * @return
     */
    private String reportAppUseCount() {
        AppCountPlus appCountPlus = new AppCountPlus();
        appCountPlus.setDeviceId(DeviceManager.getLocalDeviceID(context));
        appCountPlus.setPersonId(PersonManager.getLocalPersonID(context));
        appCountPlus.setAppId(AppManager.getLocalAppID(context));
        APIManager.getInstance(context)
                .getAppInfoApiService()
                .appCountPlus(appCountPlus)
                .subscribeOn(Schedulers.io())
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(
                        new Observer<Response>() {
                            @Override
                            public void onError(Throwable e) {
                                mRepotrStateListener.onError();
                                addState("上报应用使用次数失败:" + e.getMessage());
                            }

                            @Override
                            public void onComplete() {
                                Log.i(TAG, "onCompleted");
                            }

                            @Override
                            public void onSubscribe(Disposable d) {
                                mCompositeDisposable.add(d);
                            }

                            @Override
                            public void onNext(Response response) {
                                if (Response.STATE_SUCCESS.equals(response.getState())) {
                                    addState("上报应用使用次数成功");
                                } else {
                                    mRepotrStateListener.onError();
                                    addState("上报应用使用次数失败:" + response.getError());
                                }

                            }
                        });
        return deviceId;
    }

    @Override
    public void onDestroy() {
        super.onDestroy();
        unsubcrible();
        addState("终止统计分析服务");
    }

    private void unsubcrible() {
        if (mCompositeDisposable != null) {
            mCompositeDisposable.clear();
        }
    }

    private void addState(String state) {
        if (mStateListener != null) {
            mStateListener.reportState(state);
            Log.i(TAG, state);
        }
    }

    /**
     * 开启上报成功或退出、发生错误监听
     * 用于上层ReckAgent解绑服务
     */
    public interface RepotrStateListener {
        void onSuccess();

        void onStopReport();

        void onError();
    }
}
