package com.xdja.camera2videolibrary;

import android.app.Activity;
import android.app.Fragment;
import android.content.Intent;
import android.graphics.Bitmap;
import android.graphics.SurfaceTexture;
import android.media.MediaPlayer;
import android.os.Build;
import android.os.Bundle;
import android.os.CountDownTimer;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.RequiresApi;
import android.util.Log;
import android.view.KeyEvent;
import android.view.LayoutInflater;
import android.view.TextureView;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.RelativeLayout;
import android.widget.TextView;
import android.widget.Toast;

import com.xdja.camera2videolibrary.utils.CameraUtils;
import com.xdja.camera2videolibrary.utils.PermissionUtils;
import com.xdja.camera2videolibrary.widget.AutoFitTextureView;
import com.xdja.camera2videolibrary.widget.MyVideoView;
import com.xdja.camera2videolibrary.widget.RecordedButton;
import com.xdja.camera2videolibrary.widget.subscaleview.ImageSource;
import com.xdja.camera2videolibrary.widget.subscaleview.SubsamplingScaleImageView;

import java.util.ArrayList;

/**
 * Created by wanjing on 2017/9/21.
 * 
 * desc: 基类。
 */

public abstract class BaseFragment extends Fragment implements View.OnClickListener {

    private static final String TAG = BaseFragment.class.getSimpleName();

    /**
     * An {@link AutoFitTextureView} for camera preview.
     */
    protected AutoFitTextureView mTextureView;

    /**
     * A {@link SubsamplingScaleImageView} for image show.
     */
    private SubsamplingScaleImageView mImageView;

    /**
     * A {@link MyVideoView} for video play.
     */
    private MyVideoView mVideoView;

    /**
     * Button to record video
     */
    private RecordedButton mButtonVideo;

    /**
     * Shoot RelativeLayout on the top
     */
    private RelativeLayout mLayoutTop_Shoot;

    /**
     * RelativeLayout on the top
     */
    private RelativeLayout mLayoutTop;

    /**
     * ImageView to change flash
     */
    private ImageView mImgChangeFlash;

    /**
     * ImageView to change camera
     */
    private ImageView mImgChangeCamera;

    /**
     * ImageView to finish this fragment
     */
    private ImageView mImgClose;

    /**
     * TextView to show the results size
     */
    private TextView mTextEnsure;

    /**
     * Shoot LinearLayout on the bottom
     */
    private LinearLayout mLayoutBotton_Shoot;

    /**
     * RelativeLayout on the bottom
     */
    private RelativeLayout mLayoutBottom;

    /**
     * ImageView to finish this fragment
     */
    private ImageView mImgReturn;

    /**
     * ImageView to re-shoot
     */
    private ImageView mImgBack;

    /**
     * ImageView to finish this shoot
     */
    private ImageView mImgFinish;

    /**
     * TextView to hint
     */
    private TextView mTextHint;

    /**
     * 倒计时处理录制进度
     */
    private CountDownTimer countDownTimer;

    private Bitmap mPictureCacheBitmap;
    private String mVideoCacheAbsolutePath;

    private int max = 10000; //短视频最大录制时间，默认10000ms
    private int maxNum = 1; // 最多拍摄多少个视频/照片, 默认一张
    private int currentNum = 0; //已拍摄多少个视频/照片，默认currentNum==0
    private boolean hide = false; //是否隐藏头部布局，包含开关闪光灯+切换摄像头按钮，默认不隐藏
    private String fileDir; //保存的视频/照片存储目录，默认getExternalFilesDir(null)
    private int type = CameraUtils.TYPE_PICTURE_AND_VIDEO; //拍摄类型 0：照片和短视频，1：照片 2：短视频 默认type==0

    private ArrayList<String> results = new ArrayList<>(); //拍摄结果

    private ICameraCallBack mCameraCallBack; //

    public void setCameraCallBack(ICameraCallBack cameraCallBack) {
        this.mCameraCallBack = cameraCallBack;
    }

    @Override
    public void onCreate(@Nullable Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        Bundle bundle = getArguments();
        if (bundle != null) {
            currentNum = bundle.getInt("currentNum", 0);
            CameraParams cameraParams = (CameraParams) bundle.getSerializable("cameraParams");
            if (cameraParams != null) {
                max = cameraParams.getMax();
                maxNum = cameraParams.getMaxNum();
                hide = cameraParams.isHide();
                fileDir = cameraParams.getFileDir();
                type = cameraParams.getType();
            }
        }
    }

    @Override
    public View onCreateView(LayoutInflater inflater, ViewGroup container,
                             Bundle savedInstanceState) {
        return inflater.inflate(R.layout.fragment_camera2, container, false);
    }

    @Override
    public void onViewCreated(View view, @Nullable Bundle savedInstanceState) {
        super.onViewCreated(view, savedInstanceState);
        mTextureView = (AutoFitTextureView) view.findViewById(R.id.texture);
        mImageView = (SubsamplingScaleImageView) view.findViewById(R.id.imageView);
        mVideoView = (MyVideoView) view.findViewById(R.id.videoview);
        mButtonVideo = (RecordedButton) view.findViewById(R.id.rb_shoot);
        mLayoutTop_Shoot = (RelativeLayout) view.findViewById(R.id.rl_top_shoot);
        mImgChangeFlash = (ImageView) view.findViewById(R.id.iv_change_flash);
        mImgChangeCamera = (ImageView) view.findViewById(R.id.iv_change_camera);
        mLayoutBotton_Shoot = (LinearLayout) view.findViewById(R.id.ll_bottom_shoot);
        mLayoutBottom = (RelativeLayout) view.findViewById(R.id.rl_bottom);
        mImgBack = (ImageView) view.findViewById(R.id.iv_back);
        mImgFinish = (ImageView) view.findViewById(R.id.iv_finish);
        mImgReturn = (ImageView) view.findViewById(R.id.iv_return);
        mLayoutTop = (RelativeLayout) view.findViewById(R.id.rl_top);
        mImgClose = (ImageView) view.findViewById(R.id.iv_close);
        mTextEnsure = (TextView) view.findViewById(R.id.tv_ensure);
        mTextHint = (TextView) view.findViewById(R.id.tv_hint);

        mImgChangeFlash.setOnClickListener(this);
        mImgChangeCamera.setOnClickListener(this);
        mImgBack.setOnClickListener(this);
        mImgFinish.setOnClickListener(this);
        mImgReturn.setOnClickListener(this);
        mImgClose.setOnClickListener(this);

        mButtonVideo.setMax(max);
        mButtonVideo.setOnGestureListener(onGestureListener);

        mLayoutTop_Shoot.setVisibility(hide ? View.GONE : View.VISIBLE);

        CameraUtils.setFileDir(getActivity(), fileDir);

        // add by wanjing on 2017/10/26 10:50
        // 仅拍摄照片，所以设置不响应长按事件
        if (type == CameraUtils.TYPE_ONLY_PICTURE) {
            mButtonVideo.setResponseLongTouch(false);
        }
        // 如果仅拍摄照片或短视频，则把提示语“轻触拍照，长按摄像”隐藏掉
        if (type != CameraUtils.TYPE_PICTURE_AND_VIDEO) {
            mTextHint.setVisibility(View.GONE);
        }
        // end add on 2017/10/26 10:50.
    }

    @Override
    public void onResume() {
        super.onResume();
        View view = getView();
        if (view != null) {
            // 得到Fragment的根布局并使该布局可以获得焦点
            view.setFocusableInTouchMode(true);
            view.requestFocus();
            // 对该根布局view注册KeyListener的监听
            view.setOnKeyListener(new View.OnKeyListener() {

                @Override
                public boolean onKey(View view, int keyCode, KeyEvent keyEvent) {

                    if (keyEvent.getAction() == KeyEvent.ACTION_UP &&
                            keyCode == KeyEvent.KEYCODE_BACK) {
                        finishWithData();
                        return true;
                    }
                    return false;
                }
            });
        }

        // modify by wanjing on 2017/10/18 17:35 添加dialog非空判断，避免循环请求权限。
        if (confirmationDialog == null && mTextureView.isAvailable()) {
            openCamera(mTextureView.getWidth(), mTextureView.getHeight());
        } else {
            mTextureView.setSurfaceTextureListener(mSurfaceTextureListener);
        }
        // end modify on 2017/10/18 17:35.
    }

    @Override
    public void onDestroy() {
        if (mImageView != null) {
            mImageView.recycle();
            mImageView = null;
        }
        if (mVideoView != null) {
            mVideoView.stop();
            mVideoView.release();
            mVideoView = null;
        }
        super.onDestroy();
    }

    @Override
    public void onClick(View view) {
        int id = view.getId();
        if (id == R.id.iv_change_flash) { //开关闪关灯
            if(toggleFlash()){
                mImgChangeFlash.setImageResource(R.mipmap.video_flash_open);
            } else{
                mImgChangeFlash.setImageResource(R.mipmap.video_flash_close);
            }
        } else if (id == R.id.iv_change_camera) { //切换摄像头
            switchCamera();
            mImgChangeFlash.setImageResource(R.mipmap.video_flash_close);
        } else if (id == R.id.iv_back) { //重拍
            if (mCameraCallBack != null) {
                mCameraCallBack.onReTake();
            }

            changeButton(true);
            stopPlayingVideo();
            clearCache();
            startPreview();
        } else if (id == R.id.iv_finish) { //保存该图片/视频
            changeButton(true);
            stopPlayingVideo();

            saveCache();
            startPreview();

            // 拍摄完成一次后的处理
            if (results.size() >= maxNum - currentNum) {
                mButtonVideo.setResponseLongTouch(false);
            }

            if (maxNum == 1) {
                finishWithData();
            }
        } else if (id == R.id.iv_return || id == R.id.iv_close) {
            finishWithData();
        }
    }

    private void openCamera(int width, int height) {
        Activity activity = getActivity();
        if (null == activity || activity.isFinishing()) {
            return;
        }
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1
                && !PermissionUtils.hasPermissionsGranted(activity, PermissionUtils.VIDEO_PERMISSIONS)) {
            PermissionUtils.requestVideoPermissions(this);
            return;
        }
        openCamera(activity, width, height);
    }

    private void changeButton(boolean flag) {
        if (flag) {
            mLayoutTop.setVisibility(View.GONE);
            mLayoutBottom.setVisibility(View.GONE);
            mLayoutTop_Shoot.setVisibility(hide ? View.GONE : View.VISIBLE);
            mLayoutBotton_Shoot.setVisibility(View.VISIBLE);
        } else {
            mTextEnsure.setText(currentNum + results.size() + "/" + maxNum);

            mLayoutBotton_Shoot.setVisibility(View.GONE);
            mLayoutTop_Shoot.setVisibility(View.GONE);
            mLayoutBottom.setVisibility(View.VISIBLE);
            mLayoutTop.setVisibility(maxNum == 1 ? View.GONE : View.VISIBLE);
        }
    }

    private void clearCache() {
        if (mPictureCacheBitmap != null) {
            mPictureCacheBitmap.recycle();
            mPictureCacheBitmap = null;
        }
        if (mVideoCacheAbsolutePath != null) {
            CameraUtils.deleteFile(mVideoCacheAbsolutePath);
            mVideoCacheAbsolutePath = null;
        }
    }

    private void saveCache() {
        Activity activity = getActivity();
        if (activity == null || activity.isFinishing()) {
            return;
        }
        ArrayList<String> savedFilePaths = new ArrayList<>();
        if (mPictureCacheBitmap != null) {
            String picFilePath = CameraUtils.getPictureFilePath(activity);
            CameraUtils.savePic(mPictureCacheBitmap, picFilePath);
            savedFilePaths.add(picFilePath);
            mPictureCacheBitmap.recycle();
            mPictureCacheBitmap = null;
        }
        if (mVideoCacheAbsolutePath != null) {
            String videoFilePath = CameraUtils.getVideoFilePath(activity);
            CameraUtils.renameFile(mVideoCacheAbsolutePath, videoFilePath);
            savedFilePaths.add(videoFilePath);
            mVideoCacheAbsolutePath = null;
        }
        results.addAll(savedFilePaths);
    }

    private void finishWithData() {
        Activity activity = getActivity();
        if (activity != null) {
            Intent intent = new Intent();
            Bundle bundle = new Bundle();
            bundle.putStringArrayList(CameraUtils.EXTRA_CAMERA_RESULTS, results);
            intent.putExtras(bundle);
            activity.setResult(Activity.RESULT_OK, intent);
            activity.finish();
        }
    }

    private void stopPlayingVideo() {
        mVideoView.stop();
        mVideoView.setVisibility(View.GONE);
        mImageView.setVisibility(View.GONE);
    }

    RecordedButton.OnGestureListener onGestureListener
            = new RecordedButton.OnGestureListener() {
        @Override
        public void onLongClick() {
            if (results.size() >= maxNum - currentNum) {
                Toast.makeText(getActivity(), "最多拍摄" + maxNum + "张",
                        Toast.LENGTH_SHORT).show();
                return;
            }

            // 倒计时处理录制进度
            countDownTimer = new CountDownTimer(max, 100) {

                @Override
                public void onTick(long millisUntilFinished) {
                    mButtonVideo.setProgress(max - millisUntilFinished);
                }

                @Override
                public void onFinish() {
//                    mButtonVideo.setProgress(max);
//                    countDownTimer.cancel();
                }
            }.start();

            Activity activity = getActivity();
            if (activity == null || activity.isFinishing()) {
                return;
            }
            mVideoCacheAbsolutePath = CameraUtils.getVideoCacheFilePath(activity);
            startRecordingVideo(mVideoCacheAbsolutePath);
        }

        @Override
        public void onClick() {
            // add by wanjing on 2017/10/26 10:50
            // 仅拍摄短视频，所以设置不响应点击事件
            if (type == CameraUtils.TYPE_ONLY_VIDEO) {
                return;
            }
            // end add on 2017/10/26 10:50.
            if (results.size() >= maxNum - currentNum) {
                Toast.makeText(getActivity(), "最多拍摄" + maxNum + "张",
                        Toast.LENGTH_SHORT).show();
                return;
            }
            if(!isFastDoubleClick(1500)){
                takePicture(new PictureCallback() {
                    @Override
                    public void onPictureTaken(Bitmap bitmap) {
                        mPictureCacheBitmap = bitmap;
                        getActivity().runOnUiThread(new Runnable() {
                            @Override
                            public void run() {
                                if (mCameraCallBack != null) {
                                    mCameraCallBack.onComplete();
                                }
                                changeButton(false);
                                stopPreview();

                                mImageView.setVisibility(View.VISIBLE);
                                mImageView.setImage(ImageSource.bitmap(mPictureCacheBitmap));
                            }
                        });
                    }
                });
            }
        }


        @Override
        public void onOver() {
            if (mCameraCallBack != null) {
                mCameraCallBack.onComplete();
            }
            changeButton(false);
            countDownTimer.cancel();

            stopRecordingVideo();

            // 循环播放录制的短视频
            mVideoView.setVisibility(View.VISIBLE);
            mVideoView.setVideoPath(mVideoCacheAbsolutePath);
            mVideoView.setOnPreparedListener(new MediaPlayer.OnPreparedListener() {
                @Override
                public void onPrepared(MediaPlayer mp) {
                    mVideoView.setLooping(true);
                    mVideoView.start();
                }
            });
        }
    };


    /**
     * {@link TextureView.SurfaceTextureListener} handles several lifecycle events on a
     * {@link TextureView}.
     */
    TextureView.SurfaceTextureListener mSurfaceTextureListener
            = new TextureView.SurfaceTextureListener() {

        @Override
        public void onSurfaceTextureAvailable(SurfaceTexture surfaceTexture,
                                              int width, int height) {
            openCamera(width, height);
        }

        @Override
        public void onSurfaceTextureSizeChanged(SurfaceTexture surfaceTexture,
                                                int width, int height) {
            configureTransform(width, height);
        }

        @Override
        public boolean onSurfaceTextureDestroyed(SurfaceTexture surfaceTexture) {
            return true;
        }

        @Override
        public void onSurfaceTextureUpdated(SurfaceTexture surfaceTexture) {
        }

    };

    protected abstract void startRecordingVideo(String path);

    protected abstract void takePicture(PictureCallback callback);

    protected abstract void stopRecordingVideo();

    protected abstract void openCamera(Activity activity, int width, int height);

    protected abstract void configureTransform(int width, int height);

    protected abstract boolean toggleFlash();

    protected abstract void switchCamera();

    protected abstract void startPreview();

    protected abstract void stopPreview();

    private long mLastClickTime;
    /**
     * 防快速點擊
     * */
    private boolean isFastDoubleClick(long limit){
        long time = System.currentTimeMillis();
        if(Math.abs(time-mLastClickTime)<limit){
            return true;
        }
        mLastClickTime = time;
        return false;
    }

    private PermissionUtils.ConfirmationDialog confirmationDialog;

    @RequiresApi(api = Build.VERSION_CODES.JELLY_BEAN_MR1)
    @Override
    public void onRequestPermissionsResult(int requestCode, @NonNull String[] permissions,
                                           @NonNull int[] grantResults) {
        Log.d(TAG, "onRequestPermissionsResult");

        // modify by wanjing on 2017/10/18 17:35 修改权限请求业务逻辑。
        if (requestCode == PermissionUtils.REQUEST_VIDEO_PERMISSIONS) {
            if (PermissionUtils.hasPermissionsGranted(getActivity(), permissions)) {
                if (confirmationDialog != null) {
                    confirmationDialog.dismiss();
                    confirmationDialog = null;
                }
            } else {
                confirmationDialog = new PermissionUtils.ConfirmationDialog();
                confirmationDialog.show(getChildFragmentManager(), PermissionUtils.FRAGMENT_DIALOG);
            }
        } else {
            super.onRequestPermissionsResult(requestCode, permissions, grantResults);
        }
        // end modify on 2017/10/18 17:35.
    }
}
