/*
 * Copyright 2014 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.xdja.camera2videolibrary;

import android.app.Activity;
import android.hardware.SensorManager;
import android.hardware.camera2.CameraAccessException;
import android.os.Build;
import android.os.Bundle;
import android.support.annotation.Nullable;
import android.support.annotation.RequiresApi;
import android.view.OrientationEventListener;
import android.view.Surface;
import android.view.View;

import com.xdja.camera2videolibrary.utils.CameraUtils;

@RequiresApi(api = Build.VERSION_CODES.LOLLIPOP)
public class Camera2Fragment extends BaseFragment {

    /**
     * camera2 管理类
     */
    private Camera2Manager manager;

    public static Camera2Fragment newInstance() {
        Camera2Fragment fragment = new Camera2Fragment();

        Bundle args = new Bundle();
        CameraParams cameraParams = new CameraParams();
        cameraParams.setMethod(CameraUtils.METHOD_CAMERA2);
        args.putSerializable("cameraParams", cameraParams); //拍摄参数
        fragment.setArguments(args);

        return fragment;
    }

    public static Camera2Fragment newInstance(CameraParams cameraParams) {
        Camera2Fragment fragment = new Camera2Fragment();

        Bundle args = new Bundle();
        cameraParams.setMethod(CameraUtils.METHOD_CAMERA2);
        args.putSerializable("cameraParams", cameraParams); //拍摄参数
        fragment.setArguments(args);

        return fragment;
    }

    @Override
    public void onViewCreated(final View view, Bundle savedInstanceState) {
        super.onViewCreated(view, savedInstanceState);

        manager = new Camera2Manager(getActivity(), mTextureView);
    }

    @Override
    public void onResume() {
        manager.startBackgroundThread();
        super.onResume();
    }

    @Override
    public void onPause() {
        manager.closeDriver();
        manager.stopBackgroundThread();
        super.onPause();
    }

    @Override
    protected void startRecordingVideo(String path) {
        manager.startRecordingVideo(path);
    }

    @Override
    protected void takePicture(PictureCallback pictureCallback) {
        manager.takePicture(pictureCallback, null);
    }

    @Override
    protected void stopRecordingVideo() {
        manager.stopRecordingVideo();
    }

    @Override
    protected void openCamera(Activity activity, int width, int height) {
        try {
            manager.openDriver(activity, width, height);
        } catch (CameraAccessException e) {
//            Toast.makeText(activity, "Cannot access the camera.", Toast.LENGTH_SHORT).show();
            activity.finish();
        } catch (InterruptedException e) {
            throw new RuntimeException("Interrupted while trying to lock camera opening.");
        }
    }

    @Override
    protected void configureTransform(int width, int height) {
        manager.configureTransform(width, height);
    }

    @Override
    protected boolean toggleFlash() {
        return manager.toggleFlash();
    }

    @Override
    protected void switchCamera() {
        try {
            manager.switchCamera();
        } catch (CameraAccessException | InterruptedException e) {
            e.printStackTrace();
        }
    }

    @Override
    protected void startPreview() {
        manager.startPreview();
    }

    @Override
    protected void stopPreview() {

    }

    private OrientationEventListener mOrientationEListener;

    @Override
    public void onCreate(@Nullable Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);

        mOrientationEListener = new OrientationEventListener(getActivity(),
                SensorManager.SENSOR_DELAY_NORMAL) {
            @Override
            public void onOrientationChanged(int orientation) {
                if (orientation == OrientationEventListener.ORIENTATION_UNKNOWN) {
                    // 手机平放时，检测不到有效的角度
                    return;
                }
                if (manager != null) {
                    // 只检测是否有四个角度的改变
                    if (orientation > 350 || orientation < 10) {
                        manager.setRotation(Surface.ROTATION_0);
                    } else if (orientation > 80 && orientation < 100) {
                        manager.setRotation(Surface.ROTATION_90);
                    } else if (orientation > 170 && orientation < 190) {
                        manager.setRotation(Surface.ROTATION_180);
                    } else if (orientation > 260 && orientation < 280) {
                        manager.setRotation(Surface.ROTATION_270);
                    }
                }
            }
        };
        if (mOrientationEListener.canDetectOrientation()) {
            mOrientationEListener.enable();
        } else {
            mOrientationEListener.disable();
        }
    }

    @Override
    public void onDestroy() {
        mOrientationEListener.disable();
        super.onDestroy();
    }
}
