package com.xdja.camera2videolibrary;

import android.animation.Animator;
import android.animation.AnimatorListenerAdapter;
import android.animation.ValueAnimator;
import android.app.Activity;
import android.content.res.Configuration;
import android.graphics.Bitmap;
import android.graphics.ImageFormat;
import android.graphics.Matrix;
import android.graphics.Rect;
import android.graphics.RectF;
import android.hardware.Camera;
import android.hardware.camera2.CameraDevice;
import android.media.MediaRecorder;
import android.os.Handler;
import android.text.TextUtils;
import android.util.Log;
import android.view.MotionEvent;
import android.view.Surface;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ImageView;

import com.xdja.camera2videolibrary.utils.CameraUtils;
import com.xdja.camera2videolibrary.widget.AutoFitTextureView;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

/**
 * Created by wanjing on 2017/9/21.
 *
 * desc: camera 管理类
 */
@SuppressWarnings("deprecation")
public class CameraManager {

    private static String TAG = CameraManager.class.getSimpleName();

    private static int PICTURE_MAX_VALUE = 1080 * 960;

    private static int PICTURE_MIN_VALUE = 800 * 480;

    private static int VIDEO_MAX_VALUE = 1080 * 960;

    private static int VIDEO_MIN_VALUE = 800 * 480;

    /** 视频码率 1M */
    private static final int VIDEO_BITRATE_NORMAL = 1024;
    /** 视频码率 1.5M */
    private static final int VIDEO_BITRATE_MEDIUM = 1536;
    /** 视频码率 2M*/
    private static final int VIDEO_BITRATE_HIGH = 2048;

    /** 最大帧率 */
    private static final int MAX_FRAME_RATE = 30;
    /** 最小帧率 */
    private static final int MIN_FRAME_RATE = 16;

    /**
     * A reference to the opened {@link Camera}.
     */
    private Camera mCamera;

    /**
     * the hardware camera to access.
     */
    private int mCameraId = 0;

    /**
     * The {@link Camera.Size} of camera preview.
     */
    private Camera.Size mPreviewSize;

    /**
     * The {@link Camera.Size} of video recording.
     */
    private Camera.Size mVideoSize;

    /**
     * The {@link Camera.Size} of picture.
     */
    private Camera.Size mPictureSize;

    /**
     * MediaRecorder
     */
    private MediaRecorder mMediaRecorder;

    /**
     * Whether the app is recording video now
     */
    private boolean mIsRecordingVideo;

    private String focusMode = Camera.Parameters.FOCUS_MODE_AUTO;

    private boolean previewing;
    private boolean initialized;

    private Activity mActivity;

    private AutoFitTextureView mTextureView;

    private SensorController mSensorController;

    private ValueAnimator va;
    private ImageView imageView;

    public static void init(CameraParams cameraParams) {
        PICTURE_MAX_VALUE = cameraParams.getPictureMaxValue();
        PICTURE_MIN_VALUE = cameraParams.getPictureMinValue();
        VIDEO_MAX_VALUE = cameraParams.getVideoMaxValue();
        VIDEO_MIN_VALUE = cameraParams.getVideoMinValue();
    }

    public CameraManager(Activity activity, AutoFitTextureView textureView) {
        this.mActivity = activity;
        this.mTextureView = textureView;
        this.mSensorController = SensorController.getInstance(activity.getBaseContext());

        this.mTextureView.setOnTouchListener(new View.OnTouchListener() {
            @Override
            public boolean onTouch(View v, MotionEvent event) {
                focusOnTouch(event);
                addFocusToWindow(event);
                return true;
            }
        });
        this.mSensorController.setCameraFocusListener(new SensorController.CameraFocusListener() {
            @Override
            public void onFocus() {
                if (mSensorController.isFocusLocked()) {
                    return;
                }

                try {
                    doAutoFocus();
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        });
    }

    /**
     * Tries to open a {@link CameraDevice}. The result is listened by `mStateCallback`.
     */
    public void openDriver(Activity activity, int width, int height)
            throws IOException, InterruptedException {
        Log.i(TAG, "width: " + width + ", height: " + height);

        Camera theCamera = mCamera;
        if (theCamera == null) {
            theCamera = Camera.open(mCameraId);
            if (theCamera == null) {
                throw new IOException();
            }
            mCamera = theCamera;
        }

        theCamera.setPreviewTexture(mTextureView.getSurfaceTexture());

        Camera.Parameters parameters = theCamera.getParameters();
        if (!initialized) {
            initialized = true;
            float scale = (float) height / width;
            mPreviewSize = chooseOptimalSize(parameters, width, height, scale);
            Log.i(TAG, "preview with: " + mPreviewSize.width + ", preview height: " + mPreviewSize.height);
            mPictureSize = choosePictureSize(parameters, scale);
            Log.i(TAG, "picture with: " + mPictureSize.width + ", picture height: " + mPictureSize.height);
            mVideoSize = chooseVideoSize(parameters, scale);
            mVideoSize = mVideoSize == null ? mPictureSize : mVideoSize;
            Log.i(TAG, "video with: " + mVideoSize.width + ", video height: " + mVideoSize.height);
//            focusMode = findSettableValue(parameters.getSupportedFocusModes(),
//                    Camera.Parameters.FOCUS_MODE_AUTO, Camera.Parameters.FOCUS_MODE_MACRO);
        }
        parameters.setPreviewSize(mPreviewSize.width, mPreviewSize.height);
        parameters.setPictureSize(mPictureSize.width, mPictureSize.height);
        parameters.set("jpeg-quality", 80);
        parameters.setPictureFormat(ImageFormat.JPEG);
        parameters.setFocusMode(focusMode);
        theCamera.setDisplayOrientation(90);
        theCamera.setParameters(parameters);

        int orientation = activity.getResources().getConfiguration().orientation;
        if (orientation == Configuration.ORIENTATION_LANDSCAPE) {
            mTextureView.setAspectRatio(mPreviewSize.width, mPreviewSize.height);
        } else {
            mTextureView.setAspectRatio(mPreviewSize.height, mPreviewSize.width);
        }
        configureTransform(width, height);
        startPreview();

        mSensorController.onStart();
    }

    public void closeDriver() {
        mSensorController.onStop();

        if (mCamera != null) {
            mCamera.release();
            mCamera = null;
        }
    }

    /**
     * Start the camera preview.
     */
    public void startPreview() {
        Camera theCamera = mCamera;
        if (theCamera != null && !previewing) {
            theCamera.startPreview();
            previewing = true;
        }
    }

    /**
     * Stop the camera preview.
     */
    public void stopPreview() {
        if (mCamera != null && previewing) {
            mCamera.stopPreview();
            previewing = false;
        }
    }

    /**
     * Configures the necessary {@link Matrix} transformation to `mTextureView`.
     * This method should not to be called until the camera preview size is determined in
     * openCamera, or until the size of `mTextureView` is fixed.
     *
     * @param viewWidth  The width of `mTextureView`
     * @param viewHeight The height of `mTextureView`
     */
    public void configureTransform(int viewWidth, int viewHeight) {
        if (null == mTextureView || null == mPreviewSize || null == mActivity) {
            return;
        }
        int rotation = mActivity.getWindowManager().getDefaultDisplay().getRotation();
        Matrix matrix = new Matrix();
        RectF viewRect = new RectF(0, 0, viewWidth, viewHeight);
        RectF bufferRect = new RectF(0, 0, mPreviewSize.height, mPreviewSize.width);
        float centerX = viewRect.centerX();
        float centerY = viewRect.centerY();
        if (Surface.ROTATION_90 == rotation || Surface.ROTATION_270 == rotation) {
            bufferRect.offset(centerX - bufferRect.centerX(), centerY - bufferRect.centerY());
            matrix.setRectToRect(viewRect, bufferRect, Matrix.ScaleToFit.FILL);
            float scale = Math.max(
                    (float) viewHeight / mPreviewSize.height,
                    (float) viewWidth / mPreviewSize.width);
            matrix.postScale(scale, scale, centerX, centerY);
            matrix.postRotate(90 * (rotation - 2), centerX, centerY);
        }
        mTextureView.setTransform(matrix);
    }

    public void takePicture(final PictureCallback pictureCallback) {
        if (null == mCamera || !mTextureView.isAvailable() || null == mPreviewSize) {
            return;
        }
        mCamera.takePicture(null, null, new Camera.PictureCallback() {
            @Override
            public void onPictureTaken(byte[] data, Camera camera) {
                if (mCameraId == Camera.CameraInfo.CAMERA_FACING_FRONT) {
                    Bitmap bitmap = CameraUtils.bytes2Bitmap(data, 270);
                    pictureCallback.onPictureTaken(bitmap);
                } else {
                    Bitmap bitmap = CameraUtils.bytes2Bitmap(data, 90);
                    pictureCallback.onPictureTaken(bitmap);
                }
            }
        });
    }

    public void startRecordingVideo(String videoCacheAbsolutePath) {
        if (null == mCamera || !mTextureView.isAvailable() || null == mVideoSize
                || mIsRecordingVideo) {
            return;
        }

        try {
            if (mMediaRecorder == null) {
                mMediaRecorder = new MediaRecorder();
                mMediaRecorder.setOnErrorListener(null);
            } else {
                mMediaRecorder.reset();
            }
            mCamera.unlock();
            mMediaRecorder.setCamera(mCamera);

            mMediaRecorder.setAudioSource(MediaRecorder.AudioSource.MIC);
            mMediaRecorder.setVideoSource(MediaRecorder.VideoSource.CAMERA);
            mMediaRecorder.setOutputFormat(MediaRecorder.OutputFormat.MPEG_4);
            mMediaRecorder.setOutputFile(videoCacheAbsolutePath);
            mMediaRecorder.setVideoEncodingBitRate(VIDEO_BITRATE_MEDIUM * VIDEO_BITRATE_NORMAL);
            mMediaRecorder.setVideoFrameRate(MAX_FRAME_RATE);
            mMediaRecorder.setVideoSize(mVideoSize.width, mVideoSize.height);
            mMediaRecorder.setVideoEncoder(MediaRecorder.VideoEncoder.H264);
            mMediaRecorder.setAudioEncoder(MediaRecorder.AudioEncoder.AAC);
            if (mCameraId == Camera.CameraInfo.CAMERA_FACING_FRONT) {
                mMediaRecorder.setOrientationHint(270);
            } else {
                mMediaRecorder.setOrientationHint(90);
            }
            mMediaRecorder.setPreviewDisplay(new Surface(mTextureView.getSurfaceTexture()));
            mMediaRecorder.prepare();
            mMediaRecorder.start();
            mIsRecordingVideo = true;
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    public void stopRecordingVideo() {
        if (!mIsRecordingVideo) {
            return;
        }

        // sleep 30ms to avoid IllegalStateException: swapBuffers: EGL error: 0x300d
        try {
            Thread.sleep(30);
        } catch (InterruptedException e) {
            e.printStackTrace();
        }

        try {
            // Stop recording
            mMediaRecorder.stop();
        } catch (Exception e) {
            e.printStackTrace();
        }
        // Take camera access back from MediaRecorder
        mCamera.lock();

        // UI
        mIsRecordingVideo = false;
    }

    private static String findSettableValue(Collection<String> supportedValues, String... desiredValues) {
//		Log.i(TAG, "Supported values: " + supportedValues);
        String result = null;
        if (supportedValues != null) {
            for (String desiredValue : desiredValues) {
                if (supportedValues.contains(desiredValue)) {
                    result = desiredValue;
                    break;
                }
            }
        }
//		Log.i(TAG, "Settable value: " + result);
        return result == null ? supportedValues.toArray(new String[supportedValues.size()])[0] : result;
    }

    private static Camera.Size choosePictureSize(Camera.Parameters parameters, float scale) {
        // Collect the supported resolutions that are at least as big as the preview Surface
        List<Camera.Size> bigEnough = new ArrayList<>();
        List<Camera.Size> supportedPictureSizes = parameters.getSupportedPictureSizes();
        for (Camera.Size supportedPictureSize : supportedPictureSizes) {
            float pictureScale = (float) supportedPictureSize.width / supportedPictureSize.height;
            int pictureValue = supportedPictureSize.width * supportedPictureSize.height;
            if (Math.abs(pictureScale - scale) < 0.2
                    && pictureValue <= PICTURE_MAX_VALUE && pictureValue >= PICTURE_MIN_VALUE) {
                bigEnough.add(supportedPictureSize);
            }
        }
        if (bigEnough.size() > 0) {
            return Collections.max(bigEnough, new CompareSizesByArea());
        } else {
            Log.e(TAG, "Couldn't find any suitable picture size");
            return Collections.min(supportedPictureSizes, new CompareSizesByArea());
        }
    }

    private static Camera.Size chooseVideoSize(Camera.Parameters parameters, float scale) {
        // Collect the supported resolutions that are at least as big as the preview Surface
        List<Camera.Size> bigEnough = new ArrayList<>();
        List<Camera.Size> supportedVideoSizes = parameters.getSupportedVideoSizes();
        if (supportedVideoSizes == null) {
            return null;
        }
        for (Camera.Size supportedVideoSize : supportedVideoSizes) {
            float videoScale = (float) supportedVideoSize.width / supportedVideoSize.height;
            int videoValue = supportedVideoSize.width * supportedVideoSize.height;
            if (Math.abs(videoScale - scale) < 0.2
                    && videoValue <= VIDEO_MAX_VALUE && videoValue >= VIDEO_MIN_VALUE) {
                bigEnough.add(supportedVideoSize);
            }
        }
        if (bigEnough.size() > 0) {
            return Collections.max(bigEnough, new CompareSizesByArea());
        } else {
            Log.e(TAG, "Couldn't find any suitable video size");
            return Collections.min(supportedVideoSizes, new CompareSizesByArea());
        }
    }

    private static Camera.Size chooseOptimalSize(Camera.Parameters parameters, int width, int height,
                                                 float scale) {
        // Collect the supported resolutions that are at least as big as the preview Surface
        List<Camera.Size> bigEnough = new ArrayList<>();
        List<Camera.Size> supportedPreviewSizes = parameters.getSupportedPreviewSizes();
        for (Camera.Size supportedPreviewSize : supportedPreviewSizes) {
            if ((float) supportedPreviewSize.width / supportedPreviewSize.height == scale
                    && supportedPreviewSize.height >= width
                    && supportedPreviewSize.width >= height) {
                bigEnough.add(supportedPreviewSize);
            }
        }

        // Pick the smallest of those, assuming we found any
        if (bigEnough.size() > 0) {
            return Collections.min(bigEnough, new CompareSizesByArea());
        } else {
            Log.e(TAG, "Couldn't find any suitable preview size");
            return Collections.max(supportedPreviewSizes, new CompareSizesByArea());
        }
    }

    /**
     * Compares two {@code Size}s based on their areas.
     */
    private static class CompareSizesByArea implements Comparator<Camera.Size> {

        @Override
        public int compare(Camera.Size lhs, Camera.Size rhs) {
            // We cast here to ensure the multiplications won't overflow
            return Long.signum((long) lhs.width * lhs.height
                    - (long) rhs.width * rhs.height);
        }
    }

    /**
     * toggle flash
     *
     * @return true, flash is opened; false, flash is closed.
     */
    public boolean toggleFlash() {
        boolean flashOn = false;
        if (mCamera != null && mCameraId != Camera.CameraInfo.CAMERA_FACING_FRONT) {
            Camera.Parameters parameter = mCamera.getParameters();
            if (TextUtils.equals(parameter.getFlashMode(), Camera.Parameters.FLASH_MODE_TORCH)) {
                flashOn = false;
                parameter.setFlashMode(Camera.Parameters.FLASH_MODE_OFF);
            } else {
                flashOn = true;
                parameter.setFlashMode(Camera.Parameters.FLASH_MODE_TORCH);
            }

            mCamera.setParameters(parameter);
        }
        return flashOn;
    }

    public void switchCamera() throws IOException, InterruptedException {
        // 切换摄像头
        if (mCameraId == Camera.CameraInfo.CAMERA_FACING_FRONT) {
            stopPreview();
            closeDriver();
            mCameraId = Camera.CameraInfo.CAMERA_FACING_BACK;
            openDriver(mActivity, mTextureView.getWidth(), mTextureView.getHeight());
            startPreview();
        } else if (mCameraId == Camera.CameraInfo.CAMERA_FACING_BACK) {
            stopPreview();
            closeDriver();
            mCameraId = Camera.CameraInfo.CAMERA_FACING_FRONT;
            openDriver(mActivity, mTextureView.getWidth(), mTextureView.getHeight());
            startPreview();
        }
    }

    /**
     * 触摸对焦点击效果
     */
    private void addFocusToWindow(MotionEvent event){

        if(va == null) {
            imageView = new ImageView(mActivity.getBaseContext());
            imageView.setImageResource(R.mipmap.video_focus);
            imageView.setLayoutParams(new ViewGroup.LayoutParams(ViewGroup.LayoutParams.WRAP_CONTENT, ViewGroup.LayoutParams.WRAP_CONTENT));
            imageView.measure(0, 0);
            imageView.setX(event.getX() - imageView.getMeasuredWidth() / 2);
            imageView.setY(event.getY() - imageView.getMeasuredHeight() / 2);
            final ViewGroup parent = (ViewGroup) mTextureView.getParent();
            parent.addView(imageView);

            va = ValueAnimator.ofFloat(0, 1).setDuration(500);
            va.addUpdateListener(new ValueAnimator.AnimatorUpdateListener() {
                @Override
                public void onAnimationUpdate(ValueAnimator animation) {
                    if(imageView != null) {
                        float value = (float) animation.getAnimatedValue();
                        if (value <= 0.5f) {
                            imageView.setScaleX(1 + value);
                            imageView.setScaleY(1 + value);
                        } else {
                            imageView.setScaleX(2 - value);
                            imageView.setScaleY(2 - value);
                        }
                    }
                }
            });
            va.addListener(new AnimatorListenerAdapter() {
                @Override
                public void onAnimationEnd(Animator animation) {
                    if(imageView != null) {
                        new Handler().postDelayed(new Runnable() {
                            @Override
                            public void run() {
                                parent.removeView(imageView);
                                va = null;
                            }
                        }, 1000);
                    }
                }
            });
            va.start();
        }
    }

    /**
     * On each tap event we will calculate focus area and metering area.
     * <p/>
     * Metering area is slightly larger as it should contain more info for exposure calculation.
     * As it is very easy to over/under expose
     */
    private void focusOnTouch(MotionEvent event) {
        if (mCamera != null) {
            mSensorController.lockFocus();

            //cancel previous actions
            mCamera.cancelAutoFocus();
            Rect focusRect = calculateTapArea(event.getRawX(), event.getRawY(), 1f);
            Rect meteringRect = calculateTapArea(event.getRawX(), event.getRawY(), 1.5f);

            Camera.Parameters parameters = null;
            try {
                parameters = mCamera.getParameters();
            } catch (Exception e) {
                e.printStackTrace();
            }

            // check if parameters are set (handle RuntimeException: getParameters failed (empty parameters))
            if (parameters != null) {
                parameters.setFocusMode(Camera.Parameters.FOCUS_MODE_AUTO);

                if (parameters.getMaxNumFocusAreas() > 0) {
                    List<Camera.Area> focus = new ArrayList<>();
                    focus.add(new Camera.Area(focusRect, 1000));
                    parameters.setFocusAreas(focus);
                }

                if (parameters.getMaxNumMeteringAreas() > 0) {
                    List<Camera.Area> metering = new ArrayList<>();
                    metering.add(new Camera.Area(meteringRect, 1000));
                    parameters.setMeteringAreas(metering);
                }

                try {
                    mCamera.setParameters(parameters);
                    mCamera.autoFocus(new Camera.AutoFocusCallback() {
                        @Override
                        public void onAutoFocus(boolean success, Camera camera) {
                            mSensorController.unlockFocus();
                        }
                    });
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        }
    }

    /**
     * Convert touch position x:y to {@link Camera.Area} position -1000:-1000 to 1000:1000.
     */
    private Rect calculateTapArea(float x, float y, float coefficient) {
        int focusAreaSize = 300;
        int areaSize = Float.valueOf(focusAreaSize * coefficient).intValue();

        int centerX = (int)(x / mTextureView.getWidth() * 2000 - 1000);
        int centerY = (int)(y / mTextureView.getHeight() * 2000 - 1000);

        int left = clamp(centerX - areaSize / 2, -1000, 1000);
        int right = clamp(centerX + areaSize, -1000, 1000);
        int top = clamp(centerY - areaSize / 2, -1000, 1000);
        int bottom = clamp(centerY + areaSize, -1000, 1000);

        return new Rect(left, top, right, bottom);
    }

    private int clamp(int x, int min, int max) {
        if (x > max) {
            return max;
        }
        if (x < min) {
            return min;
        }
        return x;
    }

    private void doAutoFocus() throws Exception {
        if (mCamera == null) {
            return;
        }

        mSensorController.lockFocus();

        mCamera.cancelAutoFocus(); //只有加上了这一句，才会自动对焦

        Camera.Parameters parameters = mCamera.getParameters();
        parameters.setFocusMode(focusMode);
        mCamera.setParameters(parameters);
        mCamera.autoFocus(new Camera.AutoFocusCallback() {
            @Override
            public void onAutoFocus(boolean success, Camera camera) {
//                if (success) {
//                    mCamera.cancelAutoFocus(); //只有加上了这一句，才会自动对焦
//
//                    Camera.Parameters parameters = mCamera.getParameters();
//                    parameters.setFocusMode(focusMode);
//                    mCamera.setParameters(parameters);
//                }
                mSensorController.unlockFocus();
            }
        });
    }
}
