package com.xdja.cssp.friend.api.action;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import com.xdja.cssp.account.exception.ApiException;
import com.xdja.cssp.account.exception.BadRequestException;
import com.xdja.cssp.account.exception.ForbiddenException;
import com.xdja.cssp.account.exception.InternalServerException;
import com.xdja.cssp.account.exception.ResourceConflictException;
import com.xdja.cssp.account.exception.ResourceNotFoundException;
import com.xdja.cssp.account.filter.annotation.RequestFilter;
import com.xdja.cssp.account.filter.pojo.Request;
import com.xdja.cssp.account.service.IAccountQueryService;
import com.xdja.cssp.friend.bean.FriendBean;
import com.xdja.cssp.friend.bean.FriendReqBean;
import com.xdja.cssp.friend.bean.FriendReqQueryResult;
import com.xdja.cssp.friend.im.FriendNoticeMsg;
import com.xdja.cssp.friend.im.ImMsgThread;
import com.xdja.cssp.friend.pn.PushSender;
import com.xdja.cssp.friend.server.business.IFriendBusiness;
import com.xdja.cssp.friend.util.PinyinUtil;
import com.xdja.cssp.id.IdGenUtil;
import com.xdja.cssp.id.IdType;
import com.xdja.friend.service.model.Friend;
import com.xdja.friend.service.model.FriendReq;
import com.xdja.log.analysis.aop.annoation.AopLog;
import com.xdja.log.analysis.format.sdk.service.record.FriendRecord;
import com.xdja.platform.common.lite.kit.prop.Prop;
import com.xdja.platform.common.lite.kit.prop.PropKit;
import com.xdja.platform.rpc.consumer.refer.DefaultServiceRefer;

/**
 * 好友相关服务接口
 * @author zhouyj
 *
 */
@RestController
public class FriendAction {

	public final static int REMARK_MAX_LEN = 50;

	public final static int VERIFACTION_MAX_LEN = 50;
	public static final String NO_FRIEND_REQ = "no_friend_req";
	//没有找到好友
	private static final String NO_FRIENDS_FOUND = "no_friends_found";
	//403 不能添加自己为好友
	private static final String CANNOT_ADD_SELF = "cannot_add_self";
	//409已经是您的好友
	private static final String ALREADY_FRIEND = "already_friend";
	//好友数量达到上限
	private static final String FRIENDS_LIMIT = "friends_limit";
	//好友的好友数量达到上限
	private static final String FRIEND_FRIENDS_LIMIT = "friend_friends_limit";
	
	private static final String FRIEND_GET_ID_FAILED = "friend_get_id_failed";
	
	//403不是好友
	private static final String NOT_FRIEND = "not_friend";
	
	//404 没有找到好友请求
	private static final String NO_FRIEND_REQS_FOUND = "no_friend_reqs_found";
	
	//好友账号不存在
	private static final String FRIEND_ACCOUNT_NOT_EXISTS = "friend_account_not_exists";
	
	private Logger logger = LoggerFactory.getLogger(FriendAction.class);
	
	@Autowired
	private IFriendBusiness friendService;
	
	//账户服务
	private IAccountQueryService accountService = DefaultServiceRefer.getServiceRefer(IAccountQueryService.class);
	
	private ExecutorService executor = Executors.newCachedThreadPool();
	
	
	

	/**
	 * 查询好友请求
	 * @param lastQuerySerial 最后更新序列
	 * @return
	 */
	@AopLog(hideArgs = {1})
	@RequestFilter(ticketAllowEmpty=false)
	@RequestMapping(method = RequestMethod.GET, value= "/v1/friends")
	public List<FriendBean> queryFriends(Request request, @RequestParam(required=false) String lastQuerySerial) {
		List<FriendBean> queryFriendResults = new ArrayList<FriendBean>();
		String hostId = request.getHostId();
		String requestId = request.getRequestId();
		String account = request.getLoginfo().getAccount();
		logger.info("=================Query friends,account:{}, lastQuerySerial:{}.", account, lastQuerySerial);
		long lastQuerySerialValue;
		try {
			lastQuerySerialValue = Long.parseLong(lastQuerySerial);
		} catch (NumberFormatException e2) {
			logger.error("Query friends failure, for bad request.");
			throw new BadRequestException(hostId, requestId, ApiException.REQUEST_PARAMS_ERROR, "请求参数非法", e2);
		}
		
		List<Friend> friends;
		try {
			friends = friendService.queryFriends(account, lastQuerySerialValue);
		} catch (Exception e1) {
			logger.error("Query friends failure, friend Server exception. account:{}, lastQuerySerial{}", account, lastQuerySerial);
			throw new InternalServerException(hostId, requestId, ApiException.INTERNAL_SERVER_ERROR, "服务器内部异常", e1);
		}
		if(friends == null || friends.size() == 0) {
			logger.info("=================Query friends success, no friends found.");
			throw new ResourceNotFoundException(hostId, requestId, NO_FRIENDS_FOUND, "没有查询到好友");
		}
		
		for(Friend friend : friends) {
			FriendBean friendBean = new FriendBean();
			friendBean.setAccount(friend.getFriendAccount());
			friendBean.setState(friend.getState());
			friendBean.setUpdateSerial(friend.getUpdateSerial());
			friendBean.setRemark(friend.getRemark());
			friendBean.setRemarkPy(friend.getRemarkPy());
			friendBean.setRemarkPinyin(friend.getRemarkPinyin());
			friendBean.setInitiative(friend.getInitiative());
			queryFriendResults.add(friendBean);
		}
		
		if(queryFriendResults.size() == 0) {
			logger.info("=================Query friends success, no friends found.");
			throw new ResourceNotFoundException(hostId, requestId, NO_FRIENDS_FOUND, "没有查询到好友");
		}
		logger.info("=================Query friends success, return:{}.", queryFriendResults);
		return queryFriendResults;
	}
	
	
	
	/**
	 * 好友请求
	 * @param friendAccount 好友账号
	 * @param data
	 * @param request
	 * @param account
	 */
	@RequestFilter(ticketAllowEmpty=false)
	@AopLog(hideArgs = {1})
	@RequestMapping(method = RequestMethod.POST, value= "/v1/friendRequests/{friendAccount}", consumes="application/json")
	public void friendRequest(Request request, @PathVariable String friendAccount,  @RequestBody FriendReqBean reqBean) {
		String hostId = request.getHostId();
		String requestId = request.getRequestId();
		String account = request.getLoginfo().getAccount();
		String pnToken = request.getCardNo();
		long idGen = IdGenUtil.getId(IdType.FRIEND_REQ);
		logger.info("=================Add friend,account:{}, friendAccount:{}, friendReqBean:{}.", account, friendAccount, reqBean);
		if(reqBean == null) {
			throw new BadRequestException(hostId, requestId, ApiException.REQUEST_PARAMS_ERROR, "请求参数非法");
		}
		
		String verification = reqBean.getVerification();
		
		if(StringUtils.isBlank(friendAccount)) {
			logger.error("Bad request in friendRequest, for friendAccount is null, friendAccount: {},reqBean:{} ", friendAccount, reqBean);
			throw new BadRequestException(hostId, requestId, ApiException.REQUEST_PARAMS_ERROR, "请求参数非法");
		}
		
		/*if(!StringUtils.isNotBlank(verification) || verification.length() > VERIFACTION_MAX_LEN) {
			logger.error("Bad request in friendRequest, verification is null or empty, or param verification's length is to long, verification:{} ", verification);
			throw new BadRequestException(hostId, requestId, ApiException.REQUEST_PARAMS_ERROR, "请求参数非法");
		}
		*/
		if(account.equals(friendAccount)) {
			logger.error("Add friend error, for friendAccount is same with account. friendAccount:{}", friendAccount);
			throw new ForbiddenException(hostId, requestId, CANNOT_ADD_SELF, "不能添加自己为好友");
		}
		
		
		boolean friendExists = false;
		try {
			List<String> notExistList = accountService.checkAccountExist(friendAccount);
			friendExists = notExistList == null || notExistList.size() == 0;
		} catch (Exception e1) {
			logger.error("Add friend error, for check friendAccount is exists failure");
			logger.error("Check friendAccount is exists failure, for account server exception, friendAccount:{}", friendAccount);
			throw new InternalServerException(hostId, requestId, ApiException.INTERNAL_SERVER_ERROR, "服务器内部异常");
		}
		
		if(!friendExists) {				
			logger.error("Add friend error, friendAccount is not exists, friendAccount:{}.", friendAccount);
			throw new ResourceConflictException(hostId, requestId, FRIEND_ACCOUNT_NOT_EXISTS, "对方账号不存在");
		}
		
		//4、添加好友接口
		FriendReq friendReq = new FriendReq();
		friendReq.setReqAccount(account);
		friendReq.setRecAccount(friendAccount);
		friendReq.setVerification(verification);
		long ret;
		try {
			ret = friendService.doFriendRequest(friendReq);
		} catch (Exception e) {
			logger.error("Save friend request failure, friend server error. account:{}, friendAccount:{}.", account, friendAccount);
			throw new InternalServerException(hostId, requestId, ApiException.INTERNAL_SERVER_ERROR, "服务端内部异常", e);
		}
		
		//不能添加自己为好友
		if (ret == IFriendBusiness.CANNOT_ADD_SELF) {
			logger.error("Add friend error, for friendAccount is same with account. friendAccount:{}", friendAccount);
			throw new ForbiddenException(hostId, requestId, CANNOT_ADD_SELF, "不能添加自己为好友");
		}//对方已经是您的好友
		else if (ret == IFriendBusiness.ALREADY_FRIEND) {
			logger.error("Add friend error, already friend. friendAccount:{}", friendAccount);
			throw new ResourceConflictException(hostId, requestId, ALREADY_FRIEND, "已经是好友");
		} else if (ret == IFriendBusiness.FRIENDS_LIMIT) {
			logger.error("Add friend error, friends limit. ");
			throw new ForbiddenException(hostId, requestId, FRIENDS_LIMIT, "好友数量达到上限");
		}
		
		List<String> accounts = new ArrayList<String>();
		accounts.add(account);
		accounts.add(friendAccount);
		if(idGen<0)
		{
			logger.info("Id生成器生成错误的id，id:{}",idGen);
			throw new InternalServerException(hostId, requestId, ApiException.INTERNAL_SERVER_ERROR, "服务器内部异常");
		}
		PushSender pushSender = new PushSender(accounts, "friendReq", pnToken,idGen);
		executor.execute(pushSender);
	 	
		logger.info("===============Add friend success.");
		try {
			FriendRecord.Addfriend(account, friendAccount);
		} catch (Exception e) {
		}
		
	}
	
	
	/**
	 * 查询好友请求
	 * @param lastQuerySerial 最后更新序列
	 * @return
	 */
	@AopLog(hideArgs = {1})
	@RequestMapping(method = RequestMethod.GET, value= "/v1/friendRequests")
	@RequestFilter(ticketAllowEmpty=false)
	public List<FriendReqQueryResult> queryFriendReqs(Request request, @RequestParam(required=false) String lastQuerySerial) {
		List<FriendReqQueryResult> reqQueryResults = new ArrayList<FriendReqQueryResult>();
		String hostId = request.getHostId();
		String requestId = request.getRequestId();
		String account = request.getLoginfo().getAccount();
		logger.info("=================Query friend requests, account:{}, lastQuerySerial:{}.", account, lastQuerySerial);
		long lastQuerySerialValue;
		try {
			lastQuerySerialValue = Long.parseLong(lastQuerySerial);
		} catch (NumberFormatException e) {
			logger.error("Query friend requests friend error, for bad request, lastQuerySerial:{}.", lastQuerySerial);
			throw new BadRequestException(hostId, requestId, ApiException.REQUEST_PARAMS_ERROR, "请求参数非法");
		}
		
		List<FriendReq> friendReqs = null;
		try {
			friendReqs = friendService.queryFriendReq(account, lastQuerySerialValue);
		} catch (Exception e1) {
			logger.error("Query friend requests failure, friend Server exception. account:{}, lastQuerySerial{}", account, lastQuerySerial);
			throw new InternalServerException(hostId, requestId, ApiException.INTERNAL_SERVER_ERROR, "服务器内部异常", e1);
		}
		if(friendReqs == null || friendReqs.size() == 0) {
			logger.info("Query friend requests success, but no friend requests found. account:{}", account);
			throw new ResourceNotFoundException(hostId, requestId, NO_FRIEND_REQS_FOUND, "没有找到好友请求");
		}
		
		for(FriendReq req : friendReqs) {
			FriendReqQueryResult reqRet = new FriendReqQueryResult();
			reqRet.setReqAccount(req.getReqAccount());
			reqRet.setRecAccount(req.getRecAccount());
			reqRet.setVerification(req.getVerification());
			reqRet.setTime(req.getTime());
			reqRet.setUpdateSerial(req.getUpdateSerial());
			reqQueryResults.add(reqRet);
		}
		logger.info("=================Query friend requests success, return:{}.", reqQueryResults);
		return reqQueryResults;
	}
	
	/**
	 * 接受好友请求
	 * @param request_account 请求方的账号
	 * @param account 要接受的好友账号，也即好友的请求方账号
	 */
	@AopLog(hideArgs = {1})
	@RequestMapping(method = RequestMethod.PUT, value= "/v1/friends/{friendAccount}")
	@RequestFilter(ticketAllowEmpty=false)
	public void acceptFriend(Request request, @PathVariable String friendAccount) {
		String hostId = request.getHostId();
		String requestId = request.getRequestId();
		String account = request.getLoginfo().getAccount();
		String pnToken = request.getCardNo();
		long idGen = IdGenUtil.getId(IdType.FRIEND_REQ);
		logger.info("=================Accept friend, account:{}, friendAccount:{}.", account, friendAccount);
		//校验参数合法性
		if(StringUtils.isBlank(friendAccount)) {
			throw new BadRequestException(hostId, requestId, ApiException.REQUEST_PARAMS_ERROR, "请求参数非法");
		}

		
		boolean friendAccountExists = false;
		try {
			List<String> notExistList = accountService.checkAccountExist(friendAccount);
			friendAccountExists = (notExistList == null || notExistList.size() == 0);
		} catch (Exception e1) {
			logger.error("Accept friend error, for check friendAccount is exists failure");
			logger.error("Check friendAccount is exists failure, for account server exception, friendAccount:{}", friendAccount);
			throw new InternalServerException(hostId, requestId, ApiException.INTERNAL_SERVER_ERROR, "服务器内部异常");
		}
		
		if(!friendAccountExists) {
			logger.error("Accept friend error, friendAccount is not exists, friendAccount:{}.", friendAccount);
			throw new ResourceConflictException(hostId, requestId, FRIEND_ACCOUNT_NOT_EXISTS, "对方账号不存在");
		}
		
		long ret;
		try {
			ret = friendService.createFriend(account, friendAccount);
		} catch (Exception e1) {
			logger.error("Create friend failure, friend server exception, account:{}, friendAccount:{}", account, friendAccount);
			throw new InternalServerException(hostId, requestId, ApiException.INTERNAL_SERVER_ERROR, "服务器内部异常", e1);
		}
		if(ret == IFriendBusiness.ALREADY_FRIEND) {
			logger.error("AcceptFriend failed, aready friend, account:{}, friendAccount:{}", account, friendAccount);
			throw new ResourceConflictException(hostId, requestId, ALREADY_FRIEND, "双方已是好友");
		}else if (ret == IFriendBusiness.NO_FRIEND_REQ) {
			logger.error("AcceptFriend failed, no friend request in server, account:{}, friendAccount:{}", account, friendAccount);
			throw new ResourceConflictException(hostId, requestId, NO_FRIEND_REQ, "没有好友请求");
		}else if (ret == IFriendBusiness.FRIENDS_LIMIT) {
			logger.error("Add friend error, friends limit. ");
			throw new ForbiddenException(hostId, requestId, FRIENDS_LIMIT, "好友数量达到上限");
		} else if (ret == IFriendBusiness.FRIEND_FRIENDS_LIMIT) {
			logger.error("Add friend error, friend friends limit. ");
			throw new ForbiddenException(hostId, requestId, FRIEND_FRIENDS_LIMIT, "好友的好友数量达到上限");
		}else if (ret == IFriendBusiness.GET_ID_FAILED) {
			logger.error("createFriend friend failure, for get id error");
			throw new InternalServerException(hostId, requestId, ApiException.INTERNAL_SERVER_ERROR, "服务端内部错误");
		}
		
		
		List<String> friendAccounts = new ArrayList<String>();
		friendAccounts.add(friendAccount);

		//推送
		List<String> accounts = new ArrayList<String>();
		accounts.add(account);
		accounts.add(friendAccount);
		if(idGen<0)
		{
			throw new InternalServerException(hostId, requestId, ApiException.INTERNAL_SERVER_ERROR, "服务器内部异常");
		}
		PushSender pushSender = new PushSender(accounts, "acceptFriend", pnToken,idGen);
		executor.execute(pushSender);
		
		//发送im消息
		FriendNoticeMsg friendNoticeMsg = new FriendNoticeMsg();
		friendNoticeMsg.setAction(FriendNoticeMsg.ACTION_ACCEPT_FRIEND);
		friendNoticeMsg.setOperator(account);
		Map<String, Object> map = new HashMap<String, Object>();
		map.put("friendAccount", friendAccount);
		friendNoticeMsg.setValue(map);
		ImMsgThread imMsgThread = new ImMsgThread(friendNoticeMsg);
		executor.execute(imMsgThread);
		
		logger.info("=================Accept friend success, acocunt:{}, friendAccount:{}.", account, friendAccount);
		try {
			FriendRecord.AcceptFriend(account, friendAccount);
		} catch (Exception e) {
		}
		
	}
	
	
	
	/**
	 * 解除好友关系
	 * @param friendAccount 好友账号
	 */
	@RequestMapping(method = RequestMethod.DELETE, value= "/v1/friends/{friendAccount}")
	@RequestFilter(ticketAllowEmpty=false)
	@AopLog(hideArgs = {1})
	public void deleteFriend(Request request, @PathVariable String friendAccount) {
		String hostId = request.getHostId();
		String requestId = request.getRequestId();
		String account = request.getLoginfo().getAccount();
		String pnToken = request.getCardNo();
		long idGen = IdGenUtil.getId(IdType.FRIEND_REQ);
		logger.info("=================Delete friend, acocunt:{}, friendAccount:{}.", account, friendAccount);
		
		//校验参数合法性
		if(StringUtils.isEmpty(friendAccount)) {
			throw new BadRequestException(hostId, requestId, ApiException.REQUEST_PARAMS_ERROR, "请求参数非法");
		}
		long delRet = 0;
		try {
			delRet = friendService.deleteFriend(account, friendAccount);
		} catch (Exception e) {
			logger.error("Delete friend failure, friend server exception, account:{}, friendAccount:{}", account, friendAccount);
			throw new InternalServerException(hostId, requestId, ApiException.INTERNAL_SERVER_ERROR, "服务端异常", e);
		}
		if (delRet == IFriendBusiness.GET_ID_FAILED) {
			logger.error("Delete friend failure, for get id error");
			throw new InternalServerException(hostId, requestId, ApiException.INTERNAL_SERVER_ERROR, "服务端内部错误");
		}
		if(delRet != IFriendBusiness.NOT_FRIEND) {
			//推送
			List<String> accounts = new ArrayList<String>();
			accounts.add(account);
			accounts.add(friendAccount);
			if(idGen<0)
			{
				throw new InternalServerException(hostId, requestId, ApiException.INTERNAL_SERVER_ERROR, "服务器内部异常");
			}
			PushSender pushSender = new PushSender(accounts, "deleteFriend", pnToken,idGen);
			executor.execute(pushSender);
		}
		
		logger.info("Delete friend relation success, acocunt:{}, friendAccount:{}.", account, friendAccount);
		try {
			FriendRecord.DeleteFriend(account, friendAccount);
		} catch (Exception e) {
		}
		
	}
	
	/**
	 * 设置备注
	 * @param myAccount 操作人账号
	 * @param account 要设置备注的账号
	 * @param remark 备注
	 * @return 备注拼音信息
	 */
	@AopLog(hideArgs = {1})
	@RequestMapping(method = RequestMethod.PUT, value= "/v1/friends/{friendAccount}/remark", consumes="application/json")
	@RequestFilter(ticketAllowEmpty=false)
	public Object setRemark(Request request, @PathVariable String friendAccount, @RequestBody Map<String, Object> paramMap) {
		String hostId = request.getHostId();
		String requestId = request.getRequestId();
		String account = request.getLoginfo().getAccount();
		String pnToken = request.getCardNo();
		long idGen = IdGenUtil.getId(IdType.FRIEND_REQ);
		logger.info("=================Set friend remark, acocunt:{}, friendAccount:{}. request param:{}", account, friendAccount, paramMap);
		String remark = null;
		if(paramMap.get("remark") != null) {
			remark = (String)paramMap.get("remark");
		}
		//校验friendAccount是否合法
		if(StringUtils.isBlank(friendAccount)) {
			logger.error("friendAccount is null or empty, friendAccount: {}", friendAccount);
			throw new BadRequestException(hostId, requestId, ApiException.REQUEST_PARAMS_ERROR, "请求参数非法");
		}
		//校验参数合法性
		if(StringUtils.isEmpty(remark) || remark.length() > 50) {
			throw new BadRequestException(hostId, requestId, ApiException.REQUEST_PARAMS_ERROR, "请求参数非法");
		}
		//检查账号是否存在
		boolean friendExists = false;
		try {
			List<String> notExistList = accountService.checkAccountExist(friendAccount);
			friendExists = notExistList == null || notExistList.size() == 0;
		} catch (Exception e1) {
			logger.error("Check friendAccount is exists failure, for account server exception, friendAccount:{}", friendAccount);
			throw new InternalServerException(hostId, requestId, ApiException.INTERNAL_SERVER_ERROR, "服务器内部异常");
		}
		
		if(!friendExists) {				
			logger.error("Add friend error, friendAccount is not exists, friendAccount:{}.", friendAccount);
			throw new ResourceConflictException(hostId, requestId, FRIEND_ACCOUNT_NOT_EXISTS, "对方账号不存在");
		}
		
		long ret;
		try {
			ret = friendService.updateRemark(account, friendAccount, remark);
		} catch (Exception e) {
			logger.error("Set remark failure, Invoke friend server exception", e);
			throw new InternalServerException(hostId, requestId, ApiException.INTERNAL_SERVER_ERROR, "服务器内部异常", e);
		}
		if (ret == IFriendBusiness.GET_ID_FAILED) {
			logger.error("Set remark failure, for get id error");
			throw new InternalServerException(hostId, requestId, ApiException.INTERNAL_SERVER_ERROR, "服务器内部错误");
		}
		if(ret == IFriendBusiness.NOT_FRIEND) {
			logger.error("friendAccount:{} is not friend", friendAccount);
			throw new ForbiddenException(hostId, requestId, NOT_FRIEND, "对方不是您的好友");
		}
		
		String remarkPinyin = null ;
		String remarkPy = null;
		
		
		if(StringUtils.isNotBlank(remark)){
			remarkPinyin = PinyinUtil.getCharactersPinyin(remark);
			remarkPy = PinyinUtil.getCharactersPy(remark);
		}
		
		//推送
		List<String> accounts = new ArrayList<String>();
		accounts.add(account);
		accounts.add(friendAccount);
		if(idGen<0)
		{
			throw new InternalServerException(hostId, requestId, ApiException.INTERNAL_SERVER_ERROR, "服务器内部异常");
		}
		PushSender pushSender = new PushSender(accounts, "modifyRemark", pnToken,idGen);
		executor.execute(pushSender);
		
		Map<String, String> map = new HashMap<String, String>();
		map.put("remarkPy", remarkPy);
		map.put("remarkPinyin", remarkPinyin);
		
		logger.info("===============Set remark success, return:{}.", map);
		try {
			FriendRecord.SetFriendRemark(account, friendAccount, remark);
		} catch (Exception e) {
		}
		
		return map;
		
	}

	/**
	 * 获取系统服务标识
	 *
	 */
	public String getHostId() {
		Prop configProp = PropKit.use("system.properties");
		return configProp.get("hostId", "contact-server");
	}
	
	/**
	 * 获取请求ID
	 * @return
	 */
	public String getRequestId(){
		return UUID.randomUUID().toString();
	}

}
