package com.xdja.cssp.group.business;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Resource;

import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Component;

import redis.clients.jedis.Jedis;
import redis.clients.jedis.Pipeline;
import redis.clients.jedis.Response;

import com.xdja.cssp.group.entity.GroupMember;
import com.xdja.cssp.group.scheduling.CacheMemberBean;
import com.xdja.platform.redis.core.RedisClient;
import com.xdja.platform.redis.core.action.JedisAction;
import com.xdja.platform.redis.core.action.JedisActionNoResult;

@Component
public class GroupCacheOperator {
	private final static String GROUP_PREFIX = "group:g:";
	private final static String GROUP_MEMBER_PREFIX = "group:m:";

	private final static String GROUP_MEMBER_SEQ_PREFIX = "group:m:seq:";
	
	private static RedisClient rc;

	@Resource(name = "groupRedisClient")
	public void setRedisClient(RedisClient rc) {
		GroupCacheOperator.rc = rc;
	}
	
	
	/**
	 * 更新群组
	 * 
	 * @author: Administrator
	 * @date: 2015-7-20 下午8:15:59
	 * @param key
	 * @param memberList
	 * @throws Exception
	 */
	/*public static void updateGroupInfo(final Long groupId, final List<String> memberList) throws Exception {
		rc.execute(new JedisActionNoResult() {
			@Override
			public void action(Jedis jedis) {
				Pipeline pl = jedis.pipelined();
				for (String account : memberList) {
					pl.sadd(GROUP_MEMBER_PREFIX + account, String.valueOf(groupId));
					pl.sadd(GROUP_PREFIX + groupId, account);
				}
				pl.sync();
			}
		});
	}*/

	/**
	 * 删除群组成员或者群组
	 * 
	 * @author: mayanpei
	 * @date: 2015-7-20 下午8:15:42
	 * @param key
	 * @param memberList
	 * @throws Exception
	 */
	/*public static void deleteGroupInfo(final Long groupId, final List<String> memberList) throws Exception {

		rc.execute(new JedisActionNoResult() {
			@Override
			public void action(Jedis jedis) {
				Pipeline pl = jedis.pipelined();

				for (String account : memberList) {
					pl.srem(GROUP_MEMBER_PREFIX + account, String.valueOf(groupId));
				}
				pl.srem(GROUP_PREFIX + groupId, memberList.toArray(new String[memberList.size()]));
				pl.sync();
			}
		});
	}*/


	/**
	 * 更新群组成员序列
	 * 
	 * @author: mayanpei
	 * @date: 2015-9-6 下午5:07:43
	 * @param groupMemberSerial
	 */
	public static void updateGroupMemberSeq(final Long groupMemberSerial) {
		rc.set(GROUP_MEMBER_SEQ_PREFIX, groupMemberSerial + "");
	}

	/**
	 * 更新群组和群成员序列
	 * 
	 * @author: mayanpei
	 * @date: 2015-9-6 下午5:08:43
	 * @param groupSerail
	 * @param groupMemberSerial
	 */
	/*public static void updateSeq(final Long groupSerail, final Long groupMemberSerial) {
		rc.set(GROUP_SEQ_PREFIX, groupSerail + "");
		rc.set(GROUP_MEMBER_SEQ_PREFIX, groupMemberSerial + "");
	}*/


	/**
	 * 获取群成员序列
	 * 
	 * @author: mayanpei
	 * @date: 2015-9-6 下午5:09:22
	 * @return
	 */
	public static Long getGroupMemberSeq() {
		String str = rc.get(GROUP_MEMBER_SEQ_PREFIX);
		if (StringUtils.isBlank(str) || !StringUtils.isNumeric(str)) {
			return 0L;
		}
		return Long.parseLong(str);
	}
	
	/**
	 * 获取某个用户已在的群数量
	 * @param account 账号
	 * @return 群数量
	 */
	public static int getGroupNum(final String account) {
		Long ret = rc.execute(new JedisAction<Long>() {

			@Override
			public Long action(Jedis jedis) {
				return jedis.scard(GROUP_MEMBER_PREFIX + account);
			}
		}); 
		return ret == null?0:ret.intValue();
	}

	
	/**
	 * 获得一个群中的所有成员
	 * @param groupId 群id
	 * 
	 * 
	 */
	public static List<String> getGroupAllMembers(final String groupId){
		
		Set<String> groupMemberlist = rc.execute(new JedisAction<Set<String>>()
		{

			@Override
			public Set<String> action(Jedis jedis)
			{
				
				return jedis.smembers(GROUP_PREFIX+groupId);
				 
			}
		});
		return new ArrayList<>(groupMemberlist);
	}

	/**
	 * 获取某个群群成员数量
	 * @param groupId 群id
	 * @return 群成员数量
	 */
	public static int getGroupMemberNum(final long groupId) {
		Long ret = rc.execute(new JedisAction<Long>() {

			@Override
			public Long action(Jedis jedis) {
				return jedis.scard(GROUP_PREFIX + groupId);
			}
		}); 
		
		return ret == null?0:ret.intValue();
	}
	
	/**
	 * 添加群组
	 * @param group 群组信息
	 * @param groupMembers 群成员信息
	 */
	public static void addGroup(final long groupId, final List<GroupMember> groupMembers, 
			final long memberSerial) {
		rc.execute(new JedisActionNoResult() {
			@Override
			public void action(Jedis jedis) {
				Pipeline pl = jedis.pipelined();
				for (GroupMember groupMember : groupMembers) {
					String account = groupMember.getAccount();
					pl.sadd(GROUP_MEMBER_PREFIX + account, String.valueOf(groupId));
					pl.sadd(GROUP_PREFIX + groupId, account);
				}
//				pl.set(GROUP_SEQ_PREFIX, groupSerial + "");
				pl.set(GROUP_MEMBER_SEQ_PREFIX, memberSerial + "");
				pl.sync();
			}
		});
	}
	
	/**
	 * 删除群组缓存
	 * @param groupId 要删除的群组id
	 */
	public static void delGroup(final Long groupId) {
		rc.execute(new JedisActionNoResult() {
			@Override
			public void action(Jedis jedis) {
				Set<String> members = jedis.smembers(GROUP_PREFIX + groupId);
				Pipeline pl = jedis.pipelined();
				for(String account : members) {
					pl.srem(GROUP_MEMBER_PREFIX + account, String.valueOf(groupId));
				}
				pl.del(GROUP_PREFIX + groupId);
				pl.sync();
			}
		});
		
	}
	
	
	/**
	 * 账户退群操作
	 * @param groupId 要退的群id
	 * @param account 账号
	 * @param memberSerial 成员更新序列
	 */
	public static void quitGroup(final long  groupId, final String account, final long memberSerial) {

		rc.execute(new JedisActionNoResult() {
			@Override
			public void action(Jedis jedis) {
				Pipeline pl = jedis.pipelined();
				pl.srem(GROUP_MEMBER_PREFIX + account, String.valueOf(groupId));
				pl.srem(GROUP_PREFIX + groupId, account);
//				pl.set(GROUP_SEQ_PREFIX, groupSerial + "");
				pl.set(GROUP_MEMBER_SEQ_PREFIX, memberSerial + "");
				pl.sync();
			}
		});
	}
	
	/**
	 * 删除群组缓存
	 * @param groupId 要删除的群组id
	 */
	public static void delGroup(final Long groupId, final long memberSerial) {
		rc.execute(new JedisActionNoResult() {
			@Override
			public void action(Jedis jedis) {
				Set<String> members = jedis.smembers(GROUP_PREFIX + groupId);
				Pipeline pl = jedis.pipelined();
				for(String account : members) {
					pl.srem(GROUP_MEMBER_PREFIX + account, String.valueOf(groupId));
				}
				pl.del(GROUP_PREFIX + groupId);
//				pl.set(GROUP_SEQ_PREFIX, groupSerial + "");
				pl.set(GROUP_MEMBER_SEQ_PREFIX, memberSerial + "");
				pl.sync();
			}
		});
	}
	
	/**
	 * 移除群成员
	 * @param groupId 群id
	 * @param memberAccounts 成员账号列表
	 * @param memberSerial 群成员序列
	 */
	public static void removeGroupMembers(final long groupId, final List<String> memberAccounts,
			final long memberSerial) {
		rc.execute(new JedisActionNoResult() {
			@Override
			public void action(Jedis jedis) {
				Pipeline pl = jedis.pipelined();

				for (String account : memberAccounts) {
					pl.srem(GROUP_MEMBER_PREFIX + account, String.valueOf(groupId));
				}
				pl.srem(GROUP_PREFIX + groupId, memberAccounts.toArray(new String[memberAccounts.size()]));
//				pl.set(GROUP_SEQ_PREFIX, groupSerial + "");
				pl.set(GROUP_MEMBER_SEQ_PREFIX, memberSerial + "");
				pl.sync();
			}
		});
		
	}
	
	/**
	 * 添加群成员到缓存
	 * @param groupId 群id
	 * @param addMembers 要添加的群成员账号列表
	 * @param memberSerial 成员序列
	 */
	public static void addMembers(final long groupId, final List<String> addMembers, 
			final long memberSerial) {
		
		rc.execute(new JedisActionNoResult() {
			@Override
			public void action(Jedis jedis) {
				Pipeline pl = jedis.pipelined();
				for (String account : addMembers) {
					pl.sadd(GROUP_MEMBER_PREFIX + account, String.valueOf(groupId));
					pl.sadd(GROUP_PREFIX + groupId, account);
				}
//				pl.set(GROUP_SEQ_PREFIX, groupSerial + "");
				pl.set(GROUP_MEMBER_SEQ_PREFIX, memberSerial + "");
				pl.sync();
			}
		});
	}
	
	/**
	 * 缓存群组数据
	 * @param cacheMemberBeans
	 * @param memberSerial
	 */
	public static void cacheGroupData(final List<CacheMemberBean> cacheMemberBeans, final long memberSerial) {
		rc.execute(new JedisActionNoResult() {
			@Override
			public void action(Jedis jedis) {
				Pipeline pl = jedis.pipelined();
				for (CacheMemberBean cacheBean : cacheMemberBeans) {
					pl.sadd(GROUP_MEMBER_PREFIX + cacheBean.getAccount(), String.valueOf(cacheBean.getGroupId()));
					pl.sadd(GROUP_PREFIX + cacheBean.getGroupId(), cacheBean.getAccount());
				}
				pl.set(GROUP_MEMBER_SEQ_PREFIX, memberSerial + "");
				pl.sync();
			}
		});
	}
	
	/**
	 * 获取在同群组的所有成员账号集合
	 * @param account 待查询账号
	 * @return 与账号在同群组的所有成员账号集合
	 */
	public static Set<String> queryAllSameGroupMembers(final String account) {
		return rc.execute(new JedisAction<Set<String>>() {
			@Override
			public Set<String> action(Jedis jedis) {
				Set<String> accountSet = new HashSet<String>();
				Set<String> groups = jedis.smembers(GROUP_MEMBER_PREFIX + account);
				if(!groups.isEmpty()) {
					for(String group : groups) {
						accountSet.addAll(jedis.smembers(GROUP_PREFIX + group));
					}
				}
				return accountSet;
			}
		});
	}
}
