package com.xdja.cssp.group.business;

import java.util.List;
import java.util.Map;
import java.util.Set;

import com.xdja.cssp.group.entity.AddMembersResult;
import com.xdja.cssp.group.entity.CreateGroupRequest;
import com.xdja.cssp.group.entity.CreateGroupResult;
import com.xdja.cssp.group.entity.Group;
import com.xdja.cssp.group.entity.ModifyGroupNicknameResult;
import com.xdja.cssp.group.scheduling.CacheMemberBean;


/**
 * 群组业务接口
 * @author zhouyj
 *
 */
public interface IGroupBusiness {
	
	/**
	 * 创建群组
	 * @param group 群组信息
	 * @return 创建后包含群id的群组信息
	 */
	CreateGroupResult createGroup(CreateGroupRequest createGroupRequest);
	
	/**
	 * 修改群名称
	 * @param groupId 群组id
	 * @param newGroupName 新的群名称
	 */
	Map<String, String> modifyGroupName(long groupId, String newGroupName);
	
	/**
	 * 修改群头像
	 * @param group 包含群id、头像相关信息
	 * @param updateSerial 
	 */
	long modifyGroupAvatar(Group group);
	
	/**
	 * 退出群组
	 * @param groupId 群组id
	 * @param account 账号
	 */
	long doQuitGroup(Long groupId, String account);
	
	/**
	 * 解散群组
	 * @param groupId 群id
	 * @param account 操作者账号
	 */
	long doDismissGroup(long groupId);
	
	/**
	 * 增量更新群组信息
	 * @param account 账号
	 * @param updateSerial 更新序列
	 * @return 增量群组信息
	 */
	List<Group> queryGroups(String account, long updateSerial);
	
	/**
	 * 获取某个群组中的成员数量
	 * @param groupId 群id
	 * @return 群组中成员数量
	 */
	int getGroupMemberNum(long groupId);
	
	/**
	 * 获取某个账号的群组数
	 * @param account
	 * @return
	 */
	int getGroupNum(String account);
	
	
	/**
	 * 删除群组及群成员，永久删除，无法恢复
	 * @param account
	 */
	void deleteGroupForever(long groupId);

	/**
	 * 查询群组信息,群信息不区分是否已删除，群成员只查询未删除状态
	 * @param groupId 群id
	 * @return 群信息，包含群成员
	 */
	Group queryGroupDetail(long groupId);
	
	/**
	 * 只查询群组信息，不查询群成员
	 * @param groupId 群组id
	 * @return 群组信息，不包含群成员
	 */
	Group queryGroupInfoOnly(long groupId);
	
	/**
	 * 增量群成员信息
	 * @param paramMapList 参数信息，群组id及更新序列组成的map
	 * @return
	 */
	List<Map<String, Object>> queryMembers(
			List<Map<String, Long>> paramMapList) ;

	/**
	 * 移除群成员
	 * @param groupId 群组id
	 * @param removeAccounts 要移除的群成员账号列表
	 */
	long removeGroupMembers(long groupId, List<String> removeAccounts);

	/**
	 * 添加群成员
	 * @param groupId 群id
	 * @param inviter 邀请者
	 * @param addAccounts 成员账号列表
	 */
	AddMembersResult addGroupMembers(long groupId, int groupNum, String inviter, List<String> addAccounts);
	
	/**
	 * 查询待更新至缓存的数据
	 * @param serial
	 * @param batchSize
	 * @return map数据，key为groupId，value为群成员账号
	 */
	List<CacheMemberBean> queryDataForCache(long serial, int batchSize);

	/**
	 * 更改群昵称
	 * @param groupId 群id
	 * @param account 成员账号
	 * @param newNickname  新的群昵称
	 */
	ModifyGroupNicknameResult modifyGroupNickname(long groupId, String account, String newNickname);
	
	/**
	 * 查询所有与account在同群组的成员账号
	 * @param account 待查询账号
	 * @return 与account在同群组的成员账号集合
	 */
	Set<String> queryAllSameGroupMembers(String account);
	/**
	 * 查询一个群中的所有成员
	 * @param groupId
	 * @return 成员账号列表
	 */
	List<String> getGroupMembers(String groupId);
}
