package com.xdja.cssp.group.business.impl;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import com.xdja.cssp.contact.dao.DataSource;
import com.xdja.cssp.group.business.GroupCacheOperator;
import com.xdja.cssp.group.business.IGroupBusiness;
import com.xdja.cssp.group.common.Constants;
import com.xdja.cssp.group.dao.GroupDao;
import com.xdja.cssp.group.dao.GroupMemberDao;
import com.xdja.cssp.group.entity.AddMembersResult;
import com.xdja.cssp.group.entity.CreateGroupRequest;
import com.xdja.cssp.group.entity.CreateGroupResult;
import com.xdja.cssp.group.entity.Group;
import com.xdja.cssp.group.entity.GroupMember;
import com.xdja.cssp.group.entity.ModifyGroupNicknameResult;
import com.xdja.cssp.group.scheduling.CacheMemberBean;
import com.xdja.cssp.group.util.PinyinUtil;
import com.xdja.cssp.id.IdGenUtil;
import com.xdja.cssp.id.IdType;
import com.xdja.platform.common.lite.kit.prop.Prop;
import com.xdja.platform.common.lite.kit.prop.PropKit;

@Service
public class GroupBusinessImpl implements IGroupBusiness {
	private static long ERROR_FLAG=-1;
	//业务完成标志
	private static long GROUP_BUSSINESS_FINISHED=1;
	@Autowired
	private GroupDao groupDao;
	@Autowired
	private GroupMemberDao memberDao;

	@Override
	@DataSource("master")
	public CreateGroupResult createGroup(CreateGroupRequest createGroupRequest) {
		// 设置群组状态、群名称拼音、群创建时间
		Group group = new Group();
		//设置当Id生成器产生错误id的标志
		CreateGroupResult errorIdFlag=null;
		group.setStatus(Constants.ADD);
		String groupName = createGroupRequest.getGroupName();
		if (StringUtils.isNotBlank(groupName)) {
			group.setName(groupName);
			group.setNamePinyin(PinyinUtil.getCharactersPinyin(groupName));
			group.setNamePy(PinyinUtil.getCharactersPy(groupName));
		}
		group.setCreateTime(System.currentTimeMillis());
		
		String owner = createGroupRequest.getOwner();
		if (StringUtils.isBlank(owner)) {
			throw new IllegalArgumentException("请求参数非法，群主账号为空");
		}
		group.setOwner(owner);
		
		int groupCountLimit = getGroupCountLimit();
		int groupMemberCountLimit = getGroupMemberCountLimit();
		List<String> groupNumErrList = new ArrayList<String>();
		int ownerGroupNum = GroupCacheOperator.getGroupNum(owner);
		
		int retFlag;
		List<String> memberAccounts = createGroupRequest.getMemberAccounts();
		memberAccounts.add(owner);
		CreateGroupResult createGroupResult = new CreateGroupResult();
		if (ownerGroupNum >= groupCountLimit) {
			retFlag = CreateGroupResult.RET_OWNER_EXCESS;
		} else if(memberAccounts != null && memberAccounts.size() > groupMemberCountLimit) {
			retFlag = CreateGroupResult.RET_MEMBER_COUNT_EXCESS;
		} else {
			long groupSerail = getGroupSerial();
			long memberSerial = getGroupMemberSerial();
			if((groupSerail==ERROR_FLAG)&&(memberSerial==ERROR_FLAG))
			{
				return errorIdFlag;	
			}
				
			group.setUpdateSerial(groupSerail);
			// 保存群信息
			group = groupDao.saveGroup(group);
			List<GroupMember> groupMembers = new ArrayList<GroupMember>();
			
			for (String memberAccount: memberAccounts) {
				if (StringUtils.isBlank(memberAccount)) {
					continue;
				}
				GroupMember groupMember = new GroupMember();
				groupMember.setAccount(memberAccount);
				groupMember.setInviteAccount(owner);
				groupMember.setStatus(Constants.ADD);
				groupMember.setUpdateSerial(memberSerial);
				if (!memberAccount.equals(owner)) {
					int groupNum = GroupCacheOperator.getGroupNum(memberAccount);
					if (groupNum >= groupCountLimit) {
						groupNumErrList.add(memberAccount);
						continue;
					}
				}
				groupMembers.add(groupMember);
			}
			// 保存群成员信息
			memberDao.saveGroupMembers(group.getId(), groupMembers);
			group.setMembers(groupMembers);
			// 保存数据到redis
			GroupCacheOperator.addGroup(group.getId(), groupMembers, memberSerial);

			if (groupNumErrList.size() == 0) {
				retFlag = CreateGroupResult.RET_SUCCESS;
			} else {
				// 包括只有群主进群的情况
				retFlag = CreateGroupResult.RET_PARTIAL_SUCCESS;
			}
			createGroupResult.setGroup(group);
			Map<Integer, List<String>> errMap = new HashMap<Integer, List<String>>();
			errMap.put(CreateGroupResult.GROUP_MEMBER_EXCEED, groupNumErrList);
			createGroupResult.setErrMap(errMap);
		}

		createGroupResult.setRetFlag(retFlag);
		return createGroupResult;

	}

	/**
	 * 获取账号的群数量限制
	 * @return 一个账号所在群的最大数量
	 */
	private int getGroupCountLimit() {
		Prop props = PropKit.use("system.properties");
		return props.getInt("group.total", Constants.GROUP_COUNT_LIMIT_DEFAULT);
	}
	
	/**
	 * 获取账号的群数量限制
	 * @return 一个账号所在群的最大数量
	 */
	private int getGroupMemberCountLimit() {
		Prop props = PropKit.use("system.properties");
		return props.getInt("group.member.total", Constants.MEMBER_COUNT_LIMIT_DEFAULT);
	}

	@Override
	@DataSource("master")
	public AddMembersResult addGroupMembers(long groupId, int groupNum, String account, List<String> addAccounts) {
		//设置当Id生成器产生错误id的标志
		AddMembersResult errorIdFlag=null;
		AddMembersResult addMembersResult = new AddMembersResult();
		List<Map<String, Object>> statusMapList = memberDao.queryMemberStatus(groupId, addAccounts);
		Map<String, Integer> memberMap = new HashMap<String, Integer>();
		for (Map<String, Object> statusMap : statusMapList) {
			memberMap.put((String) statusMap.get("account"), (Integer) statusMap.get("status"));
		}
		Map<Integer, List<String>> errMap = new HashMap<Integer, List<String>>();
		List<String> groupCountErrList = new ArrayList<String>();
		List<String> existAccounts = new ArrayList<String>();
		//已经存在的并且删除的账号
		List<String> existDelAccounts = new ArrayList<String>();
		Iterator<String> it = addAccounts.iterator();
		while (it.hasNext()) {
			String addAccount = it.next();
			if(StringUtils.isNotBlank(addAccount)) {
				Integer status = memberMap.get(addAccount);
				if (status != null && (status.intValue() == Constants.ADD || status.intValue() == Constants.UPDATE)) {
					existAccounts.add(addAccount);
					it.remove();
					addAccounts.remove(addAccount);
					continue;
				}else if(status != null && status.intValue() == Constants.DELETE) {
					existDelAccounts.add(addAccount);
				}
				//某个用户已在的群数量
				int usergroupTotalNum = GroupCacheOperator.getGroupNum(addAccount);
				if (usergroupTotalNum >= getGroupCountLimit()) {
					groupCountErrList.add(addAccount);
					it.remove();
					addAccounts.remove(addAccount);
					continue;
				}
			} else {
				it.remove();
				addAccounts.remove(addAccount);
			}
		}
		
		int memberCountLimit = getGroupMemberCountLimit();
		int retFlag;
		//判断群组数量是否超限
		if(existAccounts.size() + addAccounts.size() + groupNum > memberCountLimit) {
			retFlag = AddMembersResult.RET_FAIl_GROUP_MEMBER_EXCESS;
		}else {
			if (groupCountErrList.size() == 0 && existAccounts.size() == 0) {
				retFlag = AddMembersResult.RET_SUCCESS;
			} else if(addAccounts.size() == 0){
				retFlag = AddMembersResult.RET_ALL_FAIL;
				errMap.put(AddMembersResult.GROUP_MEMBER_EXCEED, groupCountErrList);
				errMap.put(AddMembersResult.GROUP_MEMBER_ALREADY_EXIST, existAccounts);
			}else {
				retFlag = AddMembersResult.RET_PARTIAL_SUCCESS;
				errMap.put(AddMembersResult.GROUP_MEMBER_EXCEED, groupCountErrList);
				errMap.put(AddMembersResult.GROUP_MEMBER_ALREADY_EXIST, existAccounts);
			}
			long memberSerial = getGroupMemberSerial();
			long groupSerial = getGroupSerial();
			if((groupSerial==ERROR_FLAG)&&(memberSerial==ERROR_FLAG))
			{
				return errorIdFlag;	
			}
			
			memberDao.delMemberRecords(groupId, existDelAccounts);
			memberDao.addGroupMembers(groupId, account, addAccounts, memberSerial);
			groupDao.updateGroupUpdateSerial(groupId, groupSerial);
			GroupCacheOperator.addMembers(groupId, addAccounts, memberSerial);
			addMembersResult.setErrMap(errMap);
			addMembersResult.setSuceessAccounts(addAccounts);
			
		}
		addMembersResult.setRetFlag(retFlag);
		return addMembersResult;
		
	}

	@Override
	@DataSource("master")
	public Map<String, String> modifyGroupName(long groupId, String groupName) {
		Map<String, String> result = new HashMap<String, String>();
		long groupSerial = getGroupSerial();
		if(groupSerial==-1)
		{
			return result;
		}
		String groupNamePinyin = PinyinUtil.getCharactersPinyin(groupName);
		String groupNamePy = PinyinUtil.getCharactersPy(groupName);
		int res = groupDao.modifyGroupName(groupId, groupName, groupNamePinyin,
				groupNamePy, groupSerial);
		Map<String, String> pinyinMap = new HashMap<String, String>();
		if (res > 0) {
			pinyinMap.put("groupNamePinyin", groupNamePinyin);
			pinyinMap.put("groupNamePy", groupNamePy);
		}
		return pinyinMap;
	}

	@Override
	@DataSource("master")
	public long modifyGroupAvatar(Group group) {
		long updateSerial = getGroupSerial();
		if(updateSerial==-1)
		{
			return ERROR_FLAG;
		}
		groupDao.modifyGroupAvatar(group, updateSerial);
		return GROUP_BUSSINESS_FINISHED;
	}

	@Override
	@DataSource("master")
	public long doQuitGroup(Long groupId, String account) {
		long groupSerial = getGroupSerial();
		long memberSerial = getGroupMemberSerial();
		if((groupSerial==-1)&&(memberSerial==-1))
		{
			return ERROR_FLAG;
		}
		groupDao.updateGroupUpdateSerial(groupId, groupSerial);
		memberDao.updateGroupMemberStatus(account, groupId, memberSerial,
				Constants.DELETE);
		GroupCacheOperator.quitGroup(groupId, account, memberSerial);
		return GROUP_BUSSINESS_FINISHED;
	}

	@Override
	@DataSource("master")
	public long doDismissGroup(long groupId) {
		long groupSerial = getGroupSerial();
		long memberSerial = getGroupMemberSerial();
		if((groupSerial==-1)&&(memberSerial==-1))
		{
			return ERROR_FLAG;
		}
		groupDao.updateGroupStatus(groupId, Constants.DELETE, groupSerial);
		memberDao.deleteGroup(groupId, memberSerial);
		GroupCacheOperator.delGroup(groupId, memberSerial);
		return GROUP_BUSSINESS_FINISHED;
	}

	@Override
	@DataSource("slave")
	public List<Group> queryGroups(String account, long updateSerial) {
		if (updateSerial == 0) {
			return groupDao.queryAllNormalGroups(account);
		} else {
			return groupDao.queryGroups(account, updateSerial);
		}
	}

	@Override
	public int getGroupMemberNum(long groupId) {
		return GroupCacheOperator.getGroupMemberNum(groupId);
	}

	@Override
	public int getGroupNum(String account) {
		return GroupCacheOperator.getGroupNum(account);
	}

	@Override
	@DataSource("master")
	public void deleteGroupForever(long groupId) {
		groupDao.deleteGroupRecord(groupId);
		memberDao.delGroupMembersRecord(groupId);
		GroupCacheOperator.delGroup(groupId);
	}

	/**
	 * 查询正常状态的群组及群成员
	 */
	@Override
	@DataSource("slave")
	public Group queryGroupDetail(long groupId) {
		Group group = groupDao.queryGroup(groupId);
		if (group == null) {
			return null;
		}

		List<GroupMember> groupMembers = memberDao
				.queryNormalGroupMembers(groupId);
		group.setMembers(groupMembers);
		return group;
	}

	@Override
	@DataSource("slave")
	public Group queryGroupInfoOnly(long groupId) {
		return groupDao.queryGroup(groupId);
	}

	@Override
	@DataSource("slave")
	public List<Map<String, Object>> queryMembers(
			List<Map<String, Long>> paramMapList) {
		if (paramMapList == null || paramMapList.isEmpty()) {
			return null;
		}
		List<Map<String, Object>> groupMaps = new ArrayList<Map<String, Object>>();
		for (Map<String, Long> map : paramMapList) {
			Long groupId = map.get("groupId");
			Long updateSerial = map.get("seq");
			if (groupId == null || updateSerial == null || groupId < 1) {
				continue;
			}
			Map<String, Object> groupMap = new HashMap<String, Object>();
			List<GroupMember> groupMembers = memberDao.queryGroupMembers(
					groupId, updateSerial);
			if(groupMembers == null || groupMembers.isEmpty()) {
				continue;
			}
			groupMap.put("groupId", groupId);
			groupMap.put("members", groupMembers);
			groupMaps.add(groupMap);
		}
		return groupMaps;
	}

	@Override
	@DataSource("master")
	public long removeGroupMembers(long groupId, List<String> removeAccounts) {
		long groupSerial = getGroupSerial();
		long memberSerial = getGroupMemberSerial();
		if((groupSerial==-1)&&(memberSerial==-1))
		{
			return ERROR_FLAG;
		}
		groupDao.updateGroupUpdateSerial(groupId, groupSerial);
		memberDao.deleteGroupMembers(groupId, removeAccounts, memberSerial);
		GroupCacheOperator.removeGroupMembers(groupId, removeAccounts, memberSerial);
		return GROUP_BUSSINESS_FINISHED;
	}

	/**
	 * 调用id生成服务生成群组更新序列
	 * 
	 * @return
	 */
	private long getGroupSerial() {
		return IdGenUtil.getId(IdType.GROUP);
	}

	/**
	 * 调用id生成服务生成群成员序列
	 * 
	 * @return
	 */
	private long getGroupMemberSerial() {
		return IdGenUtil.getId(IdType.GROUP_MEMBER);
	}

	@Override
	@DataSource("slave")
	public List<CacheMemberBean> queryDataForCache(long serial, int batchSize) {
		return memberDao.queryMembersBatch(serial, batchSize);
	}

	@Override
	@DataSource("master")
	public ModifyGroupNicknameResult modifyGroupNickname(long groupId, String account,
			String newNickname) {
		ModifyGroupNicknameResult result=null;
		long memberSerial = getGroupMemberSerial();
		long groupSerial = getGroupSerial();
		if((memberSerial==-1)&&(groupSerial==-1))
		{
			return result;
		}
		String nicknamePy = PinyinUtil.getCharactersPy(newNickname);
		String nicknamePinyin = PinyinUtil.getCharactersPinyin(newNickname);
		memberDao.updateGroupNickName(groupId, account, newNickname, nicknamePy, nicknamePinyin, memberSerial);
		groupDao.updateGroupUpdateSerial(groupId, groupSerial);
		GroupCacheOperator.updateGroupMemberSeq(memberSerial);
		
		ModifyGroupNicknameResult ret = new ModifyGroupNicknameResult();
		ret.setNicknamePinyin(nicknamePinyin);
		ret.setNicknamePy(nicknamePy);
		return ret;
	}

	@Override
	public Set<String> queryAllSameGroupMembers(String account) {
		return GroupCacheOperator.queryAllSameGroupMembers(account);
	}

	@Override
	public List<String> getGroupMembers(String groupId)
	{
		
        List<String> groupMemberList = GroupCacheOperator.getGroupAllMembers(groupId);
		
		return groupMemberList;
	}

}
