package com.xdja.cssp.group.dao;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.List;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.jdbc.core.BeanPropertyRowMapper;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.jdbc.core.namedparam.MapSqlParameterSource;
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate;
import org.springframework.jdbc.support.GeneratedKeyHolder;
import org.springframework.jdbc.support.KeyHolder;
import org.springframework.stereotype.Repository;

import com.xdja.cssp.group.common.Constants;
import com.xdja.cssp.group.entity.Group;

@Repository
public class GroupDao {
	
	
	@Autowired
	@Qualifier("namedJdbcTemplate_dynamicDS")
	private NamedParameterJdbcTemplate namedJdbcTemplate;
	
	
	/**
	 * 添加群组信息
	 * 
	 * @author: mayanpei
	 * @date: 2015-7-6 下午2:07:37
	 * @param group
	 * @param updateSerial
	 *            增量序列
	 * @return Group 保存群信息并将新生成的群id组合成群信息返回
	 */
	public Group saveGroup(final Group group) {

		KeyHolder keyHolder = new GeneratedKeyHolder();

		StringBuffer sql = new StringBuffer("INSERT INTO t_at_group (c_name,c_avatar_id, c_thumbnail_id,c_owner,");
		sql.append(" c_name_py,c_name_pinyin,n_create_time,n_status,n_update_serial) ");
		sql.append("  VALUES(:name,:avatarId, :thumbnailId,:owner,:py,:pinyin,:time,:status,:serial)");

		MapSqlParameterSource paramSource = new MapSqlParameterSource();
		paramSource.addValue("name", group.getName());
		paramSource.addValue("avatarId", group.getAvatarId());
		paramSource.addValue("thumbnailId", group.getThumbnailId());
		paramSource.addValue("owner", group.getOwner());
		paramSource.addValue("py", group.getNamePy());
		paramSource.addValue("pinyin", group.getNamePinyin());
		paramSource.addValue("time", System.currentTimeMillis());
		paramSource.addValue("status", group.getStatus());
		paramSource.addValue("serial", group.getUpdateSerial());
		namedJdbcTemplate.update(sql.toString(), paramSource, keyHolder);
		group.setId(keyHolder.getKey().longValue());
		return group;
	}
	
	/**
	 * 删除群组信息记录
	 * @param groupId 群id
	 * @return
	 */
	public int deleteGroupRecord(long groupId) {
		String sql = "DELETE FROM t_at_group WHERE n_id = :groupId ";
		MapSqlParameterSource paramSource = new MapSqlParameterSource();
		paramSource.addValue("groupId", groupId);
		return namedJdbcTemplate.update(sql, paramSource);
	}
	
	/**
	 * 查询所有正常的群组信息
	 * @param account 账号 
	 * @param updateSerial 更新序列
	 * @return
	 */
	public List<Group> queryAllNormalGroups(String account) {
		StringBuffer groupSql = new StringBuffer("select  ag.n_id AS id ,ag.c_name AS name,")
		.append(" ag.c_avatar_id AS avatarId,ag.c_thumbnail_id as thumbnailId, ag.c_owner AS owner, ag.c_name_pinyin AS namepinyin,")
		.append(" ag.c_name_py AS  namepy , ")
		.append(" ag.n_create_time AS createTime,ag.n_status AS status,ag.n_update_serial AS generatedKey ")
		.append(" FROM t_at_group ag ")
		.append(" JOIN t_at_group_member gm on ag.n_id = gm.n_group_id ")
		.append(" WHERE gm.c_account=:account AND ag.n_status<>:status AND gm.n_status<>:status ")
		.append(" ORDER BY ag.n_create_time ASC");

		MapSqlParameterSource param = new MapSqlParameterSource();
		param.addValue("status", Constants.DELETE);
		
		param.addValue("account", account);
		return namedJdbcTemplate.query(groupSql.toString(), param, new RowMapper<Group>() {
		
			public Group mapRow(ResultSet rs, int rowNum) throws SQLException {
				Group group = new Group();
				group.setId(rs.getLong("id"));
				group.setName(rs.getString("name"));
				group.setAvatarId(rs.getString("avatarId"));
				group.setThumbnailId(rs.getString("thumbnailId"));
				group.setOwner(rs.getString("owner"));
				group.setNamePy(rs.getString("namepy"));
				group.setNamePinyin(rs.getString("namepinyin"));
				group.setCreateTime(rs.getLong("createTime"));
				group.setStatus(rs.getInt("status"));
				group.setUpdateSerial(rs.getLong("generatedKey"));
				return group;
			}
		});
		
	}
	
	/**
	 * 查询群组增量信息
	 * @param account 账号
	 * @param updateSerial 更新序列
	 * @return
	 */
	public List<Group> queryGroups(String account, long updateSerial) {
		StringBuffer groupSql = new StringBuffer("select  ag.n_id AS id ,ag.c_name AS name,")
			.append(" ag.c_avatar_id AS avatarId,ag.c_thumbnail_id as thumbnailId, ag.c_owner AS owner, ag.c_name_pinyin AS namepinyin,")
			.append(" ag.c_name_py AS  namepy , ")
			.append(" ag.n_create_time AS createTime,ag.n_status AS status,ag.n_update_serial AS generatedKey ")
			.append(" FROM t_at_group ag ").append(" JOIN t_at_group_member gm on ag.n_id = gm.n_group_id ")
			.append(" WHERE gm.c_account=:account AND ag.n_update_serial >:updateSerial ")
			.append(" ORDER BY ag.n_create_time ASC");
 
		MapSqlParameterSource param = new MapSqlParameterSource();
		// 如果是首次增量更新，把已退出或者删除的群信息过滤掉
		param.addValue("updateSerial", updateSerial);
		param.addValue("account", account);
		return namedJdbcTemplate.query(groupSql.toString(), param, new RowMapper<Group>() {
		
			public Group mapRow(ResultSet rs, int rowNum) throws SQLException {
				Group group = new Group();
				group.setId(rs.getLong("id"));
				group.setName(rs.getString("name"));
				group.setAvatarId(rs.getString("avatarId"));
				group.setThumbnailId(rs.getString("thumbnailId"));
				group.setOwner(rs.getString("owner"));
				group.setNamePy(rs.getString("namepy"));
				group.setNamePinyin(rs.getString("namepinyin"));
				group.setCreateTime(rs.getLong("createTime"));
				group.setStatus(rs.getInt("status"));
				group.setUpdateSerial(rs.getLong("generatedKey"));
				return group;
			}
		});
	}
	
	public Group queryGroup(long groupId) {
		StringBuffer sql = new StringBuffer("SELECT ag.n_id AS id ,");
		sql.append("ag.c_name AS name, ag.c_avatar_id AS avatarId,ag.c_thumbnail_id as thumbnailId, ");
		sql.append("ag.c_owner AS owner, ag.c_name_pinyin AS namePinyin,");
		sql.append("ag.c_name_py AS namePy ,ag.n_create_time AS createTime,");
		sql.append("ag.n_status AS status ,ag.n_update_serial AS updateSerial ");
		sql.append("FROM t_at_group ag WHERE n_id = :groupId ");

		MapSqlParameterSource sqlParam = new MapSqlParameterSource();
		sqlParam.addValue("groupId", groupId);
		RowMapper<Group> rowMapper = new BeanPropertyRowMapper<>(Group.class);
		List<Group> list = namedJdbcTemplate.query(sql.toString(), sqlParam, rowMapper);
		if(list.size() == 0) {
			return null;
		}else {
			return list.get(0);
		}
	}

	/**
	 * 修改群名称
	 * @param groupId 群组id
	 * @param groupName 群名称
	 * @param groupNamePinyin 群名称拼音
	 * @param groupNamePy 群名称拼音简拼
	 * @param updateSerial 群更新序列
	 */
	public int modifyGroupName(long groupId, String groupName,
			String groupNamePinyin, String groupNamePy, long updateSerial) {
		StringBuilder sql = new StringBuilder();
		sql.append("UPDATE t_at_group set c_name=:groupName, c_name_py=:groupNamePy,");
		sql.append("c_name_pinyin=:groupNamePinyin, n_status=:status,n_update_serial=:updateSerial ");
		sql.append("WHERE n_id=:groupId");
		MapSqlParameterSource sqlParam = new MapSqlParameterSource();
		sqlParam.addValue("groupName", groupName);
		sqlParam.addValue("groupNamePinyin", groupNamePinyin);
		sqlParam.addValue("groupNamePy", groupNamePy);
		sqlParam.addValue("updateSerial", updateSerial);
		sqlParam.addValue("status", Constants.UPDATE);
		sqlParam.addValue("groupId", groupId);
		
		return namedJdbcTemplate.update(sql.toString(), sqlParam);
	}

	/**
	 * 修改群名称
	 * @param groupId 群组id
	 * @param groupName 群名称
	 * @param groupNamePinyin 群名称拼音
	 * @param groupNamePy 群名称拼音简拼
	 * @param updateSerial 群更新序列
	 */
	public int modifyGroupAvatar(Group group, long updateSerial) {
		StringBuilder sql = new StringBuilder();
		sql.append("UPDATE t_at_group SET c_avatar_id=:avatarId,c_thumbnail_id=:thumbnailId,");
		sql.append("n_status=:status,n_update_serial=:updateSerial WHERE n_id=:id ");
		MapSqlParameterSource sqlParam = new MapSqlParameterSource();
		sqlParam.addValue("avatarId", group.getAvatarId())
		.addValue("thumbnailId", group.getThumbnailId())
		.addValue("status", Constants.UPDATE)
		.addValue("updateSerial", updateSerial)
		.addValue("id", group.getId());
		
		return namedJdbcTemplate.update(sql.toString(), sqlParam);
	}
	
	/**
	 * 更新群信息序列
	 * @param groupId 群id
	 * @param updateSerial 群更新序列
	 */
	public void updateGroupUpdateSerial(long groupId, long updateSerial) {
		final String sql = "UPDATE t_at_group SET n_update_serial=:updateSerial WHERE n_id=:id";
		MapSqlParameterSource param = new MapSqlParameterSource();
		param.addValue("updateSerial", updateSerial);
		param.addValue("id", groupId);
		
		namedJdbcTemplate.update(sql, param);
	}
	
	/**
	 * 更新群组状态，同时更改群组更新序列
	 * @param groupId
	 * @param status
	 * @param updateSerial
	 */
	public int updateGroupStatus(long groupId, int status, long updateSerial) {
		StringBuilder sql = new StringBuilder();
		sql.append("UPDATE  t_at_group SET n_status=:status,n_update_serial=:updateSerial ");
		sql.append("WHERE n_id=:groupId ");
		MapSqlParameterSource sqlParam = new MapSqlParameterSource();
		sqlParam.addValue("groupId", groupId);
		sqlParam.addValue("status", status);
		sqlParam.addValue("updateSerial", updateSerial);
		return namedJdbcTemplate.update(sql.toString(), sqlParam);
	}
	
	/**
	 * 根据序列查询群组信息
	 * @param updateSerial
	 * @return
	 */
	public List<Group> queryGroups(Long updateSerial) {
		StringBuffer groupSql = new StringBuffer("SELECT ag.n_id AS id ,ag.c_name AS name,");
		groupSql.append(" ag.c_avatar_id AS avatarId,ag.c_thumbnail_id as thumbnailId, ");
		groupSql.append("ag.c_owner AS owner, ag.c_name_pinyin AS namepinyin,");
		groupSql.append(" ag.c_name_py AS  namepy , ag.n_create_time AS createTime,");
		groupSql.append("ag.n_status AS status,ag.n_update_serial AS generatedKey ");
		groupSql.append(" FROM t_at_group ag WHERE 1=1 ");

		MapSqlParameterSource param = new MapSqlParameterSource();

		// 如果是首次增量更新，把已退出或者删除的群信息过滤掉
		if (updateSerial != null && updateSerial == 0) {
			groupSql.append("AND ag.n_status<>:status");
			param.addValue("status", Constants.DELETE);
		}

		/** 根据updateSerial来判断是否是增量查询 */
		if (updateSerial != null && updateSerial > 0) {
			groupSql.append(" AND ag.n_update_serial >:updateSerial");
			param.addValue("updateSerial", updateSerial);
		}

		return namedJdbcTemplate.query(groupSql.toString(), param, new RowMapper<Group>() {

			public Group mapRow(ResultSet rs, int rowNum) throws SQLException {
				Group group = new Group();
				group.setId(rs.getLong("id"));
				group.setName(rs.getString("name"));
				group.setAvatarId(rs.getString("avatarId"));
				group.setThumbnailId(rs.getString("thumbnailId"));
				group.setOwner(rs.getString("owner"));
				group.setNamePy(rs.getString("namepy"));
				group.setNamePinyin(rs.getString("namepinyin"));
				group.setCreateTime(rs.getLong("createTime"));
				group.setStatus(rs.getInt("status"));
				group.setUpdateSerial(rs.getLong("generatedKey"));
				return group;
			}
		});
	}

}
