package com.xdja.cssp.group.dao;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.springframework.jdbc.core.BeanPropertyRowMapper;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.jdbc.core.namedparam.MapSqlParameterSource;
import org.springframework.jdbc.core.namedparam.SqlParameterSource;
import org.springframework.stereotype.Repository;

import com.xdja.cssp.group.common.Constants;
import com.xdja.cssp.group.entity.GroupMember;
import com.xdja.cssp.group.scheduling.CacheMemberBean;
import com.xdja.platform.datacenter.jpa.dao.BaseJdbcDao;

@Repository
public class GroupMemberDao extends BaseJdbcDao {
	
	
	/**
	 * 保存群成员信息
	 * @param groupId
	 * @param groupMembers
	 */
	public void saveGroupMembers(long groupId, List<GroupMember> groupMembers) {
		StringBuilder addSql = new StringBuilder("INSERT INTO t_at_group_member(n_group_id,c_account,");
		addSql.append(" c_nick_name,c_nick_name_py,c_nick_name_pinyin,n_create_time,c_invite_account,n_status,n_update_serial)");
		addSql.append("  VALUES (:groupId,:account,:nickName,:py,:pinyin,:time,:inviteAccount,:status,:updateSerial) ");
		
		List<MapSqlParameterSource> params = new ArrayList<MapSqlParameterSource>();
		for (GroupMember member : groupMembers) {
			Integer status = member.getStatus();
			if(status == null) {
				member.setStatus(Constants.ADD);
			}
			MapSqlParameterSource param = new MapSqlParameterSource();
			param.addValue("groupId", groupId);
			param.addValue("account", member.getAccount());
			param.addValue("nickName", member.getNickName());
			param.addValue("py", member.getNickNamePy());
			param.addValue("pinyin", member.getNickNamePinyin());
			param.addValue("inviteAccount", member.getInviteAccount());
			param.addValue("time", System.currentTimeMillis());
			param.addValue("status", member.getStatus());
			param.addValue("updateSerial", member.getUpdateSerial());
			params.add(param);

		}
		
		if (params.size() > 0) {
			SqlParameterSource[] paramsArray = new SqlParameterSource[params.size()];
			paramsArray = params.toArray(paramsArray);
			jdbcOperator.namedJdbcTemplate.batchUpdate(addSql.toString(), paramsArray);
		}
	}
	
	/**
	 * 删除群成员，永久删除，无法恢复
	 * @param groupId 群id
	 * @return
	 */
	public int delGroupMembersRecord(long groupId) {
		String sql = "DELETE FROM T_AT_GROUP_MEMBER WHERE N_GROUP_ID = :groupId "; 
		MapSqlParameterSource paramSource = new MapSqlParameterSource();
		paramSource.addValue("groupId", groupId);
		return jdbcOperator.namedJdbcTemplate.update(sql, paramSource);
	}
	
	
	/**
	 * 查询群成员，仅查询正常状态的
	 * @param groupId 群id
	 * @return
	 */
	public List<GroupMember> queryNormalGroupMembers(long groupId) {
		StringBuffer sql = new StringBuffer();
		sql.append("SELECT gm.c_account AS account ,gm.c_nick_name AS nickName,");
		sql.append("gm.c_nick_name_py AS nickNamePy,c_nick_name_pinyin AS nickNamePinyin,");
		sql.append(" gm.n_create_time AS createTime,gm.n_status status,c_invite_account AS inviteAccount,");
		sql.append("gm.n_update_serial updateSerial FROM t_at_group_member gm ");
		sql.append("WHERE gm.n_group_id=:groupId AND gm.n_status <> :status ");
		sql.append("ORDER BY n_update_serial ASC ");
		
		MapSqlParameterSource paramSource = new MapSqlParameterSource();
		paramSource.addValue("groupId", groupId);
		paramSource.addValue("status", Constants.DELETE);
		
		RowMapper<GroupMember> rowMapper = new BeanPropertyRowMapper<>(GroupMember.class);
		return jdbcOperator.namedJdbcTemplate.query(sql.toString(), paramSource, rowMapper);
	}
	
	

	/**
	 * 增量查询群成员
	 * @param groupId 群id
	 * @return
	 */
	public List<GroupMember> queryGroupMembers(long groupId, long updateSerial) {
		StringBuffer sql = new StringBuffer();
		sql.append("SELECT gm.c_account AS account ,gm.c_nick_name AS nickName,");
		sql.append("gm.c_nick_name_py AS nickNamePy,c_nick_name_pinyin AS nickNamePinyin,");
		sql.append(" gm.n_create_time AS createTime,gm.n_status status,c_invite_account AS inviteAccount,");
		sql.append("gm.n_update_serial updateSerial FROM t_at_group_member gm ");
		sql.append("WHERE gm.n_group_id=:groupId AND gm.n_update_serial > :updateSerial ");
		sql.append("ORDER BY gm.n_update_serial ASC ");
		
		MapSqlParameterSource paramSource = new MapSqlParameterSource();
		paramSource.addValue("groupId", groupId);
		paramSource.addValue("updateSerial", updateSerial);
		
		RowMapper<GroupMember> rowMapper = new BeanPropertyRowMapper<>(GroupMember.class);
		return jdbcOperator.namedJdbcTemplate.query(sql.toString(), paramSource, rowMapper);
	}
	
	/**
	 * 更新群成员状态
	 * @param account 账号
	 * @param groupId 群组id
	 * @param updateSerial 更新序列
	 * @return
	 */
	public int updateGroupMemberStatus(String account, long groupId, long updateSerial, int status) {
		StringBuffer sql = new StringBuffer();
		sql.append(" UPDATE  t_at_group_member SET n_status=:status,n_update_serial=:updateSerial");
		sql.append("  WHERE n_group_id=:groupId AND c_account=:account ");

		MapSqlParameterSource param = new MapSqlParameterSource();
		param = new MapSqlParameterSource();
		param.addValue("status", status);
		param.addValue("updateSerial", updateSerial);
		param.addValue("groupId", groupId);
		param.addValue("account", account);
		return executeSql(sql.toString(), param);
	}
	
	/**
	 * 删除群组，修改所有群成员状态
	 * @param groupId 群id
	 * @param updateSerial 更新序列
	 * @return
	 */
	public int deleteGroup(long groupId, long updateSerial) {
		StringBuffer sql = new StringBuffer();
		sql.append(" UPDATE  t_at_group_member SET n_status=:status,n_update_serial=:updateSerial");
		sql.append("  WHERE n_group_id=:groupId ");
		
		MapSqlParameterSource param = new MapSqlParameterSource();
		param = new MapSqlParameterSource();
		param.addValue("updateSerial", updateSerial);
		param.addValue("groupId", groupId);
		param.addValue("status", Constants.DELETE);
		return executeSql(sql.toString(), param);
	}
	
	/**
	 * 删除群成员,软删除
	 * @param groupId 群组id
	 * @param accounts 账号列表
	 * @param updateSerial 更新序列
	 */
	public void deleteGroupMembers(long groupId, List<String> accounts, long updateSerial) {
		StringBuilder sql = new StringBuilder();
		sql.append("UPDATE t_at_group_member SET n_status=:status,n_update_serial=:updateSerial ");
		sql.append("WHERE c_account = :account AND n_group_id = :groupId ");
		
		List<MapSqlParameterSource> params = new ArrayList<MapSqlParameterSource>();
		for (String account : accounts) {
			MapSqlParameterSource param = new MapSqlParameterSource();
			param.addValue("groupId", groupId);
			param.addValue("account", account);
			param.addValue("status", Constants.DELETE);
			param.addValue("updateSerial", updateSerial);
			params.add(param);
		}
		
		if (params.size() > 0) {
			SqlParameterSource[] paramsArray = new SqlParameterSource[params.size()];
			paramsArray = params.toArray(paramsArray);
			jdbcOperator.namedJdbcTemplate.batchUpdate(sql.toString(), paramsArray);
		}
		
	}
	
	/**
	 * 添加群成员到群组中
	 * @param groupId 群id
	 * @param account 群账号
	 * @param addAccounts 要添加的账号
	 */
	public void addGroupMembers(long groupId, String inviter, List<String> addAccounts, long memberSerial) {
		StringBuilder addSql = new StringBuilder("INSERT INTO t_at_group_member(n_group_id,c_account,");
		addSql.append(" n_create_time,c_invite_account,n_status,n_update_serial)");
		addSql.append("  VALUES (:groupId,:account,:time,:inviteAccount,:status,:updateSerial) ");
		
		List<MapSqlParameterSource> params = new ArrayList<MapSqlParameterSource>();
		long now = System.currentTimeMillis();
		for (String account : addAccounts) {
			MapSqlParameterSource param = new MapSqlParameterSource();
			param.addValue("groupId", groupId);
			param.addValue("account", account);
			param.addValue("inviteAccount", inviter);
			param.addValue("time", now);
			param.addValue("status", Constants.ADD);
			param.addValue("updateSerial", memberSerial);
			params.add(param);

		}
		
		if (params.size() > 0) {
			SqlParameterSource[] paramsArray = new SqlParameterSource[params.size()];
			paramsArray = params.toArray(paramsArray);
			jdbcOperator.namedJdbcTemplate.batchUpdate(addSql.toString(), paramsArray);
		}
	}
	
	
	/**
	 * 删除群成员记录
	 * @param groupId 群id
	 * @param delAccounts 要删除的账号列表
	 */
	public void delMemberRecords(long groupId, List<String> delAccounts) {
		StringBuilder sql = new StringBuilder();
		sql.append("DELETE FROM t_at_group_member WHERE n_group_id = :groupId ");
		sql.append("AND c_account = :account ");
		List<MapSqlParameterSource> params = new ArrayList<MapSqlParameterSource>();
		for(String account : delAccounts) {
			MapSqlParameterSource param = new MapSqlParameterSource();
			param.addValue("groupId", groupId);
			param.addValue("account", account);
			params.add(param);
		}
		if (params.size() > 0) {
			SqlParameterSource[] paramsArray = new SqlParameterSource[params.size()];
			paramsArray = params.toArray(paramsArray);
			jdbcOperator.namedJdbcTemplate.batchUpdate(sql.toString(), paramsArray);
		}
		
	}
	
	

	/**
	 * 查询需要缓存的数据
	 * @param serial 更新序列
	 * @param batchSize 批量条数
	 * @return
	 */
	public List<CacheMemberBean> queryMembersBatch(long serial, int batchSize) {
		StringBuilder sql = new StringBuilder(); 
		sql.append("SELECT gm.n_group_id groupId, gm.c_account account, gm.n_update_serial updateSerial ");
		sql.append("FROM t_at_group_member gm ");
		sql.append("WHERE gm.n_update_serial >= :updateSerial AND gm.n_status <> :status ");
		sql.append("ORDER BY gm.n_update_serial ASC LIMIT 0, :size ");
		
		MapSqlParameterSource paramSource = new MapSqlParameterSource();
		paramSource.addValue("status", Constants.DELETE);
		paramSource.addValue("updateSerial", serial);
		paramSource.addValue("size", batchSize);
		RowMapper<CacheMemberBean> rowMapper = new BeanPropertyRowMapper<>(CacheMemberBean.class);
		return jdbcOperator.namedJdbcTemplate.query(sql.toString(), paramSource, rowMapper);
	}
	
	/**
	 * 更新群昵称
	 * @param groupId 群名称
	 * @param account 群成员账号
	 * @param nickname 群昵称
	 * @param nicknamePy 群昵称简拼
	 * @param nicknamePinyin 群昵称拼音
	 * @param memberSerial 群更新序列
	 * @return 
	 */
	public int updateGroupNickName(long groupId, String account, String nickname, 
			String nicknamePy, String nicknamePinyin, long memberSerial) {
		
		StringBuilder sql = new StringBuilder();
		sql.append("UPDATE t_at_group_member SET c_nick_name=:nickame, ");
		sql.append("c_nick_name_pinyin = :nicknamePinyin, c_nick_name_py = :nicknamePy, ");
		sql.append("n_status=:status, n_update_serial=:updateSerial ");
		sql.append("WHERE n_group_id=:id AND c_account=:account AND n_status != :deleteStatus ");
		MapSqlParameterSource param = new MapSqlParameterSource();
		param.addValue("nickame", nickname);
		param.addValue("nicknamePinyin", nicknamePinyin);
		param.addValue("nicknamePy", nicknamePy);
		param.addValue("status", Constants.UPDATE);
		param.addValue("deleteStatus", Constants.DELETE);
		param.addValue("updateSerial", memberSerial);
		param.addValue("id", groupId);
		param.addValue("account", account);
		return executeSql(sql.toString(), param);
	}
	
	/**
	 * 查询成员状态
	 * @param groupId 发群id
	 * @param memberAccounts 群成员账号
	 * @return
	 */
	public List<Map<String, Object>> queryMemberStatus(long groupId, List<String> memberAccounts) {
		StringBuilder sql = new StringBuilder();
		sql.append("SELECT c_account account,n_status status FROM t_at_group_member ");
		sql.append("WHERE n_group_id = :groupId AND c_account IN (:accounts) ");
		MapSqlParameterSource sqlParam = new MapSqlParameterSource();
		sqlParam.addValue("accounts", memberAccounts);
		sqlParam.addValue("groupId", groupId);
		return queryForList(sql.toString(), sqlParam);
	}
	
 }
