package com.xdja.contactclient.start;

import com.xdja.contact.thrift.datatype.*;
import com.xdja.contactclient.common.DefaultValues;
import com.xdja.contactclient.common.ReturnValues;
import com.xdja.contactclient.common.Utils;
import com.xdja.contactclient.rpcstubpool.RpcClientInfo;
import com.xdja.contactclient.rpcstubpool.RpcStubPool;
import com.xdja.contactclient.rpcstubpool.RpcStubPoolConfig;
import com.xdja.contactclient.rpcstubpool.RpcStubPoolFactory;
import com.xdja.contact.thrift.stub.ContactEcService;
import com.xdja.contact.thrift.stub.ContactFriendService;
import com.xdja.contact.thrift.stub.ContactGroupService;

import org.apache.thrift.TException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

public class ContactRpcClient
{
	private static Logger logger = LoggerFactory.getLogger(ContactRpcClient.class);
	private RpcStubPoolFactory rpcfactory = null;
	private RpcStubPool rpcstubpool = null;
	private RpcStubPoolConfig config = new RpcStubPoolConfig();
	private int timeout ;
	private int serverMode = 3;
	private String sHost = "";
	private int sPort = -1;

	public ContactRpcClient(String serverHost, int serverPort)
	{
		this.sHost = serverHost;
		this.sPort = serverPort;
		config.setMaxActive(100);
		config.setMaxIdle(100);
		config.setMaxWait(3000);
		config.setTestWhileIdle(false);
		config.setMinEvictableIdleTimeMillis(3600000);
		config.setTestOnBorrow(true);
		config.setTestOnReturn(false);
	}
	public ContactRpcClient(String serverHost, int serverPort, int timeout)
	{
		this.sHost = serverHost;
		this.sPort = serverPort;
		this.timeout = timeout;
		config.setMaxActive(100);
		config.setMaxIdle(100);
		config.setMaxWait(3000);
		config.setTestWhileIdle(false);
		config.setMinEvictableIdleTimeMillis(3600000);
		config.setTestOnBorrow(true);
		config.setTestOnReturn(false);
	}

	public ContactRpcClient(String serverHost, int serverPort, int serverMode, RpcStubPoolConfig config, int timeout)
	{
		this.timeout = timeout;
		this.config = config;
		this.serverMode = serverMode;
		this.sHost = serverHost;
		this.sPort = serverPort;
	}

	/**
	 * ContactRpcClient初始化
	 *
	 * @param reqId 请求索引
	 * @return 初始化成功返回true，失败返回false
	 */
	public boolean init(long reqId)
	{
		String logFlag = "ContactRpcClient.init";
		logger.info("[lid:{}][{}] serverHost:{} serverPort:{} serverMode:{}", reqId, logFlag, this.sHost, this.sPort, this.serverMode);
		rpcfactory = new RpcStubPoolFactory(this.sHost, this.sPort, this.serverMode, this.timeout);
		rpcstubpool = new RpcStubPool();
		if (!rpcstubpool.init(reqId, this.config, rpcfactory))
		{
			logger.error("[lid:{}]  rpcpool init failed!", reqId);
			return false;
		}
		return true;
	}

	@SuppressWarnings("unchecked")
	private <T> RpcClientInfo<T> getServiceClient(long reqId, String key)
	{
		String logFlag = "ContactRpcClient.getServiceClient";
		Object rpcconnect = null;
		try
		{
			rpcconnect =  rpcstubpool.getResource(reqId, key);
			return (RpcClientInfo<T>) rpcconnect;
		}
		catch (Exception e)
		{
			logger.warn("[lid:{}][{}] exception happened! detail:\n{}", reqId, logFlag, Utils.getStackTrace(e));
			rpcstubpool.returnBrokenResource(reqId, key, rpcconnect);
			return null;
		}
	}

	/**
	 * 分批次获取指定集团需要更新的人员信息
	 *
	 * @param reqId              请求索引
	 * @param caller             调用模块的标识
	 * @param ticket            账号
	 * @param personLastUpdateId 人员最后更新标识
	 * @param personSubUpdateId  人员更新截止标识
	 * @param batchSize          本批次更新的数量
	 * @return UpdateContactPersonsResult
	 * @author joniers.jia
	 */
	public Result<String> queryPersons(long reqId, String caller, String ticket, long personLastUpdateId, long personSubUpdateId, int batchSize)
	{
		String logFlag = "ContactRpcClient.queryPersons";
		logger.info("[lid:{}][{}] >> caller:{} ticket:{} personLastUpdateId:{} personSubUpdateId:{} batchSize:{}", reqId, logFlag, caller, ticket, personLastUpdateId,
				personSubUpdateId, batchSize);
		RpcClientInfo<ContactEcService.Client> ci = null;
		ResStr res = new ResStr();
		try
		{
			ci = getServiceClient(reqId, DefaultValues.CONTACT_EC_SERVICE);
			if (ci == null)
			{
				logger.warn("[lid:{}][{}] cann't get contact rpc Client", reqId, logFlag);
				return new Result<>(ReturnValues.INNER_ERROR,null);
			}
			res = ci.getClient().queryPersons(reqId, caller, ticket, personLastUpdateId, personSubUpdateId, batchSize);
			if (res.res == ReturnValues.SUCCESS)
			{
				logger.info("[lid:{}][{}] result:{}, value:{}", reqId, logFlag, res.res, res.value);
			}
			else
			{
				logger.warn("[lid:{}][{}] failed! errcode:{}", reqId, logFlag, res.res);
			}
			rpcstubpool.returnResource(reqId, DefaultValues.CONTACT_EC_SERVICE, ci);
			return new Result<String>(res.res,res.value);
		}
		catch (TException e)
		{
			rpcstubpool.returnBrokenResource(reqId, DefaultValues.CONTACT_EC_SERVICE, ci);
			logger.warn("[lid:{}][{}] exception happened! detail:\n{}", reqId, logFlag, Utils.getStackTrace(e));
			return new Result<>(ReturnValues.INNER_ERROR,null);
		}
	}
	/**
	 * 添加权限的分批次获取指定集团需要更新的人员信息
	 *
	 * @param logIndex              请求索引
	 * @param caller             调用模块的标识
	 * @param ticket            账号
	 * @param currentPage       当前页码
	 * @param pageSize          页码大小
	 * @param deptId             部门标识
	 * @return queryCondition      查询条件
	 * @author joniers.jia
	 */
	public Result<String>queryNewPersons(long logIndex, String caller, String ticket, int currentPage, int pageSize, String deptId, String queryCondition){
		String logFlag = "ContactRpcClient.queryNewPersons";
		logger.info("[lid:{}][{}] caller:{},ticket:{}, currentPage:{}, pageSize:{}, deptId:{},queryCondition:{}",
				logIndex,logFlag,caller, ticket, currentPage, pageSize, deptId, queryCondition);
		RpcClientInfo<ContactEcService.Client> ci = null;
		ResStr res = new ResStr();
		try
		{
			ci = getServiceClient(logIndex, DefaultValues.CONTACT_EC_SERVICE);
			if (ci == null)
			{
				logger.warn("[lid:{}][{}] cann't get contact rpc Client", logFlag, logFlag);
				return new Result<String>(ReturnValues.INNER_ERROR,null);
			}
			res = ci.getClient().queryNewPersons(logIndex, caller, ticket, currentPage,pageSize, deptId,queryCondition);
			if (res.res == ReturnValues.SUCCESS)
			{
				logger.info("[lid:{}][{}] result:{}, value:{}", logIndex, logFlag, res.res, res.value);
			}
			else
			{
				logger.warn("[lid:{}][{}] failed! errcode:{}", logIndex, logFlag, res.res);
			}
			rpcstubpool.returnResource(logIndex, DefaultValues.CONTACT_EC_SERVICE, ci);
			return new Result<String>(res.res,res.value);
		}
		catch (TException e)
		{
			rpcstubpool.returnBrokenResource(logIndex, DefaultValues.CONTACT_EC_SERVICE, ci);
			logger.warn("[lid:{}][{}] exception happened! detail:\n{}", logIndex, logFlag, Utils.getStackTrace(e));
			return new Result<>(ReturnValues.INNER_ERROR,null);
		}

	}
	/**
	 *
	 * 检测指定账号所属集团的通讯录更新情况
	 *
	 * @param reqId              请求索引
	 * @param caller             调用模块的标识
	 * @param ticket            账号
	 * @param deptLastUpdateId   部门最后更新标识
	 * @param personLastUpdateId 人员最后更新标识
	 * @return String
	 * @author joniers.jia
	 */
	public Result<String> updateDetect(long reqId, String caller, String ticket, long deptLastUpdateId, long personLastUpdateId)
	{
		String logFlag = "ContactRpcClient.updateDetect";
		logger.info("[lid:{}][{}] >> caller:{} ticket:{} deptLastUpdateId:{} personLastUpdateId:{}", reqId, logFlag, caller, ticket, deptLastUpdateId,
				personLastUpdateId);
		RpcClientInfo<ContactEcService.Client> ci = null;
		try
		{
			ci = getServiceClient(reqId, DefaultValues.CONTACT_EC_SERVICE);
			if (ci == null)
			{
				logger.warn("[lid:{}][{}] cann't get contact rpc Client", reqId, logFlag);
				return new Result<>(ReturnValues.INNER_ERROR,null);
			}
			ResStr res = ci.getClient().updateDetect(reqId, caller, ticket, deptLastUpdateId, personLastUpdateId);
			if (res.res == ReturnValues.SUCCESS)
			{
				logger.info("[lid:{}][{}] result:{}, value:{}", reqId, logFlag, res.res, res.value);
			}
			else
			{
				logger.warn("[lid:{}][{}] failed! errcode:{}", reqId, logFlag, res.res);
			}
			rpcstubpool.returnResource(reqId, DefaultValues.CONTACT_EC_SERVICE, ci);
			return new Result<>(res.res,res.value);
		}
		catch (TException e)
		{
			rpcstubpool.returnBrokenResource(reqId, DefaultValues.CONTACT_EC_SERVICE, ci);
			logger.warn("[lid:{}][{}] exception happened! detail:\n{}", reqId, logFlag, Utils.getStackTrace(e));
			return new Result<>(ReturnValues.INNER_ERROR,null);
		}
	}

	/**
	 * 获取指定账号所属集团的编号
	 *
	 * @param reqId   请求索引
	 * @param caller  调用模块的标识
	 * @param account account 待查询账号
	 * @return 返回指定账号所属集团的编号，失败返回null
	 * @author joniers.jia
	 */
	public Result<String> getEcCodeByAccount(long reqId, String caller, String account)
	{
		String logFlag = "ContactRpcClient.getEcCodeByAccount";
		logger.info("[lid:{}][{}] >> caller:{} account:{}", reqId, logFlag, caller, account);
		RpcClientInfo<ContactEcService.Client> ci = null;
		ResStr res = null;
		try
		{
			ci = getServiceClient(reqId, DefaultValues.CONTACT_EC_SERVICE);
			if (ci == null)
			{
				logger.warn("[lid:{}][{}] cann't get contact rpc Client", reqId, logFlag);
				return new Result<>(ReturnValues.INNER_ERROR,null);
			}
			res = ci.getClient().getEcCodeByAccount(reqId, caller, account);
			if (res.res == ReturnValues.SUCCESS)
			{
				logger.info("[lid:{}][{}] result:{}, value:{}", reqId, logFlag, res.res, res.value);
			}
			else
			{
				logger.warn("[lid:{}][{}] failed! errcode:{},value:{}", reqId, logFlag, res.res, res.value);
			}
			rpcstubpool.returnResource(reqId, DefaultValues.CONTACT_EC_SERVICE, ci);
			return new Result<>(res.res,res.value);
		}
		catch (TException e)
		{
			rpcstubpool.returnBrokenResource(reqId, DefaultValues.CONTACT_EC_SERVICE, ci);
			logger.warn("[lid:{}][{}] exception happened! detail:\n{}", reqId, logFlag, Utils.getStackTrace(e));
			return new Result<>(ReturnValues.INNER_ERROR,null);
		}
	}

	/**
	 * 查询与account在同一集团的所有账号集合
	 *
	 * @param reqId   请求索引
	 * @param caller  调用模块的标识
	 * @param account account 待查询账号
	 * @return 与account在同一集团的所有账号集合, 失败返回null
	 * @author joniers.jia
	 */
	public Set<String> queryAllSameEcAccounts(long reqId, String caller, String account)
	{
		String logFlag = "ContactRpcClient.queryAllSameEcAccounts";
		logger.info("[lid:{}][{}] >> caller:{} account:{}", reqId, logFlag, caller, account);
		RpcClientInfo<ContactEcService.Client> ci = null;
		ResSetStr res = new ResSetStr();
		try
		{
			ci = getServiceClient(reqId, DefaultValues.CONTACT_EC_SERVICE);
			if (ci == null)
			{
				logger.warn("[lid:{}][{}] cann't get contact rpc Client", reqId, logFlag);
				res.res = ReturnValues.INNER_ERROR;
				return null;
			}
			res = ci.getClient().queryAllSameEcAccounts(reqId, caller, account);
			if (res.res == ReturnValues.SUCCESS)
			{
				logger.info("[lid:{}][{}] result:{}, value:{}", reqId, logFlag, res.res, res.value);
			}
			else
			{
				logger.warn("[lid:{}][{}] failed! errcode:{},value:{}", reqId, logFlag, res.res, res.value);
			}
			rpcstubpool.returnResource(reqId, DefaultValues.CONTACT_EC_SERVICE, ci);
			return res.value;
		}
		catch (TException e)
		{
			rpcstubpool.returnBrokenResource(reqId, DefaultValues.CONTACT_EC_SERVICE, ci);
			logger.warn("[lid:{}][{}] exception happened! detail:\n{}", reqId, logFlag, Utils.getStackTrace(e));
			res.res = ReturnValues.INNER_ERROR;
			return null;
		}
	}

	/**
	 * 分批次获取指定集团需要更新的部门信息
	 *
	 * @param reqId            请求索引
	 * @param caller           调用模块的标识
	 * @param ticket           要更新的通讯录所属的集团编号
	 * @param deptLastUpdateId 部门最后更新标识
	 * @param deptSubUpdateId  部门更新截止标识
	 * @return 与account在同一集团的所有账号集合, 失败返回null
	 * @author joniers.jia
	 */
	public Result<String> queryDepts(long reqId, String caller, String ticket, long deptLastUpdateId, long deptSubUpdateId, int batchSize)
	{
		String logFlag = "ContactRpcClient.queryDepts";
		logger.info("[lid:{}][{}] >> caller:{} ticket:{} deptLastUpdateId:{} deptSubUpdateId:{} batchSize:{}", reqId, logFlag, caller, ticket, deptLastUpdateId,
				deptSubUpdateId, batchSize);
		RpcClientInfo<ContactEcService.Client> ci = null;
		ResStr res = new ResStr();
		try
		{
			ci = getServiceClient(reqId, DefaultValues.CONTACT_EC_SERVICE);
			if (ci == null)
			{
				logger.warn("[lid:{}][{}] cann't get contact rpc Client", reqId, logFlag);
				return new Result<>(ReturnValues.INNER_ERROR,null);
			}
			res = ci.getClient().queryDepts(reqId, caller, ticket, deptLastUpdateId, deptSubUpdateId, batchSize);
			if (res.res == ReturnValues.SUCCESS)
			{
				logger.info("[lid:{}][{}] result:{}, value:{}", reqId, logFlag, res.res, res.value);
			}
			else
			{
				logger.warn("[lid:{}][{}] failed! errcode:{}", reqId, logFlag, res.res);
			}
			rpcstubpool.returnResource(reqId, DefaultValues.CONTACT_EC_SERVICE, ci);
			return new Result<>(res.res,res.value);
		}
		catch (TException e)
		{
			rpcstubpool.returnBrokenResource(reqId, DefaultValues.CONTACT_EC_SERVICE, ci);
			logger.warn("[lid:{}][{}] exception happened! detail:\n{}", reqId, logFlag, Utils.getStackTrace(e));
			return new Result<>(ReturnValues.INNER_ERROR,null);
		}
	}

	/**
	 * 添加权限后的分批次获取指定集团需要更新的部门信息
	 *
	 * @param logIndex            请求索引
	 * @param caller           调用模块的标识
	 * @param ticket           要更新的通讯录所属的集团编号
	 * @param currentPage 部门最后更新标识
	 * @param pageSize  部门更新截止标识
	 * @return 与account在同一集团的所有账号集合, 失败返回null
	 * @author joniers.jia
	 */
	public Result<String>queryNewDepts(long logIndex, String caller, String ticket, int currentPage, int pageSize,boolean getAllDepts){
		String logFlag = "ContactRpcClient.queryNewDepts";
		logger.info("[lid:{}][{}], caller:{}, ticket:{},currentPage:{}, pageSize:{},",logIndex,logFlag,caller,ticket,currentPage,pageSize);
		RpcClientInfo<ContactEcService.Client> ci = null;
		ResStr res = new ResStr();
		try
		{
			ci = getServiceClient(logIndex, DefaultValues.CONTACT_EC_SERVICE);
			if (ci == null)
			{
				logger.warn("[lid:{}][{}] cann't get contact rpc Client", logIndex, logFlag);
				return new Result<String>(ReturnValues.INNER_ERROR,null);
			}
			res = ci.getClient().queryNewDepts(logIndex, caller, ticket, currentPage, pageSize,getAllDepts);
			if (res.res == ReturnValues.SUCCESS)
			{
				logger.info("[lid:{}][{}] result:{}, value:{}", logIndex, logFlag, res.res, res.value);
			}
			else
			{
				logger.warn("[lid:{}][{}] failed! errcode:{}", logIndex, logFlag, res.res);
			}
			rpcstubpool.returnResource(logIndex, DefaultValues.CONTACT_EC_SERVICE, ci);
			return new Result<String>(res.res,res.value);
		}
		catch (TException e)
		{
			rpcstubpool.returnBrokenResource(logIndex, DefaultValues.CONTACT_EC_SERVICE, ci);
			logger.warn("[lid:{}][{}] exception happened! detail:\n{}", logIndex, logFlag, Utils.getStackTrace(e));
			return new Result<String>(ReturnValues.INNER_ERROR,null);
		}
	}


	//friend部分

	/**
	 * 分页增量查询好友列表
	 *
	 * @param reqId           请求索引
	 * @param caller          调用模块的标识
	 * @param ticket         账号
	 * @param lastQuerySerial 上次查询时间
	 * @return 返回好友信息集合, 失败返回null
	 * @author joniers.jia
	 */
	public Result<String> queryFriends(long reqId, String caller, String ticket, long lastQuerySerial)
	{
		String logFlag = "ContactRpcClient.queryFriends";
		logger.info("[lid:{}][{}] >> caller:{} ticket:{} lastQuerySerial:{} ", reqId, logFlag, caller, ticket, lastQuerySerial);
		RpcClientInfo<ContactFriendService.Client> ci = null;
		try
		{
			ci = getServiceClient(reqId, DefaultValues.CONTACT_FRIEND_SERVICE);
			if (ci == null)
			{
				logger.warn("[lid:{}][{}] cann't get contact rpc Client", reqId, logFlag);
				return new Result<>(ReturnValues.INNER_ERROR,null);
			}
			ResStr res = ci.getClient().queryFriends(reqId, caller, ticket, lastQuerySerial);
			if (res.res == ReturnValues.SUCCESS)
			{
				logger.info("[lid:{}][{}] result:{}, value:{}", reqId, logFlag, res.res, res.value);
			}
			else
			{
				logger.warn("[lid:{}][{}] failed! errcode:{}", reqId, logFlag, res.res);
			}
			rpcstubpool.returnResource(reqId, DefaultValues.CONTACT_FRIEND_SERVICE, ci);
			return new Result<>(res.res,res.value);
		}
		catch (TException e)
		{
			rpcstubpool.returnBrokenResource(reqId, DefaultValues.CONTACT_FRIEND_SERVICE, ci);
			logger.warn("[lid:{}][{}] exception happened! detail:\n{}", reqId, logFlag, Utils.getStackTrace(e));
			return new Result<>(ReturnValues.INNER_ERROR,null);
		}
	}

	/**
	 * 添加好友，处理好友请求
	 *
	 * @param reqId     请求索引
	 * @param caller    调用模块的标识
	 * @param ticket    设备ID
	 * @param friendReq
	 * @return 返回处理结果  处理成功，返回好友请求ID，value为好友id， value为-3不能添加自己为好友，value为-1对方已经是您的好友
	 * @author joniers.jia
	 */
	public long friendRequest(long reqId, String caller, String ticket, String friendReq)
	{
		String logFlag = "ContactRpcClient.friendRequest";
		logger.info("[lid:{}][{}] >> caller:{} ticket:{} friendReq:{}", reqId, logFlag, caller, ticket, friendReq);
		RpcClientInfo<ContactFriendService.Client> ci = null;
		try
		{
			ci = getServiceClient(reqId, DefaultValues.CONTACT_FRIEND_SERVICE);
			if (ci == null)
			{
				logger.warn("[lid:{}][{}] cann't get contact rpc Client", reqId, logFlag);
				return ReturnValues.FRIEND_FAIL;
			}
			ResLong res = ci.getClient().friendRequest(reqId, caller, ticket, friendReq);
			if (res.res == ReturnValues.SUCCESS)
			{
				logger.info("[lid:{}][{}] result:{}, value:{}", reqId, logFlag, res.res, res.value);
			}
			else
			{
				logger.warn("[lid:{}][{}] failed! errcode:{}", reqId, logFlag, res.res);
			}
			rpcstubpool.returnResource(reqId, DefaultValues.CONTACT_FRIEND_SERVICE, ci);
			return res.value;
		}
		catch (TException e)
		{
			rpcstubpool.returnBrokenResource(reqId, DefaultValues.CONTACT_FRIEND_SERVICE, ci);
			logger.warn("[lid:{}][{}] exception happened! detail:\n{}", reqId, logFlag, Utils.getStackTrace(e));
			return ReturnValues.FRIEND_FAIL;
		}
	}

	/**
	 * 添加好友，处理好友请求
	 *
	 * @param reqId         请求索引
	 * @param caller        调用模块的标识
	 * @param ticket       账号
	 * @param friendAccount 好友账号
	 * @return 创建好友结果，成功返回好友关系id， -1-双方已经是好友 -2 没有好友请求
	 * @author joniers.jia
	 */
	public long createFriend(long reqId, String caller, String ticket, String friendAccount)
	{
		String logFlag = "ContactRpcClient.createFriend";
		logger.info("[lid:{}][{}] >> caller:{} ticket:{} friendAccount:{} ", reqId, logFlag, caller, ticket, friendAccount);
		RpcClientInfo<ContactFriendService.Client> ci = null;
		try
		{
			ci = getServiceClient(reqId, DefaultValues.CONTACT_FRIEND_SERVICE);
			if (ci == null)
			{
				logger.warn("[lid:{}][{}] cann't get contact rpc Client", reqId, logFlag);
				return ReturnValues.FRIEND_FAIL;
			}
			ResLong res = ci.getClient().createFriend(reqId, caller, ticket, friendAccount);
			if (res.res == ReturnValues.SUCCESS)
			{
				logger.info("[lid:{}][{}] result:{}, value:{}", reqId, logFlag, res.res, res.value);
			}
			else
			{
				logger.warn("[lid:{}][{}] failed! errcode:{}", reqId, logFlag, res.res);
			}
			rpcstubpool.returnResource(reqId, DefaultValues.CONTACT_FRIEND_SERVICE, ci);
			return res.value;
		}
		catch (TException e)
		{
			rpcstubpool.returnBrokenResource(reqId, DefaultValues.CONTACT_FRIEND_SERVICE, ci);
			logger.warn("[lid:{}][{}] exception happened! detail:\n{}", reqId, logFlag, Utils.getStackTrace(e));
			return ReturnValues.FRIEND_FAIL;
		}
	}

	/**
	 * 设置备注
	 *
	 * @param reqId         请求索引
	 * @param caller        调用模块的标识
	 * @param ticket       账号
	 * @param friendAccount 好友账号
	 * @param remark        备注内容
	 * @return 设置备注结果， 0-成功,-4 不是好友
	 * @author joniers.jia
	 */
	public Map<String,Object> updateRemark(long reqId, String caller, String ticket, String friendAccount, String remark)
	{
		String logFlag = "ContactRpcClient.updateRemark";
		logger.info("[lid:{}][{}] >> caller:{} ticket:{} friendAccount:{} remark:{} cardNo:{}", reqId, logFlag, caller, ticket, friendAccount, remark);
		RpcClientInfo<ContactFriendService.Client> ci = null;
		Map<String,Object> map = new HashMap<>();
		try
		{
			ci = getServiceClient(reqId, DefaultValues.CONTACT_FRIEND_SERVICE);
			if (ci == null)
			{
				logger.warn("[lid:{}][{}] cann't get contact rpc Client", reqId, logFlag);
				map.put("res",(int)ReturnValues.FRIEND_FAIL);
				return map;
			}
			ResMapStrStr res = ci.getClient().updateRemark(reqId, caller, ticket, friendAccount, remark);
			if (res.res == ReturnValues.SUCCESS)
			{
				logger.info("[lid:{}][{}] result:{}, value:{}", reqId, logFlag, res.res, res.value);
			}
			else
			{
				logger.warn("[lid:{}][{}] failed! errcode:{}", reqId, logFlag, res.res);
			}
			rpcstubpool.returnResource(reqId, DefaultValues.CONTACT_FRIEND_SERVICE, ci);
			map.put("res",res.res);
			map.put("value",res.value);
			return map;
		}
		catch (TException e)
		{
			rpcstubpool.returnBrokenResource(reqId, DefaultValues.CONTACT_FRIEND_SERVICE, ci);
			logger.warn("[lid:{}][{}] exception happened! detail:\n{}", reqId, logFlag, Utils.getStackTrace(e));
			map.put("res",(int)ReturnValues.FRIEND_FAIL);
			return map;
		}
	}

	/**
	 * 解除好友关系，软删除
	 *
	 * @param reqId         请求索引
	 * @param caller        调用模块的标识
	 * @param ticket       账号
	 * @param friendAccount 好友账号
	 * @return 操作成功返回n_id，失败返回具体的失败原因
	 * @author joniers.jia
	 */
	public long deleteFriend(long reqId, String caller, String ticket, String friendAccount)
	{
		String logFlag = "ContactRpcClient.deleteFriend";
		logger.info("[lid:{}][{}] >> caller:{} ticket:{} friendAccount:{} cardNo:{}", reqId, logFlag, caller, ticket, friendAccount);
		RpcClientInfo<ContactFriendService.Client> ci = null;
		try
		{
			ci = getServiceClient(reqId, DefaultValues.CONTACT_FRIEND_SERVICE);
			if (ci == null)
			{
				logger.warn("[lid:{}][{}] cann't get contact rpc Client", reqId, logFlag);
				return ReturnValues.FRIEND_FAIL;
			}
			ResLong res = ci.getClient().deleteFriend(reqId, caller, ticket, friendAccount);
			if (res.res == ReturnValues.SUCCESS)
			{
				logger.info("[lid:{}][{}] result:{}, value:{}", reqId, logFlag, res.res, res.value);
			}
			else
			{
				logger.warn("[lid:{}][{}] failed! errcode:{}", reqId, logFlag, res.res);
			}
			rpcstubpool.returnResource(reqId, DefaultValues.CONTACT_FRIEND_SERVICE, ci);
			return res.value;
		}
		catch (TException e)
		{
			rpcstubpool.returnBrokenResource(reqId, DefaultValues.CONTACT_FRIEND_SERVICE, ci);
			logger.warn("[lid:{}][{}] exception happened! detail:\n{}", reqId, logFlag, Utils.getStackTrace(e));
			return ReturnValues.FRIEND_FAIL;
		}
	}

	/**
	 * 增量查询好友请求列表
	 * 应用场景：所有断网联网、登录、手动刷新、每十分钟 调用
	 *
	 * @param reqId
	 * @param caller
	 * @param ticket   待查询的账号
	 * @param lastQuerySerial 更新序列
	 * @return 好友请求列表 出错返回null
	 * @author joniers.jia
	 */
	public Result<String> queryFriendReq(long reqId, String caller, String ticket, long lastQuerySerial)
	{
		String logFlag = "ContactRpcClient.queryFriendReq";
		logger.info("[lid:{}][{}] >> caller:{} ticket:{} lastQuerySerial:{}", reqId, logFlag, caller, ticket, lastQuerySerial);
		RpcClientInfo<ContactFriendService.Client> ci = null;
		try
		{
			ci = getServiceClient(reqId, DefaultValues.CONTACT_FRIEND_SERVICE);
			if (ci == null)
			{
				logger.warn("[lid:{}][{}] cann't get contact rpc Client", reqId, logFlag);
				return new Result<>(ReturnValues.INNER_ERROR,null);
			}
			ResStr res = ci.getClient().queryFriendReq(reqId, caller, ticket, lastQuerySerial);
			if (res.res == ReturnValues.SUCCESS)
			{
				logger.info("[lid:{}][{}] result:{}, value:{}", reqId, logFlag, res.res, res.value);
			}
			else
			{
				logger.warn("[lid:{}][{}] failed! errcode:{}", reqId, logFlag, res.res);
			}
			rpcstubpool.returnResource(reqId, DefaultValues.CONTACT_FRIEND_SERVICE, ci);
			return new Result<>(res.res,res.value);
		}
		catch (TException e)
		{
			rpcstubpool.returnBrokenResource(reqId, DefaultValues.CONTACT_FRIEND_SERVICE, ci);
			logger.warn("[lid:{}][{}] exception happened! detail:\n{}", reqId, logFlag, Utils.getStackTrace(e));
			return new Result<>(ReturnValues.INNER_ERROR,null);
		}
	}
	/**
	 *校验某个账号和一批账号是否好友
	 *
	 * @param reqId
	 * @param caller
	 * @param account       账号
	 * @param checkAccounts 待校验的好友账号
	 * @return 是好友的账号列表，失败返回null
	 * @author joniers.jia
	 */
	public List<String>  checkFriends(long reqId, String caller, String account, List<String> checkAccounts)
	{
		String logFlag = "ContactRpcClient.checkFriends";
		logger.info("[lid:{}][{}] >> caller:{} account:{} checkAccounts:{}", reqId, logFlag, caller, account, checkAccounts);
		RpcClientInfo<ContactFriendService.Client> ci = null;
		try
		{
			ci = getServiceClient(reqId, DefaultValues.CONTACT_FRIEND_SERVICE);
			if (ci == null)
			{
				logger.warn("[lid:{}][{}] cann't get contact rpc Client", reqId, logFlag);
				return null;
			}
			ResListStr res = ci.getClient().checkFriends(reqId, caller, account, checkAccounts);
			if (res.res == ReturnValues.SUCCESS)
			{
				logger.info("[lid:{}][{}] result:{}, value:{}", reqId, logFlag, res.res, res.value);
			}
			else
			{
				logger.warn("[lid:{}][{}] failed! errcode:{}", reqId, logFlag, res.res);
			}
			rpcstubpool.returnResource(reqId, DefaultValues.CONTACT_FRIEND_SERVICE, ci);
			return res.value;
		}
		catch (TException e)
		{
			rpcstubpool.returnBrokenResource(reqId, DefaultValues.CONTACT_FRIEND_SERVICE, ci);
			logger.warn("[lid:{}][{}] exception happened! detail:\n{}", reqId, logFlag, Utils.getStackTrace(e));
			return null;
		}
	}
	/**
	 *判断一个账号是否是另一个账号的好友
	 *
	 * @param reqId
	 * @param caller
	 * @param account       账号
	 * @param friendAccount 好友账号
	 * @return 是否好友
	 * @author joniers.jia
	 */
	public Boolean isFriend(long reqId, String caller, String account, String friendAccount)
	{
		String logFlag = "ContactRpcClient.isFriend";
		logger.info("[lid:{}][{}] >> caller:{} account:{} friendAccount:{}", reqId, logFlag, caller, account, friendAccount);
		RpcClientInfo<ContactFriendService.Client> ci = null;
		try
		{
			ci = getServiceClient(reqId, DefaultValues.CONTACT_FRIEND_SERVICE);
			if (ci == null)
			{
				logger.warn("[lid:{}][{}] cann't get contact rpc Client", reqId, logFlag);
				return false;
			}
			ResBool res = ci.getClient().isFriend(reqId, caller, account, friendAccount);
			if (res.res == ReturnValues.SUCCESS)
			{
				logger.info("[lid:{}][{}] result:{}, value:{}", reqId, logFlag, res.res, res.value);
			}
			else
			{
				logger.warn("[lid:{}][{}] failed! errcode:{}", reqId, logFlag, res.res);
			}
			rpcstubpool.returnResource(reqId, DefaultValues.CONTACT_FRIEND_SERVICE, ci);
			return res.value;
		}
		catch (TException e)
		{
			rpcstubpool.returnBrokenResource(reqId, DefaultValues.CONTACT_FRIEND_SERVICE, ci);
			logger.warn("[lid:{}][{}] exception happened! detail:\n{}", reqId, logFlag, Utils.getStackTrace(e));
			return false;
		}
	}

	/**
	 *判断一个账号是否是另一个账号的好友
	 *
	 * @param reqId
	 * @param caller
	 * @param account  待查询的账号
	 * @return 账号对应的所有好友账号集合
	 * @author joniers.jia
	 */
	public Result<Set<String>> queryAllFriends(long reqId, String caller, String account)
	{
		String logFlag = "ContactRpcClient.queryAllFriends";
		logger.info("[lid:{}][{}] >> caller:{} account:{}", reqId, logFlag, caller, account);
		RpcClientInfo<ContactFriendService.Client> ci = null;
		try
		{
			ci = getServiceClient(reqId, DefaultValues.CONTACT_FRIEND_SERVICE);
			if (ci == null)
			{
				logger.warn("[lid:{}][{}] cann't get contact rpc Client", reqId, logFlag);
				return new Result<>(ReturnValues.INNER_ERROR,null);
			}
			ResSetStr res = ci.getClient().queryAllFriends(reqId, caller, account);
			if (res.res == ReturnValues.SUCCESS)
			{
				logger.info("[lid:{}][{}] result:{}, value:{}", reqId, logFlag, res.res, res.value);
			}
			else
			{
				logger.warn("[lid:{}][{}] failed! errcode:{}", reqId, logFlag, res.res);
			}
			rpcstubpool.returnResource(reqId, DefaultValues.CONTACT_FRIEND_SERVICE, ci);
			return new Result<>(res.res,res.value);
		}
		catch (TException e)
		{
			rpcstubpool.returnBrokenResource(reqId, DefaultValues.CONTACT_FRIEND_SERVICE, ci);
			logger.warn("[lid:{}][{}] exception happened! detail:\n{}", reqId, logFlag, Utils.getStackTrace(e));
			return new Result<>(ReturnValues.INNER_ERROR,null);
		}
	}
	/**
	 * 函数名称：echo
	 * 函数功能：传递一个字符串给Thrift服务器，服务器把这个字符串原封不动的返回
	 *
	 * @param reqId
	 * @param caller
	 * @param srcStr
	 * @return String     成功返回字符串，失败返回null
	 * @author joniers.jia
	 */
	public String echo(long reqId, String caller, String srcStr)
	{
		String logFlag = "ContactRpcClient.echo";
		logger.info("[lid:{}][{}] >> caller:{} srcStr:{}", reqId, logFlag, caller, srcStr);
		RpcClientInfo<ContactFriendService.Client> ci = null;
		try
		{
			ci = getServiceClient(reqId, DefaultValues.CONTACT_FRIEND_SERVICE);
			if (ci == null)
			{
				logger.warn("[lid:{}][{}] cann't get contact rpc Client", reqId, logFlag);
				return null;
			}
			ResStr res = ci.getClient().echo(reqId, caller, srcStr);
			if (res.res == ReturnValues.SUCCESS)
			{
				logger.info("[lid:{}][{}] result:{}, value:{}", reqId, logFlag, res.res, res.value);
			}
			else
			{
				logger.warn("[lid:{}][{}] failed! errcode:{}", reqId, logFlag, res.res);
			}
			rpcstubpool.returnResource(reqId, DefaultValues.CONTACT_FRIEND_SERVICE, ci);
			return res.value;
		}
		catch (TException e)
		{
			rpcstubpool.returnBrokenResource(reqId, DefaultValues.CONTACT_FRIEND_SERVICE, ci);
			logger.warn("[lid:{}][{}] exception happened! detail:\n{}", reqId, logFlag, Utils.getStackTrace(e));
			return null;
		}
	}
	/**
	 * 创建群组
	 *
	 * @param reqId
	 * @param caller
	 * @param ticket
	 * @param createGroupRequest
	 * @return 当返回值为空时，调用id生产器出错；返回值包括为创建群组时账号不存在的，因被添加账号所在群、超过群数量限制和群组的信息
	 */
	public Result<String> createGroup(long reqId, String caller, String ticket, String createGroupRequest)
	{
		String logFlag = "ContactRpcClient.createGroup";
		//logger.info("[lid:{}][{}] >> caller:{} ticket:{} createGroupRequest:{}", reqId, logFlag, caller, ticket, createGroupRequest);
		RpcClientInfo<ContactGroupService.Client> ci = null;
		try
		{
			ci = getServiceClient(reqId, DefaultValues.CONTACT_GROUP_SERVICE);
			if (ci == null)
			{
				logger.warn("[lid:{}][{}] cann't get contact rpc Client", reqId, logFlag);
				return new Result<>(ReturnValues.INNER_ERROR,null);
			}
			ResStr res = ci.getClient().createGroup(reqId, caller, ticket, createGroupRequest);
			if (res.res == ReturnValues.SUCCESS||res.res==ReturnValues.SUC_PARTIAL)
			{
				logger.info("[lid:{}][{}] result:{}, value:{}", reqId, logFlag, res.res, res.value);
			}
			else
			{
				logger.warn("[lid:{}][{}] failed! errcode:{}", reqId, logFlag, res.res);
			}
			rpcstubpool.returnResource(reqId, DefaultValues.CONTACT_GROUP_SERVICE, ci);
			return new Result<>(res.res,res.value);
		}
		catch (TException e)
		{
			rpcstubpool.returnBrokenResource(reqId, DefaultValues.CONTACT_GROUP_SERVICE, ci);
			logger.warn("[lid:{}][{}] exception happened! detail:\n{}", reqId, logFlag, Utils.getStackTrace(e));
			return new Result<>(ReturnValues.INNER_ERROR,null);
		}
	}
	/**
	 * 查询群组信息,群信息不区分是否已删除，群成员只查询未删除状态
	 *
	 * @param reqId
	 * @param caller
	 * @param groupId  群id
	 * @return Result,里面包含errcode，value，errcode为错误码，value为群信息
	 */
	public Result<String>queryGroupDetail(long reqId, String caller, long groupId,String ticket)
	{
		String logFlag = "ContactRpcClient.queryGroupDetail";
		//logger.info("[lid:{}][{}] >> caller:{} groupId:{}", reqId, logFlag, caller, groupId);
		RpcClientInfo<ContactGroupService.Client> ci = null;
		try
		{
			ci = getServiceClient(reqId, DefaultValues.CONTACT_GROUP_SERVICE);
			if (ci == null)
			{
				logger.warn("[lid:{}][{}] cann't get contact rpc Client", reqId, logFlag);
				return new Result<>(ReturnValues.INNER_ERROR,null);
			}
			ResStr res = ci.getClient().queryGroupDetail(reqId, caller, groupId,ticket);
			if (res.res == ReturnValues.SUCCESS)
			{
				logger.info("[lid:{}][{}] result:{}, value:{}", reqId, logFlag, res.res, res.value);
			}
			else
			{
				logger.warn("[lid:{}][{}] failed! errcode:{}", reqId, logFlag, res.res);

			}
			rpcstubpool.returnResource(reqId, DefaultValues.CONTACT_GROUP_SERVICE, ci);
			return new Result<>(res.res,res.value);
		}
		catch (TException e)
		{
			rpcstubpool.returnBrokenResource(reqId, DefaultValues.CONTACT_GROUP_SERVICE, ci);
			logger.warn("[lid:{}][{}] exception happened! detail:\n{}", reqId, logFlag, Utils.getStackTrace(e));
			return new Result<>(ReturnValues.INNER_ERROR,null);
		}
	}

	/**
	 * 添加群成员无须群主确认
	 *
	 * @param reqId
	 * @param caller
	 * @param groupId     群id
	 * @param groupNum	  该参数暂时未使用，可以传递默认值0
	 * @param ticket     邀请者
	 * @param addAccounts 成员账号列表
	 *
	 */
	public Result<String> addGroupMembers(long reqId, String caller, long groupId, int groupNum, String ticket,List<String> addAccounts)
	{
		String logFlag = "ContactRpcClient.addGroupMembers";
		//logger.info("[lid:{}][{}] >> caller:{} groupId:{} , addAccounts:{} ,ticket:{}", reqId, logFlag, caller, groupId, addAccounts, ticket);
		RpcClientInfo<ContactGroupService.Client> ci = null;
		try
		{
			ci = getServiceClient(reqId, DefaultValues.CONTACT_GROUP_SERVICE);
			if (ci == null)
			{
				logger.warn("[lid:{}][{}] cann't get contact rpc Client", reqId, logFlag);
				return new Result<>(ReturnValues.INNER_ERROR,null);
			}
			ResStr res = ci.getClient().addGroupMembers(reqId, caller, groupId, groupNum, ticket, addAccounts);
			if (res.res == ReturnValues.SUCCESS||res.res == ReturnValues.ALL_FAILED)
			{
				logger.info("[lid:{}][{}] result:{}, value:{}", reqId, logFlag, res.res, res.value);
			}
			else
			{
				logger.warn("[lid:{}][{}] failed! errcode:{}", reqId, logFlag, res.res);
			}
			rpcstubpool.returnResource(reqId, DefaultValues.CONTACT_GROUP_SERVICE, ci);
			return new Result<>(res.res,res.value);
		}
		catch (TException e)
		{
			rpcstubpool.returnBrokenResource(reqId, DefaultValues.CONTACT_GROUP_SERVICE, ci);
			logger.warn("[lid:{}][{}] exception happened! detail:\n{}", reqId, logFlag, Utils.getStackTrace(e));
			return new Result<>(ReturnValues.INNER_ERROR,null);
		}
	}

	/**
	 * 需要群主确认入群
	 * @param logIndex
	 * @param caller
	 * @param groupId
	 * @param requestAccounts
	 * @param ticket
	 * @param inviterAccount
	 * @return
	 */
	public Result<String> addGroupMembersByGroupConfirm(long logIndex,String caller,String ticket,long groupId,List<String>requestAccounts,String inviterAccount)
	{
		String logFlag = "ContactRpcClient.addGroupMembersByGroupConfirm";
		//logger.info("[lid:{}][{}] >> caller:{} groupId:{} , requestAccounts:{} ,ticket:{}", logIndex, logFlag, caller, groupId, requestAccounts, ticket);
		RpcClientInfo<ContactGroupService.Client> ci = null;
		try
		{
			ci = getServiceClient(logIndex, DefaultValues.CONTACT_GROUP_SERVICE);
			if (ci == null)
			{
				logger.warn("[lid:{}][{}] cann't get contact rpc Client", logIndex, logFlag);
				return new Result<>(ReturnValues.INNER_ERROR,null);
			}
			ResStr res = ci.getClient().addGroupMembersByGroupConfirm(logIndex, caller, ticket,groupId, requestAccounts, inviterAccount);
			if (res.res == ReturnValues.SUCCESS||res.res == ReturnValues.ALL_FAILED)
			{
				logger.info("[lid:{}][{}] result:{}, value:{}", logIndex, logFlag, res.res, res.value);
			}
			else
			{
				logger.warn("[lid:{}][{}] failed! errcode:{}", logIndex, logFlag, res.res);
			}
			rpcstubpool.returnResource(logIndex, DefaultValues.CONTACT_GROUP_SERVICE, ci);
			return new Result<>(res.res,res.value);
		}
		catch (TException e)
		{
			rpcstubpool.returnBrokenResource(logIndex, DefaultValues.CONTACT_GROUP_SERVICE, ci);
			logger.warn("[lid:{}][{}] exception happened! detail:\n{}", logIndex, logFlag, Utils.getStackTrace(e));
			return new Result<>(ReturnValues.INNER_ERROR,null);
		}
	}

	/**
	 * 解散群组
	 *
	 * @param reqId
	 * @param caller
	 * @param ticket  账号
	 * @param groupId  群id
	 * @return 1:操作成功，-1:异常，-2:群组获取ID失败，-3:群不存在，-4:没有群操作权限，-5:群超限制，-6:群成员超限
	 */
	public long doDismissGroup(long reqId, String caller, String ticket, long groupId)
	{
		String logFlag = "ContactRpcClient.doDismissGroup";
		//logger.info("[lid:{}][{}] caller:{} groupId:{} ", reqId, logFlag, caller, groupId);
		RpcClientInfo<ContactGroupService.Client> ci = null;
		try
		{
			ci = getServiceClient(reqId, DefaultValues.CONTACT_GROUP_SERVICE);
			if (ci == null)
			{
				logger.warn("[lid:{}][{}] cann't get contact rpc Client", reqId, logFlag);
				return ReturnValues.INNER_ERROR;
			}
			ResLong res = ci.getClient().doDismissGroup(reqId, caller, ticket, groupId);
			if (res.res == ReturnValues.SUCCESS)
			{
				logger.info("[lid:{}][{}] result:{}, value:{}", reqId, logFlag, res.res, res.value);
			}
			else
			{
				logger.warn("[lid:{}][{}] failed! errcode:{}", reqId, logFlag, res.res);
			}
			rpcstubpool.returnResource(reqId, DefaultValues.CONTACT_GROUP_SERVICE, ci);
			return res.value;
		}
		catch (TException e)
		{
			rpcstubpool.returnBrokenResource(reqId, DefaultValues.CONTACT_GROUP_SERVICE, ci);
			logger.warn("[lid:{}][{}] exception happened! detail:\n{}", reqId, logFlag, Utils.getStackTrace(e));
			return ReturnValues.GROUP_FAILED;
		}
	}

	/**
	 * 增量更新群组信息
	 *
	 * @param reqId
	 * @param caller
	 * @param ticket      账号
	 * @param updateSerial 更新序列
	 * @return 增量群组信息
	 */
	public Result<String> queryGroups(long reqId, String caller, String ticket, long updateSerial)
	{
		String logFlag = "ContactRpcClient.queryGroups";
		//logger.info("[lid:{}][{}] >> caller:{} ticket:{} updateSerial:{}", reqId, logFlag, caller, ticket, updateSerial);
		RpcClientInfo<ContactGroupService.Client> ci = null;
		try
		{
			ci = getServiceClient(reqId, DefaultValues.CONTACT_GROUP_SERVICE);
			if (ci == null)
			{
				logger.warn("[lid:{}][{}] cann't get contact rpc Client", reqId, logFlag);
				return new Result<>(ReturnValues.INNER_ERROR,null);
			}
			ResStr res = ci.getClient().queryGroups(reqId, caller, ticket, updateSerial);
			if (res.res == ReturnValues.SUCCESS)
			{
				logger.info("[lid:{}][{}] result:{}, value:{}", reqId, logFlag, res.res, res.value);
			}
			else
			{
				logger.warn("[lid:{}][{}] failed! errcode:{}", reqId, logFlag, res.res);
			}
			rpcstubpool.returnResource(reqId, DefaultValues.CONTACT_GROUP_SERVICE, ci);
			return new Result<>(res.res,res.value);
		}
		catch (TException e)
		{
			rpcstubpool.returnBrokenResource(reqId, DefaultValues.CONTACT_GROUP_SERVICE, ci);
			logger.warn("[lid:{}][{}] exception happened! detail:\n{}", reqId, logFlag, Utils.getStackTrace(e));
			return new Result<>(ReturnValues.INNER_ERROR,null);
		}
	}
	/**
	 * 退出群组
	 *
	 * @param reqId
	 * @param caller
	 * @param groupId  群组id
	 * @return -1失败  1成功，其他返回值表示对应的错误提示
	 */
	public long quitGroup(long reqId, String caller, long groupId, String ticket)
	{
		String logFlag = "ContactRpcClient.queryGroups";
		//logger.info("[lid:{}][{}] >> caller:{} groupId:{} ticket:{}", reqId, logFlag, caller, groupId, ticket);
		RpcClientInfo<ContactGroupService.Client> ci = null;
		try
		{
			ci = getServiceClient(reqId, DefaultValues.CONTACT_GROUP_SERVICE);
			if (ci == null)
			{
				logger.warn("[lid:{}][{}] cann't get contact rpc Client", reqId, logFlag);
				return ReturnValues.INNER_ERROR;
			}
			ResLong res = ci.getClient().quitGroup(reqId, caller, groupId, ticket);
			if (res.res == ReturnValues.SUCCESS)
			{
				logger.info("[lid:{}][{}] result:{}, value:{}", reqId, logFlag, res.res, res.value);
			}
			else
			{
				logger.warn("[lid:{}][{}] failed! errcode:{}", reqId, logFlag, res.res);
			}
			rpcstubpool.returnResource(reqId, DefaultValues.CONTACT_GROUP_SERVICE, ci);
			return res.value;
		}
		catch (TException e)
		{
			rpcstubpool.returnBrokenResource(reqId, DefaultValues.CONTACT_GROUP_SERVICE, ci);
			logger.warn("[lid:{}][{}] exception happened! detail:\n{}", reqId, logFlag, Utils.getStackTrace(e));
			return ReturnValues.GROUP_FAILED;
		}
	}

	/**
	 * 移除群成员
	 *
	 * @param reqId
	 * @param caller
	 * @param ticket
	 * @param groupId        群组id
	 * @param removeAccounts 要移除的群成员账号列表
	 * @return -1失败  200成功
	 */
	public long removeGroupMembers(long reqId, String caller, String ticket, long groupId, List<String> removeAccounts)
	{
		String logFlag = "ContactRpcClient.removeGroupMembers";
		//logger.info("[lid:{}][{}] >> caller:{} ticket:{} groupId:{} removeAccounts:{} cardNo:{}", reqId, logFlag, caller, ticket, groupId, removeAccounts);
		RpcClientInfo<ContactGroupService.Client> ci = null;
		try
		{
			ci = getServiceClient(reqId, DefaultValues.CONTACT_GROUP_SERVICE);
			if (ci == null)
			{
				logger.warn("[lid:{}][{}] cann't get contact rpc Client", reqId, logFlag);
				return ReturnValues.INNER_ERROR;
			}
			ResLong res = ci.getClient().removeGroupMembers(reqId, caller, ticket, groupId, removeAccounts);
			if (res.res == ReturnValues.SUCCESS)
			{
				logger.info("[lid:{}][{}] result:{}, value:{}", reqId, logFlag, res.res, res.value);
			}
			else
			{
				logger.warn("[lid:{}][{}] failed! errcode:{}", reqId, logFlag, res.res);
			}
			rpcstubpool.returnResource(reqId, DefaultValues.CONTACT_GROUP_SERVICE, ci);
			return res.value;
		}
		catch (TException e)
		{
			rpcstubpool.returnBrokenResource(reqId, DefaultValues.CONTACT_GROUP_SERVICE, ci);
			logger.warn("[lid:{}][{}] exception happened! detail:\n{}", reqId, logFlag, Utils.getStackTrace(e));
			return ReturnValues.GROUP_FAILED;
		}
	}

	/**
	 * 修改群头像
	 *
	 * @param reqId
	 * @param caller
	 * @param ticket
	 * @param groupId
	 * @param paramMap    包含群id、头像相关信息
	 * @return   200成功
	 */
	public long modifyGroupAvatar(long reqId, String caller, String ticket, long groupId, String paramMap)
	{
		String logFlag = "ContactRpcClient.modifyGroupAvatar";
		//logger.info("[lid:{}][{}] >> caller:{} ticket:{} groupId:{} paramMap:{} ", reqId, logFlag, caller, ticket, groupId, paramMap);
		RpcClientInfo<ContactGroupService.Client> ci = null;
		try
		{
			ci = getServiceClient(reqId, DefaultValues.CONTACT_GROUP_SERVICE);
			if (ci == null)
			{
				logger.warn("[lid:{}][{}] cann't get contact rpc Client", reqId, logFlag);
				return ReturnValues.INNER_ERROR;
			}
			ResLong res = ci.getClient().modifyGroupAvatar(reqId, caller, ticket, groupId,paramMap);
			if (res.res == ReturnValues.SUCCESS)
			{
				logger.info("[lid:{}][{}] result:{}, value:{}", reqId, logFlag, res.res, res.value);
			}
			else
			{
				logger.warn("[lid:{}][{}] failed! errcode:{}", reqId, logFlag, res.res);
			}
			rpcstubpool.returnResource(reqId, DefaultValues.CONTACT_GROUP_SERVICE, ci);
			return res.res;
		}
		catch (TException e)
		{
			rpcstubpool.returnBrokenResource(reqId, DefaultValues.CONTACT_GROUP_SERVICE, ci);
			logger.warn("[lid:{}][{}] exception happened! detail:\n{}", reqId, logFlag, Utils.getStackTrace(e));
			return ReturnValues.GROUP_FAILED;
		}
	}
	/**
	 * 修改群名称
	 *
	 * @param reqId
	 * @param caller
	 * @param ticket
	 * @param groupId      群组id
	 * @param newGroupName 新的群名称
	 * @return map<groupNamePinyin, groupNamePy>
	 */
	public Result<Map<String,String>> modifyGroupName(long reqId, String caller, String ticket, long groupId, String newGroupName)
	{
		String logFlag = "ContactRpcClient.modifyGroupName";
		//logger.info("[lid:{}][{}] caller:{} ticket:{}", reqId, logFlag, caller, ticket);
		RpcClientInfo<ContactGroupService.Client> ci = null;
		try
		{
			ci = getServiceClient(reqId, DefaultValues.CONTACT_GROUP_SERVICE);
			if (ci == null)
			{
				logger.warn("[lid:{}][{}] cann't get contact rpc Client", reqId, logFlag);
				return new Result<>(ReturnValues.INNER_ERROR,null);
			}
			ResMapStrStr res = ci.getClient().modifyGroupName(reqId, caller, ticket, groupId, newGroupName);
			if (res.res == ReturnValues.SUCCESS)
			{
				logger.info("[lid:{}][{}] result:{}, value:{}", reqId, logFlag, res.res, res.value);
			}
			else
			{
				logger.warn("[lid:{}][{}] failed! errcode:{}", reqId, logFlag, res.res);
			}
			rpcstubpool.returnResource(reqId, DefaultValues.CONTACT_GROUP_SERVICE, ci);
			return new Result<>(res.res,res.value);
		}
		catch (TException e)
		{
			rpcstubpool.returnBrokenResource(reqId, DefaultValues.CONTACT_GROUP_SERVICE, ci);
			logger.warn("[lid:{}][{}] exception happened! detail:\n{}", reqId, logFlag, Utils.getStackTrace(e));
			return new Result<>(ReturnValues.INNER_ERROR,null);
		}
	}

	/**
	 * 更改群成员在群中的昵称
	 *
	 * @param reqId
	 * @param caller
	 * @param groupId     群id
	 * @param ticket     成员账号
	 * @param newNickname 新的群昵称
	 * @return 失败：null 成功返回newNickname的拼音简拼和全屏
	 */
	public Result<String> modifyGroupNickname(long reqId, String caller, long groupId, String ticket, String newNickname)
	{
		String logFlag = "ContactRpcClient.modifyGroupNickname";
		//logger.info("[lid:{}][{}] >> caller:{} ticket:{} groupId:{} newNickname:{}", reqId, logFlag, caller, ticket, groupId,newNickname);
		RpcClientInfo<ContactGroupService.Client> ci = null;
		try
		{
			ci = getServiceClient(reqId, DefaultValues.CONTACT_GROUP_SERVICE);
			if (ci == null)
			{
				logger.warn("[lid:{}][{}] cann't get contact rpc Client", reqId, logFlag);
				return new Result<>(ReturnValues.INNER_ERROR,null);
			}
			ResStr res = ci.getClient().modifyGroupNickname(reqId, caller, groupId, ticket, newNickname);
			if (res.res == ReturnValues.SUCCESS)
			{
				logger.info("[lid:{}][{}] result:{}, value:{}", reqId, logFlag, res.res, res.value);
			}
			else
			{
				logger.warn("[lid:{}][{}] failed! errcode:{}", reqId, logFlag, res.res);
			}
			rpcstubpool.returnResource(reqId, DefaultValues.CONTACT_GROUP_SERVICE, ci);
			return new Result<>(res.res,res.value);
		}
		catch (TException e)
		{
			rpcstubpool.returnBrokenResource(reqId, DefaultValues.CONTACT_GROUP_SERVICE, ci);
			logger.warn("[lid:{}][{}] exception happened! detail:\n{}", reqId, logFlag, Utils.getStackTrace(e));
			return new Result<>(ReturnValues.INNER_ERROR,null);
		}
	}

	/**
	 * 查询所有与account在同群组的成员账号
	 *
	 * @param reqId
	 * @param caller
	 * @param account  待查询账号
	 * @return 与account在同群组的成员账号集合
	 */
	public Result<Set<String>> queryAllSameGroupMembers(long reqId, String caller, String account)
	{
		String logFlag = "ContactRpcClient.queryAllSameGroupMembers";
		//logger.info("[lid:{}][{}] >> caller:{} account:{}", reqId, logFlag, caller, account);
		RpcClientInfo<ContactGroupService.Client> ci = null;
		try
		{
			ci = getServiceClient(reqId, DefaultValues.CONTACT_GROUP_SERVICE);
			if (ci == null)
			{
				logger.warn("[lid:{}][{}] cann't get contact rpc Client", reqId, logFlag);
				return new Result<>(ReturnValues.INNER_ERROR, null);
			}
			ResSetStr res = ci.getClient().queryAllSameGroupMembers(reqId, caller, account);
			if (res.res == ReturnValues.SUCCESS)
			{
				logger.info("[lid:{}][{}] result:{}, value:{}", reqId, logFlag, res.res, res.value);
			}
			else
			{
				logger.warn("[lid:{}][{}] failed! errcode:{}", reqId, logFlag, res.res);
			}
			rpcstubpool.returnResource(reqId, DefaultValues.CONTACT_GROUP_SERVICE, ci);
			return new Result<>(res.res,res.value);
		}
		catch (TException e)
		{
			rpcstubpool.returnBrokenResource(reqId, DefaultValues.CONTACT_GROUP_SERVICE, ci);
			logger.warn("[lid:{}][{}] exception happened! detail:\n{}", reqId, logFlag, Utils.getStackTrace(e));
			return new Result<>(ReturnValues.INNER_ERROR, null);
		}
	}
	/**
	 * 增量更新群中的所有成员
	 *
	 * @param reqId
	 * @param caller
	 * @param groupId
	 * @return 成员账号列表
	 */
	public Result<String> getGroupMembers(long reqId, String caller,List<Map<String, Long>> groupId,String ticket)
	{
		String logFlag = "ContactRpcClient.getGroupMembers";
		//logger.info("[lid:{}][{}] >> caller:{} ticket:{},", reqId, logFlag, caller, ticket);
		RpcClientInfo<ContactGroupService.Client> ci = null;
		try
		{
			ci = getServiceClient(reqId, DefaultValues.CONTACT_GROUP_SERVICE);
			if (ci == null)
			{
				logger.warn("[lid:{}][{}] cann't get contact rpc Client", reqId, logFlag);
				return new Result<>(ReturnValues.INNER_ERROR, null);
			}
			ResStr res = ci.getClient().getGroupMembers(reqId, caller, groupId, ticket);
			if (res.res == ReturnValues.SUCCESS)
			{
				logger.info("[lid:{}][{}] result:{}, value:{}", reqId, logFlag, res.res, res.value);
			}
			else
			{
				logger.warn("[lid:{}][{}] failed! errcode:{}", reqId, logFlag, res.res);
			}
			rpcstubpool.returnResource(reqId, DefaultValues.CONTACT_GROUP_SERVICE, ci);
			return new Result<>(res.res,res.value);

		}
		catch (TException e)
		{
			rpcstubpool.returnBrokenResource(reqId, DefaultValues.CONTACT_GROUP_SERVICE, ci);
			logger.warn("[lid:{}][{}] exception happened! detail:\n{}", reqId, logFlag, Utils.getStackTrace(e));
			return new Result<>(ReturnValues.INNER_ERROR, null);
		}
	}

	/**
	 * 获取跟该账号有关的所有联系人
	 * @param logIndex
	 * @param caller
	 * @param account
	 * @return
	 */

	public Result<Set> getAllContacts(long logIndex,String caller,String account)
	{
		String logFlag = "ContactRpcClient.getAllContacts";
		//logger.info("[lid:{}][{}] >> caller:{} account :{} ", logIndex, logFlag, caller, account);
		RpcClientInfo<ContactFriendService.Client> ci = null;
		try
		{
			ci = getServiceClient(logIndex, DefaultValues.CONTACT_FRIEND_SERVICE);
			if (ci == null)
			{
				logger.warn("[lid:{}][{}] cann't get friend rpc Client", logIndex, logFlag);
				return new Result(ReturnValues.INNER_ERROR, null);
			}
			ResSetStr res = ci.getClient().getAllContacts(logIndex,caller,account);
			if (res.res == ReturnValues.SUCCESS)
			{
				logger.info("[lid:{}][{}] result:{}, value:{}", logIndex, logFlag, res.res, res.value);
			}
			else
			{
				logger.warn("[lid:{}][{}] failed! errcode:{}", logIndex, logFlag, res.res);
			}
			rpcstubpool.returnResource(logIndex, DefaultValues.CONTACT_FRIEND_SERVICE, ci);
			return new Result(res.res,res.value);

		}
		catch (TException e)
		{
			rpcstubpool.returnBrokenResource(logIndex, DefaultValues.CONTACT_FRIEND_SERVICE, ci);
			logger.warn("[lid:{}][{}] exception happened! detail:\n{}", logIndex, logFlag, Utils.getStackTrace(e));
			return new Result(ReturnValues.INNER_ERROR,null);
		}

	}


	/**
	 * 增量更新群组信息,返回值：增量更新在群里和群解散的
	 *
	 * @param reqId
	 * @param caller
	 * @param ticket      账号
	 * @param updateSerial 更新序列
	 * @return 增量群组信息
	 */
	public Result<String> newQueryGroups(long reqId, String caller, String ticket, long updateSerial)
	{
		String logFlag = "ContactRpcClient.newQueryGroups";
		//logger.info("[lid:{}][{}] >> caller:{} ticket:{} updateSerial:{}", reqId, logFlag, caller, ticket, updateSerial);
		RpcClientInfo<ContactGroupService.Client> ci = null;
		try
		{
			ci = getServiceClient(reqId, DefaultValues.CONTACT_GROUP_SERVICE);
			if (ci == null)
			{
				logger.warn("[lid:{}][{}] cann't get contact rpc Client", reqId, logFlag);
				return new Result<>(ReturnValues.INNER_ERROR,null);
			}
			ResStr res = ci.getClient().newQueryGroups(reqId, caller, ticket, updateSerial);
			if (res.res == ReturnValues.SUCCESS)
			{
				logger.info("[lid:{}][{}] result:{}, value:{}", reqId, logFlag, res.res, res.value);
			}
			else
			{
				logger.warn("[lid:{}][{}] failed! errcode:{}", reqId, logFlag, res.res);
			}
			rpcstubpool.returnResource(reqId, DefaultValues.CONTACT_GROUP_SERVICE, ci);
			return new Result<>(res.res,res.value);
		}
		catch (TException e)
		{
			rpcstubpool.returnBrokenResource(reqId, DefaultValues.CONTACT_GROUP_SERVICE, ci);
			logger.warn("[lid:{}][{}] exception happened! detail:\n{}", reqId, logFlag, Utils.getStackTrace(e));
			return new Result<>(ReturnValues.INNER_ERROR,null);
		}
	}

	public Result<List> getGroupMembersByGroupId(long logIndex , String caller, long groupId)
	{
		String logFlag = "ContactRpcClient.getGroupMembersByGroupId";
		//logger.info("[lid:{}][{}] >> caller: {}, groupId: {}", logIndex, logFlag, caller, groupId);
		RpcClientInfo<ContactGroupService.Client> ci = null;

		try
		{
			ci = getServiceClient(logIndex, DefaultValues.CONTACT_GROUP_SERVICE);
			if (ci == null)
			{
				logger.warn("[lid:{}][{}] cann't get contact rpc Client", logIndex, logFlag);
				return new Result(ReturnValues.INNER_ERROR,null);
			}
			ResListStr res = ci.getClient().getGroupAllMembers(logIndex,caller,groupId);
			if (res.res == ReturnValues.SUCCESS)
			{
				logger.info("[lid:{}][{}] result: {}, value: {}", logIndex, logFlag, res.res, res.value);
			}
			else
			{
				logger.warn("[lid:{}][{}] failed! errcode: {}", logIndex, logFlag, res.res);
			}
			rpcstubpool.returnResource(logIndex, DefaultValues.CONTACT_GROUP_SERVICE, ci);
			return new Result(res.res,res.value);

		}
		catch (TException e)
		{
			rpcstubpool.returnBrokenResource(logIndex, DefaultValues.CONTACT_GROUP_SERVICE, ci);
			logger.warn("[lid:{}][{}] exception happened! detail:\n{}", logIndex, logFlag, Utils.getStackTrace(e));
			return new Result(ReturnValues.INNER_ERROR,null);
		}

	}

	public Result<String> getSettings(long logIndex ,String caller, String ticket)
	{
		String logFlag = "ContactRpcClient.getSettings";
		//logger.info("[lid:{}][{}] >> caller: {}", logIndex, logFlag, caller);
		RpcClientInfo<ContactGroupService.Client> ci = null;
		try
		{
			ci = getServiceClient(logIndex, DefaultValues.CONTACT_GROUP_SERVICE);
			if (ci == null)
			{
				logger.warn("[lid:{}][{}] cann't get contact rpc Client", logIndex, logFlag);
				return new Result(ReturnValues.INNER_ERROR,null);
			}
			ResStr res = ci.getClient().getSettings(logIndex,caller,ticket);
			if (res.res == ReturnValues.SUCCESS)
			{
				logger.info("[lid:{}][{}] result: {}, value: {}", logIndex, logFlag, res.res, res.value);
			}
			else
			{
				logger.warn("[lid:{}][{}] failed! errcode: {}", logIndex, logFlag, res.res);
			}
			rpcstubpool.returnResource(logIndex, DefaultValues.CONTACT_GROUP_SERVICE, ci);
			return new Result(res.res,res.value);

		}
		catch (TException e)
		{
			rpcstubpool.returnBrokenResource(logIndex, DefaultValues.CONTACT_GROUP_SERVICE, ci);
			logger.warn("[lid:{}][{}] exception happened! detail:\n{}", logIndex, logFlag, Utils.getStackTrace(e));
			return new Result(ReturnValues.INNER_ERROR,null);
		}

	}


	/**
	 * 增量更新入群请求,返回值：
	 *
	 * @param logIndex
	 * @param caller
	 * @param ticket      账号
	 * @param updateSerial 更新序列
	 * @return 增量群组信息
	 */
	public Result<String> queryGroupReqs(long logIndex, String caller, String ticket, long updateSerial)
	{
		String logFlag = "ContactRpcClient.queryGroupReqs";
		//logger.info("[lid:{}][{}] caller: {}, updateSerial: {}", logIndex, logFlag, caller, updateSerial);
		RpcClientInfo<ContactGroupService.Client> ci = null;
		try
		{
			ci = getServiceClient(logIndex, DefaultValues.CONTACT_GROUP_SERVICE);
			if (ci == null)
			{
				logger.warn("[lid:{}][{}] cann't get contact rpc Client", logIndex, logFlag);
				return new Result(ReturnValues.INNER_ERROR,null);
			}
			ResStr res = ci.getClient().queryGroupReqs(logIndex,caller,ticket,updateSerial);
			if (res.res == ReturnValues.SUCCESS)
			{
				logger.info("[lid:{}][{}] result: {}, value: {}", logIndex, logFlag, res.res, res.value);
			}
			else
			{
				logger.warn("[lid:{}][{}] failed! errcode: {}", logIndex, logFlag, res.res);
			}
			rpcstubpool.returnResource(logIndex, DefaultValues.CONTACT_GROUP_SERVICE, ci);
			return new Result(res.res,res.value);

		}
		catch (TException e)
		{
			rpcstubpool.returnBrokenResource(logIndex, DefaultValues.CONTACT_GROUP_SERVICE, ci);
			logger.warn("[lid:{}][{}] exception happened! detail:\n{}", logIndex, logFlag, Utils.getStackTrace(e));
			return new Result(ReturnValues.INNER_ERROR,null);
		}


	}

	/**
	 * 入群请求
	 *
	 * @param logIndex
	 * @param caller
	 * @param ticket      账号
	 * @param groupRequest 群请求
	 * @return 增量群组信息
	 */
	public Result<Long> groupRequest(long logIndex, String caller, String ticket,String groupRequest)
	{
		String logFlag = "ContactRpcClient.groupRequest";
		//logger.info("[lid:{}][{}] caller: {}, groupRequest: {}", logIndex, logFlag, caller, groupRequest);
		RpcClientInfo<ContactGroupService.Client> ci = null;
		try
		{
			ci = getServiceClient(logIndex, DefaultValues.CONTACT_GROUP_SERVICE);
			if (ci == null)
			{
				logger.warn("[lid:{}][{}] cann't get contact rpc Client", logIndex, logFlag);
				return new Result(ReturnValues.INNER_ERROR,null);
			}
			ResLong res = ci.getClient().groupRequest(logIndex,caller,ticket,groupRequest);
			if (res.res == ReturnValues.SUCCESS)
			{
				logger.info("[lid:{}][{}] result: {}, value: {}", logIndex, logFlag, res.res, res.value);
			}
			else
			{
				logger.warn("[lid:{}][{}] failed! errcode: {}", logIndex, logFlag, res.res);
			}
			rpcstubpool.returnResource(logIndex, DefaultValues.CONTACT_GROUP_SERVICE, ci);
			return new Result(res.res,res.value);

		}
		catch (TException e)
		{
			rpcstubpool.returnBrokenResource(logIndex, DefaultValues.CONTACT_GROUP_SERVICE, ci);
			logger.warn("[lid:{}][{}] exception happened! detail:\n{}", logIndex, logFlag, Utils.getStackTrace(e));
			return new Result(ReturnValues.INNER_ERROR,null);
		}

	}
	/**
	 * 校验二维码
	 *
	 * @param logIndex
	 * @param caller
	 * @param ticket      账号
	 * @param qrCodeInfo  二维码信息
	 * @return 增量群组信息
	 */

	public Result<Long> validateQrCode(long logIndex, String caller, String ticket,String qrCodeInfo)
	{
		String logFlag = "ContactRpcClient.validateQrCode";
		//logger.info("[lid:{}][{}] caller: {}, qrCodeInfo: {}", logIndex, logFlag, caller, qrCodeInfo);
		RpcClientInfo<ContactGroupService.Client> ci = null;
		try
		{
			ci = getServiceClient(logIndex, DefaultValues.CONTACT_GROUP_SERVICE);
			if (ci == null)
			{
				logger.warn("[lid:{}][{}] cann't get contact rpc Client", logIndex, logFlag);
				return new Result(ReturnValues.INNER_ERROR,null);
			}
			ResLong res = ci.getClient().validateQrCode(logIndex,caller,ticket,qrCodeInfo);
			if (res.res == ReturnValues.SUCCESS)
			{
				logger.info("[lid:{}][{}] result: {}, value: {}", logIndex, logFlag, res.res, res.value);
			}
			else
			{
				logger.warn("[lid:{}][{}] failed! errcode: {}", logIndex, logFlag, res.res);
			}
			rpcstubpool.returnResource(logIndex, DefaultValues.CONTACT_GROUP_SERVICE, ci);
			return new Result(res.res,res.value);

		}
		catch (TException e)
		{
			rpcstubpool.returnBrokenResource(logIndex, DefaultValues.CONTACT_GROUP_SERVICE, ci);
			logger.warn("[lid:{}][{}] exception happened! detail:\n{}", logIndex, logFlag, Utils.getStackTrace(e));
			return new Result(ReturnValues.INNER_ERROR,null);
		}

	}

	/**
	 * 获取二维码的信息
	 *
	 * @param logIndex
	 * @param caller
	 * @param ticket      账号
	 * @param groupId  群id
	 * @return 增量群组信息
	 */


	public Result<String> getQrCode(long logIndex, String caller, String ticket,long  groupId)
	{
		String logFlag = "ContactRpcClient.getQrCode";
		//logger.info("[lid:{}][{}] caller: {}, groupId: {}", logIndex, logFlag, caller, groupId);
		RpcClientInfo<ContactGroupService.Client> ci = null;
		try
		{
			ci = getServiceClient(logIndex, DefaultValues.CONTACT_GROUP_SERVICE);
			if (ci == null)
			{
				logger.warn("[lid:{}][{}] cann't get contact rpc Client", logIndex, logFlag);
				return new Result(ReturnValues.INNER_ERROR,null);
			}
			ResStr res = ci.getClient().getQrCode(logIndex,caller,ticket,groupId);
			if (res.res == ReturnValues.SUCCESS)
			{
				logger.info("[lid:{}][{}] result: {}, value: {}", logIndex, logFlag, res.res, res.value);
			}
			else
			{
				logger.warn("[lid:{}][{}] failed! errcode: {}", logIndex, logFlag, res.res);
			}
			rpcstubpool.returnResource(logIndex, DefaultValues.CONTACT_GROUP_SERVICE, ci);
			return new Result(res.res,res.value);

		}
		catch (TException e)
		{
			rpcstubpool.returnBrokenResource(logIndex, DefaultValues.CONTACT_GROUP_SERVICE, ci);
			logger.warn("[lid:{}][{}] exception happened! detail:\n{}", logIndex, logFlag, Utils.getStackTrace(e));
			return new Result(ReturnValues.INNER_ERROR,null);
		}

	}

	/**
 * 群主转移 是rpc调用 提供给OA服务用
 * @param logIndex
 * @param caller
 * @param groupOwner
 * @param authorizedAccount
 * @param groupId
 * @return
 */

public Result<Long>transferGroupOwner(long logIndex, String caller, String groupOwner, String authorizedAccount,long groupId)
{
	String logFlag = "ContactRpcClient.transferGroupOwner";
	//logger.info("[lid:{}][{}] beginning transfer groupOwner! caller:{},groupOwner:{},authorizedAccount:{},groupId:{}",logIndex,logFlag,caller,groupOwner,authorizedAccount,groupId);

	RpcClientInfo<ContactGroupService.Client> ci = null;
	try
	{
		ci = getServiceClient(logIndex, DefaultValues.CONTACT_GROUP_SERVICE);
		if (ci == null)
		{
			logger.warn("[lid:{}][{}] cann't get contact rpc Client", logIndex, logFlag);
			return new Result(ReturnValues.INNER_ERROR,null);
		}
		ResLong res = ci.getClient().transferGroupOwner(logIndex,caller,groupOwner,authorizedAccount,groupId);
		if (res.res == ReturnValues.SUCCESS)
		{
			logger.info("[lid:{}][{}] result: {}, value: {}", logIndex, logFlag, res.res, res.value);
		}
		else
		{
			logger.warn("[lid:{}][{}] failed! errcode: {}", logIndex, logFlag, res.res);
		}
		rpcstubpool.returnResource(logIndex, DefaultValues.CONTACT_GROUP_SERVICE, ci);
		return new Result(res.res,res.value);

	}
	catch (TException e)
	{
		rpcstubpool.returnBrokenResource(logIndex, DefaultValues.CONTACT_GROUP_SERVICE, ci);
		logger.warn("[lid:{}][{}] exception happened! detail:\n{}", logIndex, logFlag, Utils.getStackTrace(e));
		return new Result(ReturnValues.INNER_ERROR,null);
	}
}
	/**
	 * 提供http群主转移
	 * @param logIndex
	 * @param caller
	 * @param ticket
	 * @param authorizedAccount
	 * @param groupId
	 * @return
	 */

	public Result<Long>transferGroupOwnerByTicket(long logIndex,String caller, String ticket, String authorizedAccount, long groupId)
	{
		String logFlag = "ContactRpcClient.transferGroupOwner";
		//logger.info("[lid:{}][{}] beginning transfer groupOwner! caller:{},ticket:{},authorizedAccount:{},groupId:{}",logIndex,logFlag,caller,ticket,authorizedAccount,groupId);

		RpcClientInfo<ContactGroupService.Client> ci = null;
		try
		{
			ci = getServiceClient(logIndex, DefaultValues.CONTACT_GROUP_SERVICE);
			if (ci == null)
			{
				logger.warn("[lid:{}][{}] cann't get contact rpc Client", logIndex, logFlag);
				return new Result(ReturnValues.INNER_ERROR,null);
			}
			ResLong res = ci.getClient().transferGroupOwnerByTicket(logIndex,caller,ticket,authorizedAccount,groupId);
			if (res.res == ReturnValues.SUCCESS)
			{
				logger.info("[lid:{}][{}] result: {}, value: {}", logIndex, logFlag, res.res, res.value);
			}
			else
			{
				logger.warn("[lid:{}][{}] failed! errcode: {}", logIndex, logFlag, res.res);
			}
			rpcstubpool.returnResource(logIndex, DefaultValues.CONTACT_GROUP_SERVICE, ci);
			return new Result(res.res,res.value);

		}
		catch (TException e)
		{
			rpcstubpool.returnBrokenResource(logIndex, DefaultValues.CONTACT_GROUP_SERVICE, ci);
			logger.warn("[lid:{}][{}] exception happened! detail:\n{}", logIndex, logFlag, Utils.getStackTrace(e));
			return new Result(ReturnValues.INNER_ERROR,null);
		}
	}

	/**
 * 更新群主确认功能
 * @param logIndex
 * @param caller
 * @param ticket
 * @param openFlag
 * @param groupId
 * @return
 */

public Result<Long> updateGroupOwnerConfirm(long logIndex, String caller, String ticket, long groupId, int openFlag)
{
	String logFlag = "ContactRpcClient.updateGroupOwnerConfirm";
	//logger.info("[lid:{}][{}] beginning transfer groupOwner! caller:{},ticket:{},authorizedAccount:{},groupId:{}",logIndex,logFlag,caller,ticket,authorizedAccount,groupId);

	RpcClientInfo<ContactGroupService.Client> ci = null;
	try
	{
		ci = getServiceClient(logIndex, DefaultValues.CONTACT_GROUP_SERVICE);
		if (ci == null)
		{
			logger.warn("[lid:{}][{}] cann't get contact rpc Client", logIndex, logFlag);
			return new Result(ReturnValues.INNER_ERROR,null);
		}
		ResLong res = ci.getClient().updateGroupOwnerConfirm(logIndex,caller,ticket,groupId,openFlag);
		if (res.res == ReturnValues.SUCCESS)
		{
			logger.info("[lid:{}][{}] result: {}, value: {}", logIndex, logFlag, res.res, res.value);
		}
		else
		{
			logger.warn("[lid:{}][{}] failed! errcode: {}", logIndex, logFlag, res.res);
		}
		rpcstubpool.returnResource(logIndex, DefaultValues.CONTACT_GROUP_SERVICE, ci);
		return new Result(res.res,res.value);

	}
	catch (TException e)
	{
		rpcstubpool.returnBrokenResource(logIndex, DefaultValues.CONTACT_GROUP_SERVICE, ci);
		logger.warn("[lid:{}][{}] exception happened! detail:\n{}", logIndex, logFlag, Utils.getStackTrace(e));
		return new Result(ReturnValues.INNER_ERROR,null);
	}
}

	/**
	 * 查询群主确认功能是否开启
	 * @param logIndex
	 * @param caller
	 * @param ticket
	 * @param groupId
	 * @return
	 */

	public Result<Integer> checkGroupOwnerConfirm(long logIndex, String caller, String ticket, long groupId)
	{
		String logFlag = "ContactRpcClient.checkGroupOwnerConfirm";
		//logger.info("[lid:{}][{}] beginning transfer groupOwner! caller:{},ticket:{},authorizedAccount:{},groupId:{}",logIndex,logFlag,caller,ticket,authorizedAccount,groupId);

		RpcClientInfo<ContactGroupService.Client> ci = null;
		try
		{
			ci = getServiceClient(logIndex, DefaultValues.CONTACT_GROUP_SERVICE);
			if (ci == null)
			{
				logger.warn("[lid:{}][{}] cann't get contact rpc Client", logIndex, logFlag);
				return new Result(ReturnValues.INNER_ERROR,null);
			}
			ResInt res = ci.getClient().checkGroupOwnerConfirm(logIndex,caller,ticket,groupId);
			if (res.res == ReturnValues.SUCCESS)
			{
				logger.info("[lid:{}][{}] result: {}, value: {}", logIndex, logFlag, res.res, res.value);
			}
			else
			{
				logger.warn("[lid:{}][{}] failed! errcode: {}", logIndex, logFlag, res.res);
			}
			rpcstubpool.returnResource(logIndex, DefaultValues.CONTACT_GROUP_SERVICE, ci);
			return new Result(res.res,res.value);

		}
		catch (TException e)
		{
			rpcstubpool.returnBrokenResource(logIndex, DefaultValues.CONTACT_GROUP_SERVICE, ci);
			logger.warn("[lid:{}][{}] exception happened! detail:\n{}", logIndex, logFlag, Utils.getStackTrace(e));
			return new Result(ReturnValues.INNER_ERROR,null);
		}
	}


	/**
	 *
	 * 修改集团通信录的手机号
	 * @param logIndex
	 * @param caller
	 * @param ticket 账号
	 * @param oldPhone 老的手机号
	 * @param newPhone 新的手机号
	 * @param ecCode 集团通信录码
	 * @return res中返回操作结果，value中返回指定账号所属集团的编号
	 */


	public Result<String> updatePhoneOfAccount(long logIndex, String caller, String ticket, int ecCode, String oldPhone, String newPhone, String ext) {
		String logFlag = "ContactRpcClient.updatePhoneOfAccount";
		//logger.info("[lid:{}][{}] beginning transfer groupOwner! caller:{},ticket:{},authorizedAccount:{},groupId:{}",logIndex,logFlag,caller,ticket,authorizedAccount,groupId);

		RpcClientInfo<ContactEcService.Client> ci = null;
		try
		{
			ci = getServiceClient(logIndex, DefaultValues.CONTACT_EC_SERVICE);
			if (ci == null)
			{
				logger.warn("[lid:{}][{}] cann't get contact rpc Client", logIndex, logFlag);
				return new Result(ReturnValues.INNER_ERROR,null);
			}
			ResStr res = ci.getClient().updatePhoneOfAccount(logIndex,caller,ticket,ecCode,oldPhone,newPhone,ext);
			if (res.res == ReturnValues.SUCCESS)
			{
				logger.info("[lid:{}][{}] result: {}, value: {}", logIndex, logFlag, res.res, res.value);
			}
			else
			{
				logger.warn("[lid:{}][{}] failed! errcode: {}", logIndex, logFlag, res.res);
			}
			rpcstubpool.returnResource(logIndex, DefaultValues.CONTACT_EC_SERVICE, ci);
			return new Result(res.res,res.value);

		}
		catch (TException e)
		{
			rpcstubpool.returnBrokenResource(logIndex, DefaultValues.CONTACT_EC_SERVICE, ci);
			logger.warn("[lid:{}][{}] exception happened! detail:\n{}", logIndex, logFlag, Utils.getStackTrace(e));
			return new Result(ReturnValues.INNER_ERROR,null);
		}

	}


	/**
	 * 根据ticket查询该用户的详细信息
	 * @param logIndex 	logId
	 * @param caller	调用方标识
	 * @param ticket	ticket
	 * @return			返回详情信息
	 */
	public Result<String> getUserInfo(long logIndex, String caller, String ticket, String requestAccount) {
		String logFlag = "ContactRpcClient.getUserInfo";
		RpcClientInfo<ContactEcService.Client> ci = null;
		try {
			ci = getServiceClient(logIndex, DefaultValues.CONTACT_EC_SERVICE);
			if (ci == null)
			{
				logger.warn("[lid:{}][{}] cann't get contact rpc Client", logIndex, logFlag);
				return new Result(ReturnValues.INNER_ERROR,null);
			}
			ResStr res = ci.getClient().queryUserInfo(logIndex,caller,ticket,requestAccount);
			if (res.res == ReturnValues.SUCCESS)
			{
				logger.info("[lid:{}][{}] result: {}, value: {}", logIndex, logFlag, res.res, res.value);
			}
			else
			{
				logger.warn("[lid:{}][{}] failed! errcode: {}", logIndex, logFlag, res.res);
			}
			rpcstubpool.returnResource(logIndex, DefaultValues.CONTACT_EC_SERVICE, ci);
			return new Result(res.res,res.value);

		}
		catch (TException e)
		{
			rpcstubpool.returnBrokenResource(logIndex, DefaultValues.CONTACT_EC_SERVICE, ci);
			logger.warn("[lid:{}][{}] exception happened! detail:\n{}", logIndex, logFlag, Utils.getStackTrace(e));
			return new Result(ReturnValues.INNER_ERROR,null);
		}

	}

	/**
	 * 获得集团数据最后变更时间
	 * @param logIndex 	logId
	 * @param caller	调用方标识
	 * @param ticket	ticket
	 * @return			/
	 */
	public Result<String> queryUpdateTime(long logIndex, String caller, String ticket) {
		String logFlag = "ContactRpcClient.queryUpdateTime";
		RpcClientInfo<ContactEcService.Client> ci = null;
		try {
			ci = getServiceClient(logIndex, DefaultValues.CONTACT_EC_SERVICE);
			if (ci == null){
				logger.warn("[lid:{}][{}] cann't get contact rpc Client", logIndex, logFlag);
				return new Result(ReturnValues.INNER_ERROR,null);
			}
			ResStr res = ci.getClient().queryUpdateTime(logIndex,caller,ticket);
			if (res.res == ReturnValues.SUCCESS){
				logger.info("[lid:{}][{}] result: {}, value: {}", logIndex, logFlag, res.res, res.value);
			}else{
				logger.warn("[lid:{}][{}] failed! errcode: {}", logIndex, logFlag, res.res);
			}
			rpcstubpool.returnResource(logIndex, DefaultValues.CONTACT_EC_SERVICE, ci);
			return new Result(res.res,res.value);
		}
		catch (TException e){
			rpcstubpool.returnBrokenResource(logIndex, DefaultValues.CONTACT_EC_SERVICE, ci);
			logger.warn("[lid:{}][{}] exception happened! detail:\n{}", logIndex, logFlag, Utils.getStackTrace(e));
			return new Result(ReturnValues.INNER_ERROR,null);
		}
	}


	/**
	 * 创建全员群
	 * @param logIndex		日志ID
	 * @param caller		调用者标识
	 * @param ticket		ticket
	 * @param createAllGroupRequest	创建全员群参数
	 * @return /
	 */
	public Result<String> createAllGroup(long logIndex, String caller, String ticket, String createAllGroupRequest){
		String logFlag = "ContactRpcClient.createAllGroup";
		RpcClientInfo<ContactGroupService.Client> ci = null;
		try{
			ci = getServiceClient(logIndex, DefaultValues.CONTACT_GROUP_SERVICE);
			if (ci == null){
				logger.warn("[lid:{}][{}] cann't get contact rpc Client", logIndex, logFlag);
				return new Result<>(ReturnValues.INNER_ERROR,null);
			}
			ResStr res = ci.getClient().createAllGroup(logIndex, caller, ticket, createAllGroupRequest);
			if (res.res == ReturnValues.SUCCESS||res.res==ReturnValues.SUC_PARTIAL){
				logger.info("[lid:{}][{}] result:{}, value:{}", logIndex, logFlag, res.res, res.value);
			}else{
				logger.warn("[lid:{}][{}] failed! errcode:{}", logIndex, logFlag, res.res);
			}
			rpcstubpool.returnResource(logIndex, DefaultValues.CONTACT_GROUP_SERVICE, ci);
			return new Result<>(res.res,res.value);
		}catch (TException e){
			rpcstubpool.returnBrokenResource(logIndex, DefaultValues.CONTACT_GROUP_SERVICE, ci);
			logger.warn("[lid:{}][{}] exception happened! detail:\n{}", logIndex, logFlag, Utils.getStackTrace(e));
			return new Result<>(ReturnValues.INNER_ERROR,null);
		}
	}


	/**
	 * 全员群添加成员
	 * @param logIndex		日志ID
	 * @param caller		调用者标识
	 * @param ticket		ticket
	 * @param addAccounts	成员信息列表
	 * @return /
	 */
	public Result<String> addAtecsGroupMembers(long logIndex, String caller, long groupId, String ticket, String addAccounts){
		String logFlag = "ContactRpcClient.addAtecsGroupMembers";
		RpcClientInfo<ContactGroupService.Client> ci = null;
		try{
			ci = getServiceClient(logIndex, DefaultValues.CONTACT_GROUP_SERVICE);
			if (ci == null){
				logger.warn("[lid:{}][{}] cann't get contact rpc Client", logIndex, logFlag);
				return new Result<>(ReturnValues.INNER_ERROR,null);
			}
			ResStr res = ci.getClient().addAtecsGroupMembers(logIndex, caller, groupId, ticket, addAccounts);
			if (res.res == ReturnValues.SUCCESS){
				logger.info("[lid:{}][{}] result:{}, value:{}", logIndex, logFlag, res.res, res.value);
			}else{
				logger.warn("[lid:{}][{}] failed! errcode:{}", logIndex, logFlag, res.res);
			}
			rpcstubpool.returnResource(logIndex, DefaultValues.CONTACT_GROUP_SERVICE, ci);
			return new Result<>(res.res,res.value);
		}catch (TException e){
			rpcstubpool.returnBrokenResource(logIndex, DefaultValues.CONTACT_GROUP_SERVICE, ci);
			logger.warn("[lid:{}][{}] exception happened! detail:\n{}", logIndex, logFlag, Utils.getStackTrace(e));
			return new Result<>(ReturnValues.INNER_ERROR,null);
		}
	}

	/**
	 * 修改全员群成员在群里的昵称
	 * @param logIndex		日志ID
	 * @param caller		调用者标识
	 * @param ticket		ticket
	 * @param groupId		群ID
	 * @param memberAccount	成员账号
	 * @param newNickname	成员昵称
	 * @return	/
	 */
	public Result<String> modifyAllGroupNickname(long logIndex, String caller, String ticket, long groupId, String memberAccount, String newNickname){
		String logFlag = "ContactRpcClient.modifyAllGroupNickname";
		RpcClientInfo<ContactGroupService.Client> ci = null;
		try{
			ci = getServiceClient(logIndex, DefaultValues.CONTACT_GROUP_SERVICE);
			if (ci == null){
				logger.warn("[lid:{}][{}] cann't get contact rpc Client", logIndex, logFlag);
				return new Result<>(ReturnValues.INNER_ERROR,null);
			}
			ResStr res = ci.getClient().modifyAllGroupNickname(logIndex, caller, ticket, groupId, memberAccount, newNickname);
			if (res.res == ReturnValues.SUCCESS){
				logger.info("[lid:{}][{}] result:{}, value:{}", logIndex, logFlag, res.res, res.value);
			}else{
				logger.warn("[lid:{}][{}] failed! errcode:{}", logIndex, logFlag, res.res);
			}
			rpcstubpool.returnResource(logIndex, DefaultValues.CONTACT_GROUP_SERVICE, ci);
			return new Result<>(res.res,res.value);
		}catch (TException e){
			rpcstubpool.returnBrokenResource(logIndex, DefaultValues.CONTACT_GROUP_SERVICE, ci);
			logger.warn("[lid:{}][{}] exception happened! detail:\n{}", logIndex, logFlag, Utils.getStackTrace(e));
			return new Result<>(ReturnValues.INNER_ERROR,null);
		}
	}

	/**
	 * ContactClient关闭
	 *
	 * @param reqId 请求索引
	 */
	public void shutDown(long reqId)
	{
		String logFlag = "ContactRpcClient.shutDown";
		if (rpcstubpool == null) {
			return;
		}
		try
		{
			rpcstubpool.destroy(reqId);
			rpcstubpool = null;
		}
		catch (Exception ex)
		{
			logger.error("[lid:{}][{}]exception happened while shutting down rpcstubpool...", reqId, logFlag);
		}
	}


}
