/**
 * 信大捷安写卡控件的js封装类-基于websocket的读卡控件
 * @author 马德成
 * @date 2015-11-20
 *
 * 说明:
 *   1.该文件被引入页面后,会自动建立websocket连接,且仅初始化一次
 *   2.每次接收的消息会自动匹配发送的消息,自动注册回调列表
 *   3.该组件全部操作均为异步方式,所以都需要在回调函数中处理业务
 *   5.在回调函数中都会有success属性,当success=false时,对象会有code和message属性
 */
(function (root) {

  //常量定义
  var Const = {
    //控件最低版本号
    lowestVersion: '4.0.1.3687',

    //控件当前版本号
    version: '4.0.1.3687',

    //初始化控件常量定义
    readyConst: {
      '0': '未安装安全控件',
      '3': '安全控件版本过低',
      '4': '安全控件版本有更新',
      '5': '连接读卡服务错误',
      '6': '不支持的系统类型',
      '7': '不支持的cpu类型',
      '8': '不支持的浏览器类型',
      '9': '不支持的浏览版本',
      '10': '不支持的浏览器内核',

      //控件ID
      pluginId: 'mdc-card-pluginXdja',
      getEmbed: function () {
        return '<embed width="0" style="margin:0;padding:0;position: fixed; filter: alpha(opacity=0);" height="0" type="application/NPCardCert" id="' + this.pluginId + '" />'
      }
    },

    //错误码描述定义
    errorDesc: {
      '0': '成功', '-19': 'SM4运算失败', '-12': '证书格式错误', '-203': '容器不存在',
      '-1': '卡未插入', '-20': '重置PIN码失败', '-13': '导入证书失败', '-204': '容器证书内容错误',
      '-2': '参数错误', '-21': '创建文件失败', '-14': 'BASE64解码失败', '-100': '不支持此功能',
      '-3': '未插入管理卡', '-22': 'SM3运算失败', '-15': '解析pfx失败', '-27': '激活卡失败',
      '-4': '卡锁死', '-23': 'SM1运算失败', '-16': '导入pfx失败', '-29': '未激活',
      '-5': '口令错误', '-24': 'PSA私钥运算失败', '-17': '读卡文件失败', '-31': '手机驱动安装失败',
      '-6': '卡内产生公私钥对失败', '-25': '获取设备信息失败', '-18': '导入密钥失败', '-33': '获取设备号失败',
      '-7': '新建容器失败', '-61': '解析网关证书失败', '-26': '产生随机数失败', '-35': '导入公钥失败',
      '-8': '容器不存在', '-62': '签名失败', '-28': '获取卡激活状态失败', '-37': '公钥运算失败',
      '-9': '绑定用户卡失败', '-63': '取消签名失败', '-30': 'ADB 端口被占用', '-39': 'SM2运算失败',
      '-10': '文件不存在', '-99': '内部错误', '-36': '导入私钥失败', '-38': '私钥运算失败',
      '-11': '写文件失败', '-101': '内存申请失败', '-32': '卸载安通助手失败', '-34': '开启安通助手失败',
      '-43': '解锁加密盘失败'
    },

    //写卡控件中，卡用途
    cardUsage: {
      X_CARD_USAGE_ADMIN: 0, //0-管理卡
      X_CARD_USAGE_USER: 1, //1-用户卡
      X_CARD_USAGE_All: -1 //所有
    }
  };

  /**
   * 读卡控件的入口类
   * @class Card
   * @main Card
   */
  (!root.Card) && (root.Card = {});
  //常量定义
  root.Card.Const = root.Card.Const || Const;
  root.Card.getBytesLength = getBytesLength;

  /*
   * 字符编码数值对应的存储长度：
   * UCS-2编码(16进制) UTF-8 字节流(二进制)
   * 0000 - 007F       0xxxxxxx （1字节）
   * 0080 - 07FF       110xxxxx 10xxxxxx （2字节）
   * 0800 - FFFF       1110xxxx 10xxxxxx 10xxxxxx （3字节）
   * @param {String} str 任意非空字符串
   * @return {Number} 字节长度,空字符串返回0
   */
  function getBytesLength(str) {
    if (!str) return 0;

    var length = 0;
    var code;

    for (var i = str.length - 1; i >= 0; i--) {
      code = str.charCodeAt(i);
      if (code < 0x007f) {
        length++;
      } else if ((0x0080 <= code) && (code <= 0x07ff)) {
        length += 2;
      } else if ((0x0800 <= code) && (code <= 0xffff)) {
        length += 3;
      } else {
        length += 4;
      }
    }
    return length;
  }

  var socket, wsSocket, ws = {}, noop = function () {
  };
  if (('MozWebSocket' in window)) {
    wsSocket = window.MozWebSocket;
  } else if (('WebSocket' in window)) {
    wsSocket = window.WebSocket;
  }

  if (wsSocket) {
    //创建连接, 一个页面只能连接一次
    ws.connect = function (success, error) {
      // console.log("new: " + new Date());
      socket = new wsSocket('ws://localhost:8000/xdja/server');
      var time1 = window.setTimeout(function () {
        socket.close();
      }, 5000);
      success = success || noop;
      error = error || noop;

      //打开连接时执行成功函数
      socket.onopen = function (evt) {
        // console.log("open: " + new Date());
        window.clearTimeout(time1);
        success.call(this, new WsCardWrap(), evt);
      };

      //监听socket关闭和错误事件
      socket.onerror = function (evt) {
        // console.log("error: " + new Date());
        error.call(this, evt);
      };

      //接收消息
      socket.onmessage = function (evt) {

        //OA专用
        //自己在页面写一个名为insertNewCard的插卡监听
        if (evt.data == "Device Arrival") {
          //插入了新卡
          try {
            if (insertNewCard && typeof insertNewCard == "function") {
              //有插入新卡的方法
              insertNewCard();
            }
          } catch (e) {
          }
        } else {
          //自己在页面写一个名为pullOldCard的拔卡监听，无特定卡限制
          try {
            if (evt.data == "Device Remove") {
              //拔掉了卡
              if (pullOldCard && typeof pullOldCard == "function") {
                //拔掉了一个卡
                pullOldCard();
              }
            }
          } catch (e) {
          }
          var id = idMapping[evt.data];
          if (id) return EvtObserver(id).trigger();
        }

        var data;
        try {
          data = JSON.parse(evt.data);
        } catch (e) {
          console.log('exception==>' + evt.data + '----' + e);
        }

        if (data && data.id) { //发布消息广播
          var ret = data.state ? data.result : {
            error: {
              code: data.result,
              message: Const.errorDesc[data.result] || '未知错误',
              success: data.state
            }
          };
          EvtObserver(data.id).trigger(ret);
        }
      };


    };

    //关闭连接
    ws.close = function () {
      if (socket) socket.close();
    };

    /**
     * 获取版本号
     * @param fn 回调函数
     */
    ws.getVersion = function (fn) {
      send('Test', fn);
    };

    Card.__socket = ws;
  } else {
    return;
  }

  //当页面被刷新或关闭的时候关闭连接
  root.onclose = ws.close;

  //观察者对象
  var EvtObserver = (function () {
    var b = {};
    for (var c = 0, e = "Boolean Number String Function Array Date RegExp Object".split(" "), a = e.length; c < a; c++) {
      b["[object " + e[c] + "]"] = e[c].toLowerCase()
    }
    var d = function (l) {
      var k = {};
      var h = function (u) {
        var v = k[u] = {}, w, x;
        u = u.split(/\s+/);
        for (w = 0, x = u.length; w < x; w++) {
          v[u[w]] = true
        }
        return v
      };
      l = l ? (k[l] || h(l)) : {};
      var q = [], r = [], m, g, n, j, o, p, t = function (v) {
        var w, y, x, u, z;
        for (w = 0, y = v.length; w < y; w++) {
          x = v[w];
          u = x == null ? String(x) : b[Object.prototype.toString.call(x)] || "object";
          if (u === "array") {
            t(x)
          } else {
            if (u === "function") {
              if (!l.unique || !s.has(x)) {
                q.push(x)
              }
            }
          }
        }
      }, i = function (v, u) {
        u = u || [];
        m = !l.memory || [v, u];
        g = true;
        n = true;
        p = j || 0;
        j = 0;
        o = q.length;
        for (; q && p < o; p++) {
          if (q[p].apply(v, u) === false && l.stopOnFalse) {
            m = true;
            break
          }
        }
        n = false;
        if (q) {
          if (!l.once) {
            if (r && r.length) {
              m = r.shift();
              s.fireWith(m[0], m[1])
            }
          } else {
            if (m === true) {
              s.disable()
            } else {
              q = []
            }
          }
        }
      }, s = {
        add: function () {
          if (q) {
            var u = q.length;
            t(arguments);
            if (n) {
              o = q.length
            } else {
              if (m && m !== true) {
                j = u;
                i(m[0], m[1])
              }
            }
          }
          return this
        }, remove: function () {
          if (q) {
            var u = arguments, w = 0, x = u.length;
            for (; w < x; w++) {
              for (var v = 0; v < q.length; v++) {
                if (u[w] === q[v]) {
                  if (n) {
                    if (v <= o) {
                      o--;
                      if (v <= p) {
                        p--
                      }
                    }
                  }
                  q.splice(v--, 1);
                  if (l.unique) {
                    break
                  }
                }
              }
            }
          }
          return this
        }, has: function (v) {
          if (q) {
            var u = 0, w = q.length;
            for (; u < w; u++) {
              if (v === q[u]) {
                return true
              }
            }
          }
          return false
        }, empty: function () {
          q = [];
          return this
        }, disable: function () {
          q = r = m = undefined;
          return this
        }, disabled: function () {
          return !q
        }, lock: function () {
          r = undefined;
          if (!m || m === true) {
            s.disable()
          }
          return this
        }, locked: function () {
          return !r
        }, fireWith: function (v, u) {
          if (r) {
            if (n) {
              if (!l.once) {
                r.push([v, u])
              }
            } else {
              if (!(l.once && m)) {
                i(v, u)
              }
            }
          }
          return this
        }, fire: function () {
          s.fireWith(this, arguments);
          return this
        }, fired: function () {
          return !!g
        }
      };
      return s
    };
    var f = {};
    return function (h) {
      var g = f[h];
      if (!g) {
        g = d("once");
        g.trigger = function (i) {
          g.fire(i);
          delete f[h]
        };
        f[h] = g
      }
      return g
    }
  })();
  //生成guid
  function guid() {
    return 'xxxxxxxxxxxx4xxxyxxxxxxxxxxxxxxx'.replace(/[xy]/g, function (c) {
      var r = Math.random() * 16 | 0, v = c == 'x' ? r : (r & 0x3 | 0x8);
      return v.toString(16);
    });
  }

  //插卡和拔卡的消息和id映射
  var idMapping = {
    'Device Arrival': '00000000000000000000000000000001',
    'Device Remove': '00000000000000000000000000000002'
  };

  /**
   * 发送消息
   * @param method 调用的函数
   * @param args 函数参数
   * @param fn 函数的回调
   * @returns {boolean}
   */
  function send(method, args, fn) {
    if (socket.readyState != 1) return false;
    fn = fn || noop;

    var argv = {method: method, id: guid()};
    typeof args === 'function' ? fn = args : argv.param = args;
    //console.log('send==>' + JSON.stringify(argv));
    socket.send(JSON.stringify(argv));
    EvtObserver(argv.id).add(fn);
    return true;
  }

  /*----------以上是websocket的操作-------------*/

  //处理pin码错误
  function getPinResult(name, parame, context, callback) {
    if (typeof callback != 'function') callback = noop;
    var lockError = {'-10': true, '-16': true, '-4': true, '-5': true};
    send(name, parame, function (data) {
      if (data.error) {
        data = data.error.code;
      }
      data = data - 0;
      var ret;

      if (name == 'GetPinTryCount') {
        ret = data > 0 ? {success: true, code: data, isLock: false} : {
          success: false,
          code: data,
          message: Const.errorDesc[data] || '未知错误',
          isLock: !!lockError[data]
        };
      } else {
        ret = data ? {
          success: false,
          code: data,
          message: Const.errorDesc[data] || 'PIN码错误',
          isLock: !!lockError[data]
        } : {success: true, isLock: false};
      }
      callback.call(context, ret);
    });

    return context;
  }

  //根据卡号, 拔卡监听函数
  function pullCard(cardId, callback, isHandle) {
    if (typeof callback != 'function') callback = noop;
    if (!cardId) throw new Error('cardId为必须参数');
    var ret = {}, timer, fn = function (handle) {
      send("GetCosVer", {handle: handle}, function (resv) {
        if (resv.error) {
          clearInterval(timer);
          resv.error.stop = noop;
          callback.call((ret = resv.error), resv.error);
        }
      });
    };
    if (isHandle) {
      timer = setInterval(function () {
        fn(cardId);
      }, 800);
      fn(cardId);
      ret.stop = function () {
        if (timer) clearInterval(timer);
      }; //停止检测
      return ret;
    }

    send("GetCardHandle", {cardid: cardId}, function (data) {
      if (data.error) {
        data.error.stop = noop;
        return callback.call((ret = data.error), data.error);
      }
      timer = setInterval(function () {
        fn(data);
      }, 800);
      fn(data);
      ret.stop = function () {
        if (timer) clearInterval(timer);
      };//停止检测
    });

    return ret;
  }

  function WsCertKit(cardNo) {
    //获取卡号
    this.getCardId = function () {
      return cardNo;
    };
  }

  /**
   * 获取安全卡信息
   * @author 马德成
   * @param callback 类型 function(data) 回调函数 -可选
   * 当执行成功时data={success:true, type:1, usage:1, container:[0,3,4]}
   * type:卡类型 0-TF卡 1-USB Key
   * usage:卡用途 0-管理卡 1-用户卡
   * container:有证书的容器列表,数组
   * 当执行失败时data={success:false, code:1, message:'错误信息'}
   */
  WsCertKit.prototype.getCardInfo = function (callback) {
    if (typeof callback != 'function') callback = noop;

    var _this = this;
    send('GetCardInfo', {cardid: this.getCardId()}, function (data) {
      if (data.error) return callback.call(_this, data.error);
      var info = data.split('#');
      var type = info[0];
      if ((/.{8}74617264.*/).test(_this.getCardId())) {
        type = 4
      }
      ret = {success: true, type: type, usage: info[1], container: info.length == 3 ? info[2].match(/\d{1}/g) : []};
      callback.call(_this, ret);
    });
  };

  /**
   * 读用户交换证书
   * @author 马德成
   * @date 2015-11-10
   * @param containerId 类型 int 容器编号 -必选
   * @param callback 类型 function(data) 获取证书的回调 -可选
   * 当执行成功时data={success:true, cert:cert}
   * 当执行失败时data={success:false, code:1, message:'错误信息'}
   */
  WsCertKit.prototype.getCert = function (containerId, callback) {
    if (isNaN(containerId)) throw new Error('containerId必须为int类型参数');
    if (typeof callback != 'function') callback = noop;

    var _this = this;
    send('GetCert', {cardid: this.getCardId(), containerid: containerId}, function (data) {
      if (data.error) return callback.call(_this, data.error);
      callback.call(_this, {success: true, cert: data});
    });
  };

  /**
   * 读取用户证书
   * @param containerId 类型 int 容器编号 -必选
   * @param certType 类型 int 证书类型 0:交换证书 1:签名证书 -必选
   * @param callback 类型 function(data) 获取证书的回调 -可选
   * 当执行成功时data={success:true, cert:cert}
   * 当执行失败时data={success:false, code:1, message:'错误信息'}
   */
  WsCertKit.prototype.getCertEx = function (containerId, certType, callback) {
    if (typeof callback != 'function') callback = noop;

    var _this = this;
    send('GetCertEx', {cardid: this.getCardId(), containerid: containerId, certusage: certType}, function (data) {
      if (data.error) return callback.call(_this, data.error);
      callback.call(_this, {success: true, cert: data});
    });
    return this;
  };

  /**
   * Cos操作的包装器
   * @author 马德成
   * @date 2015-11-10
   */
  function WsCosWrap(handle) {
    WsCosWrap.prototype.handle = handle;
  }

  /**
   * 获取Cos版本
   * @author 马德成
   * @date 2015-11-10
   * @param callback 类型 function(data) 获取证书的回调 -可选
   * 当执行成功时data={success:true, version:'3.03.1'}
   * 当执行失败时data={success:false, code:1, message:'错误信息'}
   */
  WsCosWrap.prototype.getCosVer = function (callback) {
    if (typeof callback != 'function') callback = noop;
    var _this = this;
    send('GetCosVer', {handle: this.handle}, function (data) {
      if (data.error) return callback.call(_this, data.error);
      var ret = {success: true, version: data};
      callback.call(_this, ret);
    });
  };

  /**
   * 卡生成随机数
   * @param {Number} mlen 生成随机数的字符串个数（字节数的两倍）
   * @param {Function} [callback] 完成操作后的回调函数,回调参数形式如:
   * 1. 成功:{success:true, data:'随机数字符串'}
   * 2. 失败:{success:false, code:1, message:'错误信息', ex:异常对象}
   * @return {Object} 和callback参数一致
   * @method genRandom
   * @for CosWrap
   */
  WsCosWrap.prototype.genRandom = function (mlen, callback) {
    if (typeof callback != 'function') callback = noop;
    var _this = this;
    send('GenRandom', {handle: this.handle, mlen: mlen}, function (resv) {
      if (resv.error) return callback.call(_this, resv.error);
      var ret = {success: true, data: resv};
      callback.call(_this, ret);
    });
  };

  /**
   * 获取该卡的句柄
   * @author 马德成
   * @date 2015-11-10
   * @param callback 类型 function(data) 获取证书的回调 -可选
   * 当执行成功时data={success:true, handle:'00101010'}
   * 当执行失败时data={success:false, code:1, message:'错误信息'}
   */
  WsCertKit.prototype.getHandle = function (callback) {
    if (typeof callback != 'function') callback = noop;
    var _this = this;
    send("GetCardHandle", {cardid: this.getCardId()}, function (data) {
      if (data.error) return callback.call(_this, data.error);
      callback.call(new WsCosWrap(data), {success: true, handle: data});
    });
    return this;
  };

  /**
   * 验证用户口令,使用0x11角色
   * @author 马德成
   * @date 2015-11-10
   * @param pin 类型 String PIN码 -必选
   * @param callback 类型 function(data) 验证用户口令的回调 -可选
   * 当执行成功时data={success:true, isLock:false, code:1} code:PIN码错误返回重试次数（4 3 2 1）
   * 当执行失败时data={success:false, isLock:true}
   */
  WsCertKit.prototype.checkPin = function (pin, callback) {
    var role = 0x11;
    if ((/.{8}74617264.*/).test(this.getCardId())) {
      role = 0x01
    }
    return this.checkPinEx(pin, role, callback);
  };

  /**
   * 验证用户口令
   * @author 马德成
   * @date 2015-11-10
   * @param pin 类型 String PIN码 -必选
   * @param role 类型 int 用户角色 -必选
   * @param callback 类型 function(data) 验证用户口令的回调 -可选
   * 当执行成功时data={success:true, isLock:false, code:1} code:PIN码错误返回重试次数（4 3 2 1）
   * 当执行失败时data={success:false, isLock:true}
   */
  WsCertKit.prototype.checkPinEx = function (pin, role, callback) {
    if (!pin) throw new Error('pin为必须参数');
    return getPinResult('SafePin', {cardid: this.getCardId(), pin: pin, role: role}, this, callback);
  };

  /**
   * 修改pin码,只有当在卡没被锁死时,才能有用
   * @author 马德成
   * @date 2015-11-10
   * @param oldPin 类型 String 旧PIN码 -必选
   * @param newPin 类型 String 新PIN码 -必选
   * @param callback 类型 function(data) 验证用户口令的回调 -可选
   * 当执行成功时data={success:true, isLock:false, code:1} code:PIN码错误返回重试次数（4 3 2 1）
   * 当执行失败时data={success:false, isLock:true}
   */
  WsCertKit.prototype.safeChangePin = function (oldPin, newPin, callback) {
    return this.safeChangePinEx(oldPin, newPin, 0x11, callback);
  };

  /**
   * 修改pin码,只有当在卡没被锁死时,才能有用
   * @author 马德成
   * @date 2015-11-10
   * @param oldPin 类型 String 旧PIN码 -必选
   * @param newPin 类型 String 新PIN码 -必选
   * @param role 类型 int 用户角色 -必选
   * @param callback 类型 function(data) 验证用户口令的回调 -可选
   * 当执行成功时data={success:true, isLock:false, code:1} code:PIN码错误返回重试次数（4 3 2 1）
   * 当执行失败时data={success:false, isLock:true}
   */
  WsCertKit.prototype.safeChangePinEx = function (oldPin, newPin, role, callback) {
    return getPinResult('SafeChangePin', {
      cardid: this.getCardId(),
      oldPin: oldPin,
      newPin: newPin,
      role: role
    }, this, callback);
  };

  /**
   * 获取PIN码剩余次数
   * @param callback 回调函数 function(data)
   * 当执行成功时data={success:true, isLock:false, code:1} code:PIN码错误返回重试次数（4 3 2 1）
   * 当执行失败时data={success:false, isLock:true}
   */
  WsCertKit.prototype.getPinTryCount = function (callback) {
    var role = 0x11;
    if ((/.{8}74617264.*/).test(this.getCardId())) {
      role = 0x01
    }
    return this.getPinTryCountEx(role, callback);
  };

  /**
   * 获取PIN码剩余次数
   * @param role 用户角色
   * @param callback 回调函数 function(data)
   * 当执行成功时data={success:true, isLock:false, code:1} code:PIN码错误返回重试次数（4 3 2 1）
   * 当执行失败时data={success:false, isLock:true}
   */
  WsCertKit.prototype.getPinTryCountEx = function (role, callback) {
    return getPinResult('GetPinTryCount', {cardid: this.getCardId(), pinrole: role}, this, callback);
  };

  /**
   * 根据解锁码,解锁卡并修改PIN码,当卡被锁死时可以被解锁
   * @param unlockCode 解锁码
   * @param newPin 新PIN码
   * @param callback function(data) 执行结果的回调函数
   * 成功:{success:true}
   * 失败:{success:false, code:-1, message:'错误消息'}
   */
  WsCertKit.prototype.unlockCodePin = function (unlockCode, newPin, callback) {
    var role = 0x11;
    if ((/.{8}74617264.*/).test(this.getCardId())) {
      role = 0x01
    }
    return this.unlockCodePinEx(unlockCode, newPin, role, callback);
  };

  /**
   * 根据解锁码,解锁卡并修改PIN码,当卡被锁死时可以被解锁
   * @param unlockCode 解锁码
   * @param newPin 新PIN码
   * @param role 用户角色
   * @param callback function(data) 执行结果的回调函数
   * 成功:{success:true}
   * 失败:{success:false, code:-1, message:'错误消息'}
   */
  WsCertKit.prototype.unlockCodePinEx = function (unlockCode, newPin, role, callback) {
    if (typeof callback != 'function') callback = noop;
    var _this = this;

    send('ReloadPIN', {cardid: this.getCardId(), enscode: unlockCode, newpin: newPin, role: role}, function (data) {
      if (data.error) return callback.call(_this, data.error);
      data = data - 0;
      var ret = data ? {success: false, code: data, message: Const.errorDesc[data] || '未知错误'} : {success: true};
      callback.call(_this, ret);
    });
    return this;
  };

  /**
   * 导入容器证书到浏览器
   * @author 马德成
   * @date 2015-02-21
   * @param {String} pin PIN码
   * @param {Number} containerId 容器编号
   * @param {Number} certType 证书类型:
   * <pre>
   * 0: 交换(加密)证书
   * 1: 签名证书
   * </pre>
   * @param {Number} role 角色
   * @param {Function} callback 完成操作后的回调函数,回调参数形式如:
   * 1. 成功:{success:true}
   * 2. 失败:{success:false, code:1, message:'错误信息', ex:异常对象}
   * @return {Object} 和callback参数一致
   * @method importCertEx
   * @for WsCertKit
   */
  WsCertKit.prototype.importCertEx = function (pin, containerId, certType, role, callback) {
    if (typeof callback != 'function') callback = noop;
    var _this = this,
        parame = {cardid: this.getCardId(), role: role, pin: pin, containerid: containerId, certusage: certType};

    send('ImportKeyCertEx', parame, function (data) {
      if (data.error) return callback.call(_this, data.error);
      callback.call(_this, {success: !data});
    });
    return this;
  };

  /**
   * 设定浏览器记录卡ID,当关闭浏览器后则失效
   * @author 马德成
   * @date 2015-02-21
   * @param {Function} [callback] 完成操作后的回调函数,回调参数形式如:
   * 1. 成功:{success:true}
   * 2. 失败:{success:false, code:1, message:'错误信息', ex:异常对象}
   * @return {Object} CertKit
   * @method setCardId
   * @for WsCertKit
   */
  WsCertKit.prototype.setCardId = function (callback) {
    if (typeof callback != 'function') callback = noop;
    var _this = this;
    send('SetCardId', {cardid: this.getCardId()}, function (data) {
      if (data.error) return callback.call(_this, data.error);
      callback.call(_this, {success: true});
    });

    return this;
  };

  /**
   * 生成公私钥对
   * @author 马德成
   * @param password 卡用户口令
   * @param containerid 容器编号
   * @param alg 算法：0-RSA，1-SM2
   * @param certusage 算法用途：0-交换，1-签名
   * @param keybits bits 位数：RSA-一般为1024，SM2-一般为256
   * @param callback 回调函数 类型 function(data) 回调函数 -可选
   * 当执行成功时data={success:true, publicKey:'公钥：RSA-Base64编码的M，SM2-B64编码的x=...y=..'}
   * 当执行失败时data={success:false, code:1, message:'错误信息'}
   */
  WsCertKit.prototype.genKeypair = function (password, containerid, alg, certusage, keybits, callback) {
    if (typeof callback != 'function') callback = noop;
    var _this = this, parame = {
      cardid: this.getCardId(), password: password, containerid: containerid,
      alg: alg, certusage: certusage, keybits: keybits
    };

    send('GenKeypair', parame, function (data) {
      if (data.error) return callback.call(_this, data.error);
      callback.call(_this, {success: true, ublicKey: data});
    });
    return this;
  };

  /**
   * 生成0号容器,交换证书,RSA算法的公私钥对
   * @param password 卡用户口令
   * @param containerid 容器编号(可选,默认值为0)
   * @param certusage 算法用途：0-交换，1-签名 (可选,默认值为0)
   * @param callback 回调函数 类型 function(data) 回调函数 -可选
   * 当执行成功时data={success:true, publicKey:'公钥：RSA-Base64编码的M，SM2-B64编码的x=...y=..'}
   * 当执行失败时data={success:false, code:1, message:'错误信息'}
   */
  WsCertKit.prototype.genKeypairRSA = function (password, containerid, certusage, callback) {
    if (typeof containerid == 'function') {
      callback = containerid;
      containerid = null;
    } else if (typeof certusage == 'function') {
      callback = certusage;
      certusage = null;
    }
    return this.genKeypair(password, containerid || 0, 0, certusage || 0, 1024, callback);
  };

  /**
   * 生成0号容器,交换证书,RSA算法的公私钥对
   * @param password 卡用户口令
   * @param containerid 容器编号(可选,默认值为0)
   * @param certusage 算法用途：0-交换，1-签名 (可选,默认值为0)
   * @param callback 回调函数 类型 function(data) 回调函数 -可选
   * 当执行成功时data={success:true, publicKey:'公钥：RSA-Base64编码的M，SM2-B64编码的x=...y=..'}
   * 当执行失败时data={success:false, code:1, message:'错误信息'}
   */
  WsCertKit.prototype.genKeypairSM2 = function (password, containerid, certusage, callback) {
    if (typeof containerid == 'function') {
      callback = containerid;
      containerid = null;
    } else if (typeof certusage == 'function') {
      callback = certusage;
      certusage = null;
    }
    return this.genKeypair(password, containerid || 0, 1, certusage || 0, 256, callback);
  };


  /**
   * 生成公私钥对-扩展
   * @author 马德成
   * @param role 角色
   * @param password 卡用户口令
   * @param containerid 容器编号
   * @param alg 算法：0-RSA，1-SM2
   * @param certusage 算法用途：0-交换，1-签名
   * @param keybits bits 位数：RSA-一般为1024，SM2-一般为256
   * @param callback 回调函数 类型 function(data) 回调函数 -可选
   * 当执行成功时data={success:true, publicKey:'公钥：RSA-Base64编码的M，SM2-B64编码的x=...y=..'}
   * 当执行失败时data={success:false, code:1, message:'错误信息'}
   */
  WsCertKit.prototype.genKeypairEx = function (role, password, containerid, alg, certusage, keybits, callback) {
    if (typeof callback != 'function') callback = noop;
    var _this = this, parame = {
      cardid: this.getCardId(), role: role, password: password, containerid: containerid,
      alg: alg, certusage: certusage, keybits: keybits
    };

    send('GenKeypairEx', parame, function (data) {
      if (data.error) return callback.call(_this, data.error);
      callback.call(_this, {success: true, publicKey: data});
    });
    return this;
  };

  /**
   * 向卡容器写入证书
   * @param password 卡用户口令
   * @param containerid 容器编号
   * @param certusage 算法用途：0-交换，1-签名
   * @param cert 证书 十六进制编码
   * @param callback 回调函数 类型 function(data) 回调函数 -可选
   * 当执行成功时data={success:true}
   * 当执行失败时data={success:false, code:1, message:'错误信息'}
   */
  WsCertKit.prototype.writeCert = function (password, containerid, certusage, cert, callback) {
    if (typeof callback != 'function') callback = noop;
    var _this = this, parame = {
      cardid: this.getCardId(), password: password, containerid: containerid,
      certusage: certusage, cert: cert
    };

    send('WriteCert', parame, function (data) {
      if (data.error) return callback.call(_this, data.error);
      callback.call(_this, {success: true});
    });
    return this;
  };

  /**
   * 向卡容器写入证书
   * @param password 卡用户口令
   * @param containerid 容器编号 (可选,默认:0)
   * @param certusage 算法用途：0-交换，1-签名 (可选,默认:0)
   * @param cert 证书 十六进制编码
   * @param callback 回调函数 类型 function(data) 回调函数 -可选
   * 当执行成功时data={success:true}
   * 当执行失败时data={success:false, code:1, message:'错误信息'}
   */
  WsCertKit.prototype.writeCertEx = function (password, containerid, certusage, cert, callback) {
    var _cert = cert;
    if (typeof containerid == 'string') {
      _cert = containerid;
      containerid = null;
      if (typeof certusage == 'function') {
        callback = certusage;
        certusage = null;
      }
    } else if (typeof certusage == 'string') {
      _cert = certusage;
      if (typeof cert == 'function') callback = cert;
      certusage = null;
      cert = null;
    }
    return this.writeCert(password, containerid || 0, certusage || 0, _cert, callback);
  };

  /**
   * 写网关单证书（公钥）
   * @param password 卡用户口令
   * @param cert 证书 十六进制编码
   * @param callback 回调函数 类型 function(data) 回调函数 -可选
   * 当执行成功时data={success:true}
   * 当执行失败时data={success:false, code:1, message:'错误信息'}
   */
  WsCertKit.prototype.writeGateCert = function (password, cert, callback) {
    if (typeof callback != 'function') callback = noop;
    var _this = this;

    send('WriteGateCert', {cardid: this.getCardId(), password: password, cert: cert}, function (data) {
      if (data.error) return callback.call(_this, data.error);
      callback.call(_this, {success: !data});
    });
    return this;
  };


  /**
   * 写网关单证书（公钥）
   * @param password 卡用户口令
   * @param cert 证书 十六进制编码
   * @param callback 回调函数 类型 function(data) 回调函数 -可选
   * 当执行成功时data={success:true}
   * 当执行失败时data={success:false, code:1, message:'错误信息'}
   */
  WsCertKit.prototype.writeGateCert = function (password, cert, callback) {
    if (typeof callback != 'function') callback = noop;
    var _this = this;

    send('WriteGateCert', {cardid: this.getCardId(), password: password, cert: cert}, function (data) {
      if (data.error) return callback.call(_this, data.error);
      callback.call(_this, {success: true});
    });
    return this;
  };

  /**
   * 写网关双证书（公钥）
   * @param password 卡用户口令
   * @param cert 证书 十六进制编码
   * @param cert2 监管证书：pem编码 十六进制编码
   * @param callback 回调函数 类型 function(data) 回调函数 -可选
   * 当执行成功时data={success:true}
   * 当执行失败时data={success:false, code:1, message:'错误信息'}
   */
  WsCertKit.prototype.writeGateCertEx = function (password, cert, cert2, callback) {
    if (typeof callback != 'function') callback = noop;
    var _this = this;

    send('WriteGateCertEx', {cardid: this.getCardId(), password: password, cert1: cert, cert2: cert2}, function (data) {
      if (data.error) return callback.call(_this, data.error);
      callback.call(_this, {success: true});
    });
    return this;
  };

  /**
   * 导入私钥
   * 将加密私钥导入卡内（支持RSA-1024，SM2，目前不支持RSA_2048）
   * @author 马德成
   * @param {Number} role 角色
   * @param {String} pin 卡用户口令
   * @param {Number} container 容器ID
   * @param {Number} alg 算法类型:
   * <pre>
   * 。0:RSA
   * 。1:SM2
   * </pre>
   * @param {Number} usage 加密类型:
   * <pre>
   * 。0:交换
   * 。1:签名
   * </pre>
   * @param {String} privateKey 要导入的私钥Bsae64编码
   * @param {Function} [callback] 完成操作后的回调函数,回调参数形式如:
   * 1. 成功:{success:true, data:'传入证书的公钥加密之后的密文'}
   * 2. 失败:{success:false, code:1, message:'错误信息', ex:异常对象}
   * @return {Object} 和callback参数一致
   * @method importAsymKey
   * @for WsCertKit
   */
  WsCertKit.prototype.importAsymKey = function (role, pin, container, alg, usage, privateKey, callback) {
    if (typeof callback != 'function') callback = noop;
    var _this = this, parame = {
      cardid: this.getCardId(),
      role: role,
      password: pin,
      containernum: container,
      alg: alg,
      usage: usage,
      keybuf: privateKey
    };

    send('ImportAsymKey', parame, function (data) {
      if (data.error) return callback.call(_this, data.error);
      callback.call(null, {success: true, data: data});
    });
    return this;
  };

  /**
   * 私钥运算
   * 用卡内私钥解密，并由传入的证书得到公钥加密明文，将加密的密文传出
   * @author 马德成
   * @param {Number} role 角色
   * @param {String} pin Pin码
   * @param {Number} container 容器ID
   * @param {Number} alg 算法类型 0-RSA 1-SM2
   * @param {Number} usage 证书类型 0- 交换证书  1- 签名证书
   * @param {String} inBase64 老设备Base64形式的待解密数据(加密后kuep)
   * @param {String} certBase64 新设备Base64形式的证书内容
   * @param {Function} [callback] 完成操作后的回调函数,回调参数形式如:
   * 1. 成功:{success:true, dekuep:'老设备解密后的kuep'}
   * 2. 失败:{success:false, code:1, message:'错误信息', ex:异常对象}
   * @return {Object} 和callback参数一致
   * @method asymDecryption
   * @for CertKit
   */
  WsCertKit.prototype.asymDecryption = function (role, pin, container, alg, usage, inBase64, certBase64, callback) {
    if (typeof callback != 'function') callback = noop;
    var _this = this;
    send('AsymDecryption', {
      cardid: this.getCardId(),
      role: role,
      password: pin,
      containernum: container,
      alg: alg,
      usage: usage,
      instr: inBase64,
      certstr: certBase64
    }, function (data) {
      if (data.error) {
        return callback.call(_this, data.error);
      }
      var ret = {success: !!data, dekuep: data};
      callback.call(_this, ret);
    });
    return this;
  };

  /**
   * 导入私钥
   * 将加密私钥导入卡内（支持RSA-1024，SM2，目前不支持RSA_2048）
   * @author 马德成
   * @param {Number} role 角色
   * @param {String} pin 卡用户口令
   * @param {Number} container 容器ID
   * @param {Number} alg 算法类型:
   * <pre>
   * 。0:RSA
   * 。1:SM2
   * </pre>
   * @param {Number} usage 加密类型:
   * <pre>
   * 。0:交换
   * 。1:签名
   * </pre>
   * @param {String} privateKey 要导入的私钥Bsae64编码
   * @param {Function} [callback] 完成操作后的回调函数,回调参数形式如:
   * 1. 成功:{success:true, data:'传入证书的公钥加密之后的密文'}
   * 2. 失败:{success:false, code:1, message:'错误信息', ex:异常对象}
   * @return {Object} 和callback参数一致
   * @method importAsymKey
   * @for WsCertKit
   */
  WsCertKit.prototype.importAsymKey = function (role, pin, container, alg, usage, privateKey, callback) {
    if (typeof callback != 'function') callback = noop;
    var _this = this, parame = {
      cardid: this.getCardId(),
      role: role,
      password: pin,
      containernum: container,
      alg: alg,
      usage: usage,
      keybuf: privateKey
    };

    send('ImportAsymKey', parame, function (data) {
      if (data.error) return callback.call(_this, data.error);
      callback.call(null, {success: true, data: data});
    });
    return this;
  };

  /**
   * 当拔卡时,执行的操作
   * @author 马德成
   * @date 2015-11-10
   * @param callback 类型 function()当拔卡时执行的回调
   */
  WsCertKit.prototype.onpullcard = function (callback) {
    return pullCard(this.getCardId(), callback);
  };

  /**
   * 根据解锁码,解锁加密盘
   * @author LB
   * @param unlockCode 解锁码
   * @param newPin 新PIN码
   * @param callback function(data) 执行结果的回调函数
   * 成功:{success:true}
   * 失败:{success:false, message:'错误消息'}
   */
  WsCertKit.prototype.unlockUsbSecuZone = function (unlockCode, newPin, callback) {
    if (typeof callback != 'function') callback = noop;
    var _this = this;

    send('UnlockUsbSecuZone', {cardid: this.getCardId(), enscode: unlockCode, newpin: newPin}, function (data) {
      if (data.error) return callback.call(_this, data.error);
      data = data - 0;
      var ret = (data < 0) ? {success: false, code: data, message: Const.errorDesc[data] || '未知错误'} : {success: true};
      callback.call(_this, ret);
    });
    return this;
  };

  /**
   * LB
   * WsCertKit - cardSign
   * @param role 角色
   * @param pin 卡密码
   * @param container 容器ID
   * @param alg 算法类型 0-RSA 1-SM2
   * @param usage 证书类型 0-交换证书 1-签名证书
   * @param inData 待签名的数据
   * @param dataType 待签名数据的数据类型 0-inData是经过hash运算的值 1-inData没经过hash运算
   * @param callback 回调函数 形式如{success:true, signature:'签名值,经base64编码后的值'}
   */
  WsCertKit.prototype.cardSign = function (role, pin, container, alg, usage, inData, dataType, callback) {
    if (typeof callback != 'function') callback = noop;
    var role1 = role;
    if ((/.{8}74617264.*/).test(this.getCardId())) {
      role1 = 0x01
    }
    var _this = this;
    send('CardSign', {
      cardid: this.getCardId(),
      role: role1,
      password: pin,
      containernum: container,
      alg: alg,
      usage: usage,
      instr: inData,
      datatype: dataType
    }, function (data) {
      if (data.error) {
        return callback.call(_this, data.error);
      }
      var ret = {success: !!data, signature: data};
      callback.call(_this, ret);
    });
    return this;
  };

  /**
   * LB
   * WsCertKit - cardRSASign
   * 使用卡内证书进行RAS签名运算
   * @param pin 卡密码
   * @param container 容器ID
   * @param usage 证书类型 0-交换证书 1-签名证书
   * @param inData 待签名的数据(原文)
   * @param callback 回调函数 形式如{success:true, signature:'签名值,经base64编码后的值'}
   */
  WsCertKit.prototype.cardRSASign = function (pin, container, usage, inData, callback) {
    return this.cardSign(0x11, pin, container, 0, usage, inData, 1, callback);
  };


  /**
   * 使用卡内证书进行SM2签名运算
   * @author 马德成
   * @date 2015-11-28
   * @param pin 卡密码
   * @param container 容器ID
   * @param usage 证书类型 0-交换证书 1-签名证书
   * @param inData 待签名的数据(原文)
   * @param callback 回调函数 形式如{success:true, signature:'签名值,经base64编码后的值'}
   */
  WsCertKit.prototype.cardSM2Sign = function (pin, container, usage, inData, callback) {
    return this.cardSign(0x11, pin, container, 1, usage, inData, 1, callback);
  };

  /**
   * 使用卡内证书进行RAS签名运算
   * @author 马德成
   * @date 2015-11-28
   * @param pin 卡密码
   * @param container 容器ID
   * @param usage 证书类型 0-交换证书 1-签名证书
   * @param inData 待签名的数据(经过hash运算)
   * @param callback 回调函数 形式如{success:true, signature:'签名值,经base64编码后的值'}
   */
  WsCertKit.prototype.cardSha1RSASign = function (pin, container, usage, inData, callback) {
    return this.cardSign(0x11, pin, container, 0, usage, inData, 0, callback);
  };

  /**
   * 使用卡内证书进行SM2签名运算
   * @author 马德成
   * @date 2015-11-28
   * @param pwd 卡密码
   * @param container 容器ID
   * @param usage 证书类型 0-交换证书 1-签名证书
   * @param inData 待签名的数据(经过hash运算)
   * @param callback 回调函数 形式如{success:true, signature:'签名值,经base64编码后的值'}
   */
  WsCertKit.prototype.cardSha1SM2Sign = function (pwd, container, usage, inData, callback) {
    return this.cardSign(0x11, pwd, container, 1, usage, inData, 0, callback);
  };

  /**
   * SM3运算
   * @param {String} pDataIn 需要加密的数据(内存字符)
   * @param {Function} [callback] 完成操作后的回调函数,回调参数形式如:
   * 1. 成功:{success:true, data:'SM3运算结果，长度为32字节（64个内存字符）'}
   * 2. 失败:{success:false, code:1, message:'错误信息', ex:异常对象}
   * @return {Object} 和callback参数一致
   * @method sm3
   * @for WsCertKit
   */
  WsCertKit.prototype.sm3 = function (pDatain, callback) {
    if (typeof callback != 'function') callback = noop;
    var _this = this;
    var size = getBytesLength(pDatain);
    if (size % 2 != 0) {
      callback.call(_this, {success: false, code: -2, message: Const.errorDesc['-2']});
      return this;
    }

    send('SM3', {cardid: this.getCardId(), pDatain: pDatain, dataLen: size / 2}, function (data) {
      if (data.error) return callback.call(_this, data.error);
      var ret = {success: !!data, data: data};
      callback.call(_this, ret);
    });

    return this;
  };

  /**
   * 对卡号操作的包装器
   * @author 马德成
   * @date 2015-11-10
   */
  function WsCertWrap(cardNos) {
    for (var i = 0; i < cardNos.length; i++) {
      this[i] = new WsCertKit(cardNos[i]);
    }

    this.length = cardNos.length;
  }

  /**
   * 遍历卡号
   * @author 马德成
   * @date 2015-11-10
   * @param callback 类型 function(index, cardId)
   */
  WsCertWrap.prototype.each = function (callback) {
    if (typeof callback == 'function') {
      for (var i = 0; i < this.length; i++) {
        callback.call(this[i], i, this[i]);
      }
    }

    return this;
  };

  /**
   * 把另一个certWrap放到当前CertWrap里
   * 例如:
   * var cards = plugin.readUserUSBKeyCard(); //读取Ukey
   * var tfs = plugin.readUserTFCard();//读取TF卡
   * cards.push(tfs); //把读取的TF卡结果push到cards
   * @param certWrap
   * @returns {CertWrap}
   */
  WsCertWrap.prototype.push = function (certWrap) {
    if (!certWrap || !certWrap.length) return this;
    var len = this.length;

    for (var i = 0; i < certWrap.length; i++) {
      this[len++] = certWrap[i];
    }

    this.length = this.length + certWrap.length;
    return this;
  };

  /**
   * 对安全卡操作的包装器
   * @author 马德成
   * @date 2015-11-10
   */
  function WsCardWrap() {
  }

  /**
   * 当拔卡时,执行的操作
   * @author 马德成
   * @date 2015-11-10
   * @param cardId 类型 string 卡号
   * @param callback 类型 function()当拔卡时执行的回调
   * @returns 返回{stop:fn}可以停止定时器的
   */
  WsCardWrap.prototype.onpullcard = function (cardId, callback) {
    return pullCard(cardId, callback);
  };

  /**
   * 当拔卡时,执行的操作
   * @author 马德成
   * @date 2015-11-10
   * @param handle 类型 string 卡句柄
   * @param callback 类型 function()当拔卡时执行的毁掉
   */
  WsCardWrap.prototype.onpullhandle = function (handle, callback) {
    return pullCard(handle, callback, true);
  };

  /**
   * 读取卡函数
   * @author 马德成
   * @date 2015-11-10
   * @param cardType 卡类型 0:TF卡,1:USBKey,2:手机,其他:不区分 -可选(默认:-1)
   * @param cardUsage 用户类型 --0-管理卡,1-用户卡 -1:所有
   * @param callback 获取安全卡后的回调函数-----该函数没有错误码,只能通过通过属性length来判断是否读到卡
   */
  function readCard(cardType, cardUsage, callback) {
    if (typeof callback != 'function') callback = noop;
    send('GetAllCards', {type: cardType, usage: cardUsage}, function (data) {
      var ret = new WsCertWrap(data.error ? [] : data.split('#'));
      callback.call(ret, ret);
    });
    return this;
  }

  /**
   * 读取所有用户卡, 包括TF卡 和 USB Key 和 手机
   * @author 马德成
   * @date 2015-11-10
   */
  WsCardWrap.prototype.readUserCard = function (callback) {
    return readCard.call(this, -1, Const.cardUsage.X_CARD_USAGE_USER, callback);
  };

  /**
   * 读取ACE手机的芯片号
   * @author 马德成
   * @date 2015-08-14
   */
  WsCardWrap.prototype.readUserACE = function (callback) {
    return readCard.call(this, 2, Const.cardUsage.X_CARD_USAGE_USER, callback);
  };

  /**
   * 读取用户TF卡
   * @author 马德成
   * @date 2015-11-10
   */
  WsCardWrap.prototype.readUserTFCard = function (callback) {
    return readCard.call(this, 0, Const.cardUsage.X_CARD_USAGE_USER, callback);
  };

  /**
   * 读取用户USB Key
   * @author 马德成
   * @date 2015-11-10
   */
  WsCardWrap.prototype.readUserUSBKeyCard = function (callback) {
    return readCard.call(this, 1, Const.cardUsage.X_CARD_USAGE_USER, callback);
  };

  /**
   * 读取管理员卡,包括TF卡和USB Key
   * @author 马德成
   * @date 2015-11-10
   */
  WsCardWrap.prototype.readAdminCard = function (cardType, callback) {
    return readCard.call(this, -1, Const.cardUsage.X_CARD_USAGE_ADMIN, callback);
  };

  /**
   * 读取管理员卡,包括TF卡
   * @author 马德成
   * @date 2015-11-10
   */
  WsCardWrap.prototype.readAdminTFCard = function (cardType, callback) {
    return readCard.call(this, 0, Const.cardUsage.X_CARD_USAGE_ADMIN, callback);
  };

  /**
   * 读取管理员USB Key
   * @author 马德成
   * @date 2015-11-10
   */
  WsCardWrap.prototype.readAdminUSBKeyCard = function (cardType, callback) {
    return readCard.call(this, 1, Const.cardUsage.X_CARD_USAGE_ADMIN, callback);
  };

  /**
   * 在左面右下角弹出消息提示框
   * @author 马德成
   * @date 2015-02-21
   * @param message 消息内容
   * @param time 显示时长(单位:s)，如果取值0，永久显示
   * @param  type 1-oa  2-zw
   * @param callback 类型 function(data)
   * 成功,data={success:true}
   * 失败,data={success:false, code:0, message:'错误信息', ex:异常对象}
   */
  WsCardWrap.prototype.showMsgTip = function (message, time, type, callback) {
    if (typeof callback != 'function') callback = noop;
    var _this = this;
    send('ShowMsgTip', {msg: message, time: time, tip: type}, function (data) {
      if (data.error) return callback.call(_this, data.error);
      return callback.call(_this, {success: true});
    });
    return this;
  };

  /**
   * 获当前浏览器记录的卡ID,当关闭浏览器后则获取不到记录
   * @author 马德成
   * @date 2015-02-21
   * @param callback 类型 function(data)
   * 成功,data={success:true, cardId:'1df0d0f0'}
   * 失败,data={success:false, code:0, message:'错误信息', ex:异常对象}
   */
  WsCardWrap.prototype.getCard = function (callback) {
    if (typeof callback != 'function') callback = noop;
    var _this = this;
    send('GetCardId', function (data) {
      if (data.error) return callback.call(_this, data.error);
      callback.call(_this, {success: true, cardId: data});
    });
    return this;

  };

  /**
   * 根据卡号获取CertKit对象
   * @param cardNo 卡号
   * @returns {WsCertKit} 卡操作的工具类
   */
  WsCardWrap.prototype.cardKit = function (cardNo) {
    return new WsCertKit(cardNo);
  };

  /**
   * LB
   * WsCardWrap - connectACE
   */
  WsCardWrap.prototype.connectACE = function (callback) {
    if (typeof callback != 'function') callback = noop;
    var _this = this;

    send('ConnectACE', function (data) {
      if (data.error) return callback.call(_this, data.error);
      callback.call(_this, {success: data == undefined ? false : !!!data});
    });
    return this;
  };

  /**
   * LB
   * WsCardWrap - countACE
   */
  WsCardWrap.prototype.countACE = function (callback) {
    if (typeof callback != 'function') callback = noop;
    var _this = this;

    send('EnumACE', function (data) {
      if (data.error) return callback.call(_this, data.error);
      var ret = {success: true, count: data || 0};
      callback.call(ret, ret);
    });
    return this;
  };

  /**
   * 卸载APK
   * @param callback
   * @returns {WsCardWrap}
   */
  WsCardWrap.prototype.uninstallApk = function (callback) {
    if (typeof callback != 'function') callback = noop;
    var _this = this;

    send('UnInstallApk', function (data) {
      if (data.error) return callback.call(_this, data.error);
      var ret = {success: true, count: data || 0};
      callback.call(_this, ret);
    });
    return this;
  };

  /**
   * 私钥运算
   * 用卡内私钥解密，并由传入的证书得到公钥加密明文，将加密的密文传出
   * @param cardId 老卡的卡号
   * @param role 角色
   * @param pin Pin码
   * @param container 容器ID
   * @param alg 算法类型 0-RSA 1-SM2
   * @param usage 证书类型 0- 交换证书  1- 签名证书
   * @param inBase64 老设备Base64形式的待解密数据(加密后kuep)
   * @param certBase64 新设备Base64形式的证书内容
   * @param callback 回调函数 形式如{success:true, dekuep:'老设备解密后的kuep'}
   */
  WsCardWrap.prototype.asymDecryption = function (cardId, role, pin, container, alg, usage, inBase64, certBase64, callback) {
    if (typeof callback != 'function') callback = noop;
    var _this = this;
    send('AsymDecryption', {
      cardid: cardId,
      role: role,
      password: pin,
      containernum: container,
      alg: alg,
      usage: usage,
      instr: inBase64,
      certstr: certBase64
    }, function (data) {
      if (data.error) {
        return callback.call(_this, data.error);
      }
      var ret = {success: !!data, dekuep: data};
      callback.call(_this, ret);
    });
    return this;
  };

  //最终返回值
  return WsCardWrap;
})(window);
