package com.xdja.widget.emoji;

import android.content.Context;
import android.content.res.TypedArray;
import android.support.v7.widget.AppCompatEditText;
import android.text.ClipboardManager;
import android.text.Editable;
import android.text.SpannableString;
import android.text.TextUtils;
import android.text.style.DynamicDrawableSpan;
import android.util.AttributeSet;
import android.view.KeyEvent;
import android.view.inputmethod.InputConnection;
import android.view.inputmethod.InputConnectionWrapper;

import com.xdja.emoji.R;

public class EmojiconEditText extends AppCompatEditText {
    private int mEmojiconSize;
    private int mEmojiconAlignment;
    private int mEmojiconTextSize;

    public EmojiconEditText(Context context) {
        super(context);
        mEmojiconSize = (int) getTextSize();
        mEmojiconTextSize = (int) getTextSize();
    }

    public EmojiconEditText(Context context, AttributeSet attrs) {
        super(context, attrs);
        init(attrs);
    }

    public EmojiconEditText(Context context, AttributeSet attrs, int defStyle) {
        super(context, attrs, defStyle);
        init(attrs);
    }

    private void init(AttributeSet attrs) {
        TypedArray a = getContext().obtainStyledAttributes(attrs, R.styleable.Emojicon);
        mEmojiconSize = (int) a.getDimension(R.styleable.Emojicon_emojiconSize, getTextSize());
        mEmojiconAlignment = a.getInt(R.styleable.Emojicon_emojiconAlignment, DynamicDrawableSpan.ALIGN_BASELINE);
        a.recycle();
        mEmojiconTextSize = (int) getTextSize();
        setText(getText());
    }

    /**
     * Set the size of emojicon in pixels.
     */
    public void setEmojiconSize(int pixels) {
        mEmojiconSize = pixels;
        updateText();
    }

    @Override
    public boolean onTextContextMenuItem(int id) {
        switch (id) {
            case android.R.id.paste:
                ClipboardManager cmb = (ClipboardManager) this.getContext().getSystemService(Context.CLIPBOARD_SERVICE);
                if (cmb == null || cmb.getText() == null) {
                    return false;
                }
                CharSequence pasteContent = cmb.getText();
                if (!TextUtils.isEmpty(pasteContent)) {
                    try {
                        SpannableString pasteSpan = EmojiconHandler.getEmojiFormatContent(getContext(), pasteContent, mEmojiconSize, mEmojiconAlignment, mEmojiconTextSize);
                        if (!TextUtils.isEmpty(pasteSpan)) {
                            this.getText().insert(getSelectionStart(), pasteSpan); //光标处插入粘贴的内容
                            this.setFocusable(true);
                            return false;
                        }
                    } catch (NumberFormatException e) {
                        e.printStackTrace();
                    } catch (IllegalArgumentException e) {
                        e.printStackTrace();
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                }
                break;
        }
        return super.onTextContextMenuItem(id);
    }

    @Override
    public void setText(CharSequence text, BufferType type) {
        super.setText(text, type);
        updateText();
    }

    private void updateText() {
        CharSequence text = getText();
        if (!TextUtils.isEmpty(text)) {
            EmojiconHandler.addEmojis(getContext(), getText(), mEmojiconSize, mEmojiconAlignment, mEmojiconTextSize);
        }
    }

    public void insertEmojicon(Emojicon emojicon) {
        if (emojicon == null || TextUtils.isEmpty(emojicon.getEmoji())) {
            return;
        }
        try {
            CharSequence emojiText = emojicon.getEmoji();
            SpannableString pasteSpan = EmojiconHandler.getEmojiFormatContent(getContext(), emojiText, mEmojiconSize, mEmojiconAlignment, mEmojiconTextSize);
            if (!TextUtils.isEmpty(pasteSpan)) {
                this.getText().insert(getSelectionStart(), pasteSpan); //光标处插入粘贴的内容
                this.setFocusable(true);
                return;
            }
        } catch (NumberFormatException e) {
            e.printStackTrace();
        } catch (IllegalArgumentException e) {
            e.printStackTrace();
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * 调用回退键会导致TextWatcher 被回调多次,不要在文本输入的监听内使用,防止多次调用各个监听事件
     */
    public void onBackSpace() {
        KeyEvent event = new KeyEvent(0, 0, 0, KeyEvent.KEYCODE_DEL, 0, 0, 0, 0, KeyEvent.KEYCODE_ENDCALL);
        dispatchKeyEvent(event);
    }


//    @Override
//    public InputConnection onCreateInputConnection(EditorInfo outAttrs) {
//        return new EmojiInputConnection(super.onCreateInputConnection(outAttrs), true);
//    }

    public class EmojiInputConnection extends InputConnectionWrapper {

        public EmojiInputConnection(InputConnection target, boolean mutable) {
            super(target, mutable);
        }

        @Override
        public boolean commitText(CharSequence pasteContent, int newCursorPosition) {
            if (TextUtils.isEmpty(pasteContent)) {
                return super.commitText(pasteContent, newCursorPosition);
            }
            Editable edit = getEditableText();
            SpannableString pasteSpan = EmojiconHandler.getEmojiFormatContent(getContext(), pasteContent, mEmojiconSize, mEmojiconAlignment, mEmojiconTextSize);
            if (!TextUtils.isEmpty(pasteSpan)) {
                edit.insert(getSelectionStart(), pasteSpan); //光标处插入粘贴的内容
                return false;
            }
            return super.commitText(pasteContent, newCursorPosition);
        }
    }
}
