package com.xdja.eoa.commonutil;

import android.Manifest;
import android.annotation.SuppressLint;
import android.annotation.TargetApi;
import android.content.Context;
import android.content.Intent;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.content.pm.ResolveInfo;
import android.media.MediaMetadataRetriever;
import android.net.Uri;
import android.os.Build;
import android.os.Environment;
import android.os.storage.StorageManager;
import android.support.annotation.NonNull;
import android.support.v4.app.ActivityCompat;
import android.support.v4.content.FileProvider;
import android.text.TextUtils;
import android.util.Log;
import android.widget.Toast;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.Array;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import static android.os.Build.VERSION.SDK_INT;

/**
 * 统一的文件操作帮助类，其他模块的fileutil建议命名为模块名+FileUtil
 * Created by zhuruyi on 2019/10/17
 */
public class FileUtil {
	public static final Pattern DIR_SEPARATOR = Pattern.compile("/");

	final static Pattern PATTERN = Pattern.compile("(.*?)(?:\\((\\d+)\\))?(\\.[^.]*)?");

	// 格式化文件长度
	public static String formatFromSize(long size) {
		if (size == 0) {
			return "0 B";
		}
		DecimalFormat df = new DecimalFormat("#.00");
		String fileSizeString;
		if (size < 1024) {
			fileSizeString = df.format((double) size) + " B";
		} else if (size < 1048576) {
			fileSizeString = df.format((double) size / 1024) + " KB";
		} else if (size < 1073741824) {
			fileSizeString = df.format((double) size / 1048576) + " MB";
		} else {
			fileSizeString = df.format((double) size / 1073741824) + " GB";
		}
		return fileSizeString;
	}

	public static void openFileAndHint(final Context context, String filePath) {
		try {
			openFile(context, filePath);
		} catch (Exception e) {
			e.printStackTrace();
			Toast.makeText(context, "没有可打开此文件的程序", Toast.LENGTH_SHORT).show();
		}
	}

	/**
	 * 打开已安装的三方应用
	 * (lwj)
	 *
	 * @param context context
	 * @param pkgName 包名
	 */
	public static void openInstalledApp(Context context, String pkgName) {
		try {
			PackageManager packageManager = context.getPackageManager();
			Intent it = packageManager.getLaunchIntentForPackage(pkgName);
			if (it != null) {
				it.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
				context.startActivity(it);
			} else {
				PackageManager pm = context.getPackageManager();
				Intent intentQuery = new Intent(Intent.ACTION_MAIN);
				intentQuery.setPackage(pkgName);
				@SuppressLint("WrongConstant")
				List<ResolveInfo> resolveInfos = pm.queryIntentActivities(
						intentQuery, PackageManager.GET_ACTIVITIES);
				String activityName = "";
				for (int i = 0; i < resolveInfos.size(); i++) {
					ResolveInfo resolveInfo = resolveInfos.get(i);
					if (resolveInfo.activityInfo.packageName.equals(pkgName)) {
						activityName = resolveInfo.activityInfo.name;
						break;
					}
				}
				if (!TextUtils.isEmpty(activityName)) {
					Intent intent = new Intent();
					intent.setClassName(pkgName, activityName);
					intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
					context.startActivity(intent);
				} else {
					Toast.makeText(context, "该应用不支持打开", Toast.LENGTH_SHORT).show();
				}
			}
		} catch (Exception e) {
			e.printStackTrace();
			Toast.makeText(context, "该应用不支持打开", Toast.LENGTH_SHORT).show();
		}
	}

	/**
	 * 应用是否安装
	 *
	 * @param context     应用上下文
	 * @param packageName 包名
	 * @return
	 */
	public static boolean isAppInstalled(Context context, String packageName) {
		PackageInfo packageInfo = null;
		try {
			packageInfo = context.getPackageManager().getPackageInfo(
					packageName, 0);
		} catch (PackageManager.NameNotFoundException e) {
			e.printStackTrace();
			packageInfo = null;
		} catch (Exception e) {
			e.printStackTrace();
			packageInfo = null;
		}
		return packageInfo != null;
	}


	private static void grantUriPermission(Context context, Uri fileUri, Intent intent) {
		List<ResolveInfo> resolveInfoList = context.getPackageManager().queryIntentActivities(intent, PackageManager.MATCH_DEFAULT_ONLY);
		for (ResolveInfo resolveInfo : resolveInfoList) {
			String packageName = resolveInfo.activityInfo.packageName;
			context.grantUriPermission(packageName, fileUri, Intent.FLAG_GRANT_WRITE_URI_PERMISSION | Intent.FLAG_GRANT_READ_URI_PERMISSION);
		}
	}

	/**
	 * 打开文件
	 *
	 * @param context  上下文环境变量
	 * @param filePath 文件全路径
	 */
	public static void openFile(Context context, String filePath) throws Exception {

		if (TextUtils.isEmpty(filePath)) {
			Log.e("OpenFileUtil", "文件打开失败 路径为空");
			return;
		}
		Intent intent = new Intent(Intent.ACTION_VIEW);
		intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
		// 设置intent的Action属性
		intent.setAction(Intent.ACTION_VIEW);
		// 获取文件file的MIME类型
		String mimeType = getMIMEType(new File(filePath));
		if ("*/*".equals(mimeType)) {
			mimeType = getMimeType(filePath);
		}

		File file = new File(filePath);
		if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
			Uri data = FileProvider.getUriForFile(context.getApplicationContext(), context.getPackageName() + ".autoupdate.fileprovider", file);
			intent.setFlags(Intent.FLAG_GRANT_WRITE_URI_PERMISSION);
			intent.setDataAndType(data, mimeType);
			grantUriPermission(context, data, intent);
		} else {
			intent.setDataAndType(Uri.fromFile(file), mimeType);
		}

		context.startActivity(intent);
	}

	public static String getMimeType(String filePath) {
		MediaMetadataRetriever mmr = new MediaMetadataRetriever();
		String mime = "*/*";
		if (filePath != null) {
			try {
				mmr.setDataSource(filePath);
				mime = mmr.extractMetadata(MediaMetadataRetriever.METADATA_KEY_MIMETYPE);
			} catch (IllegalStateException e) {
				return mime;
			} catch (IllegalArgumentException e) {
				return mime;
			} catch (RuntimeException e) {
				return mime;
			}
		}
		return mime;
	}

	/**
	 * 根据文件后缀名获得对应的MIME类型。
	 *
	 * @param file
	 */
	public static String getMIMEType(File file) {

		String type = "*/*";
		String fName = file.getName();
		// 获取后缀名前的分隔符"."在fName中的位置。
		int dotIndex = fName.lastIndexOf(".");
		if (dotIndex < 0) {
			return type;
		}
		/* 获取文件的后缀名 */
		String end = fName.substring(dotIndex, fName.length()).toLowerCase();
		if ("".equals(end)) {
			return type;
		}
		// 在MIME和文件类型的匹配表中找到对应的MIME类型。
		for (int i = 0; i < MIME_MapTable.length; i++) {
			if (end.equals(MIME_MapTable[i][0])) {
				type = MIME_MapTable[i][1];
			}
		}
		return type;
	}

	public static String[][] MIME_MapTable = {
			// {后缀名，MIME类型}
			{".3gp", "video/3gpp"}, {".apk", "application/vnd.android.package-archive"},
			{".asf", "video/x-ms-asf"}, {".avi", "video/x-msvideo"}, {".bin", "application/octet-stream"},
			{".bmp", "image/bmp"}, {".c", "text/plain"}, {".class", "application/octet-stream"},
			{".conf", "text/plain"}, {".cpp", "text/plain"}, {".doc", "application/msword"},
			{".docx", "application/vnd.openxmlformats-officedocument.wordprocessingml.document"},
			{".xls", "application/vnd.ms-excel"},
			{".xlsx", "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet"},
			{".exe", "application/octet-stream"}, {".gif", "image/gif"}, {".gtar", "application/x-gtar"},
			{".gz", "application/x-gzip"}, {".h", "text/plain"}, {".htm", "text/html"}, {".html", "text/html"},
			{".jar", "application/java-archive"}, {".java", "text/plain"}, {".jpeg", "image/jpeg"},
			{".jpg", "image/jpeg"}, {".js", "application/x-javascript"}, {".log", "text/plain"},
			{".m3u", "audio/x-mpegurl"}, {".m4a", "audio/mp4a-latm"}, {".m4b", "audio/mp4a-latm"},
			{".m4p", "audio/mp4a-latm"}, {".m4u", "video/vnd.mpegurl"}, {".m4v", "video/x-m4v"},
			{".mov", "video/quicktime"}, {".mp2", "audio/x-mpeg"}, {".mp3", "audio/x-mpeg"},
			{".mp4", "video/mp4"}, {".mpc", "application/vnd.mpohun.certificate"}, {".mpe", "video/mpeg"},
			{".mpeg", "video/mpeg"}, {".mpg", "video/mpeg"}, {".mpg4", "video/mp4"}, {".mpga", "audio/mpeg"},
			{".msg", "application/vnd.ms-outlook"}, {".ogg", "audio/ogg"}, {".pdf", "application/pdf"},
			{".png", "image/png"}, {".pps", "application/vnd.ms-powerpoint"},
			{".ppt", "application/vnd.ms-powerpoint"},
			{".pptx", "application/vnd.openxmlformats-officedocument.presentationml.presentation"},
			{".prop", "text/plain"}, {".rc", "text/plain"}, {".rmvb", "audio/x-pn-realaudio"},
			{".rtf", "application/rtf"}, {".sh", "text/plain"}, {".tar", "application/x-tar"},
			{".tgz", "application/x-compressed"}, {".txt", "text/plain"}, {".wav", "audio/x-wav"},
			{".wma", "audio/x-ms-wma"}, {".wmv", "audio/x-ms-wmv"}, {".wps", "application/vnd.ms-works"},
			{".xml", "text/plain"}, {".z", "application/x-compress"}, {".zip", "application/x-zip-compressed"},
			{".aac", "audio/aac"}, {".jpe", "image/jpeg"},
			{".amr", "audio/amr"}, {".mid", "audio/mid"},
			{"", "*/*"}};


	@NonNull
	public static File getDiskCacheDir(Context context, String uniqueName) {
		String cachePath = null;
		if (Environment.MEDIA_MOUNTED.equals(Environment.getExternalStorageState())
				|| !Environment.isExternalStorageRemovable()) {
			File externalCacheDir = context.getExternalCacheDir();
			if (externalCacheDir != null) {
				cachePath = externalCacheDir.getPath();
			}
		}
		if (cachePath == null) {
			cachePath = context.getCacheDir().getPath();
		}
		File file = new File(cachePath + File.separator + uniqueName);
		if (!file.exists()) {
			file.mkdir();
		}
		return file;
	}

	public static String generateAccessableFileName(@NonNull final String dirPath, @NonNull final String fileName) {
		String fName = fileName;
		File targetFile = new File(dirPath, fName);
		if (!targetFile.exists()) {
			return fName;
		}
		Matcher m = PATTERN.matcher(fName);
		if (m.matches()) {
			String prefix = m.group(1);
			String last = m.group(2);
			String suffix = m.group(3);
			if (suffix == null) {
				suffix = "";
			}
			int count = last != null ? Integer.parseInt(last) : 0;
			do {
				count++;
				fName = prefix + "(" + count + ")" + suffix;
				targetFile = new File(dirPath, fName);
			} while (targetFile.exists());
		}
		return fName;
	}

	/**
	 * 获取附件图标
	 *
	 * @param filePath 附件全路径
	 * @return 附件图标
	 */
	public static int getIcon(String filePath) {
		if (TextUtils.isEmpty(filePath)) {
			return R.mipmap.fileselector_file_icon_unknow;
		}
		if (filePath.endsWith(".doc") || filePath.endsWith(".docx")) {
			return R.mipmap.fileselector_file_icon_word;
		} else if (filePath.endsWith(".xls") || filePath.endsWith(".xlsx")) {
			return R.mipmap.fileselector_file_icon_excel;
		} else if (filePath.endsWith(".txt")) {
			return R.mipmap.fileselector_file_icon_txt;
		} else if (filePath.endsWith(".ppt")) {
			return R.mipmap.fileselector_file_icon_ppt;
		} else if (filePath.endsWith(".rar") || filePath.endsWith(".zip") || filePath.endsWith(".7z")) {
			return R.mipmap.fileselector_file_icon_zip;
		} else if (filePath.endsWith(".pdf")) {
			return R.mipmap.fileselector_file_icon_pdf;
		} else if (filePath.endsWith(".apk")) {
			return R.mipmap.fileselector_file_icon_apk;
		} else if (MediaFile.isAudioFileType(filePath)) {
			return R.mipmap.fileselector_file_icon_audio;
		} else if (MediaFile.isVideoFileType(filePath)) {
			return R.mipmap.fileselector_file_icon_video;
		} else {
			return R.mipmap.fileselector_file_icon_unknow;
		}
	}

	/**
	 * Returns all available SD-Cards in the system (include emulated)
	 * <p>
	 * Warning: Hack! Based on Android source code of version 4.3 (API 18)
	 * Because there is no standard way to get it.
	 * TODO: Test on future Android versions 4.4+
	 *
	 * @return paths to all available SD-Cards in the system (include emulated)
	 */
	public static synchronized List<String> getStorageDirectories(Context context) {
		// Final set of paths
		final ArrayList<String> rv = new ArrayList<>();
		// Primary physical SD-CARD (not emulated)
		final String rawExternalStorage = System.getenv("EXTERNAL_STORAGE");
		// All Secondary SD-CARDs (all exclude primary) separated by ":"
		final String rawSecondaryStoragesStr = System.getenv("SECONDARY_STORAGE");
		// Primary emulated SD-CARD
		final String rawEmulatedStorageTarget = System.getenv("EMULATED_STORAGE_TARGET");
		if (TextUtils.isEmpty(rawEmulatedStorageTarget)) {
			// Device has physical external storage; use plain paths.
			if (TextUtils.isEmpty(rawExternalStorage)) {
				// EXTERNAL_STORAGE undefined; falling back to default.
				rv.add("/storage/sdcard0");
			} else {
				rv.add(rawExternalStorage);
			}
		} else {
			// Device has emulated storage; external storage paths should have
			// userId burned into them.
			final String rawUserId;
			if (SDK_INT < Build.VERSION_CODES.JELLY_BEAN_MR1) {
				rawUserId = "";
			} else {
				final String path = Environment.getExternalStorageDirectory().getAbsolutePath();
				final String[] folders = DIR_SEPARATOR.split(path);
				final String lastFolder = folders[folders.length - 1];
				boolean isDigit = false;
				try {
					Integer.valueOf(lastFolder);
					isDigit = true;
				} catch (NumberFormatException ignored) {
				}
				rawUserId = isDigit ? lastFolder : "";
			}
			// /storage/emulated/0[1,2,...]
			if (TextUtils.isEmpty(rawUserId)) {
				rv.add(rawEmulatedStorageTarget);
			} else {
				rv.add(rawEmulatedStorageTarget + File.separator + rawUserId);
			}
		}
		// Add all secondary storages
		if (!TextUtils.isEmpty(rawSecondaryStoragesStr)) {
			// All Secondary SD-CARDs splited into array
			final String[] rawSecondaryStorages = rawSecondaryStoragesStr.split(File.pathSeparator);
			Collections.addAll(rv, rawSecondaryStorages);
		}
		if (SDK_INT >= Build.VERSION_CODES.M && checkStoragePermission(context)) {
			rv.clear();
		}
		if (SDK_INT >= Build.VERSION_CODES.KITKAT) {
			String strings[] = getExtSdCardPathsForActivity(context);
			for (String s : strings) {
				File f = new File(s);
				if (!rv.contains(s) && canListFiles(f)) {
					rv.add(s);
				}
			}
		}
		//        if (BaseActivity.rootMode)
		//            rv.add("/");
		//        File usb = getUsbDrive();
		//        if (usb != null && !rv.contains(usb.getPath())) rv.add(usb.getPath());
		//
		//        if (SDK_INT >= Build.VERSION_CODES.KITKAT) {
		//            if (isUsbDeviceConnected()) rv.add(OTGUtil.PREFIX_OTG + "/");
		//        }
		return rv;
	}

	public static boolean canListFiles(File f) {
		try {
			return f.canRead() && f.isDirectory();
		} catch (Exception e) {
			return false;
		}
	}

	public static boolean checkStoragePermission(Context context) {

		// Verify that all required contact permissions have been granted.
		return ActivityCompat.checkSelfPermission(context, Manifest.permission.WRITE_EXTERNAL_STORAGE)
				== PackageManager.PERMISSION_GRANTED;
	}

	@TargetApi(Build.VERSION_CODES.KITKAT)
	public static String[] getExtSdCardPathsForActivity(Context context) {
		List<String> paths = new ArrayList<>();
		for (File file : context.getExternalFilesDirs("external")) {
			if (file != null) {
				int index = file.getAbsolutePath().lastIndexOf("/Android/data");
				if (index < 0) {
					Log.w("AmazeFileUtils", "Unexpected external file dir: " + file.getAbsolutePath());
				} else {
					String path = file.getAbsolutePath().substring(0, index);
					try {
						path = new File(path).getCanonicalPath();
					} catch (IOException e) {
						// Keep non-canonical path.
					}
					paths.add(path);
				}
			}
		}
		if (paths.isEmpty()) {
			paths.add("/storage/sdcard1");
		}
		return paths.toArray(new String[0]);
	}

	private static boolean isArrayEmpty(File[] listFiles) {
		return listFiles == null || listFiles.length == 0;
	}

	public static int calcDirectorFiles(File[] files) {
		if (ArraysUtils.isArrayEmpty(files)) {
			return 0;
		}
		int count = 0;
		for (File file : files) {
			if (file.isHidden()) {
				continue;
			}
			count++;
		}
		return count;
	}

	/**
	 * 通过反射调用获取内置存储和外置sd卡根路径(通用)
	 *
	 * @param mContext    上下文
	 * @param is_removale 是否可移除，false返回内部存储，true返回外置sd卡
	 * @return
	 */
	private static String getSDStoragePath(Context mContext, boolean is_removale) {

		StorageManager mStorageManager = (StorageManager) mContext.getSystemService(Context.STORAGE_SERVICE);
		Class<?> storageVolumeClazz = null;
		try {
			storageVolumeClazz = Class.forName("android.os.storage.StorageVolume");
			Method getVolumeList = mStorageManager.getClass().getMethod("getVolumeList");
			Method getPath = storageVolumeClazz.getMethod("getPath");
			Method isRemovable = storageVolumeClazz.getMethod("isRemovable");
			Object result = getVolumeList.invoke(mStorageManager);
			final int length = Array.getLength(result);
			for (int i = 0; i < length; i++) {
				Object storageVolumeElement = Array.get(result, i);
				String path = (String) getPath.invoke(storageVolumeElement);
				boolean removable = (Boolean) isRemovable.invoke(storageVolumeElement);
				if (is_removale == removable) {
					return path;
				}
			}
		} catch (ClassNotFoundException e) {
			e.printStackTrace();
		} catch (InvocationTargetException e) {
			e.printStackTrace();
		} catch (NoSuchMethodException e) {
			e.printStackTrace();
		} catch (IllegalAccessException e) {
			e.printStackTrace();
		}
		return null;
	}

	public static List<String> getStoragesPath(Context context) {
		List<String> res = new ArrayList<>();
		String extenalStoragePath = getSDStoragePath(context, false);
		if (!TextUtils.isEmpty(extenalStoragePath)) {
			res.add(extenalStoragePath);
		}
		String sdStoragePath = getSDStoragePath(context, true);
		if (!TextUtils.isEmpty(sdStoragePath)) {
			res.add(sdStoragePath);
		}
		return res;
	}

	private static final String[] ImageType = new String[]{"png", "jpg", "jpeg", "bmp"};

	/**
	 * 是否为图片格式
	 *
	 * @param filePath 需要的路径
	 * @return
	 */
	public static boolean isImageType(String filePath) {
		filePath = filePath.toLowerCase();
		for (int i = 0; i < ImageType.length; i++) {
			if (filePath.endsWith(ImageType[i])) {
				return true;
			}
		}
		return false;
	}

	/**
	 * 筛选图片格式
	 *
	 * @param mSelectedImages 路径集合
	 * @return
	 */
	public static ArrayList<String> filterImageTypeList(ArrayList<String> mSelectedImages) {
		ArrayList<String> list = new ArrayList<>();
		for (String item : mSelectedImages) {
			if (!isImageType(item)) {
				list.add(item);
			}
		}
		if (!list.isEmpty()) {
			mSelectedImages.removeAll(list);
		}
		return mSelectedImages;
	}
}
