package com.xdja.eoa.hostlib.util;

import android.content.Context;
import android.os.Bundle;
import android.os.Handler;
import android.os.Looper;
import android.text.TextUtils;
import android.util.Log;
import android.util.Pair;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.xdja.eoa.hostlib.bean.PluginConfig;
import com.xdja.eoa.hostlib.bean.PluginDownloadInfo;
import com.xdja.eoa.hostlib.inter.PluginUpdateInterface;
import com.xdja.plugin.shadow.nonedynamic.bean.InstalledShadowPluginInfo;
import com.xdja.plugin.shadow.nonedynamic.jar.JarShadowManager;

import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;

/**
 * 插件工具类 负责版本检测、apk下载安装、启动等具体业务
 * Created by @author zhuruyi on 2020/5/6
 */
public class OaHostPluginHelper {
    public static final String TAG = OaHostPluginHelper.class.getSimpleName();

    private Context applicationContext;

    private Handler mainH = new Handler(Looper.getMainLooper());

    private AtomicBoolean cancelFlag = new AtomicBoolean();

    ExecutorService fixedPool;

    private OaHostPluginHelper() {
        fixedPool = new ThreadPoolExecutor(2, 4, 30, TimeUnit.SECONDS, new LinkedBlockingQueue<Runnable>());
    }

    private static class SINGLETON {
        static OaHostPluginHelper instance = new OaHostPluginHelper();
    }

    public static OaHostPluginHelper getInstance() {
        return SINGLETON.instance;
    }

    public void init(Context applicationContext) {
        this.applicationContext = applicationContext;
    }

    /**
     * 获取当前获取已经安装的插件
     *
     * @return
     */
    public Future<List<InstalledShadowPluginInfo>> queryAllInstallPlugin() {
        Future<List<InstalledShadowPluginInfo>> future = fixedPool.submit(new Callable<List<InstalledShadowPluginInfo>>() {
            @Override
            public List<InstalledShadowPluginInfo> call() throws Exception {
                JarShadowManager jarShadowManager = JarShadowManager.getInstance(applicationContext);
                Pair<Integer, List<InstalledShadowPluginInfo>> pair = jarShadowManager.getAllInstalledPlugin();

                if (pair.first == 0) {
                    return pair.second;
                } else {//获取插件结果失败
                    return new ArrayList<>();
                }
            }
        });
        return future;
    }

    /**
     * 根据sn 和版本查询当前插件安装状态
     *
     * @param sn
     * @param version
     * @param callback
     */
    public void queryPlugin(String sn, String version, QueryCallback callback) {
        List<InstalledShadowPluginInfo> installedShadowPluginInfoList = null;
        try {
            installedShadowPluginInfoList = queryAllInstallPlugin().get();
        } catch (ExecutionException | InterruptedException e) {
            e.printStackTrace();
        }

        InstalledShadowPluginInfo result = null;

        if (installedShadowPluginInfoList != null) {
            for (InstalledShadowPluginInfo pluginInfo :
                    installedShadowPluginInfoList) {
                if (pluginInfo.partKey.equals(sn) && pluginInfo.version.equals(version)) {
                    result = pluginInfo;
                    break;
                }
            }
        }

        if (callback != null) {
            callback.onResult(result);
        }
    }

    /**
     * 打开某个插件
     *
     * @param partKey
     * @param className
     * @param bundle
     * @return {@link com.xdja.plugin.shadow.nonedynamic.jar.ErrorCode}
     */
    public synchronized int startPlugin(@NonNull String partKey, @Nullable String className, Bundle bundle) {
        return JarShadowManager.getInstance(applicationContext).startPluginActivity(partKey, className, bundle);
    }

    /**
     * 打开某个插件
     *
     * @param partKey
     * @param className
     * @return {@link com.xdja.plugin.shadow.nonedynamic.jar.ErrorCode}
     */
    public synchronized int startPlugin(@NonNull String partKey, @Nullable String className) {
        return startPlugin(partKey, className, null);
    }

    /**
     * 打开某个插件
     *
     * @param partKey
     */
    public synchronized void startPlugin(String partKey) {
        startPlugin(partKey, null);
    }


    /**
     * 插件安装
     *
     * @param localPath 本地apk目录
     * @param callback  安装结果回调
     */
    public void installPlugin(final String localPath, final CommonCallback callback) {
        if (applicationContext == null) {
            throw new IllegalStateException("must call init first in your application");
        }

        fixedPool.execute(new Runnable() {
            @Override
            public void run() {
                File pluginFile = new File(localPath);
                int resultCode = -1;
                if (pluginFile.exists()) {
                    final long localS = System.currentTimeMillis();
                    resultCode = JarShadowManager.getInstance(applicationContext).install(pluginFile);
                    Log.d(TAG, "install  plugin coast " + (System.currentTimeMillis() - localS) + " ms");
                    Log.d(TAG, "install plugin " + localPath + " , resultCode=" + resultCode);
                }

                final int result = resultCode == 0 ? CommonCallback.RESULT_OK : CommonCallback.RESULT_INSTALL_ERROR;
                mainH.post(new Runnable() {
                    @Override
                    public void run() {
                        if (callback != null) {
                            callback.onResult(result);
                        }
                    }
                });
            }
        });
    }

    /**
     * 插件卸载入口
     *
     * @param partKey
     * @param callback
     */
    public void uninstallPlugin(final String partKey, final CommonCallback callback) {
        if (applicationContext == null) {
            throw new IllegalStateException("must call init first in your application");
        }

        fixedPool.execute(new Runnable() {
            @Override
            public void run() {
                int ret = JarShadowManager.getInstance(applicationContext).uninstallPlugin(partKey);
                final int result = ret == 0 ? CommonCallback.RESULT_OK : CommonCallback.RESULT_UNINSTALL_ERROR;
                if (callback != null) {
                    callback.onResult(result);
                }
            }
        });
    }

    /**
     * 下载安装插件
     * 包含下载和安装两个过程
     */
    public void downloadAndInstall(final PluginUpdateInterface pluginUpdater, final FullFlowCallback installCallback) {
        Log.d(TAG, "will downloadAndInstall=" + pluginUpdater.getPluginDownloadInfo().getSn());
        fixedPool.submit(new Runnable() {
            @Override
            public void run() {
                if (installCallback != null) {
                    installCallback.onStart();
                }
                Log.d(TAG, "from [downloadAndInstall] => real  call doUpdate " + pluginUpdater.getPluginDownloadInfo().getSn());
                doUpdate(pluginUpdater, installCallback);
            }
        });
    }

    /**
     * 完整的插件流程， 检测下架、检测升级、检测安装、启动等完整流程
     *
     * @param installCallback
     */
    public synchronized void startPluginFullFlow(final PluginUpdateInterface pluginUpdater, final FullFlowCallback installCallback) {
        fixedPool.execute(new Runnable() {
            @Override
            public void run() {
                cancelFlag.set(false);

                if (installCallback != null) {
                    installCallback.onStart();
                }

                //检查是否需要升级
                boolean needUpdate = pluginUpdater.needUpdate(applicationContext);
                Log.d(TAG, "needUpdate =" + needUpdate);
                if (needUpdate) {
                    doUpdate(pluginUpdater, installCallback);
                } else {
                    checkInstall(pluginUpdater, installCallback);
                }
            }
        });
    }

    /**
     * 插件升级
     *
     * @param pluginUpdater   升级管理器
     * @param installCallback
     */
    private void doUpdate(final PluginUpdateInterface pluginUpdater, final FullFlowCallback installCallback) {
        final PluginDownloadInfo pluginDownloadInfo = pluginUpdater.getPluginDownloadInfo();
        if (pluginDownloadInfo == null) {
            if (installCallback != null) {
                installCallback.onResult(CommonCallback.RESULT_UNKNOWN_UPDATE_INFO);
            }
            return;
        }
        PluginZipManager.downloadPluginZipAndUnzip(applicationContext, pluginUpdater.getPluginDownloadInfo(), new PluginZipManager.DownloadCallback() {
            @Override
            public void onStart() {
                if (cancelFlag.get()) {
                    return;
                }

                Log.d(TAG, "start download " + pluginDownloadInfo.getSn());
            }

            @Override
            public void onError() {
                if (cancelFlag.get()) {
                    return;
                }
                Log.d(TAG, " downloadPluginZipAndUnzip failed");
                installCallback.onResult(CommonCallback.RESULT_DOWNLOAD_ERROR);
            }

            @Override
            public void onSuccess(String unzipDir) {
                if (cancelFlag.get()) {
                    return;
                }

                //获取插件包解压后的插件apk路径
                File unzipDirFile = new File(unzipDir);
                File[] listFile = unzipDirFile.listFiles();
                if (listFile == null) {
                    Log.d(TAG, " unzipDir has no child ");
                    return;
                }

                String pluginPath = null;
                for (File f : listFile) {
                    if (f.getName().endsWith("apk")) {
                        pluginPath = f.getAbsolutePath();
                        Log.d(TAG, " unzip child file= " + f.getAbsolutePath());
                        break;
                    }
                }

                if (TextUtils.isEmpty(pluginPath)) {
                    Log.d(TAG, " can't find plugin, may download fail or unzip fail ");
                    return;
                }

                if (installCallback != null) {
                    installCallback.onInstall();
                }

                //安装插件
                installPlugin(pluginPath, new CommonCallback() {
                    @Override
                    public void onResult(int result) {
                        if (cancelFlag.get()) {
                            return;
                        }

                        if (result == RESULT_OK) {
                            pluginUpdater.updateSuccess();

                            // 启动插件
                            if (installCallback != null) {
                                installCallback.onResult(RESULT_OK);
                            }
                        } else {
                            Log.d(TAG, "install plugin fail ");
                            // 启动插件
                            if (installCallback != null) {
                                installCallback.onResult(RESULT_INSTALL_ERROR);
                            }
                        }
                    }
                });
            }

            @Override
            public void onProgress(int progress) {
                if (cancelFlag.get()) {
                    return;
                }

                if (installCallback != null) {
                    installCallback.onDownload(progress);
                }
            }
        });
    }

    /**
     * 检测是否安装
     *
     * @param pluginUpdater
     * @param installCallback
     */
    private void checkInstall(final PluginUpdateInterface pluginUpdater, final FullFlowCallback installCallback) {
        fixedPool.submit(new Runnable() {
            @Override
            public void run() {
                //获取插件解压目录下 config.json ，解析版本信息
                String pluginDir = PluginZipManager.getPluginUnzipDir(applicationContext, pluginUpdater.getPluginDownloadInfo().getSn(), pluginUpdater.getPluginDownloadInfo().getMd5());
                String configPath = null;
                File[] children = new File(pluginDir).listFiles();
                if (children == null) {
                    if (installCallback != null) {
                        installCallback.onResult(CommonCallback.RESULT_CONFIG_ERROR);
                    }
                    return;
                }

                for (File f :
                        children) {
                    if (f.getName().endsWith("json")) {
                        configPath = f.getAbsolutePath();
                        break;
                    }
                }
                //获取插件解压目录下 config.json ，解析版本信息
                PluginConfig pluginConfig = PluginConfigParseUtil.parse(configPath);
                queryPlugin(pluginUpdater.getPluginDownloadInfo().getSn(), pluginConfig.getVersion(), new OaHostPluginHelper.QueryCallback() {
                    @Override
                    public void onResult(InstalledShadowPluginInfo oaPluginEntity) {
                        boolean pluginInstall = oaPluginEntity != null;
                        Log.d(TAG, "pluginInstall=" + pluginInstall);
                        if (pluginInstall) {
                            //启动插件
                            if (installCallback != null) {
                                installCallback.onResult(CommonCallback.RESULT_OK);
                            }
                        } else {
                            doUpdate(pluginUpdater, installCallback);
                        }
                    }
                });
            }
        });
    }

    /**
     * 取消工作
     */
    public void cancelFlow() {
        cancelFlag.set(true);
    }

    public interface QueryCallback {
        void onResult(InstalledShadowPluginInfo oaPluginEntity);
    }

    public interface CommonCallback {
        int RESULT_OK = 0;
        int RESULT_CONFIG_ERROR = -1;
        int RESULT_UNKNOWN_UPDATE_INFO = -2;
        int RESULT_DOWNLOAD_ERROR = -3;
        int RESULT_INSTALL_ERROR = -4;
        int RESULT_START_ERROR = -5;
        int RESULT_UNINSTALL_ERROR = -6;

        /**
         * 通用回调
         *
         * @param result 1
         */
        void onResult(int result);
    }

    public interface FullFlowCallback extends CommonCallback {
        /**
         * 下载进度
         *
         * @param progress
         */
        void onDownload(int progress);

        /**
         * 正在安装
         */
        void onInstall();

        /**
         * 业务流程开始
         */
        void onStart();
    }

    public static class BaseFullFlowCallback implements FullFlowCallback {

        @Override
        public void onResult(int result) {

        }

        @Override
        public void onDownload(int progress) {

        }

        @Override
        public void onInstall() {

        }

        @Override
        public void onStart() {

        }
    }
}
