package com.xdja.eoa.sdk;

import android.content.Context;
import android.util.ArrayMap;

import com.alibaba.fastjson.JSON;
import com.xdja.eoa.sdk.cb.ICallbackListener;
import com.xdja.eoa.sdk.constant.MethodConstant;
import com.xdja.eoa.sdk.constant.SdkConstant;
import com.xdja.eoa.sdk.constant.SdkLog;
import com.xdja.eoa.sdk.data.AppInfo;
import com.xdja.eoa.sdk.data.SynHelper;
import com.xdja.eoa.sdk.data.param.ApiAccessConfigParam;
import com.xdja.eoa.sdk.data.param.H5AppStartParams;
import com.xdja.eoa.sdk.data.param.LocationDetailWithRangeStartParams;
import com.xdja.eoa.sdk.data.result.AccountInfo;
import com.xdja.eoa.sdk.data.result.AuthCodeResult;
import com.xdja.eoa.sdk.module.ModuleApplication;

import java.util.concurrent.CountDownLatch;

/**
 * @author jarlen
 * @date 2020/4/30
 * SDK 工具类
 */
public class EoaModuleSdk {

    private ArrayMap<String, IModuleFacade> facadeCache;

    private static class SingletonHolder {
        private static EoaModuleSdk INSTANCE = new EoaModuleSdk();
    }

    private EoaModuleSdk() {
        facadeCache = new ArrayMap<>();
    }

    public static EoaModuleSdk getInstance() {
        return SingletonHolder.INSTANCE;
    }

    public IModuleSdkFacade getSdkFacade() {
        IModuleFacade moduleFacade = getFacade(SdkConstant.FACADE_SDK);
        if (moduleFacade == null) {
            SdkLog.w("getSdkFacade--->" + SdkConstant.FACADE_SDK + " is not registered");
            return null;
        }
        return (IModuleSdkFacade) moduleFacade;
    }

    /**
     * 注册服务
     *
     * @param facade
     */
    public void registerFacade(String facadeName, IModuleFacade facade) {
        if (facadeCache.containsKey(facadeName)) {
            SdkLog.w("registerFacade--->" + facadeName + " already register!");
            return;
        }
        facadeCache.put(facadeName, facade);
    }

    /**
     * 获取服务
     *
     * @param facadeName
     * @return 返回具体的服务，可能是null
     */
    public IModuleFacade getFacade(String facadeName) {
        if (facadeCache.containsKey(facadeName)) {
            return facadeCache.get(facadeName);
        } else {
            SdkLog.e("getFacade--->" + facadeName + " is not registered");
            return null;
        }
    }

    /**
     * 获取免登码
     *
     * @param context
     * @param moduleClass
     * @return
     */
    public static AuthCodeResult getAuthCode(Context context, Class<? extends ModuleApplication> moduleClass) {
        MethodParam methodParam = new MethodParam.Builder(moduleClass)
                .setMethod(MethodConstant.SDK_METHOD_GET_AUTH_CODE)
                .create();
        final SynHelper<AuthCodeResult> synHelper = new SynHelper<>();
        getInstance().getSdkFacade().call(context, methodParam, new ICallbackListener() {
            @Override
            public void onCallBackResult(int errorCode, String data, String errorMsg, String mark) {
                if (errorCode == MethodCallErrorCode.RET_SUCCESS) {
                    AuthCodeResult codeResult = JSON.parseObject(data, AuthCodeResult.class);
                    synHelper.setSynData(codeResult);
                }
            }
        });
        return synHelper.getSynData();
    }

    public static void configApiAccess(Context context, Class<? extends ModuleApplication> moduleClass, ApiAccessConfigParam configParam, ICallbackListener listener) {
        String params = "";
        try {
            params = JSON.toJSONString(configParam);
        } catch (Exception e) {
            e.printStackTrace();
        }
        MethodParam methodParam = new MethodParam.Builder(moduleClass)
                .setMethod(MethodConstant.SDK_METHOD_CONFIG_API_ACCESS)
                .setParams(params)
                .create();
        getInstance().getSdkFacade().call(context, methodParam, listener);
    }

    /**
     * 获取当前账户数据
     *
     * @param context
     * @param moduleClass
     * @return
     */
    public synchronized static AccountInfo getCurrentAccountInfo(Context context, Class<? extends ModuleApplication> moduleClass) {
        MethodParam methodParam = new MethodParam.Builder(moduleClass)
                .setMethod(MethodConstant.SDK_METHOD_GET_ACCOUNT_INFO)
                .create();
        final SynHelper<AccountInfo> synHelper = new SynHelper<>();
        getInstance().getSdkFacade().call(context, methodParam, new ICallbackListener() {
            @Override
            public void onCallBackResult(int errorCode, String data, String errorMsg, String mark) {
                AccountInfo accountInfo = null;
                if (errorCode == MethodCallErrorCode.RET_SUCCESS) {
                    try {
                        accountInfo = JSON.parseObject(data, AccountInfo.class);
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                    synHelper.setSynData(accountInfo);
                }
            }
        });
        return synHelper.getSynData();
    }

    /**
     * 启动应用
     *
     * @param context        应用上下文
     * @param moduleClass    ModuleApplication
     * @param targetAppSn    appSn
     * @param targetSomePage 跳转url
     */
    public static void startApp(Context context, Class<? extends ModuleApplication> moduleClass, String targetAppSn, String targetSomePage) {
        H5AppStartParams startParams = new H5AppStartParams();
        startParams.setAppSn(targetAppSn);
        startParams.setSomePage(targetSomePage);
        String param = JSON.toJSONString(startParams);
        MethodParam methodParam = new MethodParam.Builder(moduleClass)
                .setMethod(MethodConstant.SDK_METHOD_START_APP)
                .setParams(param)
                .create();
        getInstance().getSdkFacade().call(context, methodParam, null);
    }

    public static void openLocationDetailWithRange(Context context, Class<? extends ModuleApplication> moduleClass, LocationDetailWithRangeStartParams startParams) {
        String params = "";
        try {
            params = JSON.toJSONString(startParams);
        } catch (Exception e) {
            e.printStackTrace();
        }
        MethodParam methodParam = new MethodParam.Builder(moduleClass)
                .setMethod(MethodConstant.SDK_METHOD_START_LOCATION_DETAIL_WITH_RANGE)
                .setParams(params)
                .create();
        getInstance().getSdkFacade().call(context, methodParam, null);
    }

    public static AppInfo getAppInfo(Context context, Class<? extends ModuleApplication> moduleClass) {
        MethodParam methodParam = new MethodParam.Builder(moduleClass)
                .setMethod(MethodConstant.SDK_METHOD_GET_APP_INFO)
                .create();
        final SynHelper<AppInfo> synHelper = new SynHelper<>();
        final CountDownLatch countDownLatch = new CountDownLatch(1);
        getInstance().getSdkFacade().call(context, methodParam, new ICallbackListener() {
            @Override
            public void onCallBackResult(int errorCode, String data, String errorMsg, String mark) {
                AppInfo appInfo = null;
                if (MethodCallErrorCode.RET_SUCCESS == errorCode) {
                    try {
                        appInfo = JSON.parseObject(data, AppInfo.class);
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                }
                synHelper.setSynData(appInfo);
                countDownLatch.countDown();
            }
        });
        try {
            countDownLatch.await();
        } catch (InterruptedException e) {
            e.printStackTrace();
        }
        return synHelper.getSynData();
    }

}
