/**
 * Copyright 2014 Jeroen Mols
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.jmolsmobile.landscapevideocapture;

import android.app.Activity;
import android.content.Context;
import android.content.Intent;
import android.graphics.Bitmap;
import android.media.ThumbnailUtils;
import android.net.Uri;
import android.os.Bundle;
import android.provider.MediaStore.Video.Thumbnails;
import android.util.Log;
import android.view.OrientationEventListener;
import android.view.Window;
import android.view.WindowManager;
import android.widget.Toast;

import com.jmolsmobile.landscapevideocapture.camera.CameraWrapper;
import com.jmolsmobile.landscapevideocapture.configuration.CaptureConfiguration;
import com.jmolsmobile.landscapevideocapture.configuration.PredefinedCaptureConfigurations;
import com.jmolsmobile.landscapevideocapture.recorder.VideoRecorder;
import com.jmolsmobile.landscapevideocapture.recorder.VideoRecorderInterface;
import com.jmolsmobile.landscapevideocapture.view.RecordingButtonInterface;
import com.jmolsmobile.landscapevideocapture.view.VideoCaptureView;

import com.xdja.eoa.vca.R;
import java.io.File;

public class VideoCaptureActivity extends Activity implements RecordingButtonInterface, VideoRecorderInterface {

	public static final String		TAG	= "VideoCaptureActivity";

	public static final int			RESULT_ERROR				= 753245;

	private static String kB_UNIT_NAME = "KB";
	private static String B_UNIT_NAME = "B";
	private static String MB_UNIT_NAME = "MB";

	public static final String		EXTRA_OUTPUT_FILENAME		= "com.jmolsmobile.extraoutputfilename";
	public static final String		EXTRA_CAPTURE_CONFIGURATION	= "com.jmolsmobile.extracaptureconfiguration";
	public static final String		EXTRA_ERROR_MESSAGE			= "com.jmolsmobile.extraerrormessage";

	private static final String		SAVED_RECORDED_BOOLEAN		= "com.jmolsmobile.savedrecordedboolean";
	protected static final String	SAVED_OUTPUT_FILENAME		= "com.jmolsmobile.savedoutputfilename";

	private boolean					mVideoRecorded				= false;
	VideoFile						mVideoFile					= null;
	private CaptureConfiguration	mCaptureConfiguration;

	private VideoCaptureView		mVideoCaptureView;
	private VideoRecorder			mVideoRecorder;

	private final static int mFileDuration = 30;//视频最大持续时间（秒）
	private final static int mFilesize = 10;//文件最大Size
	private MyOrientationEventListener mOrientationListener;
	// The degrees of the device rotated clockwise from its natural orientation.
	private int mOrientation = OrientationEventListener.ORIENTATION_UNKNOWN;

	private class MyOrientationEventListener
			extends OrientationEventListener {
		public MyOrientationEventListener(Context context) {
			super(context);
		}

		@Override
		public void onOrientationChanged(int orientation) {
			// We keep the last known orientation. So if the user first orient
			// the camera then point the camera to floor or sky, we still have
			// the correct orientation.
			if (orientation == ORIENTATION_UNKNOWN) return;
			int newOrientation = Utils.roundOrientation(orientation, mOrientation);
			if (mOrientation != newOrientation) {
				mOrientation = newOrientation;
				if (mVideoRecorder != null) {
					mVideoRecorder.setmOrientation(mOrientation);
				}
			}
		}
	}

	@Override
	public void onCreate(final Bundle savedInstanceState) {
		super.onCreate(savedInstanceState);
		CLog.toggleLogging(this);
		requestWindowFeature(Window.FEATURE_NO_TITLE);
		getWindow().setFlags(WindowManager.LayoutParams.FLAG_FULLSCREEN, WindowManager.LayoutParams.FLAG_FULLSCREEN);
		setContentView(R.layout.activity_videocapture);
		initializeCaptureConfiguration(savedInstanceState);
		mVideoCaptureView = findViewById(R.id.videocapture_videocaptureview_vcv);
		if (mVideoCaptureView == null) return; // Wrong orientation
		mOrientationListener = new MyOrientationEventListener(this);
	}

	@Override
	protected void onResume() {
		mOrientationListener.enable();
		super.onResume();
		initializeRecordingUI();
	}

	/**
	 * 设置创建录制视频的相关参数
	 */
	private CaptureConfiguration createCaptureConfiguration() {
		final PredefinedCaptureConfigurations.CaptureResolution resolution = PredefinedCaptureConfigurations.CaptureResolution.RES_1080P;
		final PredefinedCaptureConfigurations.CaptureQuality quality = PredefinedCaptureConfigurations.CaptureQuality.LOW;
		final CaptureConfiguration config = new CaptureConfiguration(resolution, quality, mFileDuration, mFilesize);
		return config;
	}

	private void initializeCaptureConfiguration(final Bundle savedInstanceState) {
		mCaptureConfiguration = generateCaptureConfiguration();
		mVideoRecorded = generateVideoRecorded(savedInstanceState);
		mVideoFile = generateOutputFile(savedInstanceState);
	}

	private void initializeRecordingUI() {
		mVideoRecorder = new VideoRecorder(this, mCaptureConfiguration, mVideoFile, new CameraWrapper(),
				mVideoCaptureView.getPreviewSurfaceHolder());
		//录制时的取消、录制、发送按钮的点击事件
		mVideoCaptureView.setRecordingButtonInterface(this);
		Log.e(TAG, "initializeRecordingUI=========>" + mVideoRecorded);
		if (mVideoRecorded) {
			mVideoCaptureView.updateUIRecordingFinished(getVideoThumbnail());
		} else {
			mVideoCaptureView.updateUINotRecording();
		}
	}

	@Override
	protected void onPause() {
		mOrientationListener.disable();
		if (mVideoRecorder != null) {
			mVideoRecorder.stopRecording(null);
		}
		mVideoCaptureView.hideSurfaceView();
		releaseAllResources();
		super.onPause();
	}

	@Override
	public void onBackPressed() {
		if (mVideoRecorder.isRecording()){
			mVideoCaptureView.stopTimer();
			mVideoRecorder.toggleRecording();
			Log.e(TAG,"停止录制===========>");
		}else {
			finishCancelled();
		}
	}

	@Override
	public void onRecordButtonClicked() {//录制
		//del by renjingwei  在录制真正开始时  在开始计时
//		if (mVideoRecorder.isRecording()){//停止录制
//			mVideoCaptureView.stopTimer();
//			Log.e(TAG,"停止录制===========>");
//		} else {//开始录制
//			mVideoCaptureView.startTimerTask();
//			Log.e(TAG, "开始录制===========>");
//		}
		mVideoRecorder.toggleRecording();
	}

	@Override
	public void onAcceptButtonClicked() {//发送
		finishCompleted();
	}

	@Override
	public void onDeclineButtonClicked() {//取消
		finishCancelled();
	}

	@Override
	public void onRecordingStarted() {
		mVideoCaptureView.updateUIRecordingOngoing();

		mVideoCaptureView.startTimerTask();
		Log.e(TAG, "开始录制===========>");
	}

	@Override
	public void onPreviewButtonClicked(){//视频预览播放
		mVideoCaptureView.updateUIVideoPreview(mVideoFile.getFullPath());
	}

	@Override
	public void onExchangewButtonClicked(){//切换镜头
		mVideoRecorder.openFindBackOrFrontCamera();
	}

	@Override
	public void onRecordingStopped(String message) {
		if (message != null) {
			Toast.makeText(this, message, Toast.LENGTH_LONG).show();
		}

		mVideoCaptureView.stopTimer();
		Log.e(TAG,"停止录制===========>");

		mVideoCaptureView.updateVideoSize(getSizeString(getFileSize(mVideoFile.getFullPath())));
		mVideoCaptureView.updateUIRecordingFinished(getVideoThumbnail());
//		releaseAllResources();
	}

	@Override
	public void onRecordingSuccess() {
		mVideoRecorded = true;
	}

	@Override
	public void onRecordingFailed(String message) {
		finishError(message);
	}

	private void finishCompleted() {
		File file = new File(mVideoFile.getFullPath());
		Uri uri = Uri.fromFile(file);
		Intent intent = new Intent(Intent.ACTION_MEDIA_SCANNER_SCAN_FILE, uri);
		sendBroadcast(intent);
		final Intent result = new Intent();
		result.putExtra(EXTRA_OUTPUT_FILENAME, new String[] {mVideoFile.getFullPath()});
		this.setResult(RESULT_OK, result);
		finish();
	}
	@Override
	protected void onStop() {
		super.onStop();
		mVideoCaptureView.stopTimer();
	}

	private void finishError(final String message) {
		Toast.makeText(getApplicationContext(), "无法捕捉视频: " + message, Toast.LENGTH_LONG).show();
//		releaseAllResources();
		final Intent result = new Intent();
		result.putExtra(EXTRA_ERROR_MESSAGE, message);
		this.setResult(RESULT_ERROR, result);
		finish();
	}

	private void finishCancelled() {
		this.setResult(RESULT_CANCELED);
		finish();
	}

	private void releaseAllResources() {
		if (mVideoRecorder != null) {
			mVideoRecorder.releaseAllResources();
		}
	}

	@Override
	public void onSaveInstanceState(Bundle savedInstanceState) {
		savedInstanceState.putBoolean(SAVED_RECORDED_BOOLEAN, mVideoRecorded);
		savedInstanceState.putString(SAVED_OUTPUT_FILENAME, mVideoFile.getFullPath());
		super.onSaveInstanceState(savedInstanceState);
	}

	protected CaptureConfiguration generateCaptureConfiguration() {
		CaptureConfiguration returnConfiguration = createCaptureConfiguration();
		if (returnConfiguration == null) {
			returnConfiguration = new CaptureConfiguration();
			//没有定义CaptureConfiguration通过使用默认配置
			CLog.d(CLog.ACTIVITY, "No captureconfiguration passed - using default configuration");
		}
		return returnConfiguration;
	}

	private boolean generateVideoRecorded(final Bundle savedInstanceState) {
		if (savedInstanceState == null) return false;
		return savedInstanceState.getBoolean(SAVED_RECORDED_BOOLEAN, false);
	}

	protected VideoFile generateOutputFile(Bundle savedInstanceState) {
		VideoFile returnFile = null;
		if (savedInstanceState != null) {
			returnFile = new VideoFile(savedInstanceState.getString(SAVED_OUTPUT_FILENAME));
		} else {
			//为空时按统一命名格式
			returnFile = new VideoFile("");
		}
		// TODO: add checks to see if outputfile is writeable
		return returnFile;
	}

	public Bitmap getVideoThumbnail() {
		final Bitmap thumbnail = ThumbnailUtils.createVideoThumbnail(mVideoFile.getFullPath(),
				Thumbnails.FULL_SCREEN_KIND);
		if (thumbnail == null) {
			CLog.d(CLog.ACTIVITY, "Failed to generate video preview");
		}
		return thumbnail;
	}

	/**
	 * 获取文件大小
	 *
	 * @param filePath 文件路径
	 * @return -1 文件不存在 其他 文件大小
	 */
	public int getFileSize(String filePath) {
		int length = -1;
		File file = new File(filePath);
		if (file.exists()) {
			length = (int) file.length();
		}
		file = null;
		return length;
	}


	public String getSizeString(long size)
	{
		if (size == -1){
			return "";
		}

		if (size < 1024)
		{
			return String.valueOf(size) + B_UNIT_NAME;
		} else
		{
			size = size / 1024;
		}
		if (size < 1024)
		{
			return String.valueOf(size) + kB_UNIT_NAME;
		} else
		{
			size = size * 100 / 1024;
		}

		return String.valueOf((size / 100)) + "."
				+ ((size % 100) < 10 ? "0" : "") + String.valueOf((size % 100))
				+ MB_UNIT_NAME;
	}

}
