/**
 * Copyright 2014 Jeroen Mols
 * <p/>
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * <p/>
 * http://www.apache.org/licenses/LICENSE-2.0
 * <p/>
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.jmolsmobile.landscapevideocapture.recorder;

import android.media.CamcorderProfile;
import android.media.MediaRecorder;
import android.media.MediaRecorder.OnInfoListener;
import android.util.Log;
import android.view.OrientationEventListener;
import android.view.Surface;
import android.view.SurfaceHolder;

import com.jmolsmobile.landscapevideocapture.CLog;
import com.jmolsmobile.landscapevideocapture.VideoFile;
import com.jmolsmobile.landscapevideocapture.camera.CameraWrapper;
import com.jmolsmobile.landscapevideocapture.camera.OpenCameraException;
import com.jmolsmobile.landscapevideocapture.camera.PrepareCameraException;
import com.jmolsmobile.landscapevideocapture.camera.RecordingSize;
import com.jmolsmobile.landscapevideocapture.configuration.CaptureConfiguration;
import com.jmolsmobile.landscapevideocapture.preview.CapturePreview;
import com.jmolsmobile.landscapevideocapture.preview.CapturePreviewInterface;

import java.io.IOException;

public class VideoRecorder implements OnInfoListener, CapturePreviewInterface {

    private       CameraWrapper  mCameraWrapper;
    private final Surface        mPreviewSurface;
    private       CapturePreview mVideoCapturePreview;

    private final CaptureConfiguration mCaptureConfiguration;
    private final VideoFile            mVideoFile;
    private       SurfaceHolder        mPreviewHolder;

    private MediaRecorder mRecorder;
    private boolean mRecording = false;
    private final VideoRecorderInterface mRecorderInterface;

//    private static boolean isExchange = true;
    private int mOrientation = OrientationEventListener.ORIENTATION_UNKNOWN;

    public void setmOrientation(int mOrientation) {
        this.mOrientation = mOrientation;
    }

    public VideoRecorder(VideoRecorderInterface recorderInterface, CaptureConfiguration captureConfiguration, VideoFile videoFile,
                         CameraWrapper cameraWrapper, SurfaceHolder previewHolder) {
        this.mCaptureConfiguration = captureConfiguration;
        this.mRecorderInterface = recorderInterface;
        this.mVideoFile = videoFile;
        this.mCameraWrapper = cameraWrapper;
        this.mPreviewSurface = previewHolder.getSurface();
        this.mPreviewHolder  = previewHolder;
//        isExchange = true;
        initializeCameraAndPreview(previewHolder);
    }

    protected void initializeCameraAndPreview(SurfaceHolder previewHolder) {
        try {
            mCameraWrapper.openCamera();
        } catch (final OpenCameraException e) {
            e.printStackTrace();
            mRecorderInterface.onRecordingFailed("相机启动失败");
            return;
        }

        mVideoCapturePreview = new CapturePreview(this, mCameraWrapper, previewHolder);
    }

    public void toggleRecording() {
        if (isRecording()) {
            stopRecording(null);
        } else {
            startRecording();
        }
    }

    protected void startRecording() {
        mRecording = false;

        if (!initRecorder()) return;
        if (!prepareRecorder()) return;
        if (!startRecorder()) return;

        mRecording = true;
        mRecorderInterface.onRecordingStarted();
        CLog.d(CLog.RECORDER, "Successfully started recording - outputfile: " + mVideoFile.getFullPath());
    }

    public void stopRecording(String message) {
        if (!isRecording()) return;

        try {
            getMediaRecorder().stop();
            mRecorderInterface.onRecordingSuccess();
            CLog.d(CLog.RECORDER, "Successfully stopped recording - outputfile: " + mVideoFile.getFullPath());
        } catch (final RuntimeException e) {
            CLog.e(CLog.RECORDER, "Failed to stop recording");
        }
        mRecording = false;
        mRecorderInterface.onRecordingStopped(message);
    }

    public void openFindBackOrFrontCamera(){
        mCameraWrapper.openFindBackOrFrontCamera();
        try {
//            if (isExchange){
//                isExchange = false;
//            } else {
//                isExchange = true;
//            }
            mCameraWrapper.startPreview(mPreviewHolder);
            mVideoCapturePreview.setPreviewRunning(true);
        } catch (final IOException e) {
            e.printStackTrace();
            CLog.d(CLog.PREVIEW, "Failed to show preview - unable to connect camera to preview (IOException)");
            onCapturePreviewFailed();
        } catch (final RuntimeException e) {
            e.printStackTrace();
            CLog.d(CLog.PREVIEW, "Failed to show preview - unable to start camera preview (RuntimeException)");
            onCapturePreviewFailed();
        }

    }

    private boolean initRecorder() {
        try {
            mCameraWrapper.prepareCameraForRecording();
        } catch (final PrepareCameraException e) {
            e.printStackTrace();
            mRecorderInterface.onRecordingFailed("无法录制视频");
            CLog.e(CLog.RECORDER, "Failed to initialize recorder - " + e.toString());
            return false;
        }catch(Exception e1){
            e1.printStackTrace();
            CLog.e(CLog.RECORDER, "空指针引用 - " + e1.toString());
        }

        mRecorder = new MediaRecorder();
        if (mCameraWrapper!=null){
            configureMediaRecorder(getMediaRecorder(), mCameraWrapper.getCamera());
        }else {
            CLog.e(CLog.RECORDER, "错误===》err null==》" + mCameraWrapper);
        }
        CLog.d(CLog.RECORDER, "MediaRecorder successfully initialized");
        return true;
    }

    @SuppressWarnings("deprecation")
    protected void configureMediaRecorder(final MediaRecorder recorder, android.hardware.Camera camera) throws IllegalStateException, IllegalArgumentException {
        recorder.setCamera(camera);
        Log.e("configureMediaRecorder", "configureMediaRecorder=======>");
        // 将音频源设置为麦克风，视频来源为摄像头
        //在设置参数指定记录或编码器，称只有在设置输出文件的格式
        recorder.setAudioSource(mCaptureConfiguration.getAudioSource());
        recorder.setVideoSource(mCaptureConfiguration.getVideoSource());

        CamcorderProfile baseProfile = mCameraWrapper.getBaseRecordingProfile();
        baseProfile.fileFormat = mCaptureConfiguration.getOutputFormat();
        //视频宽高
        RecordingSize size = mCameraWrapper.getSupportedRecordingSize(mCaptureConfiguration.getVideoWidth(), mCaptureConfiguration.getVideoHeight());
        baseProfile.videoFrameWidth = size.width;
        baseProfile.videoFrameHeight = size.height;
        baseProfile.videoBitRate = mCaptureConfiguration.getVideoBitrate();

        baseProfile.audioCodec = mCaptureConfiguration.getAudioEncoder();
        baseProfile.videoCodec = mCaptureConfiguration.getVideoEncoder();
        //从一个记录CamcorderProfile对象的使用设置
        recorder.setProfile(baseProfile);
        // 设置录制的视频帧率。必须放在设置编码和格式的后面，否则报错
        recorder.setVideoFrameRate(baseProfile.videoFrameRate);
        //设置最大录制持续时间
        recorder.setMaxDuration(mCaptureConfiguration.getMaxCaptureDuration());
        // 设置输出文件方式： 直接本地存储
        recorder.setOutputFile(mVideoFile.getFullPath());
        //设置输出的视频播放的方向提示
        if (CameraWrapper.getBackOrFront()){//后置摄像头
            if (mOrientation != OrientationEventListener.ORIENTATION_UNKNOWN) {
                if (mOrientation == 0) {
                    recorder.setOrientationHint(90);//视频旋转90度
                } else if (mOrientation == 90) {
                    recorder.setOrientationHint(180);//视频旋转90度
                } else if (mOrientation == 180) {
                    recorder.setOrientationHint(270);//视频旋转90度
                } else if (mOrientation == 270) {
                    recorder.setOrientationHint(0);//视频旋转90度
                }
            }
        }else {
            //前置摄像头
            recorder.setOrientationHint(270);//视频旋转270度
        }

        try {
            //设置文件最大
            recorder.setMaxFileSize(mCaptureConfiguration.getMaxCaptureFileSize());
        } catch (IllegalArgumentException e) {
            CLog.e(CLog.RECORDER, "Failed to set max filesize - illegal argument: " + mCaptureConfiguration.getMaxCaptureFileSize());
        } catch (RuntimeException e2) {
            CLog.e(CLog.RECORDER, "Failed to set max filesize - runtime exception");
        }
        recorder.setOnInfoListener(this);
    }

    private boolean prepareRecorder() {
        try {
            getMediaRecorder().prepare();
            CLog.d(CLog.RECORDER, "MediaRecorder successfully prepared");
            return true;
        } catch (final IllegalStateException e) {
            e.printStackTrace();
            CLog.e(CLog.RECORDER, "MediaRecorder preparation failed - " + e.toString());
            return false;
        } catch (final IOException e) {
            e.printStackTrace();
            CLog.e(CLog.RECORDER, "MediaRecorder preparation failed - " + e.toString());
            return false;
        }
    }

    private boolean startRecorder() {
        try {
            getMediaRecorder().start();
            CLog.d(CLog.RECORDER, "MediaRecorder successfully started");
            return true;
        } catch (final IllegalStateException e) {
            e.printStackTrace();
            CLog.e(CLog.RECORDER, "MediaRecorder start failed - " + e.toString());
            return false;
        } catch (final RuntimeException e2) {
            e2.printStackTrace();
            CLog.e(CLog.RECORDER, "MediaRecorder start failed - " + e2.toString());
            mRecorderInterface.onRecordingFailed("视频录制参数不正确");
            return false;
        }
    }

    public boolean isRecording() {
        return mRecording;
    }

    protected MediaRecorder getMediaRecorder() {
        return mRecorder;
    }

    private void releaseRecorderResources() {
        MediaRecorder recorder = getMediaRecorder();
        if (recorder != null) {
            recorder.reset();
            recorder.release();
            recorder = null;
        }

    }

    public void releaseAllResources() {
        releaseRecorderResources();
        if (mVideoCapturePreview != null) {
            mVideoCapturePreview.releasePreviewResources();
        }
        if (mPreviewHolder != null) {
            mPreviewHolder.removeCallback(mVideoCapturePreview);
        }
        if (mCameraWrapper != null) {
            mCameraWrapper.releaseCamera();
            mCameraWrapper = null;
        }

        CLog.d(CLog.RECORDER, "Released all resources");
    }

    @Override
    public void onCapturePreviewFailed() {
        //无法显示相机预览Unable to show camera preview
        mRecorderInterface.onRecordingFailed("无法显示相机预览!");
    }

    @Override
    public void onInfo(MediaRecorder mr, int what, int extra) {
        switch (what) {
            case MediaRecorder.MEDIA_RECORDER_INFO_UNKNOWN:
                // NOP
                break;
            case MediaRecorder.MEDIA_RECORDER_INFO_MAX_DURATION_REACHED:
                CLog.d(CLog.RECORDER, "MediaRecorder max duration reached");
                stopRecording("停止拍摄-达到最大持续时间");
                break;
            case MediaRecorder.MEDIA_RECORDER_INFO_MAX_FILESIZE_REACHED:
                CLog.d(CLog.RECORDER, "MediaRecorder max filesize reached");
                stopRecording("停止拍摄-文件过大");
                break;
            default:
                break;
        }
    }

}