package com.xdja.eoa.sdk.module;

import android.content.Context;
import android.util.ArrayMap;

import com.alibaba.fastjson.JSON;
import com.xdja.eoa.sdk.MethodCallErrorCode;
import com.xdja.eoa.sdk.MethodConstant;
import com.xdja.eoa.sdk.SdkLog;
import com.xdja.eoa.sdk.cb.ICallbackListener;
import com.xdja.eoa.sdk.data.AppInfo;
import com.xdja.eoa.sdk.data.SynHelper;
import com.xdja.eoa.sdk.data.param.ApiAccessConfigParam;
import com.xdja.eoa.sdk.data.param.AppStartParams;
import com.xdja.eoa.sdk.data.param.LocationDetailWithRangeStartParams;
import com.xdja.eoa.sdk.data.result.AccountInfoResult;
import com.xdja.eoa.sdk.data.result.AuthCodeResult;

import java.util.concurrent.CountDownLatch;

/**
 * @author jarlen
 * @date 2020/4/30
 * 模块化SDK
 */
public class EoaModuleSdk {

    /**
     * 模块化服务缓存
     */
    private ArrayMap<String, IModuleFacade> facadeCache;

    private static class SingletonHolder {
        private static EoaModuleSdk INSTANCE = new EoaModuleSdk();
    }

    private EoaModuleSdk() {
        facadeCache = new ArrayMap<>();
    }

    public static EoaModuleSdk getInstance() {
        return SingletonHolder.INSTANCE;
    }

    public IModuleSdkFacade getSdkFacade() {
        IModuleFacade moduleFacade = getFacade(SdkConstant.FACADE_SDK);
        if (moduleFacade == null) {
            SdkLog.w("getSdkFacade--->" + SdkConstant.FACADE_SDK + " is not registered");
            return null;
        }
        return (IModuleSdkFacade) moduleFacade;
    }

    /**
     * 注册服务
     *
     * @param facade 接口服务
     */
    public void registerFacade(String facadeName, IModuleFacade facade) {
        if (facadeCache.containsKey(facadeName)) {
            SdkLog.w("registerFacade--->" + facadeName + " already register!");
            return;
        }
        facadeCache.put(facadeName, facade);
    }

    /**
     * 获取服务
     *
     * @param facadeName 接口服务名称
     * @return 返回具体的服务，可能是null
     */
    public IModuleFacade getFacade(String facadeName) {
        if (facadeCache.containsKey(facadeName)) {
            return facadeCache.get(facadeName);
        } else {
            SdkLog.e("getFacade--->" + facadeName + " is not registered");
            return null;
        }
    }

    /**
     * 获取免登码
     *
     * @param context     应用上下文
     * @param moduleClass 模块配置类
     * @return
     */
    public static AuthCodeResult getAuthCode(Context context, Class<? extends IModuleApplication> moduleClass) {
        MethodParam methodParam = new MethodParam.Builder(moduleClass)
                .setMethod(MethodConstant.SDK_METHOD_GET_AUTH_CODE)
                .create();
        final SynHelper<AuthCodeResult> synHelper = new SynHelper<>();
        getInstance().getSdkFacade().call(context, methodParam, new ICallbackListener() {
            @Override
            public void onCallBackResult(int errorCode, String data, String errorMsg, String mark) {
                if (errorCode == MethodCallErrorCode.RET_SUCCESS) {
                    AuthCodeResult codeResult = JSON.parseObject(data, AuthCodeResult.class);
                    synHelper.setSynData(codeResult);
                }
            }
        });
        return synHelper.getSynData();
    }

    /**
     * 接口鉴权
     *
     * @param context     应用上下文
     * @param moduleClass 模块配置类
     * @param configParam 接口鉴权参数
     * @param listener    接口回调
     */
    public static void configApiAccess(Context context, Class<? extends IModuleApplication> moduleClass, ApiAccessConfigParam configParam, ICallbackListener listener) {
        String params = "";
        try {
            params = JSON.toJSONString(configParam);
        } catch (Exception e) {
            e.printStackTrace();
        }
        MethodParam methodParam = new MethodParam.Builder(moduleClass)
                .setMethod(MethodConstant.SDK_METHOD_CONFIG_API_ACCESS)
                .setParams(params)
                .create();
        getInstance().getSdkFacade().call(context, methodParam, listener);
    }

    /**
     * 获取当前账户数据
     *
     * @param context     应用上下文
     * @param moduleClass 模块配置类
     * @return 账户数据
     */
    public synchronized static AccountInfoResult getCurrentAccountInfo(Context context, Class<? extends IModuleApplication> moduleClass) {
        MethodParam methodParam = new MethodParam.Builder(moduleClass)
                .setMethod(MethodConstant.SDK_METHOD_GET_ACCOUNT_INFO)
                .create();
        final SynHelper<AccountInfoResult> synHelper = new SynHelper<>();
        getInstance().getSdkFacade().call(context, methodParam, new ICallbackListener() {
            @Override
            public void onCallBackResult(int errorCode, String data, String errorMsg, String mark) {
                AccountInfoResult accountInfoResult = null;
                if (errorCode == MethodCallErrorCode.RET_SUCCESS) {
                    try {
                        accountInfoResult = JSON.parseObject(data, AccountInfoResult.class);
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                    synHelper.setSynData(accountInfoResult);
                }
            }
        });
        return synHelper.getSynData();
    }

    /**
     * 启动应用
     *
     * @param context     应用上下文
     * @param moduleClass 模块配置类
     * @param startParams 跳转参数
     */
    public static void startApp(Context context, Class<? extends IModuleApplication> moduleClass, AppStartParams startParams) {
        String param = "";
        try {
            param = JSON.toJSONString(startParams);
        } catch (Exception e) {
            e.printStackTrace();
        }
        MethodParam methodParam = new MethodParam.Builder(moduleClass)
                .setMethod(MethodConstant.SDK_METHOD_START_APP)
                .setParams(param)
                .create();
        getInstance().getSdkFacade().call(context, methodParam, null);
    }

    /**
     * 打开带有范围的地图详情
     *
     * @param context     应用上下文
     * @param moduleClass 模块配置类
     * @param startParams 打开参数
     */
    public static void openLocationDetailWithRange(Context context, Class<? extends IModuleApplication> moduleClass, LocationDetailWithRangeStartParams startParams) {
        String params = "";
        try {
            params = JSON.toJSONString(startParams);
        } catch (Exception e) {
            e.printStackTrace();
        }
        MethodParam methodParam = new MethodParam.Builder(moduleClass)
                .setMethod(MethodConstant.SDK_METHOD_START_LOCATION_DETAIL_WITH_RANGE)
                .setParams(params)
                .create();
        getInstance().getSdkFacade().call(context, methodParam, null);
    }

    /**
     * 获取应用信息
     *
     * @param context     应用上下文
     * @param moduleClass 模块配置类
     * @return 应用信息
     */
    public static AppInfo getAppInfo(Context context, Class<? extends IModuleApplication> moduleClass) {
        MethodParam methodParam = new MethodParam.Builder(moduleClass)
                .setMethod(MethodConstant.SDK_METHOD_GET_APP_INFO)
                .create();
        final SynHelper<AppInfo> synHelper = new SynHelper<>();
        final CountDownLatch countDownLatch = new CountDownLatch(1);
        getInstance().getSdkFacade().call(context, methodParam, new ICallbackListener() {
            @Override
            public void onCallBackResult(int errorCode, String data, String errorMsg, String mark) {
                AppInfo appInfo = null;
                if (MethodCallErrorCode.RET_SUCCESS == errorCode) {
                    try {
                        appInfo = JSON.parseObject(data, AppInfo.class);
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                }
                synHelper.setSynData(appInfo);
                countDownLatch.countDown();
            }
        });
        try {
            countDownLatch.await();
        } catch (InterruptedException e) {
            e.printStackTrace();
        }
        return synHelper.getSynData();
    }

}
