package com.xdja.eoa.sdk.remote;

import android.content.ContentResolver;
import android.content.Context;
import android.net.Uri;
import android.os.Bundle;
import android.os.IBinder;
import android.os.Process;
import android.os.RemoteException;

import com.xdja.eoa.sdk.remote.cb.ICallbackListener;

import static com.xdja.eoa.sdk.remote.MethodCallErrorCode.RET_ERROR_CANNOT_CALL_SERVER;
import static com.xdja.eoa.sdk.remote.MethodCallErrorCode.RET_ERROR_CANNOT_FOUND_SERVER;

/**
 * @author jarlen
 * @date 2020/3/20
 * 客户端接口
 */
public class OaRemoteSDK {

    private Uri mUri;

    private OaRemoteSDK(Uri uri) {
        this.mUri = uri;
    }

    public static final class Builder {
        private Uri uri;
        private static final Uri URI_DEFAULT = Uri.parse("content://com.xdja.eoa.sdk.SdkBinderProvider");

        public Builder baseUri(Uri uri) {
            if (uri == null) {
                throw new NullPointerException("uri is null!");
            }
            this.uri = uri;
            return this;
        }

        public OaRemoteSDK create() {
            if (uri == null) {
                uri = URI_DEFAULT;
            }
            return new OaRemoteSDK(uri);
        }
    }

    private IRemoteSdk oaSDKBinder;
    private static OaRemoteSDK INSTANCE = null;

    public synchronized static OaRemoteSDK getInstance() {
        if (INSTANCE == null) {
            INSTANCE = new OaRemoteSDK.Builder().create();
        }
        return INSTANCE;
    }

    public void call(Context context, RemoteMethodParam remoteMethodParam, ICallbackListener listener) {
        try {
            IRemoteSdk sdkBinder = getBinder(context);
            if (sdkBinder == null) {
                if (listener != null) {
                    listener.callback(RET_ERROR_CANNOT_FOUND_SERVER, "", RemoteSdkUtils.getErrorMsgByCode(RET_ERROR_CANNOT_FOUND_SERVER), remoteMethodParam.getMark());
                }
                return;
            }
            Bundle bundle = RemoteSdkUtils.getBundleForCallBinder(remoteMethodParam, listener);
            Bundle result = sdkBinder.call(bundle);
            int ret = result.getInt(RemoteSdkConstant.KEY_INT_RET);
            if (ret != MethodCallErrorCode.RET_SUCCESS) {
                if (listener != null) {
                    listener.callback(ret, "", RemoteSdkUtils.getErrorMsgByCode(ret), remoteMethodParam.getMark());
                }
            } else {
                SdkLog.d("通知接口服务提供者调用方法--->成功");
            }
        } catch (RemoteException e) {
            SdkLog.w(e);
            if (listener != null) {
                listener.callback(RET_ERROR_CANNOT_CALL_SERVER, "", RemoteSdkUtils.getErrorMsgByCode(RET_ERROR_CANNOT_CALL_SERVER), remoteMethodParam.getMark());
            }
        } catch (Exception e) {
            SdkLog.w(e);
            if (listener != null) {
                listener.callback(RET_ERROR_CANNOT_CALL_SERVER, "", RemoteSdkUtils.getErrorMsgByCode(RET_ERROR_CANNOT_CALL_SERVER), remoteMethodParam.getMark());
            }
        }
    }

    private synchronized IRemoteSdk getBinder(Context context) {
        if (oaSDKBinder != null) {
            return oaSDKBinder;
        }
        if (context == null) {
            throw new NullPointerException("context is null");
        }
        ContentResolver resolver = context.getContentResolver();
        Bundle extra = new Bundle();
        extra.putInt(RemoteSdkConstant.KEY_INT_UID, Process.myUid());
        extra.putInt(RemoteSdkConstant.KEY_INT_PID, Process.myPid());
        Bundle bundle;
        try {
            bundle = resolver.call(mUri, RemoteSdkConstant.KEY_METHOD_GET_BINDER, RemoteSdkConstant.KEY_METHOD_GET_BINDER_OA_SDK, extra);
        } catch (IllegalArgumentException e) {
            SdkLog.w(e);
            return null;
        } catch (SecurityException e) {
            SdkLog.w(e);
            return null;
        } catch (Exception e) {
            SdkLog.w(e);
            return null;
        }
        if (bundle == null) {
            return null;
        }
        IBinder binder = bundle.getBinder(RemoteSdkConstant.KEY_METHOD_GET_BINDER_OA_SDK);
        if (binder == null) {
            return null;
        }
        oaSDKBinder = IRemoteSdk.Stub.asInterface(binder);
        try {
            binder.linkToDeath(new IBinder.DeathRecipient() {
                @Override
                public void binderDied() {
                    oaSDKBinder = null;
                }
            }, 0);
        } catch (RemoteException e) {
            SdkLog.w(e);
        }
        return oaSDKBinder;
    }
}
