package com.xdja.eoa.sdk.remote;

import android.content.ContentResolver;
import android.content.Context;
import android.net.Uri;
import android.os.Bundle;
import android.os.IBinder;
import android.os.Process;
import android.os.RemoteException;
import android.util.Log;

import com.xdja.eoa.sdk.remote.cb.ICallbackListener;

/**
 * @author jarlen
 * @date 2020/3/20
 * 客户端接口
 */
public class OaRemoteSDK {

    private Uri mUri;

    private OaRemoteSDK(Uri uri) {
        this.mUri = uri;
    }

    public static final class Builder {
        private Uri uri;
        private static final Uri URI_DEFAULT = Uri.parse("content://com.xdja.eoa.sdk.SdkBinderProvider");

        public Builder baseUri(Uri uri) {
            if (uri == null) {
                throw new NullPointerException("uri is null!");
            }
            this.uri = uri;
            return this;
        }

        public OaRemoteSDK create() {
            if (uri == null) {
                uri = URI_DEFAULT;
            }
            return new OaRemoteSDK(uri);
        }
    }

    private IRemoteSdk oaSDKBinder;
    private static OaRemoteSDK INSTANCE = null;

    public static OaRemoteSDK getInstance() {
        if (INSTANCE == null) {
            synchronized (OaRemoteSDK.class) {
                if (INSTANCE == null) {
                    INSTANCE = new OaRemoteSDK.Builder().create();
                }
            }
        }
        return INSTANCE;
    }

    public int call(Context context, RemoteMethodParam remoteMethodParam, ICallbackListener listener) {
        try {
            IRemoteSdk sdkBinder = getBinder(context);
            if (sdkBinder == null) {
                return ErrorCode.RET_CANNOT_CALL_SERVER;
            }
            Bundle bundle = BundleHandle.getBundleForCallBinder(remoteMethodParam, listener);
            Bundle result = sdkBinder.call(bundle);
            return result.getInt(ParamConstant.KEY_INT_RET);
        } catch (ArithmeticException e) {
            Log.w(ParamConstant.TAG, e);
            return ErrorCode.RET_EXCEPTION;
        } catch (RemoteException e) {
            Log.w(ParamConstant.TAG, e);
            return ErrorCode.RET_EXCEPTION;
        } catch (Exception e) {
            Log.w(ParamConstant.TAG, e);
            return ErrorCode.RET_EXCEPTION;
        }
    }

    private synchronized IRemoteSdk getBinder(Context context) {
        if (oaSDKBinder != null) {
            return oaSDKBinder;
        }
        if (context == null) {
            throw new NullPointerException("context is null");
        }
        ContentResolver resolver = context.getContentResolver();
        Bundle extra = new Bundle();
        extra.putInt(ParamConstant.KEY_INT_UID, Process.myUid());
        extra.putInt(ParamConstant.KEY_INT_PID, Process.myPid());
        Bundle bundle;
        try {
            bundle = resolver.call(mUri, ParamConstant.KEY_METHOD_GET_BINDER, ParamConstant.KEY_METHOD_GET_BINDER_OA_SDK, extra);
        } catch (IllegalArgumentException e) {
            Log.w(ParamConstant.TAG, e);
            return null;
        } catch (SecurityException e) {
            Log.w(ParamConstant.TAG, e);
            return null;
        }
        if (bundle == null) {
            return null;
        }
        int ret = bundle.getInt(ParamConstant.KEY_INT_RET);
        if (ret != 0) {
            throw new ArithmeticException("" + ret);
        }
        IBinder binder = bundle.getBinder(ParamConstant.KEY_METHOD_GET_BINDER_OA_SDK);
        if (binder == null) {
            return null;
        }
        oaSDKBinder = IRemoteSdk.Stub.asInterface(binder);
        try {
            binder.linkToDeath(new IBinder.DeathRecipient() {
                @Override
                public void binderDied() {
                    oaSDKBinder = null;
                }
            }, 0);
        } catch (RemoteException e) {
            Log.w(ParamConstant.TAG, e);
        }
        return oaSDKBinder;
    }
}
