package com.xdja.update;

import android.content.Context;
import android.os.AsyncTask;
import android.support.annotation.NonNull;
import android.util.Log;

import com.xdja.update.bean.CheckResult;
import com.xdja.update.model.VersionConfigParentNodeInfo;
import com.xdja.update.model.VersionParentNodeInfo;

import java.util.concurrent.Executor;
import java.util.concurrent.Executors;


/**
 * <b>Description: 升级操作统一管理类</b>
 * Created by <a href="mailto:fjd@xdja.com">fanjiandong</a> on 2017/10/16 16:46.
 */
public class UpdateManager {

    /**
     * The constant TAG.
     */
    public static final String TAG = UpdateManager.class.getSimpleName();

    private static UpdateManager INSTANCE;

    /**
     * Gets instance.
     *
     * @return the instance
     */
    public static UpdateManager getInstance() {
        if (INSTANCE == null) {
            synchronized (UpdateManager.class) {
                if (INSTANCE == null) {
                    INSTANCE = new UpdateManager();
                }
            }
        }
        return INSTANCE;
    }

    private UpdateManager() {
    }

    private Executor exec;
    /**
     * application级别的Context上下文
     */
    private Context context;
    /**
     * baseUrl
     */
    private String baseUrl;

    /**
     * 检测升级流程
     */
    private CheckTask checkTask;
    /**
     * 文件下载流程
     */
    private DownloadTask downloadTask;

    /**
     * Sets base url.
     *
     * @param baseUrl {@link #baseUrl}
     */
    public void setBaseUrl(String baseUrl) {
        this.baseUrl = baseUrl;
    }

    /**
     * Gets base url.
     *
     * @return {@link #baseUrl}
     */
    public String getBaseUrl() {
        return baseUrl;
    }

    /**
     * Gets context.
     *
     * @return {@link #context}
     */
    public Context getContext() {
        return context;
    }

    /**
     * 初始化升级检测组件
     *
     * @param context 上下文
     */
    public void install(@NonNull Context context) {
        this.context = context.getApplicationContext();
        this.exec = Executors.newSingleThreadScheduledExecutor();
    }

    /**
     * 释放组件相关资源
     */
    public void uninstall() {
        this.context = null;
        this.exec = null;
        this.checkTask = null;
        this.downloadTask = null;
        this.baseUrl = null;
    }

    /**
     * 停止文件下载流程
     */
    public void stopDownload() {
        if (this.downloadTask != null
                && this.downloadTask.getStatus() != AsyncTask.Status.FINISHED) {
            this.downloadTask.stop();
        }
    }


    /**
     * 检测升级
     *
     * @param deviceId      设备id
     * @param checkCallback 升级回调
     */
    public void checkUpdate(String deviceId, @NonNull CheckCallback checkCallback) {
        if (isBusy()) {
            VersionUpdateResult versionUpdateResult = new VersionUpdateResult();
            versionUpdateResult.setResultCode(VersionUpdateErrorCode.Error_busy);
            checkCallback.onCheckError(versionUpdateResult);
            Log.d(TAG, "======有未执行完成的任务,本次检测任务取消=======");
        } else {
            Log.d(TAG, "======无待执行的任务,本次检测即将开始=======");
            this.checkTask = new CheckTask(context, deviceId,checkCallback);
            this.checkTask.executeOnExecutor(this.exec);
        }
    }

    public void cancelCheckUpdate() {
        if (this.checkTask != null) {
            checkTask.cancel(true);
        }
    }

    /**
     * 下载文件
     *
     * @param callback 文件下载回调
     */
    public void downloadFile(VersionConfigParentNodeInfo versionConfigParentNodeInfo, final DownloadCallback callback) {
        if (this.downloadTask != null
                && this.downloadTask.getStatus() != AsyncTask.Status.FINISHED) {
            Log.d(TAG, "======有未执行完成的任务,本次下载取消=======");
            return;
        }
        Log.d(TAG, "======无待执行的任务,本次下载即将开始=======");
        this.downloadTask = new DownloadTask(context, versionConfigParentNodeInfo, callback);
        this.downloadTask.executeOnExecutor(exec, getBaseUrl(), "");
    }

    public String alreadyDownload(String downloadDir, VersionConfigParentNodeInfo versionConfigParentNodeInfo) {
        DownloadTask downloadTask = new DownloadTask(context, versionConfigParentNodeInfo, null);
        if (downloadTask.isClientFileExists(downloadDir) && downloadTask.isFileComplete(downloadDir)) {
            return downloadTask.getFilePath(downloadDir);
        }
        return null;
    }

    /**
     * 检测当前是否有正在执行的任务
     *
     * @return 检测结果
     */
    private boolean isBusy() {
        return (this.checkTask != null
                && this.checkTask.getStatus() != AsyncTask.Status.FINISHED)
                || (this.downloadTask != null
                && this.downloadTask.getStatus() != AsyncTask.Status.FINISHED);
    }
}
