package com.xdja.hr.service.importData;

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.primitives.Ints;
import com.xdja.hr.domain.integration.FingerprintRecord;
import com.xdja.hr.domain.original.Employee;
import com.xdja.hr.pojo.Fingerprint;
import com.xdja.hr.utils.ExcelUtils;
import com.xdja.hr.utils.JsonUtils;
import org.apache.poi.openxml4j.exceptions.InvalidFormatException;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.ss.usermodel.WorkbookFactory;
import org.joda.time.DateTime;
import org.joda.time.LocalTime;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.StringUtils;

import java.io.IOException;
import java.io.InputStream;
import java.util.List;
import java.util.Map;

/**
 * User: landy
 * Date: 15/8/19
 * Time: 下午9:27
 */
class ImportFingerprint {
    private static final Logger logger = LoggerFactory.getLogger(ImportFingerprint.class);

    public static final DateTimeFormatter DATE_TIME_FORMATTER = DateTimeFormat.forPattern("yyyy-MM-dd");
    public static final DateTimeFormatter TIME_FORMATTER = DateTimeFormat.forPattern("HH:mm");

    /**
     * 解析指纹数据
     *
     * @param file
     */
    public static List<Fingerprint> parseFingerprintData(InputStream file) throws IOException, InvalidFormatException {
        Workbook wb = WorkbookFactory.create(file);
        String[][] sheetData = ExcelUtils.getSheetData(wb, 0);
        List<Fingerprint> rst = Lists.newLinkedList();

        int headerLen = sheetData[0].length;
        for (String[] one : sheetData) {
            Preconditions.checkArgument(one.length == headerLen);
        }

        String[] header = sheetData[0];

        for (int i = 1; i < sheetData.length; i++) {
            String[] row = sheetData[i];
            Map<String, String> tmp = Maps.newHashMap();
            for (int col = 0; col < header.length; col++) {
                tmp.put(header[col], row[col]);
            }

            Fingerprint fp = new Fingerprint();
            fp.setContent(JsonUtils.toJson(tmp));
            fp.setCreateDate(DATE_TIME_FORMATTER.parseDateTime(tmp.get("日期")).toDate());
            Integer employeeNo = Ints.tryParse(tmp.get("登记号码"));
            String employeeName = tmp.get("姓名");
            String department = tmp.get("部门");
            fp.setEmployee(new Employee(employeeNo, employeeName, department));

            rst.add(fp);
        }
        return rst;
    }

    /**
     * 合并日数据，合并后变成每天一条
     *
     * @param fingerprintList
     */
    public static Map<Employee, List<FingerprintRecord>> mergeDailyData(List<Fingerprint> fingerprintList) throws IOException {

        Map<Employee, List<Fingerprint>> all = Maps.newHashMap();
        for (Fingerprint one : fingerprintList) {
            if (!all.containsKey(one.getEmployee())) {
                all.put(one.getEmployee(), Lists.<Fingerprint>newLinkedList());
            }
            all.get(one.getEmployee()).add(one);
        }
        Map<Employee, List<FingerprintRecord>> rst = Maps.newHashMap();
        for (Employee employee : all.keySet()) {

            List<Fingerprint> monthData = all.get(employee);
            for (int i = 0; i < monthData.size(); i += 2) {
                Fingerprint fp1 = monthData.get(i);
                Map<String, String> one = JsonUtils.toStrMap(fp1.getContent());

                Fingerprint fp2 = monthData.get(i + 1);
                Map<String, String> other = JsonUtils.toStrMap(fp2.getContent());

                Map<String, String> morning = getMorning(one, other);
                Map<String, String> midday = getMidday(one, other);
                String th1 = morning.get("签到时间");
                String th2 = morning.get("签退时间");
                String th3 = midday.get("签到时间");
                String th4 = midday.get("签退时间");
                DateTime dateTime = new DateTime(fp1.getCreateDate());
                logger.debug(String.format("date:%s,week:%s,name:%s,no:%s,1:%s,2:%s,3:%s,4:%s", dateTime.toString("yyyy-MM-dd"), dateTime.dayOfWeek().get(), fp1.getEmployee().getEmployeeName(), employee, th1, th2, th3, th4));

                FingerprintRecord record = new FingerprintRecord();
                record.setEmployee(employee);
                record.setCreateDate(fp1.getCreateDate());
                if (StringUtils.hasText(th1)) {
                    LocalTime localTime = TIME_FORMATTER.parseLocalTime(th1);
                    record.setMorningTime(dateTime.plusHours(localTime.getHourOfDay()).plusMinutes(localTime.getMinuteOfHour()).toDate());
                }
                if (StringUtils.hasText(th2)) {
                    LocalTime localTime = TIME_FORMATTER.parseLocalTime(th2);
                    record.setMiddayTime1(dateTime.plusHours(localTime.getHourOfDay()).plusMinutes(localTime.getMinuteOfHour()).toDate());
                }
                if (StringUtils.hasText(th3)) {
                    LocalTime localTime = TIME_FORMATTER.parseLocalTime(th3);
                    record.setMiddayTime2(dateTime.plusHours(localTime.getHourOfDay()).plusMinutes(localTime.getMinuteOfHour()).toDate());
                }
                if (StringUtils.hasText(th4)) {
                    LocalTime localTime = TIME_FORMATTER.parseLocalTime(th4);
                    record.setAfternoonTime(dateTime.plusHours(localTime.getHourOfDay()).plusMinutes(localTime.getMinuteOfHour()).toDate());
                }
                if (!rst.containsKey(employee)) {
                    rst.put(employee, Lists.<FingerprintRecord>newArrayList());
                }
                rst.get(employee).add(record);
            }
        }
        return rst;
    }

    private static Map<String, String> getMorning(Map<String, String>... arr) {
        for (Map<String, String> one : arr) {
            if (one.get("对应时段").equals("上午班")) {
                return one;
            }
        }
        return null;
    }

    private static Map<String, String> getMidday(Map<String, String>... arr) {
        for (Map<String, String> one : arr) {
            if (one.get("对应时段").equals("下午班")) {
                return one;
            }
        }
        return null;
    }
}
