package com.xdja.hr.web.controller;

import com.google.common.base.Function;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.xdja.hr.domain.integration.*;
import com.xdja.hr.domain.original.Employee;
import com.xdja.hr.pojo.ValidDayComputeData;
import com.xdja.hr.pojo.ValidFingerprintData;
import com.xdja.hr.pojo.ValidFullData;
import com.xdja.hr.repository.RepositoryService;
import com.xdja.hr.service.compute.ComputeService;
import com.xdja.hr.service.compute.ComputeUtils;
import com.xdja.hr.service.compute.MonthSummaryService;
import com.xdja.hr.service.importData.ImportService;
import com.xdja.hr.utils.DayExcelView;
import com.xdja.hr.utils.JsonResult;
import com.xdja.hr.utils.MonthExcelView;
import org.apache.commons.lang3.time.DateUtils;
import org.apache.poi.openxml4j.exceptions.InvalidFormatException;
import org.joda.time.DateTime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.cglib.beans.BeanMap;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.util.Assert;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.servlet.View;

import javax.annotation.Resource;
import java.io.IOException;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * User: landy
 * Date: 15/8/2
 * Time: 上午11:14
 */
@Controller
@RequestMapping("admin")
public class AdminController {
    final Logger logger = LoggerFactory.getLogger(getClass());
    @Resource
    private RepositoryService repositoryService;
    @Resource
    private ImportService importFingerprint;
    @Resource
    ComputeService computeService;
    @Resource
    MonthSummaryService monthSummaryService;
    @ResponseBody
    @RequestMapping(value = "autoJudge", method = RequestMethod.POST)
    public JsonResult autoJudge(Date month, Integer employeeNo) {
        Assert.notNull(month, "month");
        if (employeeNo != null) {
            logger.info("重新核定员工{}的考勤数据", employeeNo);
            Employee one = new Employee(employeeNo, "", "");
            computeService.recomputeDayByMonth(month, employeeNo);
            monthSummaryService.doSummary(month, one, true);
        } else {
            logger.info("重新核定所有员工{}月的考勤数据", new DateTime(month).toString("yyyy-MM"));
            computeService.recomputeDayByMonth(month);
            monthSummaryService.doSummary(month);
        }
        logger.info("重新核定完成");
        return JsonResult.withData("success");
    }
    @ResponseBody
    @RequestMapping("overview")
    public JsonResult overview(Date month, Pageable page) {
        Page<MonthSummaryRecord> data = repositoryService.getMonthRecordRepository().findAllByCreateMonthOrderByEmployeeAsc(month, page);
        List<MonthSummaryRecord> list = data.getContent();
        List<Map<String, Object>> rst = Lists.newArrayList();
        for (MonthSummaryRecord one : list) {
            Map<String, Object> map = Maps.newHashMap(BeanMap.create(one));
            map.put("employeeNo", one.getEmployee().getEmployeeNo());
            map.put("employeeName", one.getEmployee().getEmployeeName());
            map.put("department", one.getEmployee().getDepartment());
            map.remove("employee");
            rst.add(map);
        }
        return JsonResult.gridData(data.getTotalElements(), rst);
    }

    @RequestMapping("exportMonth")
    public View exportMonth(Date month, Model model) {
        Page<MonthSummaryRecord> data = repositoryService.getMonthRecordRepository().findAllByCreateMonthOrderByEmployeeAsc(month, null);
        List<MonthSummaryRecord> list = data.getContent();
        model.addAttribute("dataList", list);
        model.addAttribute("month", month);
        return new MonthExcelView();
    }

    @RequestMapping("exportDay")
    public View exportDay(Date month, Model model) {
        List<MonthSummaryRecord> data = repositoryService.getMonthRecordRepository().findByCreateMonth(month);
        List<Integer> ids = Lists.transform(data, new Function<MonthSummaryRecord, Integer>() {
            @Override
            public Integer apply(MonthSummaryRecord input) {
                return input.getEmployee().getEmployeeNo();
            }
        });
        model.addAttribute("ids", ids);
        model.addAttribute("month", month);
        return new DayExcelView(repositoryService);
    }
    @ResponseBody
    @RequestMapping("restDay")
    public JsonResult restDay(Date month) {
        DateTime currentMonth = new DateTime(month);
        int maxDayInMonth = currentMonth.dayOfMonth().getMaximumValue();
        Date nextMonth = currentMonth.plusMonths(1).toDate();

        List<RestDayRecord> data = repositoryService.getRestDayRecordRepository().findMonthData(month, nextMonth);
        Set<Long> restDaySet = Sets.newHashSet();
        for (RestDayRecord one : data) {
            restDaySet.add(one.getCreateDate().getTime());
        }
        List<Map<String, Object>> rst = Lists.newArrayList();
        for (int i = 0; i < maxDayInMonth; i++) {
            Map<String, Object> tmp = Maps.newHashMap();
            DateTime day = currentMonth.plusDays(i);
            tmp.put("datetime", day.getMillis());
            tmp.put("isRest", restDaySet.contains(day.getMillis()));
            tmp.put("week", day.getDayOfWeek());

            rst.add(tmp);
        }

        return JsonResult.gridData(maxDayInMonth, rst);
    }
    @ResponseBody
    @RequestMapping(value = "markRestDay", method = RequestMethod.POST)
    public JsonResult markRestDay(Date restDay, Boolean isRest) {
        if (isRest) {
            //休息
            RestDayRecord record = repositoryService.getRestDayRecordRepository().findByCreateDate(restDay);
            if (record == null) {
                record = new RestDayRecord();
                record.setCreateDate(restDay);
                repositoryService.getRestDayRecordRepository().save(record);
            }
        } else {
            //工作
            RestDayRecord record = repositoryService.getRestDayRecordRepository().findByCreateDate(restDay);
            if (record != null) {
                repositoryService.getRestDayRecordRepository().delete(record);
            }
        }
        return JsonResult.withData("success");
    }
    @ResponseBody
    @RequestMapping("loadOneEmployee")
    public JsonResult loadOneEmployee(Date month, Integer employeeNo) {
        Date nextMonth = DateUtils.addMonths(month, 1);
        Employee employee = repositoryService.getEmployeeRepository().findOne(employeeNo);
        List<FingerprintRecord> fingerprintRecords = repositoryService.getFingerprintRecordRepository().findByCreateDateAndEmployee(month, nextMonth, employee);
        List<DayComputeRecord> dayComputeRecords = repositoryService.getDayComputeRecordRepository().findByCreateDateAndEmployee(month, nextMonth, employee);
        List<ValidFingerprintData> validFingerprintDataList = ComputeUtils.combineValidFingerprintData(fingerprintRecords);
        List<ValidDayComputeData> validDayComputeDatas = ComputeUtils.combineValidDayComputeData(dayComputeRecords);
        List<ValidFullData> validFullDatas = ComputeUtils.combineValidFullData(validFingerprintDataList, validDayComputeDatas);

        List<Map<String, Object>> rst = Lists.newArrayList();
        for (ValidFullData one : validFullDatas) {
            Map<String, Object> map = Maps.newHashMap();
            map.put("employeeNo", employee.getEmployeeNo());
            map.put("employeeName", employee.getEmployeeName());
            map.put("department", employee.getDepartment());
            map.put("createDate", one.getFingerprintData().getCreateDate());
            map.put("morningTime", one.getFingerprintData().getMorningTime());
            map.put("middayTime", one.getFingerprintData().getMiddayTime());
            map.put("afternoonTime", one.getFingerprintData().getAfternoonTime());
            map.put("assistSubway", one.getComputeData().getAssistSubway());
            map.put("assistEatMorning", one.getComputeData().getAssistEatMorning());
            map.put("assistEatMidday", one.getComputeData().getAssistEatMidday());
            map.put("assistEatAfternoon", one.getComputeData().getAssistEatAfternoon());
            map.put("workOvertime", one.getComputeData().getWorkOvertime());
            map.put("isWorkDay", one.getComputeData().getIsWorkDay());
            StringBuilder sb = new StringBuilder();
            for (DayStateRecord stateRecord : one.getComputeData().getDayStateRecords()) {
                sb.append("|");
                sb.append(stateRecord.getDayState().toDesc());
            }
            map.put("dayState", sb.substring(1).toString());
            rst.add(map);
        }
        return JsonResult.gridData(rst.size(), rst);
    }
    @ResponseBody
    @RequestMapping(value = "updateData", method = RequestMethod.POST)
    public JsonResult updateOriginalData(@RequestParam Integer year, @RequestParam Integer month, MultipartFile fpOriginalFile, MultipartFile fpManualFile, MultipartFile oaFile) throws IOException, InvalidFormatException {
        logger.info("数据上传,{}年{}月,原始数据为空{}，修订数据为空{}，oa数据为空{}。", year, month, fpOriginalFile.isEmpty(), fpManualFile.isEmpty(), oaFile.isEmpty());
        Date monthDate = new DateTime(year, month, 1, 0, 0).toDate();
        if (!fpOriginalFile.isEmpty()) {
            importFingerprint.importEmployeeData(fpOriginalFile.getInputStream());

            importFingerprint.importOriginalFingerprintData(fpOriginalFile.getInputStream(), monthDate);
        }
        if (!fpManualFile.isEmpty()) {
            importFingerprint.importEmployeeData(fpManualFile.getInputStream());

            importFingerprint.importManualFingerprintData(fpManualFile.getInputStream(), monthDate);
        }
        if (!oaFile.isEmpty()) {
            importFingerprint.importOAData(oaFile.getInputStream(), monthDate);
        }
        computeService.recomputeDayByMonth(monthDate);
        monthSummaryService.doSummary(monthDate);
        return JsonResult.withData("success");
    }
}
